#include "inkey.ch"
#include "gfm.ch"

///////////////
///////////////
//
//  Purpose:
//    Set/Poll GrumpFish Menu internal settings
//
//  Syntax:
//    GFMSet( [ <nSet> ], [ <xValue> ] )
//
//  Formal Arguments: (2)
//    Name        Description
//     
//    nSet        GFMSet position to assign a value to or get a value from
//    xValue      Value to assign to GFMSet[<nSet>]
//
//  Returns:
//    The old value from the GFMSet list, or the entire set array if no
//    value is passed
//
//  Examples:
//    GFMSet( _GFM_STYLE, 'P' ) // Sets the menu style to Pull down
//
//  Files:
//    None.
//
//  Description:
//    Operates very similarly to the Set() function in Clipper 5.0, with a
//    couple of differences.
//
//  Notes:
//    An array may be passed as the <nSet> value that is either a subset or
//    a complete replacement for the GFMSet list.  If the array is a subset,
//    then <xValue> may be used to specify the starting position in GFMSet
//    to assign the variable.  Refer to GFMConfig() for a good example of
//    how this can work.
//
//
//  Category:
//    Menu Handling
//
//  See Also:
//    GFMSetDisplay() GFMStack() GFMSetSys() GFMCurCol() GFMEnvSet()
//
//  Include files:
//    inkey.ch
//    gfm.ch
//
//  Revisions:
//    12/07/90	09:16:55	2.0	Original version
//		06/13/92	13:01:26	2.1	Polling the Notes, Util, and Shade settings
//										no longer clears those variables.
//
///////////////
///////////////
FUNCTION GFMSet( nSet, xValue )
   STATIC aGFMSettings := {;
   {'Menu color',       '+W/B,GR+/N,,,W/B' },;
   {'Frame color',        'BG/B' },;
   {'Background color',   'BG/N' },;
   {'Background character', '' },;
   {'Shadow',               '121' },;
   {'Top line',             0 },;
   {'Style of menu', 'P' },;
   {'Left column', 0 },;
   {'Right column', 79 },;
   {'Indent for menu', 1 }, ;
   {'Notes key', 0 }, ;
   {'Configuration key', K_F10 }, ;
   {'Utility key', K_ALT_F10 }, ;
   {'Bright Background', .t. }, ;
   {'Wrap-around menus', .t. }, ;
   {'Title for main menu', 'Main Menu' },;
   {'Footer for main menu', 'Esc=exit  ' }, ;
	{'Spacing for main menu', 2 }, ;
	{'Depth charting char', '' }, ;
	{'System name', 'Menu Handling' }, ;
	{'Resizable menus', .T. }, ;
   {'Security levels', NIL }, ;
   {'CFG File', '' }, ;
   {'Current Column', 0 }, ;
   {'Current Level', 0 }, ;
   {'Row Offset', 0 }, ;
   {'CFG Key Hit', .f. }, ;
   {'Menu Box Background', '' }, ;
   {'Menu Byte', 0 }, ;
   {'Install toggle', .f. }, ;
   {'Activity toggle', .f. }, ;
   {'DOS Command line', "" }, ;
   {'Menu byte offset', 0 }, ;
   {'Current prompt array', NIL }, ;
   {'Current coordinate array', NIL }, ;
	{'Current stable bits', NIL }, ;
	{'Resizing right now', .F. }, ;
	{'Direction moved', 1 } ;
    }
   LOCAL uReturn := NIL

   IF nSet == NIL .and. xValue == NIL
     uReturn := aClone( aGFMSettings )	// Entire array is being requested
   ELSEIF nSet # NIL
     IF valtype( nSet ) == 'N' .and. nSet > 0 .and. nSet <= _GFM_SET_COUNT
       uReturn := aGFMSettings[ nSet, _GFM_VALUE ]
       IF xValue # NIL	// a Value is being set
         IF nSet == _GFM_BACKCHAR
           // This is the only parameter where GFMSet will accept
           // either a numeric or character value, and either
           // are converted to character
           IF valtype( xValue ) == 'N'
             aGFMSettings[ nSet, _GFM_VALUE ] := chr( xValue )
           ELSE
             aGFMSettings[ nSet, _GFM_VALUE ] := xValue
           ENDIF
         ELSE
           aGFMSettings[ nSet, _GFM_VALUE ] := xValue
         ENDIF
       ENDIF
		 IF PCOUNT() > 1
	       IF nSet == _GFM_NOTEKEY
	         // Notes are being re-defined, first clear current SetKey()
	         IF uReturn # 0
	           Set Key ( uReturn ) to
	         ENDIF
	         Set Key ( xValue ) to GFMNotes
	       ELSEIF nSet == _GFM_UTILKEY
	         // The configuration hot key is being re-defined, clear it first
	         IF uReturn # 0
	           Set Key ( uReturn ) to
	         ENDIF
	         Set Key ( xValue ) to GFMUtils
	       ELSEIF nSet == _GFM_SHADE
	         w_ShadeDir( xValue )
	       ENDIF
		 ENDIF
     ELSEIF valtype( nSet ) == 'A'
       // The entire array is being set, or at least part of it
       uReturn := aGFMSettings
       IF xValue == NIL
         // Defaulting the starting position for over-writing the array
         xValue := 1
       ENDIF
       xValue--
       aEval( nSet, { | aValue, nPos | aGFMSettings[ xValue + nPos, _GFM_VALUE ] ;
             := aValue[ _GFM_VALUE ] } )
     ENDIF
   ENDIF
RETURN uReturn

//////////////////////
//////////////////////
//
//  Purpose:
//    Displays the values currently in the GFM Set
//
//  Syntax:
//    GFMSetDisplay()
//
//  No arguments specified
//
//  Returns:
//    NIL
//
//  Examples:
//    GFMSetDisplay()
//
//  Files:
//    None.
//
//  Description:
//    Displays the current run-time settings in the GFM Set to the screen
//
//  Notes:
//
//
//  Category:
//    Menu Handling
//
//  See Also:
//    GFMSet() GFMStack() GFMSetSys() GFMCurCol() GFMEnvSet()
//
//  Revisions:
//    12/07/90  09:19:22  1.0 Original version
//
//////////////////////
//////////////////////
FUNCTION GFMSetDisplay
   LOCAL aSets := GFMSet()
   w_Push()
   CLS
   aEval( aSets, { | aSet, nPos | QOut( aSet[ _GFM_PROMPT ], ' = ', ;
        aSet[ _GFM_VALUE ] ) }, 1, _GFM_RUN_COUNT  )
   InKey(0)
   w_Pop()
RETURN NIL

//////////////////
//////////////////
//
//  Purpose:
//    Set the configuration options that directly affect Clipper
//
//  Syntax:
//    GFMSetSys()
//
//  No arguments specified
//
//  Returns:
//    NIL
//
//  Examples:
//    GFMSetSys()
//
//  Files:
//    None.
//
//  Description:
//    Sets those variables in the Grumpfish Menu settings that change the
//    way Clipper internals operate
//
//  Notes:
//
//
//  Category:
//    Menu Handling
//
//  See Also:
//    GFMSetDisplay() GFMSet() GFMStack() GFMCurCol() GFMEnvSet()
//
//  Revisions:
//    12/07/90  09:20:42  1.0 Original version
//
//////////////////
//////////////////
FUNCTION GFMSetSys
   SetBlink( !GFMSet( _GFM_BRIGHT ) )
   Set( _SET_WRAP, GFMSet( _GFM_WRAP ) )
RETURN NIL

//////////////////
//////////////////
//
//  Purpose:
//    Menu Column Stack management
//
//  Syntax:
//    GFMCurCol( <nCol> )
//
//  Formal Arguments: (1)
//    Name        Description
//     
//    nCol        Column value to manipulate stack with.
//                A value of 0 clears the stack.
//                A value of -1 pops a value off the stack
//                A value > 0 is added to the stack.
//
//  Returns:
//    The last value on the column stack
//
//  Examples:
//    nCol := GFMCurCol()
//
//  Files:
//    None.
//
//  Description:
//    Used by the menuing routines to figure out the offset for the next
//    menu box.
//
//  Notes:
//
//
//  Category:
//    Menu Handling
//
//  See Also:
//    GFMSetDisplay() GFMStack() GFMSetSys() GFMSet() GFMEnvSet()
//
//
//  Include files:
//    inkey.ch
//    gfm.ch
//
//  Revisions:
//    12/07/90  09:22:11  2.0 Original version
//
//////////////////
//////////////////
FUNCTION GFMCurCol( nCol )
	STATIC aColumns := {}
	IF nCol # NIL
		IF nCol == 0
			aColumns := {}	// Clear the column stack
		ELSEIF nCol > -1
			aAdd( aColumns, nCol )	// Adding a column to the column stack
		ELSE
			aColumns := aSize( aColumns, Len( aColumns ) - 1 ) // Deleting a column from the column stack
		ENDIF
	ENDIF
	IF Len( aColumns ) == 0
		RETURN 0
	ENDIF
RETURN aColumns[ Len( aColumns ) ]

//////////////////
//////////////////
//
//  Purpose:
//    Description array of styles supported by Menu Handling
//
//  Syntax:
//    GFMStyles()
//
//  No arguments specified
//
//  Returns:
//    an array of descriptions for the styles supported
//
//  Examples:
//    nStyle := popchoice( GFMStyles(), 'Pick a menu style' )
//
//  Files:
//    None.
//
//  Description:
//    Encapsulation function to keep track of the menu styles that are
//    supported by Menu Handling.
//
//  Notes:
//
//
//  Category:
//    Menu Handling
//
//  See Also:
//
//
//  Revisions:
//    12/07/90  09:24:57  1.0 Original version
//
//////////////////
//////////////////
FUNCTION GFMStyles
RETURN {  '1-2-3 style menus          ', ;
          'Boxed 1-2-3 style menus    ', ;
          'Cascading menus            ', ;
          'Extra column/line cascading', ;
          'Pull-down menus            ', ;
          'No box pull-down           ', ;
          'Single line pull-down      ' }

#define ENV_CURSOR 1
#define ENV_ROWS   2
#define ENV_SCREEN 3
#define ENV_ROW  4
#define ENV_COL  5
#define ENV_COLOR  6

//////////////////
//////////////////
//
//  Purpose:
//    DOS Environment conservation/restoration
//
//  Syntax:
//    GFMEnvSet( [ <lSave> ] )
//
//  Formal Arguments: (1)
//    Name        Description
//     
//    lSave       Toggle for whether we're conserving or restoring
//
//  Returns:
//    NIL
//
//  Examples:
//    // To save the Environment
//    GFMEnvSet(.t.)
//
//    // To restore the environment
//    GFMEnvSet(.f.)
//
//  Files:
//    None.
//
//  Description:
//    Save various information about the current environment at the time
//    Menu Handling is loaded.
//
//  Notes:
//
//
//  Category:
//    Menu Handling
//
//  See Also:
//
//
//  Include files:
//    inkey.ch
//    gfm.ch
//
//  Revisions:
//    12/07/90  09:28:36  1.0 Original version
//
//////////////////
//////////////////
FUNCTION GFMEnvSet( lSave )
   STATIC aEnv := {}
   IF lSave == NIL
     lSave := .f.
   ENDIF
   IF lSave
     aEnv := { SetCursor(), MaxRow(), SaveScreen( 0, 0, MaxRow(), MaxCol() ), ;
               Row(), Col(), SetColor() }
   ELSEIF Len( aEnv ) > 0
     SetCursor( aEnv[ ENV_CURSOR ] )
     IF MaxRow() # aEnv[ ENV_ROWS ]
       // Clipper SetMode() must have been called by APP
       SetMode( aEnv[ ENV_ROWS ], 80 )
     ENDIF
     RestScreen( 0, 0, maxrow(), maxcol(), aEnv[ ENV_SCREEN ] )
     SetPos( aEnv[ ENV_ROW ], aEnv[ ENV_COL ] )
     SetColor( aEnv[ ENV_COLOR ] )
   ENDIF

RETURN NIL

/////////////////
/////////////////
//
//  Purpose:
//    Get/Set menu position value
//
//  Syntax:
//    GFMStack( <uNewStack>, <nResult> )
//
//  Formal Arguments: (2)
//    Name        Description
//     
//    uNewStack   If character, byte string to set stack to.  If numeric,
//                it varies depending on <nResult>.
//    nResult     If NIL, <uNewStack> is used return the ASCII value of
//                the byte at the <uNewStack> position.  If numeric, it
//                SETs the byte at the <uNewStack> position to chr( nResult )
//
//  Returns:
//    Either the value of the byte at <uNewStack>, or the whole stack
//
//  Examples:
//    nAns1 := GFMStack( GFMSet( _GFM_BYTE ) ) // Get the position
//    GFMStack( repl( chr(1), 10 ) ) // Set the entire stack
//    GFMStack( GFMSet( _GFM_BYTE ), nAns1 ) // Set the position for a menu
//
//  Files:
//    None.
//
//  Description:
//    Byte string for the current prompt positions of the menu levels.
//    Used when position saving is turned on.
//
//  Notes:
//
//
//  Category:
//    Menu Handling
//
//  See Also:
//    GFMSetDisplay() GFMSet() GFMSetSys() GFMCurCol() GFMEnvSet()
//
//
//  Include files:
//    inkey.ch
//    gfm.ch
//
//  Revisions:
//    12/07/90  09:31:18  1.0 Original version
//
/////////////////
/////////////////
FUNCTION GFMStack( uNewStack, nResult )
   STATIC cStack := ''
   IF uNewStack # NIL
      IF ValType( uNewStack ) == 'C'
         // Default the menu position stack to uNewStack
         cStack := uNewStack
      ELSEIF valtype( uNewStack ) == 'N'
         IF nResult # NIL
            // Must be overwriting one of the menu positions
            StrOver( cStack, uNewStack, Chr( nResult ) )
         ELSE
            // Must be asking for the value from the stack
            RETURN Asc( SubStr( cStack, uNewStack, 1 ) )
         ENDIF
      ENDIF
   ENDIF
RETURN cStack

FUNCTION GFMStable( nBit, lToggle )
	LOCAL cStableBits := GFMSet( _GFM_STABLE )
	LOCAL lStable := Bit( cStableBits, nBit )
	IF lToggle # NIL
		Bit( @cStableBits, nBit, lToggle )
		GFMSet( _GFM_STABLE, cStableBits )
	ENDIF
RETURN lStable

FUNCTION GFMDeStabilize( nSize )
	GFMSizeStack( {} )	// Reset the size stack
	DEFAULT nSize TO Len( GFMSet( _GFM_STABLE ) ) * 8 - 1
RETURN GFMSet( _GFM_STABLE, Repl( Chr(0), nSize / 8 + 1 ) )


FUNCTION GFMDepth
	RETURN GFMSet( _GFM_DEPTHCHARGE )

FUNCTION GFMMainDepth
	LOCAL cDepth := ""
	IF ! GFMSet( _GFM_STYLE ) $ 'PNS'
		cDepth := GFMDepth()
	ENDIF
RETURN cDepth
