#include "gfm.ch"

***************
***************
**
**  Purpose:  Main menu routine for the GrumpFish Menu program
**
**   Syntax:  GFMain( <aPrompts>, <aMessages>, <cStyle>, <cTitle>, <cFooter>,
**                    <nOption>, <aToggles>, <lTagging>, <lReBox>, 
**                    <lCanEscape> )
**
**   Parameters:
**
**      Name        Description
**       
**      aPrompts    Array of prompts for the menu
**      aMessages   Array of messages to display in the menu
**      cStyle      Style of menu. First character of the following phrases
**                  is examined to determine the type:
**                  - 1-2-3 style menus
**                  - Boxed 1-2-3 style menus (with titles and footers)
**                  - Cascading menus (with titles and footers)
**                  - Pull-down menus
**                  - No box pull-down menus (no box on main)
**                  - Single pull-down menu style (just bottom line on main)
**      cTitle      Title for box if Boxed 1-2-3 or Cascading
**      cFooter     Footer for box if Boxed 1-2-3 or Cascading
**      nOption     Default menu option
**      aToggles    Array of logical toggles, or .t.
**      lTagging    Toggle for whether or not tagging is allowed
**      cShadow     Direction for casting shadows
**      lReBox      Toggle for whether or not to draw the box
**      cFrame      Frame to use for box characters
**		  lCanEscape  .T. if the user can escape from the main menu
**
**  Returns:  The menu option selected
**
**  Example:  
**
** Overview:  Menu handling routine for all main menu styles.
**
**    Notes:  This is where support for new menu styles would be entered.
**
** Category:  Menu Handling
**
** See Also:  GFMenu()
**
***************
***************
FUNCTION GFMain( aPrompts, aMessages, cStyle, cTitle, cFooter, nOption, ;
                  aToggles, lTagging, lReBox, nTop, nLeft, nBottom, nRight, ;
						lCanEscape )
   STATIC cOldTitle := "", cOldFooter := ""
   LOCAL nResult, nPrompt
   LOCAL cFrame    := GFMFrame( cStyle, 0 )
   LOCAL nPrompts  := Len( aPrompts )
   LOCAL cShadow   := w_ShadeDir()
   
   IF aPrompts == NIL .OR. GFMExit() THEN RETURN 0
   
   IF aMessages == NIL	// All of these have defaults.
      aMessages := Array( Len( aPrompts ) )
      aFill( aMessages, '' )
   ENDIF
   
   DEFAULT cStyle TO GFMSet( _GFM_STYLE ), ;
            cTitle TO IIF( cStyle # 'N', "Main Menu", "" ), ;
            cFooter TO '', nOption TO 1, lTagging TO .F., lReBox TO .T., ;
				lCanEscape TO .T.
   
   // Setting the coordinates

	DEFAULT nTop TO GFMSet( _GFM_TOPLINE ), ;
				nLeft TO GFMSet( _GFM_LEFTCOL )

   DO CASE
   CASE cStyle $ 'PNS'  // One of the pull-downs
      DEFAULT nRight TO GFMSet( _GFM_RIGHTCOL )
      IF cStyle == 'N' // No-box pull-down menu
         DEFAULT nBottom TO nTop
		ELSE
			DEFAULT nBottom TO nTop + 2
      ENDIF
   CASE cStyle == 'B'  // Boxed 1-2-3 menu
		DEFAULT nRight TO GFMSet( _GFM_RIGHTCOL ), ;
      			nBottom TO nTop + 3
   CASE cStyle == '1'
      DEFAULT nRight TO GFMSet( _GFM_RIGHTCOL ), ;
      			nBottom TO nTop + 1
      cTitle  := ''
      cFooter := ''
      w_ShadeDir('0') // No shadow on 1-2-3
   CASE cStyle == 'C'  // Cascading
		DEFAULT nBottom TO Min( MaxRow() - 2, nTop + nPrompts + 1 ), ;
					nRight TO Min( nLeft + w_TextDim( Space( aMaxStrLen( aPrompts ) ),;
								 		cTitle, cFooter ), MaxCol() )
   CASE cStyle == 'E'  // Extra line/col Cascading
      DEFAULT nBottom TO Min( MaxRow() - 2, nTop + nPrompts + 3 ), ;
					nRight TO Min( nLeft + w_TextDim( Space( aMaxStrLen( aPrompts ) ), ;
										cTitle, cFooter ) + 3, MaxCol() )
   ENDCASE
   
   IF ( !lReBox ) .AND. ( ( !cOldTitle == cTitle ) .OR. ;
		( !cOldFooter == cFooter ) )
      w_Pop()
      lReBox := TRUE
   ENDIF
   cOldTitle := cTitle
   cOldFooter := cFooter

   IF lReBox  // Re-draw the screen
      RestScreen( 0, 0, MaxRow(), MaxCol(), GFMSet( _GFM_MENUBOX ) )
   ENDIF
   
   IF cStyle $ 'NPS'
      IF lReBox  // Re-draw the screen
         IF GFMSet( _GFM_ACTIVE )
            SetColor( GFMSet( _GFM_FRAMECOLOR ) )
         ENDIF
         w_Shade( nTop, nLeft, nBottom, nRight, cFrame, "", "" )
      ENDIF
      
      IF GFMSet( _GFM_ACTIVE )
         SetColor( GFMSet( _GFM_MENUCOLOR ) )
      ENDIF
      
      IF cStyle == 'S'  // Re-positioning the message line
         SET MESSAGE TO ( nTop + 1 )
      ENDIF
      

      nResult := nOption	// Defaulting the menu position for the PROMPT
      
      nResult := m_123( aPrompts, aMessages, nTop + IIF(cStyle=='P',1,0), ;
								nLeft + 1, nRight - 1, nResult, .T., ;
								GFMSet( _GFM_SPACING ), aToggles )
      
      SET MESSAGE TO ( MaxRow() ) // Reset message line
   
   ELSE
   
     nResult := GFMenu( aPrompts, aMessages, cStyle, cTitle, cFooter, ;
                       nTop, nLeft, nBottom, nRight, nOption, aToggles, ;
                       lTagging, lReBox, cFrame )
   ENDIF

   IF LastKey() == 27 .AND. lCanEscape
      w_Pop()
      IF GFMSet( _GFM_ACTIVE )
         SetColor( GFMSet( _GFM_MENUCOLOR ) )
      ENDIF
   ENDIF
   
   w_ShadeDir( cShadow )	// Re-set the shadow direction as needs be
RETURN nResult


***************
***************
**
**  Purpose:  Generic menuing routine for 4 different types of menus
**
**   Syntax:  GFMenu( <aPrompts>, <aMessages> [, <cStyle> [, <cTitle> 
**                    [, <cFooter> [, <nTop> [, <nLeft> [, <nBottom> [, <nRight> 
**                    [, <nOption> [, <aToggles> [, <lTagging>
**                    [, <lReBox> [, <cFrame> ] ] ] ] ] ] ] ] ] ] ] ] ] )
**
**   Parameters:
**
**      Name        Description
**       
**      aPrompts    Array of prompts for the menu
**      aMessages   Array of messages to display in the menu
**      cStyle      Style of menu. First character of the following phrases
**                  is examined to determine the type:
**                  - 1-2-3 style menus
**                  - Boxed 1-2-3 style menus
**                  - Cascading menus (with titles and footers)
**                  - Pull-down menus
**                  - No box pull-down menus (no box on first level, that is)
**      cTitle      Title for box IF Boxed 1-2-3 or Cascading
**      cFooter     Footer for box IF Boxed 1-2-3 or Cascading
**      nTop        Top row of box
**      nLeft       Left column of box
**      nBottom     Bottom row of box
**      nRight      Right column of box
**      nOption     Default menu option
**      aToggles    Array of logical toggles, or .t.
**      lTagging    Toggle for whether or not tagging is allowed
**      lReBox      Toggle for whether or not to draw the box
**      cFrame      Frame to use for box characters
**
**  Returns:  The number of the PROMPT chosen, or 0 IF [Esc] was pressed.
**
**  Example:  GFMenu( prompts, messages, '1' )  // For 1-2-3 style menus
**
**    Files:  None.
**
** Overview:  General-purpose menu handling routine for many different kinds of
**            menu styles.
**
**    Notes:  The long list of parameters is necessary, given the control that
**            may be performed.
**
** Category:  Menu Handling
**
** See Also:  
**
***************
***************
FUNCTION GFMenu( aPrompts, aMessages, cStyle, cTitle, cFooter, nTop, ;
                 nLeft, nBottom, nRight, nOption, aToggles, lTagging, ;
                 lReBox, cFrame )
   LOCAL nResult, nWidth, nMessRow
	LOCAL cShadow := w_ShadeDir()
   
   IF aPrompts == NIL .OR. GFMExit() THEN RETURN 0
   
   DEFAULT aMessages TO "", ;
         cStyle TO IIF(GFMSet(_GFM_ACTIVE),GFMSet(_GFM_STYLE),'C')

   cStyle := Upper( SubStr( cStyle, 1, 1 ) )
   IF ! cStyle $ '1BCEPNS'
      cStyle := 'C'  // Defaulting the style code
   ENDIF
   
   IF cStyle $ 'PN1S'  // Pull-downs, or 1-2-3 w/o box (no titles, footers)
     cTitle := cFooter := ''
   ENDIF
   
   // Calculating number of lines for menu
   nResult := len( aPrompts ) + IIF( cStyle == 'E', 2, 0 )
   
   IF SubStr( cStyle, 1, 1 ) $ '1B'
     nWidth := 78
   ELSE
     nWidth := w_TextDim( aMaxStrLen( aPrompts ), cTitle, cFooter ) + ;
               iif( cStyle == 'E', 2, 0 ) // Add 2 for extra columns
   ENDIF
   
   DEFAULT nTop TO Max( ( MaxRow() - nResult ) / 2 - 1, 0 ), ;
         nLeft TO Max( 2, ( MaxCol() - nWidth ) / 2 - 1 ), ;
         nBottom TO Min( MaxRow() - 2, nTop + nResult + 1 ), nOption TO 1, ;
         aToggles TO .F., lTagging TO .F., lReBox TO .T., cFrame TO w_Frame()
         
   IF nRight == NIL .OR. cStyle $ 'CE'
     nRight := Min( nLeft + nWidth + 1, MaxCol() )	// re-size right column
   ENDIF
   
   nRight := Min( nRight, MaxCol() )					// Make sure box fits
   
   IF cStyle == '1'  // 1-2-3 style
     w_ShadeDir('0') // Turn off shadows
     cFrame  := space( 9 )
   ENDIF
   
   IF lReBox
   
     // Draw and shade the box
     IF GFMSet( _GFM_ACTIVE )
       SetColor( GFMSet( _GFM_FRAMECOLOR ) )
     ENDIF
   
     w_Shade( nTop, nLeft, nBottom, nRight, cFrame, cTitle, cFooter )
   
     IF GFMSet( _GFM_ACTIVE )
       SetColor( GFMSet( _GFM_MENUCOLOR ) )
     ENDIF
   
   ENDIF

   IF cStyle $ 'CPENS'  // Cascading, pull-down, or Extra line/col
   
     	IF cStyle == 'E'  // Extra line/col
			nResult := gchoice( aPrompts, aMessages, nTop + 2, nLeft + 2, nBottom - 2, ;
										nRight - 2, nOption, aToggles, lTagging )
     	ELSE
			nResult := gchoice( aPrompts, aMessages, nTop + 1, nLeft + 1, nBottom - 1, ;
										nRight - 1, nOption, aToggles, lTagging )
     	ENDIF
   
   ELSEIF cStyle $ '1B'  // 1-2-3 or Boxed 1-2-3
   
     IF cStyle == 'B'
   
       nTop++
       nLeft++
       nBottom--
       nRight--
   
     ENDIF
   
	  nMessRow := Set( _SET_MESSAGE, nTop + 1 )
     nResult := m_123( aPrompts, aMessages, nTop, nLeft, nRight, nOption, .t., ;
	  							GFMSet( _GFM_SPACING ), aToggles )
	  Set( _SET_MESSAGE, nMessRow )
         
   ENDIF
   
   
	IF nResult > 0
		IF GFMSet( _GFM_BYTE ) > 0
		  // Assume that GFMstack is declared, and save menu option to stack
		  GFMStack( GFMSet( _GFM_BYTE ), nResult )
		ENDIF
	ELSE
		@ MaxRow(), 0
	ENDIF
   
	w_ShadeDir( cShadow )
RETURN nResult

