#include "setcurs.ch"
#include "inkey.ch"

// Use the /n switch when compiling
//           SF0 ^F1 a-q ^F4 ^F5  F2 SF3 AF2 ^CR a-r
static aExitKeys := { K_SH_F10, K_CTRL_F1, K_ALT_Q, K_CTRL_F4, K_CTRL_F5, K_F2, ;
                      K_SH_F3, K_ALT_F2, K_CTRL_RET, K_ALT_R }
static cTarget   := ''
static cReplace  := ''
static nExitRow  := 0, nExitCol := 0, nExitMarks := 0, nEditMargin := 75

/////////////
/////////////
//
//  Purpose:  Make notes for any part of a system
//
//  Syntax:
//    not8( [ <cAction>, [ <cndIdentity> ], [ <cTxt> ], [ <cTitle> ] )
//
//  Formal Arguments:
//
//      Name        Description
//       
//      cAction      Action code. First letter of action words:
//                  - Add
//                  - Delete
//                  - Edit
//                  - Find
//                  - Get
//                  - View
//      uIdentity   Unique identifier for this note.  If no identifier
//                  is specified, the identifier is built by adding
//                  the alias and &( indexkey(1) ).
//      cText       Text to add/replace note with
//      cTitle      Title for note
//
//  Returns:  .t. if a memo field was accessed, or memo field if 'G' is
//            passed.
//
//  Example:  not8( 'E' )  && To edit a memo for this record
//
//    Files:  None.
//
// Overview:  General purpose notes system that can be used for on-line
//            help, and more.
//
//    Notes:  
//
// Category:  Help System
//
// See Also:  
//
/////////////
/////////////
function not8( cAction, uIdentity, cText, cTitle )
field key, notes, coords in _notes
local lOK, dbStack := pushdbf(), nR1, nR2, nC1, nC2, nRow, nCol

if cAction == NIL
  cAction := 'E'
else
  cAction := upper( cAction )
endif

if uIdentity == NIL
  uIdentity := substr( alias() + '          ', 1, 10 )
  if indexord() == 0
    uIdentity += str( recno() )
  else
    uIdentity += totext( &( indexkey(0) ) )
  endif
else
  uIdentity  := totext( uIdentity )
endif

if cTitle == NIL .or. empty( cTitle )
  cTitle := 'Notes'
endif

if select('_notes') == 0 
  // Notes was not found

  if ! file( 'notes.dbf' )  
    // Could not find the notes file, so create it!
    dbCreate( 'notes.dbf', { { 'KEY', 'C', 70, 0 }, ;
                           { 'NOTES', 'M', 10, 0 }, ;
                           { 'COORDS', 'C', 4, 0 }, ;
                           { 'TITLE', 'C', 30, 0 } } )
  endif

  select( freearea( 150 ) )
  use notes alias _notes
  if ! file( 'notes.ntx' )
    index on key to notes
  else
    set index to notes
  endif

endif

select _notes
set order to 1
seek uIdentity

if eof()

  if cAction $ 'EA'  // Edit, so add may be done

    // Look for an empty note record
    seek ' '
    if eof()
      lOK := addrec()
    else
      lOK := crlock()
    endif

    if ! lOK

      popdbf( @dbStack )
      return .f.

    endif

    _notes->key := uIdentity
    _notes->title := cTitle
    _notes->coords := chr( 4 ) + chr( 3 ) + chr( maxrow() - 4 ) + ;
                      chr( maxcol() - 3 )

    // Flush as many buffers as possible
    unlock
    skip 0
    commit

  else

    if cAction # 'F'  // Not find - display message
      w_error( setcolor(), 30, "Key:" + key, "No notes found", ;
              "Press any key to return" )
    endif

    popdbf( @dbStack )
    return iif( cAction == 'G', '', .f. )

  endif

endif

nRow := row()
nCol := col()

if cAction == 'D'
  // Delete the notes record
  if crlock()
    // Clear out the key and notes
    _notes->key := ''
    _notes->notes := ''
    _notes->title := ''
    unlock
    skip 0
    commit
  endif
elseif cAction == 'G'  // Notes are being fetched
  cText := _notes->notes
else

  nR1 := asc( _notes->coords )              // Top Row
  nC1 := asc( substr( _notes->coords, 2 ) ) // Left column
  nR2 := asc( substr( _notes->coords, 3 ) ) // Bottom row
  nC2 := asc( substr( _notes->coords, 4 ) ) // Right column

  if cAction == 'V'
    TExtEditor( _notes->notes, nR1, nC1, nR2, nC2, frame(3), ;
            "Viewing " + _notes->title, "Press [Esc] to exit", .f. )
  elseif cAction $ 'EA'
    if crlock()
      if cText # NIL // Text was passed - add it to existing notes
        if cAction == 'A'
          _notes->notes := _notes->notes + cText
        else  // overwrite the notes with the passed text
          _notes->notes := cText
        endif
      else
        cText := TextEditor( _notes->notes, @nR1, @nC1, @nR2, @nC2, ;
                         frame(3), "Editing " + trim( _notes->title ), ;
                         "Press [F1] for key list" )
        if lastkey() # 27
          if empty( cText ) .and. ask4yn( 'Text was removed.  Delete note?' )
            WipeRec()
          else
            if ! _notes->notes == cText
              _notes->notes := cText
            endif
            // Edit wasn't aborted
            _notes->coords := chr( nR1 ) + chr( nC1 ) +;
                              chr( nR2 ) + chr( nC2 )
          endif
        endif
      endif
      unlock
      skip 0
      commit
    endif
  endif
endif

popdbf( @dbStack )
// Reposition the cursor
if cAction $ 'AVE'
  SetPos( nRow, nCol )
endif
return iif( cAction=='G', cText, .t. )

///////////////
///////////////
//
//  Purpose:  Allow better editing control on memos
//
//   Syntax:  TextEditor( @<cText> [, <nR1> [, <nC1> [, <nR2> [, <nC2> [, <cFrame>
//                    [, <cTitle> [, <cFooter> [, <lEdit>
//                    [, <nMargin> ] ] ] ] ] ] ] ] ] )
//
//   Parameters:
//
//      Name        Description
//       
//      cText       Text to edit, if passed by reference, will be modified
//      nR1         Top row of window
//      nC1         Left column of window
//      nR2         Left row of window
//      nC2         Right column of window
//      cFrame      Frame for box
//      cTitle      Title for box
//      cFooter     Footer for box
//      lEdit       Whether or not editing is allowed
//      nMargin     Margin for memo
//
//  Returns:  the modified memo
//
//  Example:  replace notes with lEditor( notes )
//
//    Files:  None.
//
// Overview:  
//
//    Notes:  
//
// Category:  Help System
//
// See Also:  
//
///////////////
///////////////
function TextEditor( cText, nR1, nC1, nR2, nC2, cFrame, cTitle, cFooter, lEdit, nMargin )
local nQuitKey := 1, aExitKeys, cTemp, nLines, nCursor, bKey23, bKey127, bKey28

if cText == NIL
  cText := ''
endif

// Rough guess at the number of lines
nLines := max( 5, len( cText ) / 75 )
if nR1 == NIL
  nR1 := ( MaxRow() - nLines ) / 2
endif
if nC1 == NIL
  nC1 := ( MaxCol() - 75 ) / 2 - 1
endif
if nR2 == NIL
  nR2 := min( nR1 + nLines, MaxRow() - 2 )
endif
if nC2 == NIL
  nC2 := min( MaxCol(), nC1 + 75 )
endif
if cFrame == NIL
  cFrame := frame( 3 )
endif
if cTitle == NIL
  cTitle := 'Text Editing'
endif
if cFooter == NIL
  cFooter := "^F1=DOS F1=Help Esc=abort"
endif
if lEdit == NIL
  lEdit := .t.
endif
if nMargin == NIL
  nMargin := nC2 - nC1 - 1
endif

nEditMargin := nMargin
w_shade( nR1, nC1, nR2, nC2, cFrame, cTitle, cFooter )

cText := MemoEdit( cText, nR1 + 1, nC1 + 1, nR2 - 1, nC2 - 1, ;
                  lEdit, "_medtudf_", nMargin, 4, nExitRow, ;
                  nExitCol, min( nR2 - nR1, nExitRow ) )
/*
// Edit the memo
nCursor := SetCursor( SC_NORMAL )
do while nQuitKey # 0
  cTemp := ''

  // Setting up the help key
  bKey28 := SetKey( K_F1, { | cProc, cLine, cVar | ;
                    _mdthelp_( cProc, cLine, cVar ) } )
  // Setting the key up for ^Y, but saving what it was
  bKey23 := SetKey( K_CTRL_Y, { | cProc, cLine, cVar | ;
                    CtrlY( cProc, cLine, cVar ) } )
  // ^Backspace compatibility
  bKey127 := SetKey( 127, { | cProc, cLine, cVar | ;
                    CtrlT( cProc, cLine, cVar ) } )

  cText := MemoEdit( cText, nR1 + 1, nC1 + 1, nR2 - 1, nC2 - 1, ;
                    lEdit, "_medtudf_", nMargin, 4, nExitRow, ;
                    nExitCol, min( nR2 - nR1, nExitRow ) )

  SetKey( K_CTRL_Y, bKey23 )
  SetKey( 127, bKey127 )
  SetKey( K_F1, bKey28 )

  nQuitKey := KeyHandler( lastkey(), @cText, @nR1, @nC1, @nR2, @nC2, cFrame, ;
              cTitle, cFooter )

enddo
/**/

SetCursor( nCursor )
w_Pop()

cText := strtran( cText, '' )
nExitMarks := 0
return cText

static function CtrlY
keyboard chr( 25 )
return NIL

static function CtrlT
keyboard chr( 20 )
return NIL


///////////////////
///////////////////
//
//     Purpose:
//
//
//      Syntax:
//      KeyHandler( <nQuitKey>, @<cText>, @<nR1>, @<nC1>, @<nR2>, @<nC2>, ;
//                  <cFrame>, <cTitle>, <cFooter> )
//
//   Arguments:
//      Name        Description
//       
//      nQuitKey    last key that was pressed
//      cText       text that is being edited
//      nR1         Top row of editing box
//      nC1         Left column of editing box
//      nR2         Bottom row of editing box
//      nC2         Right column of editing box
//      cFrame      Frame for box
//      cTitle      Title of box
//      cFooter     Footer for box
//
//     Returns:
//      the result key
//
//     Example:
//
//
//       Files:
//      None.
//
// Description:
//
//
//       Notes:
//
//
//    Category:
//      Help System
//
//    See Also:
//      editor()
//
///////////////////
///////////////////
static function KeyHandler( nQuitKey, cText, nR1, nC1, nR2, nC2, cFrame, cTitle, cFooter )
local nStart, nEnd, nAnchor, cActKey, cTemp
local aBlocks := { 'Delete marked block', 'Move marked block', ;
                   'Copy marked block', 'Remove marks' }
local aCases := { 'UPPERCASE BLOCK  ', 'lowercase block  ', ;
                  'Proper Case Block' }
local cLineFeed := chr(10)
local nIter

if nQuitKey == NIL
  nQuitKey := lastkey()
endif

if nQuitKey == K_CTRL_F4 .or. nQuitKey == K_SH_F3 // Ctrl-F4 or Shift-F3
  if nExitMarks < 2
    w_error( "warning", 0, ;
              "Use Alt-F4 to mark the beginning and end of a block", ;
              "No aBlocks are marked!", "Press a key" )
  else
    nStart := at( '', cText )
    nEnd := fat( '', cText, nStart )
    if nQuitKey == K_CTRL_F4
      if ( cActKey := popchoice( aBlocks, 'Block operations', 'Esc=abort' ) ) > 0
        cActKey := upper( substr( aBlocks[ cActKey ], 1, 1 ) )
      else
        cActKey := NIL
      endif
    else
      if ( cActKey := popchoice( aCases, 'Switch case', 'Esc=abort' ) ) > 0
        cActKey := upper( substr( aCases[ cActKey ], 1, 1 ) )
      else
        cActKey := NIL
      endif
    endif

    if cActKey # NIL .and. cActKey $ 'MCULP'
      // First, copy the block to a buffer
      cTemp := substr( cText, nStart + 1, nEnd - nStart - 1 )

      if cActKey $ 'MC'
        // Find out the destination
        nAnchor := atnext( cLineFeed, cText, nExitRow - 1 ) + nExitCol

        // Insert the block
        cText := substr( cText, 1, nAnchor ) + cTemp + ;
                  substr( cText, nAnchor + 1 )

        if cActKey == 'M'  // Switch to "D"elete mode
          nStart := at( '', cText )
          nEnd := fat( '', cText, nStart )
          cActKey := 'D'
        endif
      else
        if cActKey == 'L'
          cTemp := lower( cTemp )
        elseif cActKey == 'U'
          cTemp := upper( cTemp )
        elseif cActKey == 'P'
          cTemp := capfirst( cTemp )
        endif
        cText := substr( cText, 1, nStart ) + cTemp + substr( cText, nEnd )
      endif
    endif

    if cActKey == 'D'
      cText := substr( cText, 1, nStart - 1 ) + ;
               substr( cText, nEnd + 1 )
      nExitMarks := 0
    elseif cActKey == 'R'
      nExitMarks := 0
      cText := strtran( cText, '' )
      nExitCol -= 2
    endif

  endif

elseif nQuitKey == K_CTRL_F5

  w_error( setcolor(), 0, "Not available in this version" )
  ****** outreply() - ADD FILE OUTPUT CAPABILITY!!!!!!!!!!!!!!!

elseif nQuitKey == K_F2 .or. nQuitKey == K_ALT_F2 && F2 .or. Alt-F2

  cTarget := trim( ask4string( "String to find:", 80, cTarget ) )
  if lastkey() # 27
    nStart := atnext( cLineFeed, cText, nExitRow - 1 ) + nExitCol + ;
               iif( nQuitKey == -31, 0, 1 )
    if ( nEnd := fat( cTarget, cText, nStart ) ) == 0
      w_error( setcolor(), 5, "'"+cTarget+"'", "Could not find text!", ;
                "Press a key" )
      return nQuitKey
    else
      nExitRow := chrcount( cLineFeed, substr( cText, 1, nEnd ) ) + 1
      nExitCol := ( nEnd - atnext( cLineFeed, cText, nExitRow - 1 ) ) - 1
    endif

    if nQuitKey = -31  && Alt-F2 - search and replace
      cReplace := trim( ask4string( "Replacement string:", 80, cReplace ) )

      if lastkey() # 27
        nIter := ask4number( "# of occurences to replace:", 4, 0, 9999 )
        if lastkey() # 27
          cText := strtran( cText, cTarget, cReplace, nEnd - 1, nIter )
        endif

      endif

    endif

  endif

elseif nQuitKey == K_ALT_R // Scroll-lock on, or Alt-R pressed

  w_unshade()
  ReSizeBox( @nR1, @nC1, @nR2, @nC2 )
  nEditMargin := nC2 - nC1 - 1
  keytoggle( 'S', 'N' )  // Turn-off scroll-lock
  if lastkey() # 27  // Re-draw the whole box
    w_Pop()
    w_shade( nR1, nC1, nR2, nC2, cFrame, cTitle, cFooter )
  else
    w_shade()
  endif

else

  nQuitKey := 0

endif

return nQuitKey

******************
******************
**
**     Purpose:
**      Exception handler for editor calling memoedit()
**
**      Syntax:
**      _medtudf_( <nMode>, <nLine>, <nColumn> )
**
**   Arguments:
**      Name        Description
**       
**      nMode       MemoEdit() mode
**      nLine       Line currently at in memoedit
**      nColumn     Column current at in memoedit
**
**     Returns:
**      an action number to memoedit
**
**    Examples:
**      see editor()
**
**       Files:
**      None.
**
** Description:
**
**
**       Notes:
**
**
**    Category:
**      Help System
**
**    See Also:
**      editor()
**
******************
******************
function _medtudf_( nMode, nLine, nColumn )
local nKey := lastkey(), nReturn := 0

if nMode == 0 .or. nMode == 3
  if w_cols() > 33
    @ w_top() - 1, w_right() - 15 say iif(Set( _SET_INSERT )," I "," T ")
    ?? "R:"+str(nLine,4),"C:"+str(nColumn,3)
  endif

else

  if nKey == K_F7
    nReturn  := 23
  elseif ascan( aExitKeys, nKey ) # 0
    nExitRow := nLine
    nExitCol := nColumn
    nReturn  := 23
  elseif nKey == K_ALT_F4
    if nExitMarks == 2
      w_error( "warning", 0, "You may only mark one block at a time", ;
                "A block is already marked!", "Press a key" )
      nReturn := 32
    else 
      keyboard ''
      nExitMarks++
    endif
  elseif nKey == K_F1
    _mdthelp_()
  elseif nKey == K_INS
    @ w_top() - 1, w_right() - 15 say iif(ReadInsert()," I "," T ")
  elseif nKey == K_ESC
    nReturn := K_ESC
  endif

endif

return nReturn

******************
******************
**
**  Purpose:  Provide key list for Editor()
**
**   Syntax:  _mdthelp_()
**
**  No parameters specified
**
**  Returns:  .t.
**
**  Example:  
**
**    Files:  None.
**
** Overview:  
**
**    Notes:  
**
** Category:  Help System
**
** See Also:  
**
******************
******************
function _mdthelp_
local nR1 := MaxRow() /2 - 6
local nR2 := nR1 + 12
local nC1 := MaxCol() /2 - 35
local nC2 := nC1 + 70

w_shade( nR1, nC1, nR2, nC2, Frame(3), "Editing keys", ;
        "Press the key to use" )

@ w_top(  0 ), w_left( 0 ) say '[Alt]="a", [Shift]="s", [Ctrl]="^"'
@ w_top(  2 ), w_left( 1 ) say '[F1]   This screen'
@ w_top(  3 ), w_left( 1 ) say '[F2]   Search for text'
@ w_top(  4 ), w_left( 1 ) say '[aF2]  Search and replace text'
@ w_top(  5 ), w_left( 1 ) say '[aF4]  Mark block beginning/end'
@ w_top(  6 ), w_left( 1 ) say '[^F4]  Block operations menu - Copy, Move, Delete'
@ w_top(  7 ), w_left( 1 ) say '[sF3]  Case switching on marked block'
@ w_top(  8 ), w_left( 1 ) say '[F7]   Exit and save'
@ w_top(  9 ), w_left( 1 ) say '[Esc]  Exit without save'
@ w_top( 10 ), w_left( 1 ) say '[aR]   Resize editing window'
wait4key()
if nextkey() == K_ESC
  inkey()
endif
w_Pop()

return .t.

STATIC FUNCTION Wait4Key()
	DO WHILE NextKey() == 0 ; ENDDO
RETURN NextKey()

