#include "gfm.ch"
// The level to get back to where the key was pressed is 3
#define CALL_BACK         3
                          
#define SCREEN_LEVEL      1
#define TABLE_LEVEL       2
#define FIELD_LEVEL       3
#define SYSTEM_LEVEL      4
#define MAINMENU_LEVEL    5
#define SUBMENU_LEVEL     6

///////////////////
///////////////////
//
//   Purpose:
//    Provide look-up key for notes based on system status
//
//   Syntax:
//    GFMHelpKey()
//
//   No arguments specified
//
//   Returns:
//    The look-up key to use in the notes file
//
//   Example:
//    cKey := GFMHelpKey()
//
//   Files:
//    None.
//
//   Description:
//    Potentially interactive routine to determine how to look up a note
//    for a GrumpFish Menu system
//
//   Notes:
//    Makes some assumptions (albeit safe ones) about the menuing facility
//    of GrumpMenu for the menu-specific help.
//
//Category:
//    Help System
//
//   See Also:
//    GFMNotes()
//
///////////////////
///////////////////
FUNCTION GFMHelpKey
	LOCAL cProcedure := ProcName( CALL_BACK )
	LOCAL nLine := ProcLine( CALL_BACK )
	LOCAL oGet := GetActive()
	LOCAL cKey
	LOCAL cVariable
	LOCAL aChoices := {}
	LOCAL aCodes   := {}
	LOCAL nR1 := row(), nC1 := col(), nC2 := w_right()
	LOCAL cPrompt
	LOCAL aPrompts
	LOCAL nChoice := 0
	LOCAL cTitle
	LOCAL xReturn
	
	DO CASE
	CASE cProcedure $ 'ACHOICEM_123' .and. GFMSet( _GFM_LEVEL ) # 0
	  // We're in the main menu level or a sub-menu
	  cPrompt := GFMGetPrompt( nR1, nC1, nC2 )
	  nChoice := SUBMENU_LEVEL
	CASE GFMSet( _GFM_LEVEL ) == 0 .AND. Full( GFMSet( _GFM_PROMPT_ARRAY ) )
	  cPrompt := GFMGetPrompt( nR1, nC1, nC2 )
	  aAdd( aChoices, 'Entire System' )
	  aAdd( aCodes, SYSTEM_LEVEL )
	  aAdd( aChoices, cPrompt )
	  aAdd( aCodes, MAINMENU_LEVEL )
	OTHE
	  aAdd( aChoices, 'Screen displayed' )
	  aAdd( aCodes, SCREEN_LEVEL )
	  IF ! empty( alias() )
	    aAdd( aChoices, 'Table in use' )
	    aAdd( aCodes, TABLE_LEVEL )
	  ENDIF
	  IF oGet # NIL
	    // We're on a get screen
	    cVariable := oGet:name
	    aInsert( aChoices, cVariable, 1 )
		 aInsert( aCodes, FIELD_LEVEL, 1 )
	  ENDIF
	ENDCASE
	
	IF len( aChoices ) > 0
	  // Have multiple choices GAG!! ACK!!!
	  IF ( nChoice := PopChoice( aChoices, 'Get Help on:', 'Esc=abort' ) ) # 0
	    nChoice := aCodes[ nChoice ]
	  ENDIF
	ENDIF
	
	IF nChoice == FIELD_LEVEL
	  cKey := alias()+'->'+cVariable
	  cTitle := "Field Help"
	ELSEIF nChoice == SCREEN_LEVEL
	  cKey := cProcedure
	  cTitle := "Screen Help"
	ELSEIF nChoice == TABLE_LEVEL
	  cKey := 'ALIAS=' + alias()
	  cTitle := "Table Help"
	ELSEIF nChoice == SYSTEM_LEVEL
	  cTitle := cKey := 'System Overview'
	ELSEIF nChoice == SUBMENU_LEVEL
	  cKey := 'GFM:' + l2bin( GFMSet( _GFM_BYTE ) ) + cPrompt
	  cTitle := cPrompt
	ELSEIF nChoice == MAINMENU_LEVEL
	  cKey := 'GFM:' + l2bin( 0 ) + cPrompt
	  cTitle := cPrompt
	ENDIF
	
	IF cKey # NIL THEN xReturn := { cKey, cTitle }
RETURN xReturn


/////////////////////
/////////////////////
//
//   Purpose:
//    Read the screen, and find the prompt
//
//   Syntax:
//    GFMGetPrompt( <nR1>, <nC1>, <nC2> )
//
//   Formal Arguments:
//    Name        Description
//     
//    nR1         Row to read prompt from
//    nC1         Column to begin reading prompt at
//    nC2         Column to stop reading prompt at
//
//   Returns:
//    The prompt that most closely corresponds to what is on the screen
//
//   Example:
//    GFMGetPrompt( row(), w_left(), w_right() )
//
//   Files:
//    None.
//
//   Description:
//    Support FUNCTION for GFMHelpKey()
//
//   Notes:
//      
//
//Category:
//    Help System
//
//   See Also:
//    GFMHelpKey()
//
/////////////////////
/////////////////////
FUNCTION GFMGetPrompt( nR1, nC1, nC2 )
	LOCAL aPrompts  := GFMSet( _GFM_PROMPT_ARRAY )
	LOCAL cMatch    := ReadScreen( nR1, nC1, nC2 )
	LOCAL nMatch    := 0
	LOCAL nTemp     := 0
	LOCAL nMaxMatch := 0
	LOCAL xReturn

	IF Full( aPrompts )  // Find the closest match for the text on the screen
		aEval( aPrompts, { | cText, nPos | nTemp := MatchTo( cText, cMatch ), ;
	       if( nTemp > nMaxMatch, nMaxMatch := nTemp, NIL ), ;
	       if( nTemp == nMaxMatch, nMatch := nPos, NIL ) } )
	ENDIF
	
	IF nMatch # 0 THEN xReturn := aPrompts[ nMatch ]
RETURN xReturn

///////////////////
///////////////////
//
//   Purpose:
//    Read a line of the screen into a text string
//
//   Syntax:
//    ReadScreen( [ <nR1> [, <nC1> [, <nC2> ] ] ] )
//
//   Formal Arguments:
//    Name        Description
//     
//    nR1         Row to read line from [ defaults to row() ]
//    nC1         Column to starting reading line [ defaults to col() ]
//    nC2         Column to stop reading line [ defaults to MaxCol() ]
//
//   Returns:
//    The text on the screen at the specified position
//
//   Example:
//    cText := readscreen( row(), col(), MaxCol() )
//
//   Files:
//    None.
//
//   Description:
//    Reads the screen at the specified position, and returns the text sans
//    attribute bytes via a call to scr2txt(), a C function.
//
//   Notes:
//      
//
//Category:
//    Help System
//
//   See Also:
//
//
///////////////////
///////////////////
FUNCTION ReadScreen( nR1, nC1, nC2 )
	LOCAL cScreen, cReturn := ''
	LOCAL nPos, nLength
	DEFAULT nR1 TO Row(), ;
				nC1 TO Col(), ;
	  			nC2 TO MaxCol()
	
	cScreen := SaveScreen( nR1, nC1, nR1, nC2 )
	nLength := Len( cScreen )
	for nPos := 1 to nLength step 2
		cReturn += substr( cScreen, nPos, 1 )
	next nPos
RETURN cReturn

