#include "gfm.ch"
#include "inkey.ch"

STATIC lGFMPulling := .F.

//////////////////
//////////////////
//
//  Purpose:  Calculate coordinate for the current menu level
//
//   Syntax:  GFMCoords( <cStyle>, <nLines>, <nWidth>, <nLevel>,
//                       <nR1>, <nC1>, <nR2>, <nC2> )
//
//   Parameters:
//
//      Name        Description
//       
//      cStyle      Type of menu
//      nLines      # of prompts
//      nWidth      width of longest prompt
//      nLevel      menu level
//      nR1         top row of box to RETURN
//      nC1         left column of box to RETURN
//      nR2         bottom row of box to RETURN
//      nC2         right column of box to RETURN
//
//  Returns:  The coordinates of the menu box to use
//
//  Example:  GFMCoords( GFMStyle(), len( aPrompts ), len( aPrompts[ 1 ] ), ;
//                       GFMLevel(), @nR1, @nC1, @nR2, @nC2 )
//
//    Files:  None.
//
// Overview:  Calculates the dimensions for the menu box for whatever menu
//            style is being used.  Any updates to the menu styles should
//            "ripple" down from here to GFMRow() and GFMCol().
//
//    Notes:  This is where support for new menu styles should be added.
//
// Category:  Menu Handling
//
// See Also:  GFMRow() GFMCol()
//
//////////////////
//////////////////
FUNCTION GFMCoords( cStyle, nLines, nWidth, nLevel, nR1, nC1, nR2, nC2 )
	LOCAL aMenuSize := GFMSizeGet( GFMSet( _GFM_BYTE ) )
	IF Full( aMenuSize )	// Saved menu size
		nR1 := MenuSizeTop( aMenuSize )
		nC1 := MenuSizeLeft( aMenuSize )
		nR2 := MenuSizeBottom( aMenuSize )
		nC2 := MenuSizeRight( aMenuSize )
	ELSE
	   nC1 := GFMCol( cStyle, nLevel )
	   nR1 := GFMRow( cStyle, 1, nLevel )
	   IF cStyle $ "PCNS"  // Pull-down, Cascading, No-box or Single pull-down
	      nR2 := nR1 + nLines + 1
	      nC2 := nC1 + nWidth + 1
	   ELSEIF cStyle == 'E'  // Extra line/column
	      nR2 := nR1 + nLines + 3
	      nC2 := nC1 + nWidth + 3
	   ELSEIF cStyle $ "1B"  // 1-2-3 style menu
	      IF cStyle == "B"
	         nR2 := nR1 + 3
	      ELSE
	         nR2 := nR1 + 1
	      ENDIF
	      nC2 := MaxCol()
	   ENDIF
		GFMFitScreen( @nR1, @nC1, @nR2, @nC2 )
	   GFMSet( _GFM_CURRENT_COORDS, { nR1, nC1, nR2, nC2 } )
	ENDIF
RETURN .t.

PROCEDURE GFMFitScreen( nTop, nLeft, nBottom, nRight )
	LOCAL nRows := Min( nBottom - nTop, MaxRow() - 1 )
	LOCAL nCols := Min( nRight - nLeft, MaxCol() )
	IF nTop < 0
		nBottom := nRows
		nTop := 0
	ENDIF
	IF nLeft < 0
		nLeft := 0
		nRight := nCols
	ENDIF
	IF nBottom > MaxRow() - 1
		nBottom := MaxRow() - 1
		nTop := Max( nBottom - nRows, GFMSet( _GFM_TOPLINE ) )
	ENDIF
	IF nRight > MaxCol()
		nRight := MaxCol()
		nLeft := Max( nRight - nCols, GFMSet( _GFM_LEFTCOL ) )
	ENDIF
RETURN

////////////////
////////////////
//
//  Purpose:  Determine the column the prompts should begin at
//
//   Syntax:  GFMCol( <cStyle>, <nLevel> )
//
//   Parameters:
//
//      Name        Description
//       
//      cStyle      Type of menu
//      nLevel      Current menu nLevel
//
//  Returns:  the column the prompt should be at
//
//  Example:  GFMCol( GFMStyle(), 3 )  // For the column position for nLevel 3
//
//    Files:  None.
//
// Overview:  This routine is designed to return the correct column position
//            for whatever menu style happens to be active.  By using this
//            Core routine for this calculation, all other code is simplified
//            tremendously.
//
//    Notes:  If any menu types are added, this is where the change for
//            supporting its display location should be made.
//
// Category:  Menu Handling
//
// See Also:  GFMRow() GFMCol()
//
////////////////
////////////////
FUNCTION GFMCol( cStyle, nLevel )
	LOCAL nCol := w_Right() + 2
   IF cStyle $ "CE"  // Cascading menus
      nCol := GFMSet( _GFM_LEFTCOL ) + nLevel * GFMSet( _GFM_INDENT )
   ELSEIF cStyle $ "B1"   //  Boxed 1-2-3
      nCol := GFMSet( _GFM_LEFTCOL )
   ELSEIF nLevel == 1
      nCol := Col()
   ENDIF
RETURN nCol

////////////////
////////////////
//
//  Purpose:  Determine the row the current menu option should be on
//
//   Syntax:  GFMRow( <cStyle>, <nOption>, <nLevel> )
//
//   Parameters:
//
//      Name        Description
//       
//      cStyle      Type of menu
//      nOption     Menu option number
//      nLevel      Level of this menu
//
//  Returns:  the row the prompt should be at
//
//  Example:  GFMRow( GFMStyle(), 1, 2 )  // Row for first prompt on nLevel 2
//
//    Files:  None.
//
// Overview:  This routine is designed to RETURN the correct column position
//            for whatever menu style happens to be active.  By using this
//            Core routine for this calculation, all other code is simplified
//            tremendously.
//
//    Notes:  If any menu types are added, this is where the change for
//            supporting its display location should be made.
//
// Category:  Menu Handling
//
// See Also:  GFMCol()  GFMCoords()
//
////////////////
////////////////
FUNCTION GFMRow( cStyle, nOption, nLevel )
	LOCAL nRow := w_Top() + GFMSet( _GFM_ROWOFFSET ) - 1
	DO CASE
	CASE cStyle $ "CEB"
      nRow := GFMSet( _GFM_TOPLINE ) + nLevel * GFMSet( _GFM_INDENT )
   CASE cStyle == "1"  // 1-2-3 style menus
      nRow := GFMSet( _GFM_TOPLINE )
   CASE nLevel == 1	// Otherwise, default to pull-downs
  	   nRow := Max( GFMSet( _GFM_TOPLINE ), w_Bottom() ) + iif(cStyle=='N',0,1)
	ENDCASE
RETURN nRow

//////////////////
//////////////////
//
//  Purpose:  Turn off special keyboarding for some menu types
//
//   Syntax:  GFMKeyOff()
//
//  No parameters specified
//
//  Returns:  .t.
//
//  Example:  GFMKeyOff()
//
//    Files:  None.
//
// Overview:  Sets the keyboard definitions off for when you're in any nLevel
//            below nLevel one.
//
//    Notes:  Only used for the Pull-Down style menus.
//
// Category:  Menu Handling
//
// See Also:  GFMKeyOn()
//
//////////////////
//////////////////
FUNCTION GFMKeyOff( cStyle )
   DEFAULT cStyle TO GFMSet( _GFM_STYLE )
   IF cStyle $ "SPN"  // Pull down - make sure the keys are cleared
		GFMMainShade()
      set key 19 to
      set key 4 to
   ENDIF
RETURN .t.

/////////////////
/////////////////
//
//  Purpose:  Turn on special keyboarding for some menu types
//
//   Syntax:  GFMKeyOn()
//
//  No parameters specified
//
//  Returns:  .t.
//
//  Example:  GFMKeyOn()
//
//    Files:  None.
//
// Overview:  Sets the keyboard definitions on for when you're in any nLevel
//            below nLevel one.
//
//    Notes:  Only used for the Pull-Down style menus.
//
// Category:  Menu Handling
//
// See Also:  GFMKeyOff()
//
/////////////////
/////////////////
FUNCTION GFMKeyOn( cStyle )
   DEFAULT cStyle TO GFMSet( _GFM_STYLE )
	lGFMPulling := .F.
   IF cStyle $ "SPN" // Pulldown - make sure the keys are set
      SetKey( K_LEFT, {|| Left_Arrow( Chr( K_ESC ) + Chr( K_ENTER ) ) } )
      SetKey( K_RIGHT, {|| Right_Arrow( Chr( K_ESC ) + Chr( K_ENTER ) ) } )
   ENDIF
RETURN .t.


STATIC FUNCTION left_arrow( cKeyStuff )
	DEFAULT cKeyStuff TO Chr( K_ESC ) + Chr( K_ENTER )
   // CLEAR TYPEAHEAD
   // Prompts and nAns0 are dynamically scoped from the main menu routine
   IF MEMVAR->nAns0 == 1
      MEMVAR->nAns0 := MEMVAR->nPrompts
   ELSE
      MEMVAR->nAns0--
   ENDIF
	GFMSet( _GFM_DIRECTION, _GFM_LEFT )
	lGFMPulling := .T.
   KeyStuff( cKeyStuff )
RETURN .t.

STATIC FUNCTION right_arrow( cKeyStuff )
	DEFAULT cKeyStuff TO Chr( K_ESC ) + Chr( K_ENTER )
	
   // CLEAR TYPEAHEAD
   // Both nAns0 and nPrompts are dynamically scoped from the main FUNCTION
   IF MEMVAR->nAns0 == MEMVAR->nPrompts
      MEMVAR->nAns0 := 1
   ELSE
      MEMVAR->nAns0++
   ENDIF
	lGFMPulling := .T.
	GFMSet( _GFM_DIRECTION, _GFM_RIGHTCOL )
   KeyStuff( cKeyStuff )
RETURN .T.

STATIC PROCEDURE GFMMainShade()
	LOCAL cShade := GFMSet( _GFM_SHADE )
	IF lGFMPulling THEN cShade := "0"
	w_ShadeSet( Max( w_Handle() - 1, 1 ), cShade )
RETURN

/////////////////
/////////////////
//
//  Purpose:  Determine the box frame for the current nLevel
//
//   Syntax:  GFMFrame( <cStyle>, <nLevel> )
//
//   Parameters:
//
//      Name        Description
//       
//      cStyle      Menu style that is active
//      nLevel      Level of the menu
//
//  Returns:  The appropriate string for the box frame
//
//  Example:  @ nR1, nC1, nR2, nC2 box GFMFrame( GFMSet( _GFM_STYLE ), 1 )
//            //to draw a box with a frame for nLevel 1.
//
//    Files:  None.
//
// Overview:  Framing FUNCTION for the menu boxes.
//
//    Notes:  If you don't like the framing as it's currently implemented,
//            here's the place to change it!
//
// Category:  Menu Handling
//
// See Also:  frame()
//
/////////////////
/////////////////
FUNCTION GFMFrame( cStyle, nLevel )
   LOCAL cFrame := Frame( nLevel )
	cStyle := Upper( SubStr( cStyle, 1, 1 ) )
   IF nLevel == 0
      IF cStyle $ 'N1'
         cFrame := Space( 9 )   // No box
      ELSEIF cStyle $ 'CEB' // Cascading or Boxed 1-2-3
         cFrame := Frame( 2 )
      ELSEIF cStyle == 'S'   // Single pull-down - just bottom line
         cFrame := "      "
		ELSE
	      cFrame := Frame( 1 )
      ENDIF
   ELSEIF nLevel == 1 .and. cStyle $ 'PS'  // One of the special frames
      IF cStyle == 'P'
         cFrame := "³ "
      ELSEIF cStyle == 'S'
         cFrame := "ķȺ "
      ENDIF
   ENDIF
RETURN cFrame


/////////////////
/////////////////
//
//     Purpose:
//      Keyboard stuffing
//
//      Syntax:
//      keystuff( [ <cTxt> ] )
//
//   Arguments:
//      Name        Description
//       
//      txt         Text to stuff into keyboard
//
//     Returns:
//      .t. if <txt> is passed, .f. otherwise
//
//     Example:
//      keystuff( chr(21) )
//
//       Files:
//      None.
//
// Description:
//      Changes the KEYBOARD command into a FUNCTION (for prototyping use)
//
//       Notes:
//
//
//    Category:
//      Menu Handling
//
//    See Also:
//
//
/////////////////
/////////////////
FUNCTION KeyStuff( cText )
   IF cText == NIL
     RETURN .f.
   ENDIF
   KEYBOARD cText
RETURN .t.

///////////////////
///////////////////
//
//     Purpose:
//      RETURN the version information for GrumpFish Menu
//
//      Syntax:
//      GFMversion()
//
//    No arguments specified
//
//     Returns:
//      This version of GrumpFish Menu
//
//     Example:
//      ? 'You are using', GFMversion()
//
//       Files:
//      None.
//
// Description:
//      Handy for version information control
//
//       Notes:
//
//
//    Category:
//      Menu Handling
//
//    See Also:
//
//
///////////////////
///////////////////
FUNCTION GFMversion
RETURN GFM_VERSION

//////////////////
//////////////////
//
// Purpose:
//    Determine a security level
//
// Syntax:
//    GFMSecure( <uSecurity> [, <lSet> ] )
//
// Formal Arguments: (1)
//    Name        Description
//     
//    uSecurity   Security level
//    lSet        Toggle for setting the security level
//
// Returns:
//    .T. if the user has a high-enough security level, .F. otherwise
//
// Examples:
//    GFMSecure( 5, .T. )  // Sets the user's security level to 5
//
// Files:
//    None.
//
// Description:
//    Security level testing that accesses the internal GFM setting
//    _GFM_SECURITY to determine if the user has a high-enough security
//    level to access a menu option.
//
// Notes:
//    This routine is written very ambiguously so any type of variable
//    could be tested for the security level.  This method works for
//    both numeric and character values.  In order to speed it up, the
//    test of the type of variable could be removed, and one or the other
//    method could be used.
//
// Class:
//    Menu Handling
//
// See Also:
//
//
// Include files:
//    gfm.ch
//    setcurs.ch
//
// Revisions:
//    02/12/91 23:17:22 1.0   Original version
//
//////////////////
//////////////////
FUNCTION GFMSecure( uSecurity, lSet )
   LOCAL uTest := GFMSet( _GFM_SECURITY )
   LOCAL lAccess := .T.
   IF ValType( lSet ) == 'L' .AND. lSet
      GFMSet( _GFM_SECURITY, uSecurity )
   ELSEIF uTest # NIL .AND. uSecurity # NIL // A security level has been set
      IF ValType( uTest ) == 'C'
         lAccess := uSecurity $ GFMSet( _GFM_SECURITY )
      ELSE
         lAccess := uSecurity <= GFMSet( _GFM_SECURITY )
      ENDIF
   ENDIF
RETURN lAccess

/////////////////////
/////////////////////
//
//	Section:
//		Source Code
//
//	Category:
//		Menu Handling
//
//	Purpose:
//		Protect from automatically invoking a menu that shouldn't be
//
//	Syntax:
//		GFMNotPulled() -> lNotPulled
//
//	Returns:
//		.T. if this is not automatically being pulled, .F. otherwise
//
//	Examples:
//		See your generated code.
//
//	Files:
//		
//
//	Description:
//		Provides more intelligent handling of main menu options generated from
//		Grumpfish Menu.
//
//	Notes:
//		Also shades the main menu to maintain consistent appearance.
//
//	See Also:
//		GFMKeyOn() GFMKeyOff()
//
//	Include files:
//		gfm.ch
//		inkey.ch
//
//	Revisions:
//		06/23/92	22:28:41	1.0	Original version
//
/////////////////////
/////////////////////
FUNCTION GFMNotPulled
	LOCAL lPulled := .F.
	IF lGFMPulling
		lGFMPulling := .F.
		w_ShadeSet( w_Handle(), GFMSET( _GFM_SHADE ) )
		w_Shade()
		lPulled := .T.
		CLEAR TYPEAHEAD
		SET KEY K_LEFT TO
		SET KEY K_RIGHT TO
   ENDIF

RETURN ! lPulled

PROCEDURE MarkArea( nTop, nLeft, nBottom, nRight )
	ReSizeBox( @nTop, @nLeft, @nBottom, @nRight )
RETURN
