/*
     Program: COLORPAL()
     System: GRUMPFISH LIBRARY
     Author: Greg Lief
     Copyright (c) 1988-89, Greg Lief
     Clipper 5.x Version
     Compile instructions: clipper colorpal /n/w/a
     Allows user to select color from interactive palette
     Procs & Fncts: COLORPAL()
     Calls: COLOR_N2S()   (function in COLORS.PRG)
*/

// begin preprocessor directives

#include "inkey.ch"

// end preprocessor directives
// begin global declarations

static foreground := 'N  B  G  BG R  BR GR W  N+ B+ G+ BG+R+ BR+GR+W+ '

// end global declarations


function colorpal(curr_color, pal_top, pal_left, redrawfunc)
static palette := []
local mrow, mcol, keypress, xx, oldcolor := setcolor(), ;
      use_sample := .f., num_colors, palbuffer, pal_bot, ;
      colorstrng := 'N  B  G  BG R  BR GR W  N+ B+ G+ BG+R+ BR+GR+W+ '
local oldcurs := setcursor(0), oldscrn := savescreen(0, 0, maxrow(), maxcol())

// set start-up color to current color if no parameter passed
curr_color := if(curr_color == NIL, setcolor(), upper(curr_color))

// establish top row and leftmost column for palette if not passed
pal_top  := IF(pal_top = NIL, IF(setblink(), 16, 8), ;
            IF(pal_top < 6, 6, MIN(pal_top, IF(setblink(), 16, 8))))
pal_bot  := pal_top + IF(setblink(), 8, 16)
pal_left := IF(pal_left = NIL, maxcol()-15, MIN(pal_left, maxcol()-15))

/*
   establish color sample UDF.  there are three options:
   a) user did not pass fourth parameter --- use default
   b) user passed a code block as fourth parameter - use that unchanged
   c) user passed character string as parameter (such as "whatever()") --
      compile it to a code block
*/
if valtype(redrawfunc) == 'C'
   redrawfunc:=&("{ | | " + redrawfunc + "}")
elseif valtype(redrawfunc) != 'B'
   redrawfunc := { | a, b | colorsample(a, b) }
   use_sample := .T.
endif

// convert the string to a number
curr_color := color_s2n(curr_color)

/*
   create character string of text/color attributes for color palette
   if it was not already created on a previous visit to colorpal()
   note: we must do a secondary test based on the current setting of
   SETBLINK(), because if it was changed, the palette must either be
   doubled to halved accordingly (SETBLINK(.T.)=128 colors, .F. = 256 colors)
*/
if len(palette) != ( num_colors := if(setblink(), 127, 255)+1) * 2
   palette := []
   for xx = 0 to num_colors
      palette += chr(4) + chr(xx)
   next
endif
restscreen(pal_top, pal_left, pal_bot - 1, pal_left+15, palette)
setcolor("+W/BR")
scroll(pal_bot, pal_left, pal_bot, pal_left+15, 00)
@ pal_bot, pal_left      say chr(24)+chr(25)+chr(27)+chr(26)
@ pal_bot, pal_left + 7  say chr(17) + chr(217)
@ pal_bot, pal_left + 13 say 'Esc'

// determine starting row and column within palette
mrow := pal_top + int(curr_color / 16)
mcol := pal_left + curr_color % 16

// commence main keypress loop
keypress := 0
do while keypress != K_ESC .and. keypress != K_ENTER
   setcolor(Color_N2S(curr_color))
   // draw blinking diamond to mark current color and get keypress
   @ mrow,mcol say chr(219)
   keypress := inkey(0)

   // clear blinking diamond
   @ mrow,mcol say chr(4)

   // process keystroke
   do case
      case keypress == K_DOWN
         // if we are at the bottom, jump to the top
         IF mrow == pal_bot - 1
            curr_color -= (pal_bot - 1 - (mrow := pal_top) ) * 16
         else
            mrow++
            curr_color += 16
         endif

      case keypress == K_UP
         // if we are at the top, jump to the bottom
         if mrow == pal_top
            curr_color += ( (mrow := pal_bot - 1) - pal_top) * 16
         else
            mrow--
            curr_color -= 16
         endif

      case keypress == K_RIGHT
         if mcol < pal_left + 15
            mcol++
            curr_color++
         else
            mcol := pal_left
            curr_color -= 15
         endif

      case keypress == K_LEFT
         if mcol > pal_left
            mcol--
            curr_color--
         else
            mcol := pal_left + 15
            curr_color += 15
         endif

      case keypress == K_PGDN
         curr_color += (pal_bot - 1 - mrow) * 16
         mrow := pal_bot - 1

      case keypress == K_PGUP
         curr_color -= (mrow - pal_top) * 16
         mrow := pal_top

      case keypress == K_ENTER .or. keypress == K_ESC
         exit

      // user pressed something else - give 'em a raspberry
      otherwise
         tone(50, 2)

   endcase
   setcolor(color_n2s(curr_color))
   dispbegin()
   if ! use_sample
      palbuffer := savescreen(pal_top, pal_left, pal_bot, pal_left+15)
   endif
   eval(redrawfunc, pal_top, pal_left)
   if ! use_sample
      restscreen(pal_top, pal_left, pal_bot, pal_left+15, palbuffer)
   endif
   dispend()
enddo

// restore environment
setcursor(oldcurs)              // restore previous cursor
restscreen(0, 0, maxrow(), maxcol(), oldscrn)

// if user pressed esc to abort, return the old color
return (if(keypress == K_ESC, oldcolor, color_n2s(curr_color)))

* end function ColorPal()
*--------------------------------------------------------------------*


/*
   ColorSample(): draw sample box showing current selection
*/
static function ColorSample(pal_top, pal_left)
DispBox(pal_top-6, pal_left, pal_top-3, pal_left+15,'ͻȺ ')
@ pal_top-5, pal_left+3 say "Sample of"
@ pal_top-4, pal_left+1 say "current color"
return NIL

* end static function ColorSample()
*--------------------------------------------------------------------*

* eof colorpal.prg
/*
   Program: COLORS.PRG
   System: GRUMPFISH LIBRARY
   Author: Greg Lief
   Copyright (c) 1988-90, Greg Lief
   Clipper 5.x Version
   Compile instructions: clipper colors /n/w/a

   Conversion functions for colors - numeric to character string
                                     and vice versa
   Functions: COLOR_S2N()
            : COLOR_N2S()
*/


/*
   Color_N2S(): convert color number (0-127) to dBASE color string
*/
static function color_n2s(colorno)
local background := 'N B G BGR BRGRW ', blinking
if valtype(colorno) = "C"
   colorno := bin2i(colorno)
endif
blinking := (colorno > 127)
colorno := colorno % 128
return (if(blinking, '*', '') + ;
        trim(substr(foreground, (colorno % 16) * 3 + 1, 3)) + '/' + ;
        trim(substr(background, int(colorno / 16) * 2 + 1, 2)))

* end function Color_N2S()
*--------------------------------------------------------------------*


/*
   Color_S2N(): convert dBASE color string to number
*/
static function color_s2n(colorno)
local mslash, mcomma, blinking, mfore, mback

// set start-up color to current color if no parameter passed
colorno := if(colorno == NIL, setcolor(), upper(colorno))

// parse this string to determine foreground and background colors
// first determine how many characters are in the foreground color
// by locating the first slash in the string
mslash := at('/', colorno)

// background color will lie between the slash and the first comma,
mcomma := at(',', colorno)

// but we also must allow for color parameters passed without a comma
mcomma := if(mcomma = 0, len(colorno) + 1, mcomma)

// break out the foreground and background colors
mfore := substr(colorno, 1, mslash - 1)
if '*' $ mfore
   mfore := strtran(mfore, '*', '')
   blinking := .t.
else
   blinking := .f.
endif
mback := substr(colorno, mslash + 1, mcomma - mslash - 1)
// convert the string to a number
return int(at(mfore, foreground) / 3) + int(at(mback, foreground) / 3) * 16 + ;
       if(blinking, 128, 0)

* end function Color_S2N()
*--------------------------------------------------------------------*

* eof colors.prg
