#include "gfm.ch"
#include "inkey.ch"
#include "setcurs.ch"

///////////////
///////////////
//
//  Purpose:  Set the GrumpMenu configuration toggle on for top level
//
//   Syntax:  GFMCFGSet( <cProcedure>, <nLine>, <cVariable> )
//
//   Parameters:
//
//      Name        Description
//       
//      cProcedure  Procedure name - standard parameter in a SET KEY command
//      nLine       Procedure line - standard parameter in a SET KEY command
//      cVariable   read variable  - standard parameter in a SET KEY command
//
//  Returns:  .t.
//
//  Example:  set key GFMSet( _GFM_CFGKEY ) to GFMCFGSet
//
//    Files:  None.
//
// Overview:  Used in the main menu loop to control when configuration is
//            being executed.
//
//    Notes:  Keyboards the Escape character to get out of the main menu
//            level, but IF _GFM_CFGHIT is set to .t., configuration is
//            executed before the main menu is exited.
//
// Category:  Menu Handling
//
// See Also:  gfmconfig()
//
///////////////
///////////////
FUNCTION GFMCFGSet( cProcedure, nLine, cVariable )
   GFMSet( _GFM_CFGHIT, .t. )
   KeyStuff( chr(27) )
RETURN .t.

//////////////////
//////////////////
//
//  Purpose:  Allows end user to customize GrumpFish menu
//
//   Syntax:  GFMConfig()
//
//  No parameters specified
//
//  Returns:  .t.
//
//  Example:  GFMConfig()
//
//    Files:  None.
//
// Overview:  Sets the internal variables for the user to customize the menu
//
//    Notes:  Have to implement saving the configuration, and reading it in
//            as well (part of the general purpose gfmhello()) that isn't yet
//            implemented.
//
// Category:  Menu Handling
//
// See Also:  setgfm()
//
//////////////////
//////////////////
FUNCTION GFMConfig
   LOCAL cStandard, cUnselected, cEnhanced
   LOCAL nDispRow, nDispCol
   LOCAL aSets 		:= GFMSet()
   LOCAL cTitle		:= GFMSet( _GFM_SYSTEM ) + ' Cosmetics Screen'
   LOCAL cFooter		:= 'Press [F1] for options, [Esc] to abort'
   LOCAL nR1 			:= 1, nC1 := 1, nC2 := 60, nR2 := 5 + _GFM_RUN_COUNT
   LOCAL cBoxColor  	:= GFMSet( _GFM_FRAMECOLOR )
   LOCAL cBackColor 	:= GFMSet( _GFM_BACKCOLOR )
   LOCAL nBackChr   	:= Asc( GFMSet( _GFM_BACKCHAR ) )
   LOCAL cShade     	:= GFMSet( _GFM_SHADE )
   LOCAL nTopLine   	:= GFMSet( _GFM_TOPLINE )
   LOCAL cStyle     	:= GFMSet( _GFM_STYLE )
   LOCAL nLeftCol   	:= GFMSet( _GFM_LEFTCOL )
   LOCAL nRightCol  	:= GFMSet( _GFM_RIGHTCOL )
   LOCAL nIndent    	:= GFMSet( _GFM_INDENT )
   LOCAL nNoteKey   	:= GFMSet( _GFM_NOTEKEY )
   LOCAL nCFGKey    	:= GFMSet( _GFM_CFGKEY )
   LOCAL nUtilKey   	:= GFMSet( _GFM_UTILKEY )
   LOCAL lBlink     	:= GFMSet( _GFM_BRIGHT )
   LOCAL lWrap      	:= GFMSet( _GFM_WRAP )
   LOCAL cGFMTitle  	:= PadR( GFMSet( _GFM_TITLE ), 40 )
   LOCAL cGFMFooter 	:= PadR( GFMSet( _GFM_FOOTER ), 40 )
	LOCAL nGFMSpacing := GFMSet( _GFM_SPACING )
   LOCAL cSaveFile	:= Space( 72 )
   LOCAL lScoreboard := set( _SET_SCOREBOARD, .f. )  // Status of scoreboard
   LOCAL nCursor     := SetCursor( SC_NORMAL ) // Status of cursor
   LOCAL GetList		:= {}
	LOCAL bHelpKey		:= SetKey( K_F1 )

	IF GFMSet( _GFM_INSTALL )
		cSaveFile := PadR( GFMSet( _GFM_CFGFILE ), 72 )	// In Install?
	ENDIF

   w_Pop()			// Clear the current menu screen
	@ MaxRow(), 0	// Clear the message line
   w_Push()			// Save the background screen
	SET KEY K_F1 TO DoValid

   cEnhanced   := GFMSet( _GFM_MENUCOLOR )
   cStandard   := SubStr( cEnhanced, 1, Pos(',', cEnhanced ) - 1 )
   cUnselected := StrExtract( cEnhanced, ',', 5 )
   cEnhanced   := StrExtract( cEnhanced, ',', 2 )
   SetColor( GFMSet( _GFM_MENUCOLOR ) )
   w_Shade( nR1, nC1, nR2, nC2, Frame( 1 ), cTitle, cFooter )
   nDispCol    := nC1 + 1
   nDispRow    := nR1
   @ ++nDispRow, nDispCol say GFMFormat( aSets[ _GFM_STYLE, _GFM_PROMPT ] ) ;
      get cStyle pict '!' valid PickList( @cStyle, GFMStyles(),, aSets[ _GFM_STYLE, ;
     _GFM_PROMPT ] ) .and. GFMGetSet( _GFM_STYLE, SubStr( cStyle, 1, 1 ) )
   @ ++nDispRow, nDispCol say GFMFormat( "Menu foreground color" ) ;
      get cStandard valid PickColor( @cStandard ) .and. ;
     GFMSet( _GFM_MENUCOLOR, cStandard + ',' + cEnhanced + ',N,N,' + cUnselected ) # NIL
   @ ++nDispRow, nDispCol say GFMFormat( "Menu hi-light color" ) ;
      get cEnhanced valid PickColor( @cEnhanced ) .and. ;
     GFMSet( _GFM_MENUCOLOR, cStandard + ',' + cEnhanced + ',N,N,' + cUnselected ) # NIL
   @ ++nDispRow, nDispCol say GFMFormat( "Unselected Get Color" ) ;
      get cUnselected valid PickColor( @cUnselected ) .and. ;
     GFMSet( _GFM_MENUCOLOR, cStandard + ',' + cEnhanced + ',N,N,' + cUnselected ) # NIL
   @ ++nDispRow, nDispCol say GFMFormat( aSets[ _GFM_FRAMECOLOR, _GFM_PROMPT ] ) ;
      get cBoxColor valid PickColor( @cBoxColor ) .and. ;
     GFMGetSet( _GFM_FRAMECOLOR, cBoxColor )
   @ ++nDispRow, nDispCol say GFMFormat( aSets[ _GFM_BACKCOLOR, _GFM_PROMPT ] ) ;
      get cBackColor valid PickColor( @cBackColor ) .and. ;
     GFMGetSet( _GFM_BACKCOLOR, cBackColor )
   @ ++nDispRow, nDispCol say GFMFormat( aSets[ _GFM_BACKCHAR, _GFM_PROMPT ] ) ;
     get nBackChr pict '999' valid IsBackChr( nBackChr ) .and. ;
     GFMGetSet( _GFM_BACKCHAR, nBackChr )
   @ ++nDispRow, nDispCol say GFMFormat( aSets[ _GFM_SHADE, _GFM_PROMPT ] ) ;
      get cShade pict '@!' valid PickShade( @cShade ) .and. ;
     GFMGetSet( _GFM_SHADE, cShade )
   @ ++nDispRow, nDispCol say GFMFormat( aSets[ _GFM_TOPLINE, _GFM_PROMPT ] ) ;
      get nTopLine pict '99' ;
     valid nTopLine >= 0 .and. nTopLine < maxrow() /2 .and. ;
            GFMGetSet( _GFM_TOPLINE, nTopLine )
   @ ++nDispRow, nDispCol say GFMFormat( aSets[ _GFM_LEFTCOL, _GFM_PROMPT ] ) ;
      get nLeftCol pict '99' ;
      valid nLeftCol >= 0 .and. nLeftCol <= maxcol() /2 .and.;
      GFMGetSet( _GFM_LEFTCOL, nLeftCol )
   @ ++nDispRow, nDispCol say GFMFormat( aSets[ _GFM_RIGHTCOL, _GFM_PROMPT ] ) ;
      get nRightCol pict '999' ;
      valid LastKey() == K_UP .OR. ( nRightCol >= nLeftCol .and. ;
			nRightCol <= MaxCol() .and. GFMGetSet( _GFM_RIGHTCOL, nRightCol ) )
   @ ++nDispRow, nDispCol say GFMFormat( aSets[ _GFM_INDENT, _GFM_PROMPT ] ) ;
      get nIndent pict '9' ;
     valid nIndent >= 0 .and. GFMGetSet( _GFM_INDENT, nIndent )
   @ ++nDispRow, nDispCol say GFMFormat( aSets[ _GFM_NOTEKEY, _GFM_PROMPT ] ) ;
      get nNoteKey pict '999' valid PickExtKey( @nNoteKey ) .and. ;
      GFMGetSet( _GFM_NOTEKEY, nNoteKey )
   @ ++nDispRow, nDispCol say GFMFormat( aSets[ _GFM_CFGKEY, _GFM_PROMPT ] ) ;
      get nCFGKey pict '999' valid PickExtKey( @nCFGKey ) .and. ;
      GFMGetSet( _GFM_CFGKEY, nCFGKey )
   @ ++nDispRow, nDispCol say GFMFormat( aSets[ _GFM_UTILKEY, _GFM_PROMPT ] ) ;
      get nUtilKey pict '999' valid PickExtKey( @nUtilKey ) .and. ;
      GFMGetSet( _GFM_UTILKEY, nUtilKey )
   @ ++nDispRow, nDispCol say GFMFormat( aSets[ _GFM_BRIGHT, _GFM_PROMPT ] ) ;
      get lBlink pict '@! Y' valid GFMGetSet( _GFM_BRIGHT, lBLink )
   @ ++nDispRow, nDispCol say GFMFormat( aSets[ _GFM_WRAP, _GFM_PROMPT ] ) ;
      get lWrap pict '@! Y' valid GFMGetSet( _GFM_WRAP, lWrap )
   @ ++nDispRow, nDispCol say GFMFormat( aSets[ _GFM_TITLE, _GFM_PROMPT ] ) ;
      get cGFMTitle pict '@S20' valid GFMGetSet( _GFM_TITLE, alltrim( cGFMTitle ) )
   @ ++nDispRow, nDispCol say GFMFormat( aSets[ _GFM_FOOTER, _GFM_PROMPT ] ) ;
      get cGFMFooter pict '@S20' valid GFMGetSet( _GFM_FOOTER, alltrim( cGFMFooter ) )
   @ ++nDispRow, nDispCol say GFMFormat( aSets[ _GFM_SPACING, _GFM_PROMPT ] ) ;
      get nGFMSpacing pict '99' valid nGFMSpacing > 0 .AND. ;
		GFMGetSet( _GFM_SPACING, nGFMSpacing )
   @ ++nDispRow, nDispCol say GFMFormat( "Save settings to" ) ;
      get cSaveFile pict '@S13'

   READ
	SetKey( K_F1, bHelpKey )
   w_shadedir( cShade )							 	// Re-set shade direction

   IF lastkey() # 27
		IF ! empty( cSaveFile := Trim( cSaveFile ) )
			GFMCFGwt( cSaveFile )
		ENDIF
		IF aSets[ _GFM_STYLE, _GFM_VALUE ] # GFMSet( _GFM_STYLE ) .OR. ;
	  		aSets[ _GFM_TOPLINE, _GFM_VALUE ] # GFMSet( _GFM_TOPLINE ) .OR. ;
	  		aSets[ _GFM_LEFTCOL, _GFM_VALUE ] # GFMSet( _GFM_LEFTCOL ) .OR. ;
	  		aSets[ _GFM_RIGHTCOL, _GFM_VALUE ] # GFMSet( _GFM_RIGHTCOL ) .OR. ;
	  		aSets[ _GFM_INDENT, _GFM_VALUE ] # GFMSet( _GFM_INDENT )
			GFMDeStabilize()	// Destabilize all menu positions
		ENDIF
   ELSE
      GFMSet( aSets )	// Re-set the GFM settings
		GFMSetSys()
   ENDIF

   Set( _SET_SCOREBOARD, lScoreboard )
   SetCursor( nCursor )
   w_Pop()		// Restore this screen
   GFMHello()	// Use the settings
RETURN .t.

STATIC PROCEDURE DoValid()
	LOCAL Get := GetActive()
	IF !Empty( Get:postBlock ) .AND. ValType( Get:postBlock ) == 'B'
		Eval( Get:postBlock )
	ENDIF
RETURN

FUNCTION GFMFormat( cText )
RETURN PadR( cText, _GFM_PROMPT_WIDTH + 5 , '.' ) + ':'

#xtrans AssignGet( <v> ) => Get:varPut(<v>);Get:updateBuffer();Get:display()
#define POINT_RIGHT Chr(16)
#define POINT_LEFT  Chr(17)

STATIC FUNCTION GivePicks( Get )
	LOCAL lGive := LastKey() == K_F1 .AND. Get # NIL
	IF lGive
		Point2Get()
	ENDIF
RETURN lGive

STATIC FUNCTION Point2Get()
	STATIC cScreen1, cScreen2
   LOCAL Get 		:= GetActive()
   LOCAL nLength  := Len( Trans( Get:varGet(), ;
									IIF( Empty( Get:picture ), "@X", Get:picture ) ) )
   LOCAL nFront   := Get:col - Len( POINT_RIGHT )
   LOCAL nBack    := Get:col + nLength

	IF cScreen1 == NIL // Toggle the pointer on
		cScreen1 := SaveScreen( Get:row, nFront, Get:row, Get:col )
		cScreen2 := SaveScreen( Get:row, nBack, Get:row, nBack + Len( POINT_LEFT ) )
   	@ Get:row, nFront SAY POINT_RIGHT
   	@ Get:row, nBack SAY POINT_LEFT
	ELSE
	   RestScreen( Get:row, nFront, Get:row, Get:col, cScreen1 )
	   RestScreen( Get:row, nBack, Get:row, nBack + Len( POINT_LEFT ), cScreen2 )
		cScreen1 := cScreen2 := NIL
	ENDIF
RETURN cScreen1 == NIL


///////////////
///////////////
//
//  Purpose:  Valid FUNCTION for color pallette
//
//   Syntax:  PickColor( <cColor> )
//
//   Parameters:
//
//      Name        Description
//       
//      cColor      Color setting string
//
//  Returns:  .t. or .f.
//
//  Example:
//
//    Files:  None.
//
// Overview:
//
//    Notes:
//
// Category:
//      Menu Handling
//
// See Also:
//      PickShade() PickExtKey()
//
///////////////
///////////////
STATIC FUNCTION PickColor( cColor )
   LOCAL cTemp
   LOCAL Get   := GetActive()
	LOCAL cSaveColor := SetColor()
	IF GivePicks( Get )
   	cTemp := PadR( ColorPal( cColor ), 6 )
		SetColor( cSaveColor )
		Point2Get()
   	IF lastkey() # 27
   	   AssignGet( cTemp )
   	ENDIF
	ENDIF
RETURN .T.

/////////////////
/////////////////
//
//  Purpose:  Test for background character setting
//
//   Syntax:  IsBackChr( <nChar> )
//
//   Parameters:
//
//      Name        Description
//       
//      nChar       Background character setting
//
//  Returns:  .t.
//
//  Example:
//
//    Files:  None.
//
// Overview:
//
//    Notes:
//
// Category:
//      Menu Handling
//
// See Also:
//
//
/////////////////
/////////////////
STATIC FUNCTION IsBackChr( nChar )
   IF lastkey() == K_F1
     ChrSample()
     RETURN .f.
   ELSEIF nChar < 0 .or. nChar > 255
     w_error( setcolor(), 5, "Enter a value between 0 and 255", ;
              "You can not use that as a background!", "Press a key" )
     RETURN .f.
   ENDIF
RETURN .t.

//////////////////
//////////////////
//
//  Purpose:  Display sample screen backgrounds
//
//   Syntax:  ChrSample()
//
//  No parameters specified
//
//  Returns:  .t.
//
//  Example:  ChrSample()
//
//    Files:  None.
//
// Overview:  Used in configuration program
//
//    Notes:
//
// Category:
//      Menu Handling
//
// See Also:
//
//////////////////
//////////////////
STATIC FUNCTION ChrSample
   LOCAL cColor := setcolor( GFMSet( _GFM_BACKCOLOR ) )

   w_box( 3, 3, 23, 75, "ķӺ ", "Background Samples", ;
         "Press a key to return" )
   @  5, 5 SAY repl('',9)+" "+repl('',9)+" "+repl('',9)+" "+repl('',9)+" "+;
               repl('',9)+" "+repl('',9)+" "+repl('',9)
   @  6, 5 SAY repl('',9)+" "+repl('',9)+" "+repl('',9)+" "+repl('',9)+" "+;
               repl('',9)+" "+repl('',9)+" "+repl('',9)
   @  7, 5 SAY repl('',3)+"176"+repl('',3)+" "+repl('',3)+"177"+repl('',3)+;
               " "+repl('',3)+"178"+repl('',3)+" "+repl('',3)+"179"+;
               repl('',3)+" "+repl('',3)+"196"+repl('',3)+" "+repl('',3)+;
               "186"+repl('',3)+" "+repl('',3)+"205"+repl('',3)
   @  8, 5 SAY repl('',9)+" "+repl('',9)+" "+repl('',9)+" "+repl('',9)+" "+;
               repl('',9)+" "+repl('',9)+" "+repl('',9)
   @  9, 5 SAY repl('',9)+" "+repl('',9)+" "+repl('',9)+" "+repl('',9)+" "+;
               repl('',9)+" "+repl('',9)+" "+repl('',9)
   @ 11, 5 SAY repl('',9)+" "+repl(chr(7),9)+" "+repl(chr(8),9)+" "+;
               repl(chr(9),9)+" "+repl('',9)+" "+repl('',9)+" "+repl('',9)
   @ 12, 5 SAY repl('',9)+" "+repl(chr(7),9)+" "+repl(chr(8),9)+" "+;
               repl(chr(9),9)+" "+repl('',9)+" "+repl('',9)+" "+repl('',9)
   @ 13, 5 SAY repl('',3)+"254"+repl('',3)+" "+repl(chr(7),4)+"7"+;
               repl(chr(7),4)+" "+repl(chr(8),4)+"8"+repl(chr(8),4)+" "+;
               repl(chr(9),4)+"9"+repl(chr(9),4)+" "+repl('',3)+"250"+;
               repl('',3)+" "+repl('',3)+"197"+repl('',3)+" "+repl('',3)+;
               "174"+repl('',3)
   @ 14, 5 SAY repl('',9)+" "+repl(chr(7),9)+" "+repl(chr(8),9)+" "+;
               repl(chr(9),9)+" "+repl('',9)+" "+repl('',9)+" "+repl('',9)
   @ 15, 5 SAY repl('',9)+" "+repl(chr(7),9)+" "+repl(chr(8),9)+" "+;
               repl(chr(9),9)+" "+repl('',9)+" "+repl('',9)+" "+repl('',9)
   @ 17, 5 SAY repl('',9)+" ~~~~~~~~~ "+repl(chr(15),9)+" "+repl(chr(1),9)+" "+;
               repl(chr(2),9)+" "+repl(chr(3),9)+" "+repl(chr(4),9)
   @ 18, 5 SAY repl('',9)+" ~~~~~~~~~ "+repl(chr(15),9)+" "+repl(chr(1),9)+" "+;
               repl(chr(2),9)+" "+repl(chr(3),9)+" "+repl(chr(4),9)
   @ 19, 5 SAY repl('',3)+"175"+repl('',3)+" ~~~126~~~ "+repl(chr(15),4)+"15"+;
               repl(chr(15),3)+" "+repl(chr(1),4)+"1"+repl(chr(1),4)+" "+;
               repl(chr(2),4)+"2"+repl(chr(2),4)+" "+repl(chr(3),4)+"3"+;
               repl(chr(3),4)+" "+repl(chr(4),4)+"4"+repl(chr(4),4)
   @ 20, 5 SAY repl('',9)+" ~~~~~~~~~ "+repl(chr(15),9)+" "+repl(chr(1),9)+" "+;
               repl(chr(2),9)+" "+repl(chr(3),9)+" "+repl(chr(4),9)
   @ 21, 5 SAY repl('',9)+" ~~~~~~~~~ "+repl(chr(15),9)+" "+repl(chr(1),9)+" "+;
               repl(chr(2),9)+" "+repl(chr(3),9)+" "+repl(chr(4),9)
   inkey(0)
   w_Pop()
   setcolor( cColor )
RETURN .t.


/////////////////
/////////////////
//
//  Purpose:  Verify that the shade direction is recognized
//
//   Syntax:  PickShade( @<cShade> )
//
//   Parameters:
//
//      Name        Description
//       
//      cShade      Passed shade direction
//
//  Returns:  .t. or .f.
//
//  Example:
//
//    Files:  None.
//
// Overview:
//      Routine to pick the shade direction
//
//    Notes:
//
// Category:
//      Menu Handling
//
// See Also:
//
/////////////////
/////////////////
STATIC FUNCTION PickShade( cShade )
   LOCAL nR1 := maxrow()/2 - 3
   LOCAL nC1 := maxcol()/2 - 20
   LOCAL nR2 := nR1 + 6
   LOCAL nC2 := nC1 + 40
   LOCAL cTemp := SubStr( cShade, 1, 1 )
   LOCAL cTitle := 'Use the numeric keypad to change the shadow direction'
   LOCAL cFooter := 'The [NumLock] key should be on'
   LOCAL lNumLock := keytest('N')
   LOCAL nKey
   LOCAL Get := GetActive()

	IF GivePicks( Get )
   	KeyToggle( 'N', 'Y' )	// Turn the NumLock key on in case it's off

   	nR1 := maxrow()/2 - 3
   	nC1 := maxcol()/2 - 20
   	nR2 := nR1 + 6
   	nC2 := nC1 + 40
   	w_shade( nR1, nC1, nR2, nC2, frame( 1 ), cTitle, cFooter )
   	@ nR1 + 2, nC1 + 2 say 'Press [Esc] to abort'
   	@ nR1 + 3, nC1 + 2 say 'Press [] to accept'
   	@ nR1 + 4, nC1 + 2 say 'Press "T" for Time Shading'
   	DO WHILE ( nKey := inkey(0) ) # K_ESC .and. nKey # K_ENTER
   	   IF chr( nKey ) $ 'tT0123456789'
   	      cTemp := upper( chr( nKey ) ) + SubStr( cShade, 2 )
   	      w_shadedir( cTemp )
   	      @ w_top(), w_left() say 'Shadow direction: ' + cTemp
   	      w_Pop()
   	      w_shade( nR1, nC1, nR2, nC2, frame( 1 ), cTitle, cFooter )
   	      @ nR1 + 2, nC1 + 2 say 'Press [Esc] to abort'
   	      @ nR1 + 3, nC1 + 2 say 'Press [] to accept'
   	      @ nR1 + 4, nC1 + 2 say 'Press "T" for Time Shading'
   	   ENDIF
   	enddo
   	w_Pop()

		Point2Get()
   	IF nKey # K_ESC
			AssignGet( cShade := cTemp )
		ELSE
   		w_shadedir( cShade )							 	// Re-set shade direction
   	ENDIF

   	KeyToggle( 'N', iif( lNumLock, 'Y', 'N' ) )	// Re-set NumLock key
	ENDIF
RETURN .T.


STATIC FUNCTION PickExtKey( nKey )
   LOCAL nTemp, Get := GetActive()
	IF GivePicks( Get )
   	nTemp := w_error( setcolor(), 0, ;
						'Choose a FUNCTION key or Alt-Key combination~' + ;
						'Use [Space] to set the key to 0 (turn it off)~' + ;
						'Use any other key to leave it the same', ;
						'Please select a FUNCTION key', ;
						'Press a key now' )
   	IF nTemp < 0 .or. nTemp > 255 .or. nTemp == 32
   	   IF nTemp == 32
   	      nTemp := 0
   	   ENDIF
   	   nKey := nTemp
   	ENDIF
		Point2Get()
		IF nTemp # K_ESC
			AssignGet( nKey )
		ENDIF
	ENDIF
RETURN .T.


STATIC FUNCTION GFMGetSet( nSet, uValue )
   GFMSet( nSet, uValue )
RETURN .T.

//////////////
//////////////
//
//  Purpose:  Validate input
//
//   Syntax:  PickList( @<entry>, <acList>, <cDelim>, <cPrompt> )
//
//   Parameters:
//
//      Name        Description
//       
//      entry       Value that was entered
//      acList      List of possible values
//      cDelim      Delimiting character for list - defaults to ','
//      cPrompt     Prompt for the box
//
//  Returns:  .t.
//
//  Example:
//
//    Files:  None.
//
// Overview:
//
//    Notes:
//
// Category:
//      Menu Handling
//
// See Also:
//
//////////////
//////////////
STATIC FUNCTION PickList( cEntry, acList, cDelim, cPrompt )
   LOCAL aChoices := {}, nResponse := 0, nWidth
   LOCAL Get := GetActive(), lOK := .t.
   IF acList == NIL .OR. cEntry == NIL
      RETURN lOK
   ENDIF

	IF GivePicks( Get )
   	DEFAULT cDelim TO ",", cPrompt TO 'Choices'

   	IF ValType( acList ) == 'C'
   	   aChoices := DelimToA( acList, cDelim )
   	ELSE
   	   aChoices := acList	// It's an array already.
   	ENDIF

		// Pad them to the correct length
		nWidth := Max( aMaxStrLen( aChoices ), Len( get:buffer ) )
  	   aEval( aChoices, { | c, n | aChoices[ n ] := PadR( c, nWidth ) } )
		Point2Get()
   	IF ( nResponse := PopChoice( aChoices, cPrompt, "  Esc",,,,,,,;
   	      aScan( aChoices, cEntry ) ) ) # 0
   	   cEntry := aChoices[ nResponse ]
   	   AssignGet( PadR( cEntry, Len( get:buffer ) ) )
   	ENDIF
	ENDIF
RETURN lOK
