/*

   Program: SAVEENV.PRG
   System: GRUMPFISH LIBRARY
   Author: Greg Lief
   Copyright (c) 1988-90, Greg Lief
   Clipper 5.x version
   Compile instructions: clipper saveenv /n/w/a

   Functions: GFSaveEnv(), GFRestEnv()
   Save and restore basic environment items (cursor size & position,
   color, screen contents)

*/

// begin global declarations

#include "grump.ch"

#define TOP      coords[1]
#define LEFT     coords[2]
#define BOTTOM   coords[3]
#define RIGHT    coords[4]

#define NTOP     envstack_[ele, SCREEN][1]
#define NLEFT    envstack_[ele, SCREEN][2]
#define NBOTTOM  envstack_[ele, SCREEN][3]
#define NRIGHT   envstack_[ele, SCREEN][4]
#define NSCREEN  envstack_[ele, SCREEN][5]

#define ROW     1
#define COLUMN  2
#define CURSOR  3
#define COLOR   4
#define SCREEN  5

static envstack_ := {}       // environmental stack

// end global declarations

/*
    Function:   GFSaveEnv()
    Purpose:    Save current cursor row/column/size, color, and screen
    Syntax:     GFSaveEnv( <coords>, <cursorsize>, <color> )
                All parameters are optional
    Parameters: <coords> can be one of three things:
                         NIL -- do not save the screen
                         Array -- four numeric expressions representing
                                  the screen coordinates to be saved
                         Other -- save the entire screen

                <cursorsize> is a numeric representing the new cursor
                size to use.  If not passed, the cursor size will be
                left unchanged. Please refer to SETCURS.CH for the
                appropriate definitions. Use a comma if you want to
                skip this parameter.

                <color> is a character string representing the new color
                setting to use. If not passed, the color will be left
                unchanged.

    Examples:   #include "setcurs.ch"
                gfsaveenv(, SC_NONE, 'w/b')

             Do not save screen, turn off cursor, change color to W/B

                gfsaveenv(.t., , 'r/w')

             Save full screen, don't change cursor, change color to R/W

                gfsaveenv( { 10, 10, 14, 69 }, SC_NONE)

             Save screen buffer between @ 10, 10 and @ 14, 69, turn off
             cursor, leave color unchanged.
*/
function GFSaveEnv(scrn_save, curs_size, newcolor)
local coords := if(scrn_save == NIL .or. valtype(scrn_save) == "A", ;
                scrn_save, { 0, 0, maxrow(), maxcol() } )
aadd(envstack_, { row(), col(), setcursor(curs_size), setcolor(newcolor), ;
     if(valtype(coords) == "A", { TOP, LEFT, BOTTOM, RIGHT, ;
        savescreen(TOP, LEFT, BOTTOM, RIGHT) }, NIL) } )
return len(envstack_)

* end function GFSaveEnv()
*--------------------------------------------------------------------*


/*
       GFRestEnv(): restore previous environmental settings
*/
function GFRestEnv(ele)
// use LIFO (last item in array) if no parameter was passed
default ele to len(envstack_)
// preclude empty array!
if len(envstack_) > 0 .and. ele <= len(envstack_)
   // reset row/column position
   setpos(envstack_[ele, ROW], envstack_[ele, COLUMN])
   // reset cursor state
   setcursor(envstack_[ele, CURSOR])
   // reset color
   setcolor(envstack_[ele, COLOR])
   // restore screen if it was saved
   if envstack_[ele, SCREEN] != NIL
      restscreen(NTOP, NLEFT, NBOTTOM, NRIGHT, NSCREEN)
   endif
   // truncate length of array only if using LIFO, i.e., no param passed
   if ele == len(envstack_) .and. pcount() == 0
      truncate(envstack_)
   endif
endif
return NIL

* end function GFRestEnv()
*--------------------------------------------------------------------*

* eof saveenv.prg
