/*
    Program: IsPostCode()
    System: GRUMPFISH LIBRARY
    Author: Greg Lief
    Copyright (c) 1988-90, Greg Lief
    Clipper 5.0 Version
    Compile instructions: clipper postcode /n/w/a

    Provide basic validation for Canadian postal codes
       1) verify that length is seven
       2) verify that format is X#X #X#
       3) check first character against array of territory codes

    Calls: Err_Msg()      (function in ERRMSG.PRG)
*/
function ispostcode(cstate, cpostalcode)
static postcodes_ := { "NFANewfoundland", "NSBNova Scotia", ;
                       "PECPrince Edward Island", "NBENew Brunswick", ;
                       "PQGQuebec (Eastern)", "PQHMontreal", ;
                       "PQJQuebec (Western)", "ONKOntario (Eastern)", ;
                       "ONLOntario (Central)", "ONMToronto", ;
                       "ONNOntario (Southwestern)", "ONPOntario (Northern)", ;
                       "MBRManitoba", "SKSSaskatchewan", "ABTAlberta", ;
                       "BCVBritish Columbia", "NTXNorthwest Territories", ;
                       "YTYYukon" }
local ret_val := .f., mpostcode, ele, ptr, improv
static mnumbers := '0123456789', badletters := 'DFIOQU'
do case
   // first check length - must be 7
   case len(trim(cpostalcode)) != 7
      err_msg( { "Canadian postal codes must be 7 characters long!" } )
   case ! (isalpha(substr(cpostalcode, 1, 1)) .and. ;
           substr(cpostalcode, 2, 1) $ mnumbers .and. ;
           isalpha(substr(cpostalcode, 3, 1)) .and. ;
           substr(cpostalcode, 4, 1) == [ ] .and. ;
           substr(cpostalcode, 5, 1) $ mnumbers .and. ;
           isalpha(substr(cpostalcode, 6, 1)) .and. ;
           substr(cpostalcode, 7, 1) $ mnumbers)
        err_msg( { "Canadian postal codes must be of the format 'X#X #X#'" } )
   otherwise
      // the letters D/F/I/O/Q/U cannot appear in the postal code -
      // check that first
      cpostalcode := upper(cpostalcode)
      if left(cpostalcode, 1) $ badletters            .or. ;
               substr(cpostalcode, 3, 1) $ badletters .or. ;
               substr(cpostalcode, 6, 1) $ badletters
         err_msg( { "Invalid postal code" } )
      else
         // up to this point everything checks out okay... now make
         // sure that the first character matches the geographic area
         ele := ascan(postcodes_, upper(cstate))
         if ele > 0
            // only use first word of the province -- i.e., "Ontario"
            // rather than "Ontario (eastern)" -- because if they
            // entered the wrong first digit, we don't really know
            // which section they wanted
            if ( ptr := at('(', postcodes_[ele]) ) > 0
               improv := substr(postcodes_[ele], 4, ptr - 5)
            else
               improv := substr(postcodes_[ele], 4)
            endif
            do while cstate == substr(postcodes_[ele], 1, 2) .and. ;
                     substr(cpostalcode, 1, 1) != substr(postcodes_[ele], 3, 1)
               ele++
            enddo
            if substr(cpostalcode, 1, 1) != substr(postcodes_[ele], 3, 1)
               err_msg( { "Invalid postal code for " + improv } )
            else
               ret_val := .t.
            endif
         else
            err_msg( { "Invalid province code" } )
         endif
      endif
endcase
return ret_val

* end function IsPostCode()
*--------------------------------------------------------------------*

* eof postcode.prg
