/*
    Program: POPPHONE.PRG
    System: GRUMPFISH LIBRARY
    Author: Greg Lief
    Copyright (c) 1988-90, Greg Lief
    Clipper 5.01 Version
    Compile instructions: clipper popphone /n/w/a
    Procs & Fncts: PHONEMAINT()
                 : PRINTPHONE()
                 : PH_HEAD()
                 : DIALER()
                 : MODEMSETUP()
                 : PHONEBOOK()
                 : FRONTPAGE()

            Calls: ERR_MSG()     (function in ERRMSG.PRG)
                 : YES_NO2()     (function in YESNO2.PRG)
                 : HELPACT()     (function in HELPBROW.PRG)
                 : PRINTOK()     (function in PRINTOK.PRG)
                 : SHADOWBOX()   (function in SHADOWB.PRG)
                 : WAITON()      (function in WAITON.PRG)
                 : WAITOFF()     (function in WAITON.PRG)

             Uses: PHONE.DBF

          Indexes: PHONE.NTX
*/

#include "grump.ch"
#include "inkey.ch"
#include "error.ch"
#include "fileio.ch"
#include "setcurs.ch"


#define PBXFLAG      gfphone[1]
#define HANGUP       gfphone[2]
#define MODEMINIT    gfphone[3]
#define TONEDIAL     gfphone[4]
#define COMMPORT     gfphone[5]
#define AREACODE     gfphone[6]
#define MODEMDELAY   gfphone[7]
#define CONFIGFILE   "gfphone.cfg"

/*
   if I left the following as a bona fide function, that would mean that
   the local variables LSEARCHING and CSEARCHSTR would have to be made
   file-wide static, and that disturbed me, so here we are...
*/
#xtranslate endsearch() => lSearching := .f. ; ;
            cSearchStr := ''  ; ;
            setpos(oBrowse:nBottom + 1, (maxcol() + 1) / 2 - 11) ; ;
            dispout(replicate(chr(205), 22))

static gfphone := { .F., 'ATH', 'ATZ', 'T', '1', '   ', 1.0 }
static redialnum                         // last number dialed (for redialing)
static headings_ := { { "First Name", "!XXXXXXXXXXXXXX"} , ;
                      { "Last Name",  "!XXXXXXXXXXXXXX"} , ;
                      { "Phone", "###-###-####"}         , ;
                      { "Ext", "####" }                  , ;
                      { "Fax", "###-###-####"}           , ;
                      { "Mobile", "###-###-####"}        , ;
                      { "Company", "!XXXXXXXXXXXXXXXXXXX"} , ;
                      { "Address", "!XXXXXXXXXXXXXXXXXXXXXXXX" } , ;
                      { "City", "!XXXXXXXXXXXXXX" }      , ;
                      { "St",  "@!" }                    , ;
                      { "Zip" , "@!" }                   , ;
                      { "LD", "Y" } }


function popphone(gfproc, line, var)
local nHotKey := 0
local bOldblock
local cFile
local cSearchStr := []
local xx
local key
local marker
local maincolor := ColorSet(C_PHONEBOOK_WINDOW1, .T.)
local lSearching := .f.
local lOldDelete := set(_SET_DELETED, .T.)
local oBrowse
local nWorkArea := select()
local lOldScore := set(_SET_SCOREBOARD, .f.)
local lHad2Open := .f.
local nFields
memvar phonedir      // global PUBLIC that may have been set in calling program


// if the file GFPHONE.CFG exists, read in previously saved modem settings
if file(CONFIGFILE)
   if loadarray(xx, CONFIGFILE)
      gfphone := xx
   endif
endif

// determine whether this was called via hot-key; if so, disable it
if (gfproc <> NIL)
   bOldblock := setkey(nHotKey := lastkey(), NIL)
endif

// open phone.dbf... first determine path, then confirm existence of file
cFile := if(type('phonedir') == 'U', '', phonedir + '\') + 'phone'
if ! file(cFile + '.dbf')
   waiton('Initializing phone database... please wait')
   dbcreate(cFile + ".dbf", { {"FNAME",   "C", 15, 0} , ;
                              {"LNAME",   "C", 15, 0} , ;
                              {"PHONE_NO","C", 12, 0} , ;
                              {"EXT",     "C",  4, 0} , ;
                              {"FAX",     "C", 12, 0} , ;
                              {"MOBILE",  "C", 12, 0} , ;
                              {"COMPANY", "C", 20, 0} , ;
                              {"ADDRESS", "C", 25, 0} , ;
                              {"CITY",    "C", 15, 0} , ;
                              {"STATE",   "C",  2, 0} , ;
                              {"ZIP",     "C", 10, 0} , ;
                              {"LONGDIST","L",  1, 0} } )
   waitoff()
endif
if ! file(cFile + indexext())
   use (cFile) new exclusive
   index on upper(phone->lname + phone->fname) to (cFile)
   use
endif

// only open PHONE.DBF if it is not already open!
if select("phone") == 0
   lHad2Open := .t.
   use (cFile) new shared index (cFile)
else
   if phone->( indexkey(0) ) <> "upper(phone->lname + phone->fname)"
      err_msg({ "Missing index, aborting (POPPHONE)" } )
      return nil
   else
      select phone
   endif
endif

lOldDelete := set(_SET_DELETED, .T.)
lOldScore := set(_SET_SCOREBOARD, .f.)
GFSaveEnv(.t., SC_NONE)    // shut off cursor
GFSaveGets()

// verify that PHONE.DBF contains MOBILE and FAX number fields
// which were added with version 4.0 -- if not, then remove them
// from the static HEADINGS_ array
if fieldpos("FAX") == 0
   adel(headings_, 6)
   adel(headings_, 5)
   asize(headings_, len(headings_) - 2)
endif

// create new browse object
oBrowse := TBrowseDB( 6, 3, maxrow() - 6, 75)
oBrowse:headSep := ""
oBrowse:colorSpec := maincolor + ',' + "+W/N"
nFields := fcount()
for xx := 1 to nFields
   oBrowse:addColumn( TBColumnNew( headings_[xx, 1], fieldblock( field(xx) ) ) )
next
oBrowse:freeze := 2   // freeze first two columns (fname and lname)
xx := maxrow()
scroll(xx - 2, 00, maxrow(), maxcol(), 0)
@ xx - 1,08 ssay 'phonebook'
@ xx - 1,26 ssay 'list'
@ xx - 1,39 ssay 'dial phone'
@ xx - 1,58 ssay 'redial'
@ xx - 1,73 ssay 'setup'
@ xx,    03 ssay "Add"
@ xx,    14 ssay "Edit"
@ xx,    26 ssay "Delete"
@ xx,    40 ssay "Search"
@ xx,    55 ssay "to move"
@ xx,    72 ssay "to exit"
setcolor('i')
@ xx - 1,02 ssay 'Alt-B'
@ xx - 1,20 ssay 'Alt-L'
@ xx - 1,33 ssay 'Alt-P'
@ xx - 1,52 ssay 'Alt-R'
@ xx - 1,67 ssay 'Alt-S'
@ xx,    01 ssay "A"
@ xx,    12 ssay "E"
@ xx,    24 ssay "D"
@ xx,    38 ssay "S"
@ xx,    52 ssay CHR(24)+CHR(25)
@ xx,    68 ssay "Esc"
setcolor(maincolor)
shadowbox(oBrowse:nTop - 1, oBrowse:nLeft - 1, oBrowse:nBottom + 1, ;
          oBrowse:nRight + 1, 1, "Phonebook" + ;
          if(type("PHONEUSER") == "C", " for " + m->phoneuser, "") )
do while key <> K_ESC
   dispbegin()
   do while (key := inkey()) == 0 .and. ! oBrowse:stabilize()
   enddo
   dispend()
   if oBrowse:stable
      key := ginkey(0, "KEY")
   endif
   // deal with the keypress
   do case
      case key == K_UP
         endsearch()
         oBrowse:up()
      case key == K_LEFT
         oBrowse:left()
      case key == K_RIGHT
         oBrowse:right()
      case key == K_DOWN
         endsearch()
         oBrowse:down()
      case key == K_CTRL_PGUP
         endsearch()
         oBrowse:goTop()
      case key == K_CTRL_PGDN
         endsearch()
         oBrowse:goBottom()
      case key == K_CTRL_LEFT
         oBrowse:panLeft()
      case key == K_CTRL_RIGHT
         oBrowse:panRight()
      case key == K_PGUP .or. key == K_HOME
         endsearch()
         oBrowse:pageUp()
      case key == K_PGDN .or. key == K_END
         endsearch()
         oBrowse:pageDown()
      case key == K_ENTER .and. lSearching       // end search mode
         endsearch()
      case key == K_ENTER .and. oBrowse:stable
         if rec_lock() .and. editcell(oBrowse)
            // if key field was edited, refresh entire screen
            if oBrowse:colPos < 3
               oBrowse:refreshAll()
            else
               oBrowse:refreshCurrent()
            endif
            unlock
         endif
      case lSearching .and. isalpha(chr(key))   // letter key - search 'em, Dan-O
         marker := recno()
         if dbseek( cSearchStr + upper(chr(key)) )
            cSearchStr += upper(chr(key))
            @ oBrowse:nBottom+1, (maxcol()+1)/2 - 10 ssay padc(cSearchStr, 20)
            oBrowse:refreshAll()
         else
            go marker
         endif
      case (key == 65 .or. key == 97)  .and. oBrowse:stable   // add
         endsearch()
         PhoneMaint("A")
         oBrowse:refreshAll()
      case (key == 69 .or. key == 101) .and. oBrowse:stable   // edit
         endsearch()
         PhoneMaint("E")
         oBrowse:refreshAll()
      case (key == 68 .or. key == 100) .and. oBrowse:stable   // delete
         endsearch()
         if yes_no('This record will be deleted from the file',;
                   'Do you want to do this')
            if rlock()
               delete
               skip -1
            else
               err_msg( { NETERR_MSG } )
            endif
         endif
         oBrowse:refreshAll()

      case key == 83 .OR. key == 115        // search
         lSearching := .t.
         @ oBrowse:nBottom+1, (maxcol()+1) / 2 - 11 ssay "[" + space(20) + "]"

      case key == K_BS .and. lSearching     // truncate the search string
         if len(cSearchStr) > 0
            cSearchStr := substr(cSearchStr, 1, len(cSearchStr) - 1)
            seek cSearchStr
            if len(cSearchStr) > 0
               @ oBrowse:nBottom+1, (maxcol()+1)/2 - 10 ssay padc(cSearchStr, 20)
            else
               endsearch()
            endif
            oBrowse:refreshAll()
         endif

      case key == K_ALT_L               // print list
         endsearch()
         printphone(cFile)

      case key == K_ALT_B               // phone book
         endsearch()
         phonebook()

      case key == K_ALT_P               // dial phone #
         endsearch()
         // if MOBILE field exists, offer that as a choice
         if fieldpos("MOBILE") > 0
            xx := yes_no2({ "Dial which number?" },," Voice "," Mobile ")
            if lastkey() <> K_ESC
               Dialer(if(xx, phone->phone_no, phone->mobile))
            endif
         else
            Dialer(phone->phone_no)
         endif

      case key == K_ALT_R               // redial
         endsearch()
         if redialnum == NIL
            Err_Msg( { 'No redial number has been established yet' } )
         else
            Dialer(redialnum)
         endif

      case key == K_ALT_S               // modem setup
         ModemSetup()

   endcase
enddo
// clean up
if lHad2Open
  dbCloseArea()
endif
select(nWorkArea)

// restore hot-key
if nHotKey <> 0
   setkey(nHotKey, bOldblock)
endif
GFRestEnv()
GFRestGets()
set(_SET_SCOREBOARD, lOldScore)  // go ahead and keep score if you must
set(_SET_DELETED, lOldDelete)
return NIL

* end function PopPhone()
*--------------------------------------------------------------------*


/*
  EditCell(): edit one field
*/
static function EditCell(b)
local c := b:getColumn(b:colPos)
local bOldinsert := setkey( K_INS, {|| setcursor( ;
                   if(readinsert(! readInsert()), SC_NORMAL, SC_SPECIAL1))} )
// set initial cursor size based on current mode
setcursor( if(readInsert(), SC_SPECIAL1, SC_NORMAL) )
readmodal( { getnew(Row(), Col(), c:block, c:heading, '@K', b:colorSpec) } )
setcursor(SC_NONE)
setkey(K_INS, bOldinsert)
return (lastkey() <> K_ESC)

* end static function EditCell()
*--------------------------------------------------------------------*


/*
  PhoneMaint(): add/edit records in phone.dbf
*/
static function phonemaint(mode)
local num_flds := fcount()
local ahold := array(num_flds)
local marker := recno()
local app_ok
local xx
local getlist := {}
local nTop := int( maxrow() - num_flds) / 2 - 1
gfsaveenv( .t., SC_NORMAL)
scroll(maxrow() - 2, 00, maxrow(), maxcol(), 0)
xx := maxrow()
@ xx - 1, 25 ssay 'save edits'
@ xx - 1, 43 ssay 'exit without saving'
@ xx - 1, 18 ssay 'Ctrl-W' color 'I'
@ xx - 1, 39 ssay 'Esc' color 'I'
ColorSet(C_PHONEBOOK_WINDOW2)
ShadowBox(nTop, 18, nTop + num_flds + 1, 61, 2, ;
          if(mode == "A", "Add New", "Edit") + " Record")

// use the phantom record to grab initial values if adding
if mode == 'A'
   go 0
endif

// dump field contents to an array
for xx := 1 to num_flds
   ahold[xx] := fieldget(xx)
   @ nTop + xx, 20 ssay headings_[xx, 1]
   @ nTop + xx, 33 get ahold[xx] picture headings_[xx, 2]
next
setcursor(SC_INSERT)
read
setcursor(SC_NONE)
if lastkey() <> K_ESC
   if mode == 'A'      // adding record
      append blank
      app_ok := ! neterr()
   else
      app_ok := rlock()
   endif
   if app_ok
      // now dump array contents to the fields of the blank record
      for xx := 1 to num_flds
         fieldput(xx, ahold[xx])
      next
   else
      err_msg( { NETERR_MSG } )
   endif
elseif mode == 'A'
   go marker
endif
gfrestenv()
return NIL

* end static function PhoneMaint()
*--------------------------------------------------------------------*


/*
   PrintPhone(): print phone list
*/
static function PrintPhone(mindex)
local buffer
local nOrder := 1
local getlist := {}
local lFax
local lMobile
ColorSet(C_MESSAGE)
buffer := shadowbox(19, 10, 22, 69, 2, 'List options')
@ 20,12 ssay 'Sort list by (1) last name, (2) company, (3) phone no'
@ 21,22 ssay '(press ESC to exit without printing)'
@ 20,67 get nOrder picture '#'
setcursor(1)
read
setcursor(0)
byebyebox(buffer)
if lastkey() <> K_ESC
   // verify that the FAX and MOBILE fields exist
   lFax := ( fieldpos("FAX") > 0 )
   lMobile := ( fieldpos("MOBILE") > 0 )
   if nOrder == 2
      index on upper(phone->company) to phonetmp
   elseif nOrder == 3
      index on phone->phone_no to phonetmp
   endif
   go top
   waiton()
   if printok()
      Ph_Head(.t., lFax, lMobile)
      do while ! eof()
         if prow() > 59
            Ph_Head(.f., lFax, lMobile)
         endif
         @ prow()+2,1 say phone->company
         @ prow(),22  say trim(phone->fname) + ' ' + phone->lname
         @ prow(),54  say phone->phone_no
         if lMobile
           @ prow(), 68 say phone->mobile
         endif
         @ prow()+1,1 say phone->address
         @ prow(),27  say trim(phone->city) + ' ' + phone->state + ' ' + ;
                          phone->zip
         if lFax
            @ prow(),58  say phone->fax
         endif
         @ prow(),74  say phone->ext
         skip
      enddo
      eject
   endif
   waitoff()
   set device to screen
   set index to (mindex)
   ferase('phonetmp' + indexext())
endif
return NIL

* end static function PrintPhone()
*--------------------------------------------------------------------*


/*
  Ph_Head(): heading for phone list report
*/
static function Ph_Head(lReset, lFax, lMobile)
memvar phonehead   // global that may have been set in calling program
static nPage := 1
if lReset
   nPage := 1
endif
@ 0,1 say dtoc(date())
CENTER(0, if(type('phonehead') == 'U', 'Phone List', phonehead))
@ 0,71 say 'Page ' + ltrim(str(nPage++))
@ prow()+2,1 say 'Company Name'
@ prow(),22  say 'Contact Name'
@ prow(),54  say 'Phone Number'
if lMobile
   @ prow(), 68 say 'Mobile'
endif
@ prow()+1,1 say 'Address'
@ prow(),27  say 'City  State  Zip'
if lFax
   @ prow(),58  say 'Fax Number'
endif
@ prow(),74  say 'Ext.'
@ prow()+1,1 say replicate('-',79)
return NIL

* end static function Ph_Head()
*--------------------------------------------------------------------*


/*
  Dialer(): self-explanatory
*/
static function Dialer(numtodial)
local numbr, xx, yy, oldcolor, buffer, mareacode, oldcons, nhandle
redialnum := numtodial   // save this number for redialing
mareacode := substr(numtodial, 1, 3)

// see if this area code matches the local area code
if mareacode == AREACODE
   numtodial := if(phone->longdist, '1-', '') + substr(numtodial, 5)
endif
numtodial := ltrim(trim(numtodial))

// clean up the phone number
if len(numtodial) == 12
   numtodial := '1-' + numtodial
elseif substr(numtodial, 1, 1) == '-'
   numtodial := substr(numtodial, 2)
endif
yy := len(numtodial)
numbr := strtran(numtodial, '-', '')

// add the requisite half second pause if using a pbx system
nhandle := fopen('com' + COMMPORT, FO_WRITE)
if nhandle == F_ERROR
   err_msg( { "Could not open COM " + COMMPORT } )
else
   numbr := if(PBXFLAG, "9 ~ ", "") + numbr
   oldcolor := ColorSet(C_MESSAGE)
   buffer := shadowbox(19, 12, 22, 67, 2)
   @ 21,14 ssay "Wait for ring, then pick up phone and press spacebar"
   SCRNCENTER(20, "dialing " + numtodial, '+' + setcolor())
   oldcons := set(_SET_CONSOLE, .F.)
   fwrite(nhandle, '+++' + chr(13))
   inkey(MODEMDELAY)
   fwrite(nhandle, trim(MODEMINIT) + chr(13))
   inkey(MODEMDELAY)
   fwrite(nhandle, 'ATD' + TONEDIAL + numbr + chr(13))
   inkey(0)
   fwrite(nhandle, trim(HANGUP) + chr(13))
   ByeByeBox(buffer)
   setcolor(oldcolor)
   set(_SET_CONSOLE, oldcons)
   fclose(nhandle)
endif
return NIL

* end static function Dialer()
*--------------------------------------------------------------------*


/*
   ModemSetup(): establish/edit modem parameters
*/
static function ModemSetup
local oldscrn, oldcolor := ColorSet(C_PHONEBOOK_WINDOW2)
local getlist := {}
oldscrn := shadowbox(08, 18, 16, 62, 2, 'Communications Parameters')
HANGUP    := padr(HANGUP, 20)
MODEMINIT := padr(MODEMINIT, 20)
@ 09, 20 ssay 'PBX (y/n)'
@ 10, 20 ssay 'Modem set-up string'
@ 11, 20 ssay 'Modem hang-up string'
@ 12, 20 ssay '[T]one / [P]ulse'
@ 13, 20 ssay 'COM port'
@ 14, 20 ssay 'Local area code'
@ 15, 20 ssay 'Timing delay'
@ 09, 41 get PBXFLAG picture "Y"
@ 10, 41 get MODEMINIT
@ 11, 41 get HANGUP
@ 12, 41 get TONEDIAL picture '!' valid TONEDIAL $ 'PT'
@ 13, 41 get COMMPORT valid COMMPORT $ '12345'
@ 14, 41 get AREACODE picture '###'
@ 15, 41 get MODEMDELAY picture '#.#'
setcursor(1)
read
setcursor(0)
HANGUP    := trim(HANGUP)
MODEMINIT := trim(MODEMINIT)
ByeByeBox(oldscrn)
setcolor(oldcolor)
// if user did not escape out, ask for confirmation and write GFPHONE.CFG
if lastkey() <> K_ESC
   if yes_no2( { "Save these modem settings?" }, maxrow()/2, ;
                 " Save ", " Don't Save ")
      if if(! file(CONFIGFILE), .T., ;
             yes_no2( { "Overwrite configuration file?" } , ;
                       maxrow() / 2, " Overwrite ", " Cancel "))
         gsavearray(gfphone, CONFIGFILE)
      endif
   endif
endif
return NIL

* end static function ModemSetup()
*--------------------------------------------------------------------*


/*
   PhoneBook(): prints pocket-sized phone directory
*/
static function PhoneBook
local xx
local yy
local cLetter
local lNewLetter
local cName
local cCompany
local cAddress
local cCity
local nCurrLine := 1
local pages := 1
local maxrec
local nPage
local aTemp := {}
local nCurrElem
local mlines
local lines
local nEle
local lLaserjet
local lFax := ( fieldpos("FAX") > 0 )
local lMobile := ( fieldpos("MOBILE") > 0 )
lLaserjet := yes_no2( { "What type of printer are you using?" }, ;
                     maxrow() / 2, " HP Laserjet ", " IBM Graphics ")
if lastkey() <> K_ESC
   waiton("Creating Phone Directory")
   go top
   cLetter := chr(32)
   do while ! eof()
      lNewLetter := .f.   // flag set true if alpha category changes
      // determine person and company names to use for this entry
      if len(trim(phone->lname)) > 0
         cName := ltrim(trim(phone->lname)) + ;
                  if(len(trim(phone->lname)) > 0, ', ' + ;
                  ltrim(trim(phone->fname)), [])
         cCompany := space(3) + ltrim(trim(phone->company))
      else
         cName := ltrim(trim(phone->company))
         cCompany := []
      endif

      // tack voice number onto name line
      cName += space(29 - len(cName)) + ;
              if(val(phone->phone_no) == 0, "   No Number", ;
                 "V: " + phone->phone_no)

      // tack fax number onto company line
      if lFax .and. val(phone->fax) > 0
         cCompany += space(29 - len(cCompany)) + "F: " + ;
                     phone->fax
      endif

      cAddress := space(3) + ltrim(trim(phone->address))

      // tack mobile number onto company line
      if lMobile .and. val(phone->mobile) > 0
         cAddress += space(29 - len(cAddress)) + "M: " + ;
                     phone->mobile
      endif

      cCity := space(3) + ltrim(trim(phone->city)) + " " + ;
               ltrim(trim(phone->state)) + " " + ;
               ltrim(trim(phone->zip))
      lines := 1

      // did alphabetical category change?
      if cLetter <> substr(phone->lname, 1, 1)
         cLetter := substr(phone->lname, 1, 1)
         lines++
         lNewLetter := .t.
      endif
      // determine how many lines this entry will need, so that we
      // can thus determine whether or not they will fit on this page
      lines += if(len(cCompany) > 1, 1, 0) + ;
               if(len(cAddress) > 1, 1, 0) + ;
               if(len(cCity) > 3, 1, 0)
      // this entry will go on next page - add blank lines to complete page
      if lines + nCurrLine > 22
         pages++
         for xx := nCurrLine to 22
            aadd(aTemp, '')
         next
         nCurrLine := 1
      endif
      // if we started a new alpha category, must put the heading in now
      if lNewLetter
         nCurrLine++
         aadd(aTemp, space(20) + '- ' + cLetter + ' -')
      endif
      nCurrLine++
      aadd(aTemp, cName)
      if len(cCompany) > 1
         nCurrLine++
         aadd(aTemp, cCompany)
      endif
      if len(cAddress) > 1
         nCurrLine++
         aadd(aTemp, cAddress)
      endif
      if len(cCity) > 3
         nCurrLine++
         aadd(aTemp, cCity)
      endif
      skip
   enddo
   // add enough blank records in temp file to make last page full length
   if nCurrLine < 22
      // also, it will be a heck of a lot easier if we have an
      // even number of pages, so we'll check on that right now
      mlines := (23 - nCurrLine) + if(pages%2 <> 0, 22, 0)
      // increment number of pages if we previously had an odd number
      if pages % 2 <> 0
         pages++
      endif
      for xx := 1 to mlines
         aadd(aTemp, '')
      next
   endif

   // now begins the arduous task of printing the book
   if printok()
      devout(if(lLaserjet, chr(27) + "&k2S", chr(15)))
      maxrec := len(aTemp) / 2
      nPage := 0
      frontpage()
      nCurrElem := 1
      do while nCurrElem < maxrec
         nPage++
         for xx := 1 to 27
            @ prow()+1, 0 say "."
            if xx == 1 .or. xx == 27
               for yy := 1 to 46
                  @ prow(), pcol()+1 say "."
               next
            else
               if xx == 26
                  @ prow(), 12 say nPage picture '###'
                  @ prow(), 68 say pages + 1 - nPage picture '###'
               else
                  if xx > 2 .and. xx < 25
                     @ prow(), 2 say aTemp[nCurrElem]
                  endif
                  if (xx > 8 .and. xx < 12) .or. (xx > 16 .and. xx < 20)
                     @ prow(), 47 say "|"
                  endif
                  if xx > 2 .and. xx < 25
                     if (nEle := len(aTemp) - ;
                                  (22 * (int(nCurrElem / 22) + 1)) + ;
                                  (nCurrElem % 22) ) <> 0
                        @ prow(), 49 say aTemp[nEle]
                     endif
                     nCurrElem++
                  endif
               endif
            endif
            @ prow(), 94 say "."
         next
         // since we are putting two pages per sheet,
         // only eject every other time
         if nPage % 2 <> 0
            eject
         endif
      enddo
      devout(if(lLaserjet, chr(27) + "&k0S", chr(18)))
      eject
      set device to screen
   endif
   waitoff()
endif
return NIL

* end static function PhoneBook()
*--------------------------------------------------------------------*


/*
   FrontPage(): print phone page of phone directory
*/
static function FrontPage
local cUserName
local mdate, xx, yy
if type("PHONEUSER") == "C"
   cUserName := m->phoneuser
endif
for xx := 1 to 27
   @ prow()+1, 0 say "."
   do case
      case xx == 1 .or. xx == 27
         for yy := 1 to 46
            @ prow(), pcol()+1 say "."
         next
      // draw lines for staples in middle of page
      case (xx > 8 .and. xx < 12) .or. (xx > 16 .and. xx < 20)
         @ prow(), 47 say "|"
   endcase
   do case
      case xx == 11
         @ prow(), 61 say "Telephone Directory"
      case xx == 13 .and. cUserName <> NIL
         @ prow(), 69 say "for"
      case xx == 15 .and. cUserName <> NIL
         @ prow(), 70 - int( len(cUserName) / 2 ) say cUserName
      case xx == 17
         mdate := cmonth(date()) + ' ' + ltrim(str(day(date()))) + ', ' + ;
                  ltrim(str(year(date())))
         @ prow(), 70 - int( len(mdate) / 2 ) say mdate
   endcase
   @ prow(), 94 say "."
next
return []

* end static function FrontPage()
*--------------------------------------------------------------------*

* eof popphone.prg
