/*
    Program: POPNOTE.PRG
    System: GRUMPFISH LIBRARY
    Author: Greg Lief
    Copyright (c) 1988-93, Greg Lief
    Clipper 5.x Version

    NOTE: MEMOEDIT() cannot be made mouseable because it is still a
          "closed-off" wait state... sorry.
    Compile instructions: clipper popnote /n /w

    NOTE: To use this in conjunction with Grumpfish Speller, recompile
          this file with the following syntax:

          clipper popnote /n /w /dSPELLER

          Then be sure to link in SPELL.LIB.

          The ALT-Z keypress will be used to invoke the spellchecker
          (sorry, ALT-S is already taken).  If you want to use a
          different key, look for and change the manifest constants
          SPELLER_KEY and SPELLER_KEY_HELP_PROMPT to whatever you
          desire.  These can be found just below this file header.

    Procs & Fncts: FILEWRITE()
                 : NOTEPAD_HD
                 : GRUMPFUN()
                 : NOTEHELP()
                 : GETVAR()
                 : PROBLEM()
                 : NEWFILE()
                 : STAT_MSG()
                 : SETTINGS()
                 : GETFILE()
*/

#ifdef SPELLER
   #define SPELLER_KEY               K_ALT_Z
   #define SPELLER_KEY_HELP_PROMPT   "Alt-Z"
#endif

#include "grump.ch"
#include "inkey.ch"
#include "fileio.ch"
#include "memoedit.ch"

static pastebuff := []            // buffer for pasting
static cLastFile  := []           // last file edited
static leftmargin := 1, linewidth := 80, rightmargin := 80, pagelength
static blockmarks                 // number of block markers currently placed
static blocksel                   // selected block command
static aOperations := { 'Move block', 'Copy block', 'Kill block', 'Print block', ;
                        'Uppercase', 'Lowercase', 'Erase markers' }
static aFields := {}              // array to hold fields for insertion
static oldscrn                    // entry screen
static nRelCol, nRelRow           // offset row/col position for memoedit()
static nStartRow                  // starting row for memoedit()
static wordwrap := .t.            // wordwrap status
static cDir                       // current search directory
static cWildCard                  // current search wildcard pattern
static memoedkey                  // last key -- must be visible in main
static filename                   // current file being edited
static lFileChange                // flag for whether current file was changed
static nLines                     // total lines in memo--used for elevator bar

// manifest constants for blocksel values
#define MOVEBLOCK     1
#define COPYBLOCK     2
#define KILLBLOCK     3
#define PRINTBLOCK    4
#define UPPERBLOCK    5
#define LOWERBLOCK    6
#define ERASEMARKERS  7

// pseudo-function to strip directory from filename
#xtranslate StripDir(<file>) => trim(substr(<file>, rat('\', <file>) + 1))

// pseudo-function to save current position in the memo
#xtranslate SavePosition() => nStartRow := nRelRow := gfline ; nRelCol := gfcol

function popnote(gfproc, line, var)
local nHotKey
local aTemp
local nWorkArea
local nMaxArea
local xx
local yy
local mfile
local lOldScore
local nFields
local cWorkFile
local nMarker1
local nMarker2
local cTemp
local cSearch
local cReplace
local copyblock
local oldtext
local condblock
local nHandle
local nMarker
local bOldblock
local lOldInsert := set(_SET_INSERT)
memvar curr_dir, wildcard, notefile  // globals possibly declared elsewhere

// determine whether this was called via hot-key; if so, disable it
if (gfproc <> NIL)
   bOldblock := setkey(nHotKey := lastkey(), NIL)
endif
GFSaveEnv()
oldscrn := savescreen(0, 0, maxrow(), maxcol())
// clear out aFields array from last time
asize(aFields, 0)

// initialize fields array if a database is open (fcount() > 0)
if (nFields := fcount()) > 0
   nWorkArea := select()      // save current work area
   select 0
   nMaxArea := select() - 1   // determine highest numbered active work area
   nFields := 0              // accumulator for total number of fields
   for xx = 1 to nMaxArea
      aTemp := (xx)->(dbstruct())   // add # of fields in that work area
      yy := 0
      // loop through this array and add each element to aFields array
      // preface each field name by the alias name
      aeval(aTemp, { | a | yy++, ;
                   aadd(aFields, (xx)->(alias()) + '->' + a[1]) } )
   next
   select(nWorkArea)         // pop back to the current work area
endif

// if global variable CURR_DIR exists, use that as default directory
cDir := IF(type('curr_dir') == "U", '', ;
           curr_dir + if(right(curr_dir, 1)<>'\','\',''))

// if global variable WILDCARD exists, use that as default wildcard
cWildCard := if(type('wildcard') == "U", '*.*', wildcard)
lOldScore := set(_SET_SCOREBOARD, .F.)   // no keeping score here!
ColorSet(C_NOTEPAD_BOX)

// initialize the variables
nRelRow := nRelCol := blockmarks := 0
nStartRow := 1
pagelength := 60
lFileChange := .f.
filename := ''

// paint note pad box
@ 0, 0, maxrow(), maxcol() box BOXFRAMES[4]
@ maxrow() - 2,  0 ssay chr(199)
@ maxrow() - 2,  1 ssay replicate( chr(196), maxcol() - 1)
@ maxrow() - 2, maxcol() ssay chr(182)

// if cLastFile contains a filename, use that
if ! empty(cLastFile)
   filename := cLastFile

// if there is a global NOTEFILE variable, use that
elseif type("notefile") == "C"
   filename := cDir + notefile

// otherwise, prompt the user to select a file or type in a filename
else
   setcursor(1)
   filename := upper(GetVar('Enter file name to edit:','C'))
   do case
      case empty(filename)                      // no file specified, full list
         filename := getfile(.t.)
      case "*" $ filename .or. "?" $ filename   // wildcard specification
         filename := getfile(.t., filename)
      otherwise                                 // filename entered
         filename := cDir + filename
   endcase
endif
if ! empty(filename)  // i.e., if they didn't hit ESC at the file directory
   ColorSet(C_NOTEPAD_BOX)
   @ 0, maxcol() - 13 ssay 'Alt-H = help'
   ColorSet(C_NOTEPAD_WINDOW)
   cTemp := NewFile()
   memoedkey := 0
   // begin main edit loop
   do while memoedkey <> K_ESC .and. memoedkey <> K_ALT_X .and. ;
            memoedkey <> K_ALT_Q
      // determine total lines in this memo for status indicator
      nLines := mlcount(cTemp, rightmargin)
      scroll(01, 01, maxrow() - 3, maxcol() - 1, 0)
      setcursor(1)
      cWorkFile := memoedit(cTemp, 1, leftmargin, maxrow() - 3, ;
                   min(rightmargin, maxcol() - 1), .t., 'GFEditFunc', ;
                   linewidth, '', nStartRow, nRelCol, nRelRow, nRelCol)

      // various actions to take upon exiting memoedit()
      do case

#ifdef SPELLER

         case memoedkey == SPELLER_KEY        // Invoke spelling checker
            gfsaveenv(.t., 0)
            cTemp := SpellCheck(cWorkFile)    // Spell check the text
            if lastkey() == K_ESC             // If user ESCAPED out, then
               cTemp := cWorkFile             // restore the original text
            endif
            gfrestenv()

#endif

         case memoedkey == K_ALT_A       // append new file and continue
            if ! empty(mfile := GetFile(.f.))
               if file(mfile)
                  cTemp := strtran(cWorkFile, chr(26)) + memoread(mfile)
               elseif len(trim(mfile)) > 0
                  problem('file ' + mfile + ' not found')
               endif
            else
               cTemp := cWorkFile
            endif

         case blockmarks == 3          // block command
            blockmarks := 0
            nMarker1 := at(chr(254), cWorkFile)
            nMarker2 := rat(chr(254), cWorkFile)
            do case

               case blocksel == MOVEBLOCK
                  // copy marked block to scrap buffer
                  copyblock := substr(cWorkFile, nMarker1+1, nMarker2-nMarker1-1)
                  // delete marked block
                  cWorkFile := substr(cWorkFile, 1, nMarker1 - 1) + ;
                               substr(cWorkFile, nMarker2 + 1)
                  // then paste the scrap buffer at the new location
                  cTemp := substr(cWorkFile, 1, at(chr(255), cWorkFile) - 1) +;
                           copyblock + ;
                           substr(cWorkFile, at(chr(255), cWorkFile) + 1)

               case blocksel == COPYBLOCK
                  // copy marked block to scrap buffer
                  copyblock := substr(cWorkFile, nMarker1+1, nMarker2-nMarker1-1)
                  // save text after the marker
                  oldtext   := substr(cWorkFile, at(chr(255), cWorkFile) + 1)
                  // add in the marked block
                  cTemp := substr(cWorkFile, 1, at(chr(255), cWorkFile) - 1) +;
                           copyblock + oldtext

               case blocksel == KILLBLOCK
                  cTemp := substr(cWorkFile, 1, nMarker1 - 1) + ;
                           substr(cWorkFile, nMarker2 + 1)

               case blocksel == PRINTBLOCK
                  nMarker := recno()
                  printit(substr(cWorkFile, nMarker1 + 1, nMarker2 - nMarker1 - 1))
                  if nMarker <> 0
                     go nMarker
                  endif

               case blocksel == UPPERBLOCK .or. blocksel == LOWERBLOCK
                  // copy marked block to scrap buffer & convert appropriately
                  if blocksel == UPPERBLOCK
                     copyblock := upper(substr(cWorkFile, nMarker1 + 1, ;
                                              nMarker2 - nMarker1 - 1))
                  else
                     copyblock := lower(substr(cWorkFile, nMarker1 + 1, ;
                                               nMarker2 - nMarker1 - 1))
                  endif
                  cTemp := substr(cWorkFile, 1, nMarker1-1) + copyblock + ;
                           substr(cWorkFile, nMarker2+1)

               // erase block markers
               case blocksel == 7
                  cTemp := cWorkFile

            endcase
            cTemp := strtran(cTemp, chr(254), '')  // erase block markers

         case memoedkey == K_ALT_R      // search and replace
            cSearch := trim(GetVar('enter search string:', 'C'))
            // if user entered null string, forget the whole thing
            if len(cSearch) > 0
               cReplace := trim(GetVar('Enter replacement string:', 'C'))
               if GetVar('Replace all? (y/n)','L')
                  cTemp := strtran(cWorkFile, cSearch, cReplace)
               else
                  if ( xx := GetVar('Occurrences to replace:', 'N')) > 0
                     cTemp := strtran(cWorkFile, cSearch, cReplace, 1, xx)
                  else
                     cTemp := cWorkFile   // to retain working text
                  endif
               endif
            else
               cTemp := cWorkFile         // to retain working text
            endif

         case memoedkey == K_ALT_S .or. memoedkey == K_ALT_Q  // save file
            if len(cWorkFile) > 0 .or. ;
                   (len(cWorkFile) == 0 .and. len(cTemp) > 0)
               if filename == 'tempfile'
                  filename := upper(GetVar('enter file name to save to:','C'))
               endif
               Stat_Msg('Saving ' + filename)
               FileWrite(filename, cWorkFile)
               cTemp := cWorkFile
            endif

         case memoedkey == K_ALT_W      // write to new file
            mfile := upper(GetVar('File name to write to:','C'))
            if file(mfile)
               Problem('The file ' + mfile + ' already exists')
            elseif len(trim(mfile)) > 0
               Stat_Msg('Writing to ' + mfile)
               FileWrite(mfile, cWorkFile)
            endif
            cTemp := cWorkFile  // so that we retain the working text

         case memoedkey == K_ALT_N           // change files
            if ! empty( mfile := GetFile(.t.) )
               if len(trim(mfile)) > 0
                  if len(cWorkFile) == 0   // delete this file handle if no text
                     ferase(filename)
                  endif
                  filename := mfile
                  cTemp := NewFile()
                  blockmarks := 0          // reset block markers
               endif
            else
               cTemp := cWorkFile
            endif

         case memoedkey == K_ALT_P            // print file
            cTemp := cWorkFile
            if ! isprinter()
               Problem('printer not ready')
            else

               // establish condition for do..while printing loop
               nMarker2 := recno()
               condblock := NIL
               if nMarker2 <> 0
                  go nMarker2
               endif

               // if you would like to call the interactive query builder
               // (setfilt) to filter out records for mail merge, all you
               // need to do is compile this program with the following
               // syntax:  CLIPPER POPNOTE /N/W/A/DMAILMERGE

   #ifdef MAILMERGE
               nMarker1 = at('{', cTemp)
               if nMarker1 > 0
                  if GetVar('Print for more than one record (y/n)', 'L')
                     setfilt()        // establish filter criteria
                     condblock := { | | ! eof() }
                  endif
               endif
   #endif

               PrintIt(cWorkFile, condblock)
               // reset record pointer, turn off filter if database open
               if nMarker2 <> 0
                  go nMarker2           // defined at the top of this block
                  if nFields > 0
                     set filter to
                  endif
               endif

            endif

         otherwise
            cTemp := cWorkFile      // to retain the working text

      endcase
   enddo

   // delete last file handle if there is no text
   if file(filename)
      nHandle := fopen(filename, FO_READ)
      cTemp := fseek(nHandle, 0, FS_END)
      fclose(nHandle)
      if cTemp == 0
         ferase(filename)
      endif
   endif
   cLastFile := filename       // save filename for next time
endif

// restore hot-key
if nHotKey <> NIL
   setkey(nHotKey, bOldblock)
endif
GFRestEnv()
set(_SET_INSERT, lOldInsert)
restscreen(0, 0, maxrow(), maxcol(), oldscrn)
return NIL

* end function PopNote()
*--------------------------------------------------------------------*


/*
  PrintIt(): print either the entire file or a portion
*/
static function printit(mtext, condition)
local linewidth, printline, currline, lines, oldtext, page := 1, skipline, ;
      oldmargin, oldcursor := setcursor(0), nMarker, nMarker2, mtype, xx := 1
memvar setup_str, reset_str, notehead  // globals from calling program

/*
   XX is a pointer to the codeblock that sets the condition for looping
   below -- if we are not doing a mail merge, the code block will
   essentially call for one time through the loop by specifying XX == 1
   if we are doing a mail merge, the value of XX will be meaningless
   sounds crazy, but this is actually the best way to do it, and i sure
   did not want to resort to macros
*/
default condition to { | | xx == 1 }
oldmargin := set(_SET_MARGIN, leftmargin)
linewidth := rightmargin - leftmargin + 1
Stat_Msg('Printing ' + filename + ', page ' + ltrim(str(page)))
set device to print

// if SETUP_STR is defined in your calling routine, send it to the printer now
if type('setup_str') <> 'U'
   devout(setup_str)
endif

// begin main loop
lines := mlcount(mtext, linewidth)
do while eval(condition) .and. lastkey() <> K_ESC
   // if NOTEHEAD is defined in your calling routine, print a header
   if type('notehead') <> "U"
      notepad_hd(page)
   endif
   currline := 1
   do while currline <= lines .and. inkey() <> K_ESC
      printline := memoline(mtext, linewidth, currline)
      skipline := .f.
      // check for embedded fieldname
      if ( nMarker := at('{', printline) ) > 0 .and. ;
         ( nMarker2 := at('}', printline) ) > 0 .and. nMarker2 > nMarker + 1
         do while nMarker > 0 .and. nMarker + 1 < nMarker2
            oldtext := substr(printline, nMarker + 1, nMarker2 - nMarker - 1)
            // convert this field appropriately
            do case
               case (mtype := type(oldtext)) == 'N'
                  oldtext := &("{ | | ltrim(str(" + oldtext + "))}")
               case mtype == 'L'
                  oldtext := &("{ | | if(" + oldtext + ", 'yes', 'no') }")
               case mtype == 'D'
                  oldtext := &("{ | | dtoc(" + oldtext + ") }")
               case mtype $ 'UM'
                  oldtext := &("{ | | '' }")
               otherwise
                  oldtext := &("{ | | trim(" + oldtext + ") }")
            endcase
            printline := substr(printline, 1, nMarker-1) + eval(oldtext) + ;
                         substr(printline, nMarker2 + 1)
            nMarker := at('{', printline)
            nMarker2 := at('}', printline)
         enddo
         // if by chance the inserted field is blank, then do not print
         // this line down below!
         if empty(printline)
            skipline := .t.
         endif
      endif
      if ! skipline
         @ prow()+1, 0 say trim(printline)
         if prow() == pagelength - 1
            page++
            set device to screen
            Stat_Msg('Printing ' + filename + ', page ' + ltrim(str(page)))
            set device to print
            if type('notehead') <> "U"
               Notepad_Hd(page)
            else
               eject
            endif
         endif
      endif
      currline++
   enddo
   eject
   // if no database open, fall out
   if empty(alias())
      exit
   else
      skip
      xx++
   endif
enddo     // end main loop
// if RESET_STR is defined in your calling routine, send it to the printer now
if type('reset_str') <> 'U'
   devout(reset_str)
endif
set device to screen
if lastkey() == K_ESC
   Stat_Msg('Printing aborted')
endif
set(_SET_MARGIN, oldmargin)  // reset previous margin
return NIL

* end static function PrintIt()
*--------------------------------------------------------------------*


/*
  notepad_hd: heading when printing file
*/
static function notepad_hd(page)
@ 0,0 say ''
@ 1,0 say StripDir(filename)
CENTER(1, dateword())
@ 1, linewidth - 8 say 'Page ' + ltrim(str(page))
@ 3, 0 say ''
return NIL

* end static function Notepad_Hd()
*--------------------------------------------------------------------*


/*
  gfeditfunc(): fancy UDF for memoedit()
*/
function gfeditfunc(gfstatus, gfline, gfcol)
local ret_val := 0, scrnbuff, buffer, sel, trightmargin, switchfile, ;
      quickquit, mdir
static lFirstLoop
static nstatrow
memoedkey := lastkey()
ColorSet(C_NOTEPAD_BOX)
do case
case gfstatus == ME_INIT
   @ maxrow() - 1, maxcol() - 11 ssay '<wrap>' + if(readinsert(), '<ins>', '')
   // use full block cursor for insert mode
   setcursor(if(readinsert(), 3, 1))
   lFirstLoop := .t.
   // if there are more lines than will fit in one screen,
   // draw indicator showing relative position in the memo
   if nLines > maxrow() - 3
      @ 0, maxcol() say chr(24)
      @ maxrow(), maxcol() say chr(25)
      @ 1, maxcol(), maxrow() - 1, maxcol() box replicate(chr(178), 9)
      @ 1, maxcol() say chr(219)
      nstatrow := 1                      // set status row for next time
   endif
case gfstatus == ME_IDLE
   // if there are more lines than will fit in one screen,
   // draw indicator showing relative position in the memo
   if nLines > maxrow() - 3
      // first, adjust total # of lines if they have added to the memo
      nLines := max(nLines, gfline)
      // prepare to draw status indicator on the right side
      @ nstatrow, maxcol() say chr(178)  // remove old status indicator
      // determine new row position for status indicator
      if gfline == nLines
         nstatrow := maxrow() - 1
      else
         // note: use of MAX() keeps status indicator off row 0
         nstatrow := max(int((gfline/nLines) * (maxrow() - 1)), 1)
      endif
      @ nstatrow, maxcol() say chr(219)  // redraw status indicator at new row
   endif
   if lFirstLoop .and. ! lFileChange    // on 1st loop we don't want the asterisk
      lFirstLoop := .f.
      @ 0, maxcol() - 15 ssay chr(196)
   elseif ! lFileChange
      // file was changed if they hit backspace, enter, or printable characters
      lFileChange = (memoedkey == K_BS .or. memoedkey == K_ENTER .or. memoedkey > 31)
      if lFileChange
         @ 0, maxcol() - 15 ssay '*'
      endif
   endif
   Settings(gfline, gfcol)
   Stat_Msg(StripDir(filename))

otherwise

   do case

      case memoedkey == K_CTRL_V      // toggled insert on/off
         @ maxrow() - 1, 74 ssay if(! readinsert(), "<Ins>", space(5))
         setcursor(if(readinsert(), 1, 3))
         ret_val := 22

      case memoedkey == K_ALT_A       // append file
         lFileChange := .t.
         ret_val := 23

      case memoedkey == K_ALT_B       // mark a block
         if blockmarks == 2
            SavePosition()
            ColorSet(C_MESSAGE)
            scrnbuff := shadowbox(08, 32, 16, 46, 2)
            blocksel := achoice(09, 33, 15, 45, aOperations, .t.)
            ByeByeBox(scrnbuff)

            // if moving or copying block, we must mark current position with
            // chr(255), which looks for all intents/purposes like a space
            if blocksel == COPYBLOCK .or. blocksel == MOVEBLOCK
               if ! readinsert()
                  readinsert(.t.)
                  keyboard chr(255) + chr(K_CTRL_W) + chr(22)
               else
                  keyboard chr(255) + chr(K_CTRL_W)
               endif
            endif
            blockmarks := if(blocksel > 0, 3, 2)
            ret_val := if(blocksel > 2, 23, 32)
         else
            blockmarks++

            // switch to insert mode (if not already) so as not
            // overwrite text, then stuff keyboard w/ insert key
            // to switch back -- this is the only way i found to do it
            if ! readinsert()
               readinsert(.t.)
               keyboard chr(254) + chr(22)
            else
               keyboard chr(254)
            endif
            ret_val := 32
         endif

      case memoedkey == K_ALT_F           // insert field
         if len(aFields) > 0
            ColorSet(C_MESSAGE)
            scrnbuff := shadowbox(4, 29, 18, 50, 2, 'fields')
            sel := achoice(5, 30, 17, 49, aFields, .t.)
            ByeByeBox(scrnbuff)
            // switch to insert mode (if not already) so as not
            // overwrite text, then stuff keyboard w/ name of field
            if sel > 0
               sel := "{" + aFields[sel] + "}"
               if ! readinsert()
                  readinsert(.t.)
                  keyboard sel + chr(22)
               else
                  keyboard sel
               endif
            endif
         endif

      case memoedkey == K_ALT_G        // go to line
         nRelRow := GetVar('Go to line number:', 'N')
         nStartRow := nRelRow
         ret_val := 23

      case memoedkey == K_ALT_H        // help screen
         NoteHelp()
         ret_val := 32

      case memoedkey == K_ALT_I        // insert buffer
         if ! empty(pastebuff)
            // switch to insert mode (if not already) so as not
            // overwrite text, then stuff keyboard w/ insert key
            if ! readinsert()
               readinsert(.t.)
               keyboard pastebuff + chr(22)
            else
               keyboard pastebuff
            endif
         endif
         ret_val := 32

      case memoedkey == K_ALT_K         // kut and paste
         buffer := cutnpaste(@oldscrn)
         if lastkey() <> K_ESC
            pastebuff := buffer
         endif
         ret_val := 32

      case memoedkey == K_ALT_L         // change page length for printing
         pagelength := GetVar('Enter new page length:', 'N')
         ret_val = 32

      case memoedkey == K_ALT_M         // change margins for printing
         SavePosition()
         do while .t.
            leftmargin := max(GetVar('Enter new left margin:', 'N'), 1)
            trightmargin := GetVar('Enter new right margin:', 'N')
            // if user hit enter for right margin (0), use previous setting
            rightmargin := if(trightmargin == 0, rightmargin, trightmargin)
            if rightmargin > leftmargin
               exit
            endif
            Problem('Illegal margins, try again')
         enddo
         linewidth := rightmargin - leftmargin + 1
         Settings(gfline, gfcol)
         ret_val := 23

      case memoedkey == K_ALT_N         // edit new file
         switchfile := .t.
         if lFileChange
            switchfile := GetVar('File changed, exit anyway (y/n)', 'L')
         endif
         if switchfile
            nRelRow := nRelCol := 0
            nStartRow := 1
            lFileChange := .f.
            ret_val := 23
         else
            ret_val := 32
         endif

      case memoedkey == K_ALT_P         // print file
         SavePosition()
         ret_val := 23

      case memoedkey == K_ALT_Q         // save file and quit
         ret_val := 23

      case memoedkey == K_ALT_R         // search and replace
         SavePosition()
         lFileChange := .t.
         ret_val := 23

      case memoedkey == K_ALT_S         // save file
         SavePosition()
         @ 0, maxcol() - 15 ssay chr(196)  // remove 'file changed' indicator
         lFileChange := .f.
         ret_val := 23

      case memoedkey == K_ALT_T         // toggle word wrap
         wordwrap := ! wordwrap
         @ maxrow() - 1, maxcol() - 11 ssay if(wordwrap, '<wrap>', space(6))
         ret_val := 34

      case memoedkey == K_ALT_V         // view buffer
         ColorSet(C_MESSAGE)
         scrnbuff := shadowbox(9, 10, 15, 69, 2, 'Scrap Buffer')
         pastebuff := memoedit(pastebuff, 10, 11, 14, 68, .f.)
         ByeByeBox(scrnbuff)

      case memoedkey == K_ALT_W         // write text to new file
         SavePosition()
         ret_val := 23

      case memoedkey == K_ALT_X .or. memoedkey == K_ESC  // exit
         quickquit := .t.
         if lFileChange
            quickquit := GetVar('File changed, exit anyway (y/n)', 'L')
         elseif memoedkey == K_ESC
            quickquit := GetVar('Abort editing (y/n)', 'L')
         endif
         if quickquit
            ret_val := 27
         else
            ret_val := 32
         endif

      case memoedkey == K_ALT_Y           // change directory
         mdir := upper(GetVar('Enter directory and/or wildcard:', 'C'))
         do case

               // user input wildcard mask only
            case '.' $ mdir .and. ! ('\' $ mdir)
               cWildCard := mdir

               // user input directory only
            case ! ('*' $ mdir) .and. len(trim(mdir)) > 0
               /* add a backslash if they either did not begin it with a
                  backslash or inserted a drive designator */
               cDir := if(left(mdir,1) == '\' .or. substr(mdir, 2, 1) == ':', ;
                       '', '\') + mdir + if(right(trim(mdir),1) == '\', '', '\')
               cWildCard := '*.*'
               // add new subdirectory to current working filename
               filename := cDir + StripDir(filename)

               // user input directory and wildcard mask - parse string
            case len(trim(mdir)) > 0
               cWildCard := substr(mdir, rat('\', mdir) + 1)
               /* add a backslash if they either did not begin it with a
                  backslash or inserted a drive designator */
               cDir := if(left(mdir, 1) == '\' .or. substr(mdir, 2, 1) == ':', ;
                       '', '\') + substr(mdir, 1, rat('\', mdir))
               // add new subdirectory to current working filename
               filename := cDir + StripDir(filename)

         endcase
         ret_val := 32

#ifdef SPELLER

      case memoedkey == K_ALT_Z
         ret_val := 23

#endif

   endcase
endcase
ColorSet(C_NOTEPAD_WINDOW)
return (ret_val)

* end function GFEditFunc()
*--------------------------------------------------------------------*


/*
  NoteHelp(): display list of active keys
*/
static function NoteHelp
gfsaveenv(.t., 0)   // turn off cursor
dispbegin()
setcolor(C_NOTEPAD_BOX)
scroll(01, 01, maxrow() - 3, maxcol() - 1, 0)
@ 2, 7 ssay 'Alt-A  (a)ppend file'
@ 2, 42 ssay 'Alt-B  '+if(blockmarks<2, 'place (B)lock marker','(B)lock operation')
if len(aFields) > 0
   @ 3, 42 ssay 'Alt-F insert (F)ield'
endif
@ 3,  7 ssay 'Alt-G  (G)oto line'
@ 4,  7 ssay 'Alt-H  show (H)elp screen'
@ 4, 42 ssay 'Alt-I  (I)nsert buffer'
@ 5,  7 ssay 'Alt-K  (K)ut and paste'
@ 5, 42 ssay 'Alt-L  page (L)ength'
@ 6,  7 ssay 'Alt-M  change (M)argins'
@ 6, 42 ssay 'Alt-N  edit (N)ew file'
@ 7,  7 ssay 'Alt-P  (P)rint file'
@ 7, 42 ssay 'Alt-Q  (Q)uit w/ save'
@ 8,  7 ssay 'Alt-R  search and (R)eplace'
@ 8, 42 ssay 'Alt-S  (S)ave file & continue'
@ 9,  7 ssay 'Alt-T  (T)oggle wordwrap on/off'
@10, 42 ssay 'Alt-V  (V)iew buffer contents'
@10,  7 ssay 'Alt-W  (W)rite to file'
@10, 42 ssay 'Alt-X  e(X)it without saving'
@11,  7 ssay 'Alt-Y  change director(Y)'
#ifdef SPELLER
   @11, 42 ssay padr(SPELLER_KEY_HELP_PROMPT, 7) + 'check spelling'
#endif
@12,  7 ssay '^E     go up one line'
@12, 42 ssay '^X     go down one line'
@13,  7 ssay '^S     go left one char'
@13, 42 ssay '^D     go right one char'
@14,  7 ssay '^A     go left one word'
@14, 42 ssay '^F     go right one word'
@15,  7 ssay 'Home    go to start of line'
@15, 42 ssay 'End    go to end of line'
@16,  7 ssay '^Home   top corner of window'
@16, 42 ssay '^End   bottom corner of window'
@17, 7 ssay '^PgUp   go to start of file'
@17, 42 ssay '^PgDn  go to end of file'
@18, 7 ssay 'Esc     abort edit, no save'
@18, 42 ssay '^Y     delete current line'
@ 1,33 ssay 'MNEMONIC KEYS'
@ 21,24 ssay 'press any key to resume editing' color "*" + setcolor()
dispend()
ginkey(0)
gfrestenv()
return NIL

* end static function NoteHelp()
*--------------------------------------------------------------------*


/*
  filewrite: write text to new or existing file
*/
static function filewrite(gffilename, text)
if ! memowrit(gffilename, text)
   Problem('Write error - ' + ltrim(str(doserror())))
endif
return NIL

* end static function FileWrite()
*--------------------------------------------------------------------*


/*
  GetVar(): get a variable from the user
*/
static function GetVar(msg, vartype)
local ret_val := [], keypress
setcolor('+' + ColorSet(C_NOTEPAD_BOX, .T.))
scroll(maxrow() - 1, 02, maxrow() - 1, 67, 0)
@ maxrow() - 1, 02 ssay msg + ' '
if vartype == 'L'     // special case for yes/no questions
   do while ! upper(chr(inkey())) $ 'YN'
   enddo
   ret_val := (chr(lastkey()) $ 'Yy')
else
   do while .t.
      keypress := ginkey(0)
      do case

            // exit loop
         case keypress == K_ESC .or. keypress == K_ENTER
            ret_val := if(vartype == 'N', val(ret_val), ret_val)
            exit

            // scrub last character
         case (keypress == K_BS .or. keypress == K_LEFT) .and. len(ret_val) > 0
            ret_val := substr(ret_val, 1, len(ret_val) - 1)
            @ row(), col()-1 ssay chr(32)
            devpos(row(), col()-1)

            // other printable characters
         case keypress > 31 .and. keypress < 127
            ret_val += chr(keypress)
            @ row(),col() ssay chr(keypress)
      endcase
   enddo
endif
Stat_Msg(StripDir(filename))      // clear message
return (ret_val)

* end static function NoteHelp()
*--------------------------------------------------------------------*


/*
  Problem(): display error message
*/
static function Problem(msg)
ColorSet(C_NOTEPAD_BOX)
@ maxrow() - 1, 02 ssay padr(msg, 65)
tone(MUSIC_ERROR, 1)
tone(MUSIC_ERROR, 1)
inkey(2)
ColorSet(C_NOTEPAD_WINDOW)
return NIL

* end static function Problem()
*--------------------------------------------------------------------*


/*
  newfile(): create new file handle
*/
static function newfile
local nHandle
if ! file(filename)
   nHandle := fcreate(filename, FC_NORMAL)
   if nHandle == -1
      Problem('File creation error - ' + ltrim(str(doserror())))
   else
      fclose(nHandle)
   endif
endif
return memoread(filename)

* end static function NewFile()
*--------------------------------------------------------------------*


/*
  Stat_Msg(): display status message
*/
static function Stat_Msg(msg)
ColorSet(C_NOTEPAD_BOX)
@ maxrow() - 1, 02 ssay padr(msg, 41)
ColorSet(C_NOTEPAD_WINDOW)
return NIL

* end static function Stat_Msg()
*--------------------------------------------------------------------*


/*
  Settings(): self-explanatory
*/
static function Settings(nline, ncolumn)
ColorSet(C_NOTEPAD_BOX)
@ maxrow() - 1,43 ssay padr('L ' + ltrim(str(nline)), 6)
@ maxrow() - 1,50 ssay padr('C ' + ltrim(str(ncolumn + 1)), 4)
@ maxrow() - 1,55 ssay 'LM ' + str(leftmargin, 2)
@ maxrow() - 1,61 ssay 'RM ' + str(rightmargin, 3)
return NIL

* end static function Settings()
*--------------------------------------------------------------------*


/*
  GetFile(): pop up file directory for user
*/
static function GetFile(adding, cTempMask)
local numfiles
local ele
local files_ := directory(cDir + cWildCard)
local xx
local browse
local key
local column
local searchstr := []
local telem
local ret_val
// if wildcard spec was passed, use that as override
if cTempMask <> NIL
   files_ := directory(cDir + cTempMask)
else
   files_ := directory(cDir + cWildCard)
endif
if empty(files_)
   problem("No files found!")
else
   GFSaveEnv(.t., 0)
   numfiles := len(files_)
   for xx := 1 to numfiles
      // strip out binary files and any file > 64K
      if substr(files_[xx,1], at('.', files_[xx,1]) + 1, 3) $ ;
             'EXE-COM-DBF-NTX-DBT-NDX-OBJ-FRM-LIB-LBL-MEM-ZIP-ARC' .or. ;
             files_[xx,2] > 65535
         adel(files_, xx)
         numfiles--
         xx--
      endif
   next
   asize(files_, numfiles)
   // sort array alphabetically by filename
   asort(files_,,, { | x, y | x[1] < y[1] } )
   if adding
      aadd(files_, { "NEW FILE", 0, date(), time() } )
      numfiles++
   endif
   ColorSet(C_MESSAGE)
   shadowbox(2, 18, 16, 61, 2, cDir+if(cTempMask <> NIL, cTempMask, cWildCard))
   SINGLEBOX(19, 13, 21, 66)
   @ 20, 15 ssay 'Move highlight bar to desired file and press enter'
   browse := TBrowseNew(3, 19, 15, 60)
   browse:colorSpec := setcolor()
   browse:headSep := ""
   browse:colSep  := ""
   ele := 1
   browse:goTopBlock := { || ele := 1 }
   browse:goBottomBlock := { || ele := numfiles }
   browse:skipBlock := { |SkipCnt| Gilligan(@ele, SkipCnt, numfiles) }
   column := TBColumnNew("Name", { | | files_[ele, 1] })
   column:width := 12
   browse:AddColumn( column )
   column := TBColumnNew("  Size", { | | files_[ele, 2] })
   column:width := 10
   browse:AddColumn( column )
   column := TBColumnNew("  Date", { | | files_[ele, 3] })
   column:width := 8
   browse:AddColumn( column )
   column := TBColumnNew("  Time", { | | files_[ele, 4] })
   column:width := 8
   browse:AddColumn( column )
   do while key <> K_ESC .and. key <> K_ENTER
      dispbegin()
      do while (key := inkey()) == 0 .and. ! browse:stabilize()
      enddo
      dispend()
      if browse:stable
         key := ginkey(0, "KEY")
      endif
      do case
         case key == K_LEFT
            browse:left()
         case key == K_RIGHT
            browse:right()
         case key == K_UP
            searchstr := []
            browse:up()
         case key == K_DOWN
            searchstr := []
            browse:down()
         case key == K_PGUP .or. key == K_HOME   // top o' window
            searchstr := []
            browse:pageUp()
         case key == K_PGDN .or. key == K_END    // bottom o' window
            searchstr := []
            browse:pageDown()
         case Isalpha(chr(key))      // letter key - search 'em, Dan-O
            if (telem := ascan(files_, ;
                         { |a| upper(a[1]) = upper(searchstr + chr(key)) })) > 0
               searchstr += chr(key)
               ele := telem
               browse:refreshAll()
            endif
         case key == K_BS             // truncate the search string
            if len(searchstr) > 0
               searchstr := substr(searchstr, 1, len(searchstr) - 1)
               if (telem := ascan(files_, ;
                            { | a | upper(a[1]) = upper(searchstr) })) > 0
                  ele := telem
                  browse:refreshAll()
               endif
            endif
      endcase
   enddo
   do case
      case key == K_ESC      // user pressed esc to abort
         ret_val := ''
      case ele == len(files_) .and. adding    // user selected 'new file'
         ret_val := cDir + upper(GetVar('Enter file name:', 'C'))
      otherwise
         ret_val := cDir + files_[ele][1]
   endcase
   GFRestEnv()
endif
return (ret_val)

* end static function GetFile()
*--------------------------------------------------------------------*


/*
  pastebuff: paste the current popnote buffer
*/
function pastebuff(p, l, v)
if ! empty(pastebuff)
   keyboard pastebuff
endif
return NIL

* end function PasteBuff()
*--------------------------------------------------------------------*

* eof popnote.prg
