/*
    Program: POPCALC.PRG
    System: GRUMPFISH LIBRARY
    Author: Greg Lief
    Copyright (c) 1988-93, Greg Lief
    Clipper 5.x version
    Compile instructions: clipper popcalc /n /w

    Note: For backward compatibility I have included logic to
          detect public variables GFCALCLEFT, GFCALCTOP, and
          GFCALCCOL.  If you do not need this logic, I would
          strongly advise you to recompile this with the following
          syntax, which will slightly improve performance:

          clipper popcalc /n /w /dNO_PUBLICS

    Procs & Fncts: MEMKEYS()
                 : PASTE
                 : PAPER_TAPE()
                 : DRAWCALCBX()

            Calls: SHADOWBOX()   (function in $SHADOWB.PRG)
                   CHECKMOVE()   (function in $MOVING.PRG)

*/

// begin preprocessor directives

#include "grump.ch"
#include "inkey.ch"

// end preprocessor directives

// begin global declarations

static num := 0
static nCalcTop := 5
static nCalcLeft := 28
static nDecimals := 4

#translate ShowMemKeys() => @ nCalcTop+6, nCalcLeft+2 SAY 'R + -' ;;
                            @ nCalcTop+7, nCalcLeft+2 SAY 'C * /'

// end global declarations


function popcalc(gfproc, line, var)
static cPicture := '###########.####'   // PICTURE string for displaying numbers
local num1 := 0
local mem := 0
local nDecPlace := 0
local tnum
local lJustCalced := .f.
local cOldScrn
local op := 0
local opstr := '+-*/^%'
local hotkey := 0
local nKey
local lPaperTape := .f.
local olddecimals := set(_SET_DECIMALS, nDecimals)
local maincolor := ColorSet(C_CALCULATOR_BOX, .T.)
local bOldblock

GFSaveEnv(, 0)         // shut off cursor

// determine whether this was called via hot-key; if so, disable it
if (gfproc <> NIL)
   bOldblock := setkey(hotkey := lastkey(), NIL)
endif

/*
  Variables:

   NUM       = primary number to be manipulated
   NUM1      = second number when operator has been introduced
   MEM  = stored in memory
   OP   = current operator: 1=add, 2=sub, 3=mult, 4=div, 5=exp
   NDECIMALS = number of decimals to display
   NDECPLACE = decimal place counter - if nDecPlace is 0, decimal point
   has not been activated; otherwise indicates # of digits to the right of it

   User-configurable color/screen coordinates: Current hot-key to change
   color is Alt-F10, although you can easily change this (see line 236
   below).  Press any arrow key, Home, End, PgUp, PgDn to move window.
*/

#ifndef NO_PUBLICS    // see note in file header above

// use public variables to initialize position and color
if type('m->gfcalcleft') <> "U"
   nCalcLeft := min(m->gfcalcleft, maxcol() - 25)
endif
if type('m->gfcalctop') <> "U"
   nCalcTop := min(m->gfcalctop, maxrow() - 12)
endif
if type('m->gfcalccol') <> "U"
   ColorSet(C_CALCULATOR_BOX, ( maincolor := m->gfcalccol ) )
endif

#endif

cOldScrn := savescreen(nCalcTop, nCalcLeft, nCalcTop + 12, nCalcLeft + 26)

DrawCalcBx()

// main loop
do while nKey <> K_ESC

   // display M if there is a number currently in memory
   @ nCalcTop+3, nCalcLeft+21 ssay if(mem <> 0, 'M', chr(32))
   @ nCalcTop+3, nCalcLeft+4  say num picture cPicture
   setcolor(maincolor)

   // display 1st number if an operator has been introduced
   if num1 <> 0
      @ nCalcTop+1,nCalcLeft+4  say num1 picture cPicture
      @ nCalcTop+1,nCalcLeft+21 ssay substr(opstr,op,1)
   else
      @ nCalcTop+1,nCalcLeft+4 ssay space(18)
   endif

   nKey := ginkey(0)

   do case

      // numeric key was pressed
      case nKey > 47 .and. nKey < 58
         tnum := val(chr(nKey))        // determine numeric value
         do case

            case (num=0 .or. lJustCalced) .and. nDecPlace == 0
               num := tnum
               lJustCalced := .f.

            case num <> 0 .and. nDecPlace == 0 .and. num < 10000000000 //integer
               num := num * 10 + tnum

            case nDecPlace > 0 .and. nDecPlace <= nDecimals    // real number
               num += (tnum / (10 ^ nDecPlace))
               nDecPlace++

            case lJustCalced .and. nDecPlace <= nDecimals
               num := (tnum / (10 ^ nDecPlace))
               lJustCalced := .f.

         endcase

         // backspace was pressed
      case nKey == K_BS .and. (num <> 0 .or. nDecPlace > 0)
         if nDecPlace == 0             // if number is still an integer
            num := int(num / 10)       // kill the least significant digit
         else
            nDecPlace--
            if nDecPlace > 0
               num := ltrim(str(num, 16, nDecPlace))
               num := val(substr(num, 1, len(num) - 1))
            else
               num := 0
            endif
         endif

         // decimal point (period) was pressed
      case nKey == 46 .and. nDecPlace == 0
         nDecPlace++

         // operator pressed (43='+' 45='-' 42='*' 47='/' 94='^', 37='%')
         // with no pending operations
      case (nKey == 43 .or. nKey == 45 .or. nKey == 42 .or.     ;
              nKey == 47 .or. nKey == 94 .or. nKey == 37) .and. ;
              op == 0 .and. num1 == 0
         num1 := num
         num := nDecPlace := 0
         op := at(chr(nKey), opstr)
         lJustCalced := .f.
         if lPaperTape
            Paper_Tape(str(num1) + ' ' + substr(opstr, op, 1))
         endif

         // '=', return or operator was pressed -- perform calculation
      case (nKey == 61 .or. nKey == 13 .or. nKey == 43 .or. nKey == 45 ;
           .or. nKey == 42 .or. nKey == 47 .or. nKey == 94) .and. op > 0
         if lPaperTape
            Paper_Tape(str(num) + ' ' + substr(opstr, op, 1))
         endif
         do case
            case op == 1            // addition
               num += num1
            case op == 2            // subtraction
               num = num1 - num
            case op == 3            // multiplication
               num *= num1
            case op == 4 .and. num <> 0   // division (don't allow by zero)
               num = num1 / num
            case op == 5            // exponent
               num = num1 ^ num
            case op == 6            // percentage
               num = num1 / 100 * num
         endcase
         nDecPlace := 0
         op := at(chr(nKey), opstr)
         lJustCalced := .t.
         if op == 0
            num1 := 0
            if lPaperTape
               Paper_Tape(str(num) + ' =')
               Paper_Tape()
            endif
         else
            num1 := num
            num := 0
         endif

         // 'c' was pressed to clear number
      case nKey == 67 .or. nKey == 99
         num := num1 := nDecPlace := op := 0

         // 'd' was pressed to change decimals
      case nKey == 68 .or. nKey == 100
         setcolor('+' + maincolor)
         @ nCalcTop+1, nCalcLeft+2 ssay 'How many decimals?'
         nKey := ginkey(0)
         scroll(nCalcTop+1, nCalcLeft+2, nCalcTop+1, nCalcLeft+20, 0)
         if nKey > 47 .and. nKey < 58
            nDecimals := val(chr(nKey))
            cPicture := replicate('#', 15 - nDecimals) + '.' + ;
                    if(nDecimals > 0, replicate('#', nDecimals), '')
            set(_SET_DECIMALS, nDecimals)
         endif

         // 'r' was pressed to round number (truncate some decimals)
      case nKey == 82 .or. nKey == 114
         setcolor('+' + maincolor)
         @ nCalcTop+1, nCalcLeft+2 ssay 'Decimals to round to?'
         nKey := ginkey(0)
         scroll(nCalcTop+1, nCalcLeft+2, nCalcTop+1, nCalcLeft+23, 0)
         if (nKey > 47 .and. nKey < 58)
            nKey := val(chr(nKey))
            if nKey <= nDecimals
               num := round(num, nKey)
            endif
         endif

         // 'e' was pressed to clear entry
      case nKey == 69 .or. nKey == 101
         num := nDecPlace := 0

         // 'p' was pressed for paper-tape function
      case nKey == 80 .or. nKey == 112
         lPaperTape := (! lPaperTape)
         @ nCalcTop+3, nCalcLeft+23 ssay if(lPaperTape, 'P', ' ')
         if lPaperTape
            Paper_Tape()
         endif

         // 'm' was pressed to access memory functions
      case nKey == 77 .or. nKey == 109
         setcolor('+*' + maincolor)
         ShowMemKeys()
         nKey := ginkey(0)  // we need to get another key before taking action
         do case
            case nKey == 43       // '+' -- add number to memory
               mem += num
            case nKey == 45       // '-' -- subtract number from memory
               mem -= num
            case nKey == 42       // '*' -- multiple memory by number
               mem *= num
            case nKey == 47       // '/' -- divide memory by number
               mem /= num
            case nKey == 82 .or. nKey == 114   // 'r' -- recall memory
               num := mem
            case nKey == 67 .or. nKey == 99    // 'c' -- clear memory
               mem := 0
         endcase
         setcolor('+' + maincolor)
         ShowMemKeys()

      case nKey == K_ALT_F10   // change color!
         maincolor := ColorPal(maincolor, 16, IF(nCalcLeft > 39, 0, 64))
         // if color was changed, redraw calculator box
         if lastkey() <> 27
            ColorSet(C_CALCULATOR_BOX, maincolor)
            DrawCalcBx()
         endif

      case nKey < SPACEBAR .and. nKey <> K_ESC
         checkmove(nKey, 12, 26, @nCalcTop, @nCalcLeft, @cOldScrn)

   endcase
   ColorSet(C_CALCULATOR_WINDOW)
enddo

// restore hot-key
if hotkey <> 0
   setkey(hotkey, bOldblock)
endif

// restore previous environment
GFRestEnv()
restscreen(nCalcTop, nCalcLeft, nCalcTop+12, nCalcLeft+26, cOldScrn)
set(_SET_DECIMALS, olddecimals)
return NIL

* end function PopCalc()
*--------------------------------------------------------------------*


/*
   Paste() -- paste current calculator value
*/
function paste(a, b, c)   // internal Clipper parameters
local get := getactive()
if get <> NIL
   if get:type == "N"
      get:varPut(num)
   elseif get:type == "C"
      get:varPut(str(num))
   endif
   get:changed := .t.
endif
return NIL

* end function Paste()
*--------------------------------------------------------------------*


/*
   Paper_Tape() -- send calculation to printer
*/
static function Paper_Tape(mvar)
default mvar to ''
if isprinter()
   set device to printer
   @ prow()+1,1 say mvar
   set device to screen
endif
return NIL

* end static function Paper_Tape()
*--------------------------------------------------------------------*


/*
   DrawCalcBx() -- draw calculator box
*/
static function drawcalcbx
ColorSet(C_CALCULATOR_BOX)
shadowbox(nCalcTop, nCalcLeft, nCalcTop+11, nCalcLeft + 24, 3)
SINGLEBOX(nCalcTop+2, nCalcLeft+2, nCalcTop+4, nCalcLeft+22)
@ nCalcTop+5, nCalcLeft+3 ssay 'emory'
@ row()+1,    nCalcLeft+2 ssay 'R + - '
@ row()+1,    nCalcLeft+2 ssay 'C * / '
@ row()+1,    nCalcLeft+1 ssay replicate(chr(196), 7) + chr(217)
@ row()+1,    nCalcLeft+3 ssay 'lear   rint   c ntry'
@ row()+1,    nCalcLeft+3 ssay 'ound   ecimals'
@ nCalcTop+5, nCalcLeft+10 ssay '+  7  8  9  -'
@ row()+1,    nCalcLeft+10 ssay '*  4  5  6  /'
@ row()+1,    nCalcLeft+10 ssay '%  1  2  3  ^'
@ row()+1,    nCalcLeft+13 ssay '=  0  .'
ColorSet(C_CALCULATOR_WINDOW)
@ nCalcTop+3, nCalcLeft+3 ssay space(19)
@ row()+2,    nCalcLeft+2 ssay 'M'
@ row()+4,    nCalcLeft+2 ssay 'C'
@ row(),      col()+6 ssay 'P'
@ row(),      col()+8 ssay 'E'
@ row()+1,    nCalcLeft+2 ssay 'R'
@ row(),      col()+6 ssay 'D'
@ row(),      col()+10 ssay 'Esc'
return NIL

* end static function DrawCalcBx()
*--------------------------------------------------------------------*

* eof popcalc.prg
