/*
    Program: NET_USE()
    System: GRUMPFISH LIBRARY
    Author: Greg Lief
    Copyright (c) 1988-93, Greg Lief
    Clipper 5.x Version
    Compile instructions: clipper netuse /n/w/a
    Opens files for exclusive and shared use in multi-user systems
*/

#include "grump.ch"
#include "error.ch"
#include "inkey.ch"

function net_use(cFile, lExclusive, cIndex1, cIndex2, cIndex3, cIndex4, ;
                 cIndex5, cIndex6, cIndex7, cIndex8)
local lFirstLoop := .t.
local lRetVal := .f.
local box_no
local cWaitMsg
local cAlias
local ptr
local nKey
local bOldHandler := errorblock()
local x
local nIndexes
local lReadonly := .f.
local aIndexes
local oShucks
local cDriver

// if you didn't pass an array of index names, create one!
if valtype(cIndex1) <> "A"
   aIndexes := { cIndex1, cIndex2, cIndex3, cIndex4, ;
                 cIndex5, cIndex6, cIndex7, cIndex8 }
   nIndexes := 0
   do while ++nIndexes < 8 .and. aIndexes[nIndexes] <> NIL
   enddo
   nIndexes--
else
   aIndexes := cIndex1
   nIndexes := len(aIndexes)
endif

default lExclusive to .f.
cFile := upper(cFile)
cWaitMsg := cFile + '.DBF'  // used in kaleidoscope if user has to wait

// post custom error handler to take care of file handle/DBT problems
errorblock( { | e | file_error(e, bOldHandler) } )

// if they passed the word "ALIAS" as part of filename,
// it means they want to use a different alias
if ( ptr := at("ALIAS", cFile) ) > 0
   cAlias := trim(substr(cFile, ptr + 6))
   if ( ptr := at(' ', cAlias) ) > 0
      cAlias := substr(cAlias, 1, ptr - 1)
   endif
   cFile := strtran(cFile, " ALIAS " + cAlias, '')
endif

// if they passed the word "VIA" as part of filename,
// it means they want to use a different database driver
if ( ptr := at("VIA", cFile) ) > 0
   cDriver := trim(substr(cFile, ptr + 4))
   if ( ptr := at(' ', cDriver) ) > 0
      cDriver := substr(cDriver, 1, ptr - 1)
   endif
   cFile := strtran(cFile, " VIA " + cDriver, '')
endif

// if they passed the word "READONLY" as part of filename,
// must open file as readonly
if ( ptr := at("READONLY", cFile) ) > 0
   lReadonly := .t.
   cFile := substr(cFile, 1, ptr - 1)
endif

do while nKey <> K_ESC .and. ! lRetVal
   // potential problematic code is marked by BEGIN SEQUENCE
   begin sequence
      dbUseArea( .T., cDriver, cFile, if(cAlias <> NIL, cAlias, ), ;
                 ! lExclusive, lReadonly)
   recover using oShucks
      // if error is anything other than a sharing violation
      // force an exit from DO WHILE loop and return False
      if oShucks:genCode <> EG_OPEN .or. oShucks:osCode <> 32
         nKey := K_ESC
      endif
   end sequence
   if nKey <> K_ESC
      if ! neterr()      // file opened successfully - now open indexes
         if cIndex1 <> NIL
            x := 0
            begin sequence
               do while ++x <= nIndexes .and. aIndexes[x] <> NIL
                  dbsetindex(aIndexes[x])
               enddo
            recover
               nKey := K_ESC   // to force exit from main DO WHILE loop
               x := 0          // to force lRetVal False three lines down
            end sequence
            lRetVal := (x > nIndexes) // if True, forces exit from main DO loop
            // if we could not open all indeces, close the database
            if ! lRetVal
               dbCloseArea()
            endif
         else
            lRetVal := .t.
         endif
      else
         if lFirstLoop
            if yes_no('File ' + cFile + ' cannot be opened at this time', ;
                      'Would you like to wait')
               gfsaveenv(.t.)
               kaleid(.t., cWaitMsg)
               lFirstLoop := .f.
               nKey := lastkey()
            else
               nKey := K_ESC   // force exit from main loop
            endif
         else
            kaleid(.f., cWaitMsg)
            nKey := lastkey()
         endif
      endif
   endif
enddo
if ! lFirstLoop
   gfrestenv()
   if lRetVal   // file opened successfully - give user aural feedback
      CHARGE
   endif
endif
errorblock(bOldHandler)   // reset previous error handler
return lRetVal

* end function Net_Use()
*--------------------------------------------------------------------*

/*
     Function: File_Error()
     Author:   Greg Lief
     Date:     12/3/91
     Purpose:  Trap missing DBT / file handle errors
*/
static function file_error(oShucks, bOldHandler)
local aMsgs
if oShucks:genCode == EG_OPEN
   // don't display any message for sharing violation (osCode 32)
   if oShucks:osCode <> 32
      aMsgs := { "Cannot open " + oShucks:filename }
      do case
         case oShucks:osCode == 2
            aadd(aMsgs, "(File not found)")
         case oShucks:osCode == 3
            aadd(aMsgs, "(Path not found)")
         case oShucks:osCode == 4
            aadd(aMsgs, "File handle limit exceeded")
            aadd(aMsgs, "Check CONFIG.SYS/CLIPPER env variable")
      endcase
      err_msg(aMsgs)
   else
      neterr(.t.)
   endif
   break oShucks
elseif oShucks:genCode == EG_CORRUPTION
   err_msg( { oShucks:filename + " corrupted, cannot open" } )
   break oShucks
endif
return eval(bOldHandler, oShucks)

* end static function File_Error()
*--------------------------------------------------------------------*

* eof netuse.prg
