/*
   Function: LITE_MENU2()
   System:   Grumpfish Library
   Author:   Greg Lief
   Copyright (c) 1988-90 Greg Lief
   Dialect:  Clipper 5.01
   Compile with: clipper litemen2 /n/w/a
   Purpose:  Replacement for MENU TO... and ACHOICE()

   Warning:  You must include the following line in your program if
             you want to use this function:

                #include "grumpm.ch"

   Caveat:   Pretty minor, actually.  In the same fashion that Clipper
             carries around a PUBLIC variable GETLIST for GETs, this
             function needs to use the variable MENULIST.  Therefore,
             to avoid compiler warnings, you should include this line
             of code in your program somewhere:

                memvar menulist
*/

// begin preprocessor directives

#include "grump.ch"
#include "inkey.ch"

#define  DUMMY           254

#define  ROW             1
#define  COL             2
#define  PROMPT          3
#define  MESSAGE         4
#define  MESSAGECOLOR    5
#define  WHEN            6
#define  ACTION          7
#define  TRIGGERCOLOR    8
#define  VARIABLE        9
#define  TRIGGERPOSITION 10

#define  FORWARD         1
#define  BACKWARD       -1

static unavacolor := '+n/n'   // color for unavailable menu options

// end preprocessor directives

function lite_menu2(menuarray, plaincolor, selection, varname, bevent, ;
                   ntimeout, bexit, trigcolor, cleft, cright, lCenter, ;
                   lClearmsgs, nMessrow, nMesscol)
local marray_ := aclone(menuarray)
local num_elem := len(marray_)
local xx
local nkey := 0
local luvletters := ''
local fallout := .f.
local ptr
local hilitcolor
local direction := FORWARD
local nstart
local lMessages := .f.

gfsaveenv(, 0)

// if no time-out was specified, use a ridiculous default
default ntimeout to 100000000

// set default colors for unselected, selected, and unavailable options
default plaincolor to setcolor()
xx := at(',', plaincolor)
ptr := rat(',', plaincolor)
if xx == ptr     // this means only unselected & selected colors were passed
   hilitcolor := substr(plaincolor, xx + 1)
else
   hilitcolor := substr(plaincolor, xx + 1, ptr - xx - 1)
   /*
      if a color string was passed that contains two commas, then use
      whatever lies to the right of the second comma as the color for
      unavailable menu options -- but if we are using the current
      color setting, instead use the default (+N/N) as defined above
   */
   if plaincolor != setcolor()
      unavacolor := substr(plaincolor, ptr + 1)
   endif
endif

// establish default color for highlighting trigger letters
default trigcolor to "+" + plaincolor

/*
   determine initial highlighted item default to 1 -- also perform
   error-checking to ensure they didn't specify an invalid selection
*/
if selection == NIL .or. (selection < 1 .or. selection > num_elem)
   selection := 1
endif

// display the options and build the string of trigger letters
dispbegin()
for xx = 1 to num_elem

   // determine whether we are using messages or not
   if ! lMessages .and. marray_[xx, MESSAGE] != NIL
      lMessages := .t.
      // set defaults for various message-related items
      default lClearmsgs to .t.

      // if user didn't specify whether or not to center the message,
      // use the default SET MESSAGE CENTER setting (however, message
      // column is specified, then assume they don't want centering)
      if nMesscol != NIL
         lCenter := .f.
      else
         nMesscol := 0
         default lCenter to set(_SET_MCENTER)
      endif

      // if user didn't specify message row, use the system default
      // (or use the bottom row if the default is still zero)
      if nMessrow == NIL .and. ( nMessrow := set(_SET_MESSAGE) ) == 0
         nMessrow := maxrow()
      endif
   endif

   // if there are any code blocks in the place of menu prompts,
   // evaluate them now to create the character string prompts
   if valtype(menuarray[xx, PROMPT]) == "B"
      marray_[xx, PROMPT] := eval(menuarray[xx, PROMPT])
   endif

   // next make sure that any WHEN clause for this option is
   // satisfied, and that the option can therefore be selected
   if marray_[xx, WHEN] == NIL .or. eval( marray_[xx, WHEN] )
      /*
         the default is to add the first non-space character.
         However, if there is a tilde embedded in this menu
         option, use the letter directly following it.
      */
      if (ptr := at("~", marray_[xx, PROMPT]) ) > 0
         luvletters += upper(substr(marray_[xx, PROMPT], ptr + 1, 1))
      else
         luvletters += upper(left(ltrim(marray_[xx, PROMPT]), 1))
         // determine how much white space precedes the actual prompt
         ptr := len(marray_[xx, PROMPT]) - len(ltrim(marray_[xx, PROMPT])) + 1
      endif
      /*
         Add actual trigger letter position to nested array for this
         menu option.  I discovered the hard way that this is necessary
         in the event that the menu prompt begins with a blank space.
      */
      // but first, a little kludge... users with old GRUMPM.CH
      // files might not have the ninth element in the array
      if len(marray_[xx]) < VARIABLE
         aadd(marray_[xx], NIL )
      endif
      aadd(marray_[xx], ptr)
   else
      // add dummy placeholder for unavailable options
      luvletters += chr(DUMMY)
   endif
   // strip out any tildes now
   marray_[xx, PROMPT] := strtran(marray_[xx, PROMPT], "~", "")
   ShowOption(marray_[xx], plaincolor, trigcolor)
next
dispend()

/*
   if LUVLETTERS is full of dummy placeholders, that means there
   are no selectable menu options at this time - fall out
*/
if luvletters == replicate(chr(DUMMY), num_elem)
   selection := 0
else
   // commence main key-grabbing loop
   do while nkey != K_ENTER .and. nkey != K_ESC
      // make sure that current option is available for selection
      if marray_[selection, WHEN] == NIL .or. eval(marray_[selection, WHEN])

         // then display current option in highlight color
         @ marray_[selection, ROW], marray_[selection, COL] ssay ;
                 marray_[selection, PROMPT] color hilitcolor

         // if we are using messages, either display the message (if
         // there is one) or clear the message row
         if lMessages
            if marray_[selection, MESSAGE] == NIL .and. lClearmsgs
               scroll(nMessrow, nMesscol, nMessrow, maxcol(), 0)
            elseif lCenter
               if lClearmsgs
                  @ nMessrow, 0 ssay ;
                            padc(marray_[selection, MESSAGE], maxcol()+1)     ;
                            color if(marray_[selection, MESSAGECOLOR] == NIL, ;
                                  plaincolor, marray_[selection, MESSAGECOLOR])
               elseif marray_[selection, MESSAGE] != NIL
                  scrncenter(nMessrow, marray_[selection, MESSAGE],         ;
                             if(marray_[selection, MESSAGECOLOR] == NIL,    ;
                             plaincolor, marray_[selection, MESSAGECOLOR])  )
               endif
            elseif lClearmsgs
               @ nMessrow, nMesscol ssay ;
                      padr(marray_[selection, MESSAGE], maxcol()+1)        ;
                      color if(marray_[selection, MESSAGECOLOR] == NIL,    ;
                            plaincolor, marray_[selection, MESSAGECOLOR])
            else
               @ nMessrow, nMesscol ssay marray_[selection, MESSAGE] color ;
                               if(marray_[selection, MESSAGECOLOR] == NIL, ;
                               plaincolor, marray_[selection, MESSAGECOLOR])
            endif
         endif
         if fallout
            exit
         else
            /*
               begin keypress wait loop -- necessary to accommodate
               recurring event and keyboard inactivity timeout
            */
            nstart := seconds()
            do while (nkey := inkey()) == 0 .and. seconds() - nstart < ntimeout
               if bevent != NIL
                  eval(bevent)
               endif
            enddo

            // we timed out!
            if nkey == 0
               // if no exitevent was specified, use screen blanker
               if bexit == NIL
                  blankscr3(-1)
               else
                  eval(bexit)
               endif
            endif
            do case

               /*
                  First, check for left and right arrow keypresses.
                  If the LEFT and RIGHT clauses were used in conjunction
                  with the MENU TO command, we must stuff characters into
                  the buffer now.  This would generally be used with a
                  pull-down menu system.
               */
               case nkey == K_LEFT .and. cleft != NIL
                  keyboard cleft

               case nkey == K_RIGHT .and. cright != NIL
                  keyboard cright

               /*
                  next, check for action block attached to last keypress
                  if there is one, evaluate it and pass it the name of
                  the MENU TO variable along with the current highlighted
                  option (e.g., SEL[5] if you are on the fifth option)
               */
               case setkey(nkey) != NIL
                  eval(setkey(nkey), procname(1), procline(1), varname + ;
                       "[" + ltrim(str(selection)) + "]")

               // go down one line, observing wrap-around conventions
               case nkey == K_DOWN .or. nkey == K_RIGHT
                  direction := FORWARD
                  ShowOption(marray_[selection], plaincolor, trigcolor)
                  if selection == num_elem
                     selection := 1
                  else
                     selection += direction
                  endif

               // go up one line, observing wrap-around conventions
               case nkey == K_UP .or. nkey == K_LEFT
                  direction := BACKWARD
                  ShowOption(marray_[selection], plaincolor, trigcolor)
                  if selection == 1
                     selection := num_elem
                  else
                     selection--
                  endif

               // jump to top option
               case nkey == K_HOME .or. nkey == K_PGUP
                  // no point in changing color if we're already there
                  if selection != 1
                     ShowOption(marray_[selection], plaincolor, trigcolor)
                     selection := 1
                     direction := FORWARD
                  endif

               // jump to bottom option
               case nkey == K_END .or. nkey == K_PGDN
                  // no point in changing color if we're already there
                  if selection != num_elem
                     ShowOption(marray_[selection], plaincolor, trigcolor)
                     selection := num_elem
                     direction := BACKWARD
                  endif

               // first letter - jump to appropriate option
               case upper(chr(nkey)) $ luvletters
                  ShowOption(marray_[selection], plaincolor, trigcolor)
                  selection := at(upper(chr(nkey)), luvletters)
                  fallout := .t.

            endcase
         endif
      else
         // keep moving in current direction
         selection += DIRECTION
         // wrap-around
         if selection == 0
            selection := num_elem
         elseif selection > num_elem
            selection := 1
         endif
      endif
   enddo
   // if there is an action block attached to this selection, run it
   if lastkey() != K_ESC
      if marray_[selection][ACTION] != NIL
         // reset cursor because next module might expect it to be on
         gfrestenv()
         /*
           if there is a variable associated with this menu option,
           we must save the value returned from the ACTION block
           NOTE: we must ensure that array actually contains the
           VARIABLE element, which was a late addition
         */
         if len(marray_[selection]) >= VARIABLE .and. ;
                    marray_[selection][VARIABLE] != NIL
            eval(marray_[selection][VARIABLE], eval(marray_[selection][ACTION]))
         else
            eval(marray_[selection][ACTION])
         endif
         // now that that's over with, turn the cursor back off
         gfsaveenv(, 0)
      endif
   else
      selection := 0  // since they Esc'd out, return a zero
   endif
endif
gfrestenv()
return selection

* end function Lite_Menu2()
*--------------------------------------------------------------------*


/*
  Function: ShowOption()
  Purpose:  Display current prompt in mixed colors
*/
static function ShowOption(item, plaincolor, trigcolor)
local ptr
// first make sure that this option is available for selection
if item[WHEN] == NIL .or. eval( item[WHEN] )
   if item[TRIGGERPOSITION] != 1
      @ item[ROW], item[COL] ssay item[PROMPT] color plaincolor
      @ item[ROW], item[COL] + item[TRIGGERPOSITION] - 1 ssay ;
           substr(item[PROMPT], item[TRIGGERPOSITION], 1) ;
           color if( empty(item[TRIGGERCOLOR]), trigcolor, item[TRIGGERCOLOR] )
   else
      @ item[ROW], item[COL] ssay left(item[PROMPT], 1) ;
        color if( empty(item[TRIGGERCOLOR]), trigcolor, item[TRIGGERCOLOR] )
      dispout(substr(item[PROMPT], 2), plaincolor)
   endif
else
   // display unavailable menu options in gray on black
   @ item[ROW], item[COL] ssay item[PROMPT] color unavacolor
endif
return nil

* end static function ShowOption()
*--------------------------------------------------------------------*

* eof litemen2.prg
