/*
    Function: IsConfig()
    System: GRUMPFISH LIBRARY
    Author: Greg Lief - inspired by Eric Lewis
    Copyright (c) 1990 - All Rights Reserved
    Clipper 5.x version
    Compile instructions: clipper isconfig /n/w/a

    Checks either CONFIG.SYS or SHELL.CFG file for either
    FILES / FILE HANDLES and/or BUFFERS / CACHE BUFFERS statements

    Calls: GFReadLine()   (function contained in READLINE.PRG)
         : Yes_No()       (function contained in YESNO.PRG)
         : WaitOn()       (function contained in WAITING.PRG)
         : ShadowBox()    (function contained in SHADOWBO.PRG)
         : ByeByeBox()    (function contained in BYEBYE.PRG)
*/

// begin preprocessor directives

#include "grump.ch"
#include "fileio.ch"

#define TEMP_FILE  "temp.cfg"

// end preprocessor directives

function isconfig(cFile, nFiles, nBuffers)
local cDrive
local lNoFiles := .t.
local lNoBuffers := .t.
local lRetval := .t.
local cString := space(1)
local nInHandle
local nOutHandle
local buffer
local cFileString
local cBuffString
local lOldexact
local cBackupFile
GFSaveEnv(, 0)
lOldexact := set(_SET_EXACT, .f.)    // turn off SET EXACT
cFile := if(cFile == NIL, "CONFIG.SYS", ;
         if(! upper(cFile) $ "CONFIG.SYSSHELL.CFG", "CONFIG.SYS", upper(cFile)))
default nFiles to 50
default nBuffers to 10
cFileString := "FILE" + if(cFile = "CONFIG.SYS", "S", " HANDLES")
cBuffString := if(cFile = "SHELL.CFG", "CACHE ", "") + "BUFFERS"

// determine which drive is the booting drive by looking at the comspec.
cDrive := substr(gete("COMSPEC"), 1, at(":", gete("COMSPEC")) - 1)
if file(cDrive + ":\" + cfile)

   // read in from the config.sys file out to the temp.sys file
   nInHandle := fopen(cDrive + ":\" + cfile, FO_READWRITE)
   nOutHandle := fcreate(cDrive + ":\" + TEMP_FILE, FC_NORMAL)

   do while gfreadline(@cString, nInHandle)
      cString := upper(cString)

      // check for the files and/or buffers lines and the correct number
      do case
         case cFileString $ cString
            // check not only for adequate # of files, but that this
            // line begins with the word FILES or FILE HANDLES
            if cString <> cFileString .or. ;
                    val(substr(cString, at("=", cString) + 1)) < nFiles
               // write a new "FILE HANDLES=" line
               fwrite(nOutHandle, cFileString + "=" + ltrim(str(nFiles)) + CRLF)
               lRetval := .f.
            else
               fwrite(nOutHandle, cString + CRLF)
            endif
            lNoFiles := .f.

         case cBuffString $ cString
            // check not only for adequate # of buffers, but that this
            // line begins with the word BUFFERS or CACHE BUFFERS
            if upper(cString) <> cBuffString .or. ;
                     val(substr(cString, at("=", cString) + 1)) < nBuffers
               // check to see if using Expanded Memory switch ("/X")
               if "/X" $ cString
                  buffer := " /X"
               elseif "," $ cString
                  buffer := substr(cString, at(",", cString))
               else
                  buffer := ''
               endif
               // Write a new "BUFFERS=" line
               FWRITE(nOutHandle, cBuffString + "=" + LTRIM(STR(nBuffers)) + ;
                      buffer + CRLF)
               lRetval := .f.
            else
               fwrite(nOutHandle, cString + CRLF)
            endif
            lNoBuffers := .f.

         otherwise
            // write the string out to the new file
            if cString <> chr(26)
               fwrite(nOutHandle, cString + CRLF)
            endif
      endcase
   enddo

   if lNoFiles .or. lNoBuffers
      // move to the end of the file less two bytes
      // if the file ends with a cr/lf, erase them
      fseek(nOutHandle, -2, FS_END)
      // if the file does not end with a cr/lf, skip to the very end
      cString := freadstr(nOutHandle, 2)
      if cString <> CRLF
         fseek(nOutHandle, 2, FS_RELATIVE)
      endif

      // check to see if a file handles= line was found
      if lNoFiles
         // write a new "file handles=" line
         fwrite(nOutHandle, cFileString + "=" + ltrim(str(nFiles)) + CRLF)
         lRetval := .f.
      endif

      // check to see if a buffers= line was found
      if lNoBuffers
         // write a new cache buffers= line
         fwrite(nOutHandle, cBuffString + "=" + ltrim(str(nBuffers)) + CRLF)
         lRetval := .f.
      endif
   endif

   // close the input file
   fclose(nInHandle)

   // check to see if the file needs an eof
   fseek(nOutHandle, 0, 2)
   if freadstr(nOutHandle, 1) <> chr(26)
      fwrite(nOutHandle, chr(26))
   endif

   // close the output file
   fclose(nOutHandle)

   // confirm copy
   if ! lRetval
      if yes_no2( { cFile + " requires changes in order to run this program", ;
                    "Would you like me to make these changes now?" }, , ;
                    " Change " + cfile, " Ignore ")
         cBackupFile := cDrive + ":\" + substr(cfile,1,at(".",cfile)-1) + ".bak"
         if file(cBackupFile)
            ferase(cBackupFile)
         endif
         frename(cDrive + ":\" + cfile, cBackupFile)
         frename(cDrive + ":\" + TEMP_FILE, cDrive + ":\" + cfile)
      else
         lRetval := .t.
         waiton( { "The " + cFile + " file must contain the following lines:", ;
                   "'" + cFileString + "=" + ltrim(str(nFiles)) + "'", ;
                   "'" + cBuffString + "=" + ltrim(str(nBuffers)) + "'" } )
         inkey(5)
         waiton(.F.)
      endif
   endif
   ferase(cDrive + ":\" + TEMP_FILE)
else
   // the file does not exist, so create one
   nOutHandle := fcreate(cDrive + ":\" + cfile, FC_NORMAL)
   fwrite(nOutHandle, cFileString + "=" + ltrim(str(nFiles)) + CRLF + ;
                     cBuffString + "=" + ltrim(str(nBuffers)) + CRLF)
   lRetval := .f.

   // check to see if the file needs an eof
   fseek(nOutHandle, 0, 2)
   if freadstr(nOutHandle, 1) <> chr(26)
      fwrite(nOutHandle, chr(26))
   endif

   // close the output file
   fclose(nOutHandle)
endif

if ! lRetval
   ColorSet(C_MESSAGE)
   buffer := ShadowBox(10, 0, 14, 79, 2)
   SCRNCENTER(11, "Changes have been to the " + cFile + " file." )
   SCRNCENTER(11, "Please restart the computer by turning it off")
   SCRNCENTER(12, "or pressing Ctrl-Alt-Del at the same time.")
   SCRNCENTER(13, "Then start this program again.")
   ginkey(0)
   ByeByeBox(buffer)
endif
GFRestEnv()
set(_SET_EXACT, lOldexact)    // restore previous SET EXACT setting
return lRetval

* end function IsConfig()
*--------------------------------------------------------------------*

* eof isconfig.prg
