/*
   Program: INDEXBAR()
   System: GRUMPFISH LIBRARY
   Author: Greg Lief
   Copyright (c) 1988-90, Greg Lief
   Clipper 5.x version
   Compile instructions: clipper indexbar /n/w/a

   Procs & Fncts: IndexBar()
                : GFShowBar()
                : GFKillBar()

           Calls: SHADOWBOX()   (function in SHADOWB.PRG)

   IndexBar() displays a graph showing percentage complete when
   indexing files.  When indexing is complete, IndexBar() will
   then strip the graph UDF() call from the index file so that you
   don't have to look at the graph when the index is updated.

*/

// begin preprocessor directives

#include "grump.ch"
#include "fileio.ch"

// end preprocessor directives

// begin global declarations

static num_recs        // total number of records in current database
                       // I put it here instead of referring to RECCOUNT()
                       // because it speeds things up a klick or two

// end global declarations

/*
    Function: IndexBar()
    Syntax:   IndexBar(<cFile>, <cKey>, <nRow>, <lShowcount>, <lUnique>, <cMsg>)
              <cFile> = name of index file... to be displayed FYI
              <cKey>  = self-explanatory -- if not passed, will
                        be assumed to be the same as the index filename
              <nRow> = row at which to place window (default: centered)
              <lShowcount> = whether or not to display record counter
              <lUnique> = whether or not to create UNIQUE index
              <cMsg>    = message to use rather than the default
                          (which is "now creating index file <cfile>")
    Example:  IndexBar('customer', 'lname')
    Returns:  Logical: .T. if index is built, .F. if not
    Notes:    will not process empty database files
*/
function indexbar(cfile, mkey, nRow, showcount, lunique, cmsg)
local ret_val := .f.
local xx
local oldscrn
local bkey
local nLeft
default mkey to cfile
default showcount to .f.
default lunique to .f.
GFSaveEnv()
// make sure that the index key expression is valid to preclude a crash!
if ! valtype(mkey) $ "UE"
   bKey := makeblock(mkey)       // see GRUMP.CH for MakeBlock()
   default nRow to ( maxrow() / 2 ) - 3
   dbClearIndex()
   // no point in going thru this rigmarole on an empty database (which
   // would give us zero divide, a/k/a "the great divide")
   if lastrec() > 0
      num_recs := reccount()
      // if no extension specified, add default now
      if ! "." $ cfile
         AddExtension(cfile, substr(indexext(), 2))
      endif
      ColorSet(C_MESSAGE)
      nLeft := ( maxcol() / 2 ) - 30
      dispbegin()
      oldscrn := shadowbox(nRow, nLeft, nRow + 6, nLeft + 63, 2)
      @ nRow + 3, nLeft + 2  ssay ''
      @ nRow + 3, nLeft + 61 ssay ''
      @ nRow + 4, nLeft + 2  ssay '' + replicate("", 9) + ';'
      @ nRow + 5, nLeft + 2  ssay '0'
      for xx := 1 to 10
         @ nRow + 5, nLeft + xx * 6 ssay str(10 * xx, 3)
      next
      SCRNCENTER(nRow + 1, if(cmsg != NIL, cmsg, ;
                              "Creating index file " + cfile + "...") )
      ColorSet(C_APICK_STATUSBAR)
      @ nRow + 3, nLeft + 3 ssay replicate(chr(177), 58)
      setpos(nRow + 3, nLeft + 3)
      dispend()
      ColorSet(C_APICK_INDICATOR)
      go top
      /*
         I know it looks dumb, but the parentheses around the indexkey
         (2nd parameter) force dbCreateIndex() to evaluate the block
         (3rd parameter) for each record in the DBF.  If you remove
         the parentheses, the block will only be evaluated once, which
         means that the status bar will not be drawn.
      */
      dbCreateIndex(cfile, "(" + mkey + ")", ;
                    {|| gfshowbar(bkey, recno(), showcount)}, lunique)
      //  ret_val := gfkillbar(cfile, mkey)
      dbClearIndex()
      dbSetIndex(cfile)
      byebyebox(oldscrn)
   else
      dbCreateIndex(cfile, mkey, bkey, lunique)
      dbClearIndex()
      dbSetIndex(cfile)
   endif
   ret_val := .t.
endif
GFRestEnv()
return ret_val

* end function IndexBar()
*--------------------------------------------------------------------*


/*
    Function: GFShowBar()
    Syntax:   GFShowBar(<index key>)
              <index key> = self-explanatory
    Purpose:  Display status graph whilst indexing
    Example:  See above -- NOT MEANT FOR USE AS A STANDALONE
    Returns:  the index key for each record
*/
static function gfshowbar(bkey, nrec, showcount)
local oldcol
static ngraphlen
if ngraphlen == NIL  // initialize first time only
   ngraphlen := 0
endif
if nrec <= num_recs
   if showcount
      ColorSet(C_MESSAGE)
      oldcol := col()  // save column position for restoration below
      SCRNCENTER(row() - 1, if(nrec != num_recs, ;
                               "Indexing Record " + ltrim(str(nrec)) + ;
                               " of " + ltrim(str(num_recs)), ;
                               space(18) + "Finished !" + space(18)))
      setpos(row()+1, oldcol)     // reset to proper column
      ColorSet(C_APICK_INDICATOR)
   endif

   // draw more of the graph if the relative position has changed
   do while ngraphlen < int(nrec / num_recs * 58)
      dispout(chr(219))
      ngraphlen++
   enddo

   // if we're done, reset graph length variable for next time
   if nrec == num_recs
      ngraphlen := NIL
   endif
endif
return eval(bkey)

* end static function GFShowBar()
*--------------------------------------------------------------------*

#ifdef INACTIVE_FOR_NOW

/*
    Function: GFKillBar()
    Syntax:   GFKillBar(<filename>)
              <filename> = name of index file to be modified - enclose
              in quotes
    Purpose:  Strip out the call to SHOWGRAPH()
    Example:  GFKillBar('name.ntx')
    Returns:  Logical: True (.T.) if successful, False (.F.) if not
*/
static function gfkillbar(cfile, mkey)
local defaultdir := set(_SET_DEFAULT)
local handle, buffer, pos, ret_val := .f.
/*
   only attempt to open the index low-level if we are guaranteed that no
   indexes are currently open!  trying to do a low-level write on an open
   .ntx file will cause lost clusters, general confusion, and a major
   headache for you the developer, so try to avoid it like the plague
*/
if indexord() == 0
   handle := fopen(defaultdir + if(! empty(defaultdir), '\', '') + ;
                   cfile, FO_READWRITE)
   if handle != -1
      // skip past index file header->22 bytes for NTX, 24 bytes for NDX
      fseek(handle, if(indexext() == ".NTX", 22, 24))
      buffer := mkey + replicate(chr(0), 254 - len(mkey))
      if fwrite(handle, buffer) == 254
         ret_val := .t.
      endif
      fclose(handle)
   endif
endif
return(ret_val)

* end static function GFKillBar()
*--------------------------------------------------------------------*

#endif

* eof indexbar.prg
