/*
   Program: HELPDEV.PRG
   System: GRUMPFISH LIBRARY
   Author: Greg Lief
   Copyright (c) 1988-90, Greg Lief
   Clipper 5.01 Version
   Compile instructions: clipper helpdev /n/a

   Procs & Fncts: HELPDEV()
                : HELPSCREEN()
                : PICKACOLOR()
                : DRAW_FTHD()
                : EDITHELP()
                : HELPHEAD()
   Creates/displays user-defined help screens
*/

#include "box.ch"
#include "grump.ch"
#include "grumph.ch"
#include "inkey.ch"
#include "achoice.ch"

static cMainScreen                      // saved entry screen
static helptext                         // help text
static aHelpDefs := { 0,   ;            // top row of help window
                      0,   ;            // left column of help window
                      20,  ;            // bottom row of help window
                      79,  ;            // right column of help window
                      2,   ;            // box outline type (single)
                      7,   ;            // color for box outline (W/N)
                      112, ;            // color for title (N/W)
                      112, ;            // color for footer (N/W)
                      15,  ;            // color for help text (+W/N)
                      "Help Screen", ;  // default title
        "  Alt-P print  Esc exit", ;  // default footer
                      .t., ;            // index entry?
                      "help" }          // default filename (HELP.DBF)

// manifest constant for readability only...
#define HELPFILE   aHelpDefs[_HELP_FILENAME]

function helpdev(cProc, mline, mvar)
local nHotKey
local nWorkArea := select()
local lOldInsert
local mid
local lOldScore := set(_SET_SCOREBOARD, .F.)
local bOldBlock
local bOldF2
local nKey
memvar helpcode   // would be declared in the calling program if it exists

/*
   usually if HELPDEV() was called from anything other than GINKEY(),
   cProc will be a null string.  We would then have to use the procedure
   name three levels deep.  However, in the event that the procedure
   name passed to HELPDEV() is an internal Clipper function (beginning
   with double underscore "__"), then we must use the procedure name
   four levels deep.
*/
if empty(cProc)
   cProc := procname(3)
elseif left(cProc, 2) = '__'
   cProc := procname(4)
endif
bOldBlock := setkey(nHotKey := lastkey(), NIL)
GFSaveEnv(, 0)                      // turn off cursor
if ! file(HELPFILE + '.dbf')
   waiton('initializing help database... please wait')
   dbcreate(HELPFILE + ".dbf", { {"THEPROC",    "C", 10, 0} , ;
                                 {"VAR",        "C", 20, 0} , ;
                                 {"TEXT",       "M", 10, 0} , ;
                                 {"TOPROW",     "N",  2, 0} , ;
                                 {"BOTROW",     "N",  2, 0} , ;
                                 {"LT_COL",     "N",  2, 0} , ;
                                 {"RT_COL",     "N",  2, 0} , ;
                                 {"BOXCOLOR",   "N",  3, 0} , ;
                                 {"TITCOLOR",   "N",  3, 0} , ;
                                 {"TXTCOLOR",   "N",  3, 0} , ;
                                 {"BOXNO",      "N",  1, 0} , ;
                                 {"TITLE",      "C", 30, 0} , ;
                                 {"FTCOLOR",    "N",  3, 0} , ;
                                 {"FOOTER",     "C", 30, 0} , ;
                                 {"INDEXENTRY", "L", 1, 0} } )
   use (HELPFILE) new exclusive
   index on _field->theproc + _field->var to (HELPFILE)
   waitoff()
else
   use (HELPFILE) new
endif
cMainScreen := savescreen()
if ! file(HELPFILE + indexext())    // must rebuild help index file
   if flock()
      waiton("Indexing help database")
      index on _field->theproc + _field->var to (HELPFILE)
      unlock
   else
      waiton("Database is being reindexed... please wait or press Esc to abort")
      do while ! flock() .and. inkey() <> K_ESC
      enddo
   endif
   waitoff()
else
   set index to (HELPFILE)
endif

// seek on helpcode first if such a variable is defined
if dbseek( if(type('helpcode') == "U", padr(cProc, 10) + mvar, helpcode ) )
   shadowbox((HELPFILE)->toprow, (HELPFILE)->lt_col, (HELPFILE)->botrow, ;
             (HELPFILE)->rt_col, (HELPFILE)->boxno,,,;
             Color_N2S((HELPFILE)->boxcolor))
   mid := (HELPFILE)->lt_col + (int((HELPFILE)->rt_col - (HELPFILE)->lt_col) / 2)
   @ (HELPFILE)->toprow, mid - int(len(trim((HELPFILE)->title))/2) ;
            ssay trim((HELPFILE)->title) color Color_N2S((HELPFILE)->titcolor)
   @ (HELPFILE)->botrow, mid - int(len(trim((HELPFILE)->footer))/2) ;
            ssay trim((HELPFILE)->footer) color Color_N2S((HELPFILE)->ftcolor)
   setcolor(Color_N2S((HELPFILE)->txtcolor))
   do while nKey <> K_ESC
      memoedit((HELPFILE)->text, (HELPFILE)->toprow + 1, (HELPFILE)->lt_col + 1, ;
              (HELPFILE)->botrow - 1, (HELPFILE)->rt_col - 1, .f., 'EditHelp')
      nKey := lastkey()
   enddo
else
   DispBox(0, 0, maxrow() - 4, maxcol(), B_SINGLE + ' ', '+w/n')
   SCRNCENTER(0, '<< help screen >>', '+w/n')
   SCRNCENTER(maxrow() / 2 - 1, 'no help text defined...', '+w/n')
   SCRNCENTER(maxrow() / 2, 'Press F2 to create or any other key to go back', '+w/n')
   bOldF2 := setkey(K_F2, NIL)   // shut off the F2 key for below...
   if ginkey(0) == K_F2
      restscreen(,,,, cMainScreen)
      lOldInsert := readinsert(.T.)    // toggle insert mode on, save status
      setcolor(aHelpDefs[_HELP_BOXCOLOR])
      shadowbox(aHelpDefs[_HELP_TOP], aHelpDefs[_HELP_LEFT], aHelpDefs[_HELP_BOTTOM], aHelpDefs[_HELP_RIGHT], aHelpDefs[_HELP_OUTLINE])
      mid := aHelpDefs[_HELP_LEFT] + (int(aHelpDefs[_HELP_RIGHT] - aHelpDefs[_HELP_LEFT]) / 2)
      @ aHelpDefs[_HELP_BOTTOM], mid - 12 ssay ' Ctrl-W save   Esc quit '
      setcursor(2)
      helptext := memoedit([], aHelpDefs[_HELP_TOP] + 1, aHelpDefs[_HELP_LEFT] + 1, ;
                           aHelpDefs[_HELP_BOTTOM] - 1, aHelpDefs[_HELP_RIGHT] - 1, .t.)
      readinsert(lOldInsert)
      setcursor(0)
      if ! empty(helptext)
         append blank
         if ! neterr()

            (HELPFILE)->theproc := if(type('helpcode') == "U", ;
                                   cProc, substr(helpcode, 1, 10))
            (HELPFILE)->var     := if(type('helpcode') == "U", ;
                                   mvar, substr(helpcode, 11, 10))
            (HELPFILE)->text    := helptext

            // redraw bottom row of box, clearing out Save/Quit msg
            @ aHelpDefs[_HELP_BOTTOM], mid - 12 ssay ;
                  replicate(substr(BOXFRAMES[aHelpDefs[_HELP_OUTLINE]], 6, 1), 24)
            HelpScreen("A")
         else
            Err_Msg( { NETERR_MSG } )
         endif
      endif
   endif
   setkey(K_F2, bOldF2)        // restore F2 key
endif

// restore hot-key
setkey( nHotKey, bOldBlock )
restscreen(,,,, cMainScreen)
use
select(nWorkArea)
set(_SET_SCOREBOARD, lOldScore)    // in case you were keeping SCORE
GFRestEnv()
return NIL

* end of function HelpDev()
*--------------------------------------------------------------------


/*
   Function: HelpScreen()
   allow user to select box coordinates, box type, title,
   and colors for this help screen
*/
static function HelpScreen(mode)     // A = adding new, E = editing existing
local xx, yy, key, buffer2, buffer3, firstloop, redraw, palettetop
local getlist := {}
local mainloop := .t.
static boxtypes := { , , , , , 'no border' }
static palette := []    // string containing all color attributes
static lIndexentry

// load lIndexentry flag and boxtypes array on first pass only...
// NOTE: lIndexentry is for downward compatibility only because
// older versions of HELP.DBF will not have the INDEXENTRY field
if lIndexentry == NIL
   lIndexentry := ( (HELPFILE)->( fieldpos("indexentry") ) > 0 )
   for xx := 1 to 5
      boxtypes[xx] := substr(BOXFRAMES[xx], 1, 8)
   next
endif

if mode == 'E'
   aHelpDefs[_HELP_TOP]      := (HELPFILE)->toprow
   aHelpDefs[_HELP_LEFT]  := (HELPFILE)->lt_col
   aHelpDefs[_HELP_BOTTOM]   := (HELPFILE)->botrow
   aHelpDefs[_HELP_RIGHT] := (HELPFILE)->rt_col
   aHelpDefs[_HELP_OUTLINE]     := (HELPFILE)->boxno
   aHelpDefs[_HELP_BOXCOLOR]    := (HELPFILE)->boxcolor
   aHelpDefs[_HELP_TITLECOLOR]  := (HELPFILE)->titcolor
   aHelpDefs[_HELP_FOOTERCOLOR] := (HELPFILE)->ftcolor
   aHelpDefs[_HELP_TEXTCOLOR]   := (HELPFILE)->txtcolor
   aHelpDefs[_HELP_TITLE]       := (HELPFILE)->title
   aHelpDefs[_HELP_FOOTER]      := (HELPFILE)->footer
   if lIndexentry
      aHelpDefs[_HELP_INDEXENTRY]  := (HELPFILE)->indexentry
   endif
endif

do while mainloop
   ColorSet(C_MESSAGE)
   buffer2 := ShadowBox(maxrow()-3, 11, maxrow(), 68, 2)
   SCRNCENTER(maxrow()-2, 'use arrow keys to drag top left corner of help box')
   SCRNCENTER(maxrow()-1, 'press Enter when finished')

   // get top left coordinate for help box
   firstloop := .t.
   key := 0
   do while key <> K_ENTER
      redraw := .t.
      key := inkey(0)
      if firstloop
         ByeByeBox(buffer2)
         firstloop := .f.
      endif
      dispbegin()
      do case

         case key == K_DOWN .and. aHelpDefs[_HELP_TOP] < maxrow()-4 .and. ;
                                 aHelpDefs[_HELP_TOP] < aHelpDefs[_HELP_BOTTOM] - 4
            restscreen(,,,, cMainScreen)
            aHelpDefs[_HELP_TOP]++

         case key == K_UP .and. aHelpDefs[_HELP_TOP] > 0
            aHelpDefs[_HELP_TOP]--

         case key == K_RIGHT .and. aHelpDefs[_HELP_LEFT] < maxcol() - 9 .and. ;
                                  aHelpDefs[_HELP_LEFT] < aHelpDefs[_HELP_RIGHT] - 11
            restscreen(,,,, cMainScreen)
            aHelpDefs[_HELP_LEFT]++

         case key == K_TAB .and. aHelpDefs[_HELP_LEFT] < maxcol() - 14 .and. ;
                                aHelpDefs[_HELP_LEFT] < aHelpDefs[_HELP_RIGHT] - 16
            restscreen(,,,, cMainScreen)
            aHelpDefs[_HELP_LEFT] += 5

         case key == K_SH_TAB .and. aHelpDefs[_HELP_LEFT] > 5
            restscreen(,,,, cMainScreen)
            aHelpDefs[_HELP_LEFT] -= 5

         case key == K_LEFT .and. aHelpDefs[_HELP_LEFT] > 0
            restscreen(,,,, cMainScreen)
            aHelpDefs[_HELP_LEFT]--

         case key == K_CTRL_LEFT .and. aHelpDefs[_HELP_LEFT] > 0
            restscreen(,,,, cMainScreen)
            aHelpDefs[_HELP_LEFT] := 0

         case key == K_HOME           // reposition at top left corner
            aHelpDefs[_HELP_LEFT] := aHelpDefs[_HELP_TOP] := 0

         otherwise
            redraw := .f.
      endcase
      if redraw
         @ aHelpDefs[_HELP_TOP],aHelpDefs[_HELP_LEFT],aHelpDefs[_HELP_BOTTOM],aHelpDefs[_HELP_RIGHT] box BOXFRAMES[aHelpDefs[_HELP_OUTLINE]] ;
                  color Color_N2S(aHelpDefs[_HELP_BOXCOLOR])
         setcolor(Color_N2S(aHelpDefs[_HELP_TEXTCOLOR]))
         memoedit(helptext, aHelpDefs[_HELP_TOP]+1, aHelpDefs[_HELP_LEFT]+1, aHelpDefs[_HELP_BOTTOM]-1, ;
                  aHelpDefs[_HELP_RIGHT]-1, .f., .f.)
      endif
      dispend()
   enddo

   // get bottom right coordinate of help box
   ColorSet(C_MESSAGE)
   buffer2 := ShadowBox(maxrow()-3, 11, maxrow(), 68, 2)
   SCRNCENTER(maxrow() - 2, ;
              'Use arrow keys to drag bottom right corner of help box')
   SCRNCENTER(maxrow() - 1, 'Press Enter when finished')
   firstloop := .t.
   key := 0
   do while key <> K_ENTER
      redraw := .t.
      key := inkey(0)
      if firstloop
         ByeByeBox(buffer2)
         firstloop := .f.
      endif
      dispbegin()
      do case

         case key == K_DOWN .and. aHelpDefs[_HELP_BOTTOM] < maxrow()
            aHelpDefs[_HELP_BOTTOM]++

         case key == K_UP .and. aHelpDefs[_HELP_BOTTOM] > 6 .and. aHelpDefs[_HELP_BOTTOM] > aHelpDefs[_HELP_TOP] + 4
            restscreen(,,,, cMainScreen)
            aHelpDefs[_HELP_BOTTOM]--

         case key == K_RIGHT .and. aHelpDefs[_HELP_RIGHT] < maxcol()
            aHelpDefs[_HELP_RIGHT]++

         case key == K_LEFT .and. aHelpDefs[_HELP_RIGHT] > 11 .and. ;
                                 aHelpDefs[_HELP_RIGHT] > aHelpDefs[_HELP_LEFT] + 11
            restscreen(,,,, cMainScreen)
            aHelpDefs[_HELP_RIGHT]--

         case key == K_END           // reposition at bottom right corner
            aHelpDefs[_HELP_RIGHT] := maxcol()
            aHelpDefs[_HELP_BOTTOM] := maxrow()

         case key == K_TAB .and. aHelpDefs[_HELP_RIGHT] < maxcol() - 5
            aHelpDefs[_HELP_RIGHT] += 5

         case key == K_SH_TAB .and. aHelpDefs[_HELP_RIGHT] > 16 .and. ;
                                 aHelpDefs[_HELP_RIGHT] > aHelpDefs[_HELP_LEFT] + 16
            restscreen(,,,, cMainScreen)
            aHelpDefs[_HELP_RIGHT] -= 5

         case key == K_CTRL_RIGHT .and. aHelpDefs[_HELP_RIGHT] < maxcol()
            aHelpDefs[_HELP_RIGHT] := maxcol()

         otherwise
            redraw := .f.
      endcase
      if redraw
         @ aHelpDefs[_HELP_TOP], aHelpDefs[_HELP_LEFT], aHelpDefs[_HELP_BOTTOM], aHelpDefs[_HELP_RIGHT] box BOXFRAMES[aHelpDefs[_HELP_OUTLINE]] ;
                   color Color_N2S(aHelpDefs[_HELP_BOXCOLOR])
         setcolor(Color_N2S(aHelpDefs[_HELP_TEXTCOLOR]))
         memoedit(helptext, aHelpDefs[_HELP_TOP] + 1, aHelpDefs[_HELP_LEFT] + 1, aHelpDefs[_HELP_BOTTOM] - 1, ;
                  aHelpDefs[_HELP_RIGHT] - 1, .f., .f.)
      endif
      dispend()
   enddo

   // drag box around
   ColorSet(C_MESSAGE)
   buffer2 := ShadowBox(maxrow()-3, 14, maxrow(), 65, 2)
   SCRNCENTER(maxrow() - 2, 'Use arrow keys to drag box to desired location')
   SCRNCENTER(maxrow() - 1, 'Press Enter when finished')
   firstloop := .t.
   key := 0
   do while key <> K_ENTER
      redraw := .t.
      key := inkey(0)
      if firstloop
         ByeByeBox(buffer2)
         buffer2 := savescreen(aHelpDefs[_HELP_TOP], aHelpDefs[_HELP_LEFT], aHelpDefs[_HELP_BOTTOM], aHelpDefs[_HELP_RIGHT])
         firstloop := .f.
      endif
      dispbegin()
      do case

         case key == K_HOME
            aHelpDefs[_HELP_BOTTOM] -= aHelpDefs[_HELP_TOP]
            aHelpDefs[_HELP_RIGHT] -= aHelpDefs[_HELP_LEFT]
            aHelpDefs[_HELP_TOP] := aHelpDefs[_HELP_LEFT] := 0

         case key == K_PGUP
            aHelpDefs[_HELP_BOTTOM] -= aHelpDefs[_HELP_TOP]
            aHelpDefs[_HELP_TOP] := 0

         case key == K_END
            aHelpDefs[_HELP_TOP] += maxrow()- aHelpDefs[_HELP_BOTTOM]
            aHelpDefs[_HELP_LEFT] += maxcol() - aHelpDefs[_HELP_RIGHT]
            aHelpDefs[_HELP_BOTTOM] := maxrow()
            aHelpDefs[_HELP_RIGHT] := maxcol()

         case key == K_PGDN
            aHelpDefs[_HELP_TOP] += maxrow() - aHelpDefs[_HELP_BOTTOM]
            aHelpDefs[_HELP_BOTTOM] := maxrow()

         case key == K_UP .and. aHelpDefs[_HELP_TOP] > 0
            aHelpDefs[_HELP_TOP]--
            aHelpDefs[_HELP_BOTTOM]--

         case key == K_DOWN .and. aHelpDefs[_HELP_BOTTOM] < maxrow()
            aHelpDefs[_HELP_TOP]++
            aHelpDefs[_HELP_BOTTOM]++

         case key == K_LEFT .and. aHelpDefs[_HELP_LEFT] > 0
            aHelpDefs[_HELP_LEFT]--
            aHelpDefs[_HELP_RIGHT]--

         case key == K_RIGHT .and. aHelpDefs[_HELP_RIGHT] < maxcol()
            aHelpDefs[_HELP_LEFT]++
            aHelpDefs[_HELP_RIGHT]++

         case key == K_CTRL_LEFT .and. aHelpDefs[_HELP_LEFT] > 0
            aHelpDefs[_HELP_RIGHT] -= aHelpDefs[_HELP_LEFT]
            aHelpDefs[_HELP_LEFT] := 0

         case key == K_CTRL_RIGHT .and. aHelpDefs[_HELP_RIGHT] < maxcol()
            aHelpDefs[_HELP_LEFT] += maxcol() - aHelpDefs[_HELP_RIGHT]
            aHelpDefs[_HELP_RIGHT] := maxcol()

         otherwise
            redraw = .f.
      endcase
      if redraw
         restscreen(,,,, cMainScreen)
         restscreen(aHelpDefs[_HELP_TOP], aHelpDefs[_HELP_LEFT], aHelpDefs[_HELP_BOTTOM], aHelpDefs[_HELP_RIGHT], buffer2)
      endif
      dispend()
   enddo

   // redraw header and footer if we are editing an existing screen
   if mode == "E"
      Draw_FtHd()
   endif

   // select box outline
   ColorSet(C_MESSAGE)
   buffer2 := ShadowBox(maxrow()-3, 18, maxrow(), 61, 2)
   SCRNCENTER(maxrow()-2, 'Use arrow keys to select help box outline')
   SCRNCENTER(maxrow()-1, 'Press Enter when finished')
   buffer3 := shadowbox(10, 35, 17, 44, 2)
   aHelpDefs[_HELP_OUTLINE] := max(achoice(11, 36, 17, 43, boxtypes, '', 'PickABox', ;
                 aHelpDefs[_HELP_OUTLINE], aHelpDefs[_HELP_OUTLINE]), 1)
   ByeByebox(buffer3)
   @ aHelpDefs[_HELP_TOP], aHelpDefs[_HELP_LEFT], aHelpDefs[_HELP_BOTTOM], aHelpDefs[_HELP_RIGHT] ;
                box substr(BOXFRAMES[aHelpDefs[_HELP_OUTLINE]], 1, 8) color Color_N2S(aHelpDefs[_HELP_BOXCOLOR])
   Draw_FtHd()
   ByeByeBox(buffer2)

   // header and footer
   ColorSet(C_MESSAGE)
   aHelpDefs[_HELP_TITLE] := padr(aHelpDefs[_HELP_TITLE], 30)
   aHelpDefs[_HELP_FOOTER] := padr(aHelpDefs[_HELP_FOOTER], 30)
   buffer2 := ShadowBox(maxrow()-3, 10, maxrow(), 69, 2)
   @ 22, 12 ssay 'Enter help screen header:'
   @ 23, 12 ssay 'Enter help screen footer:'
   @ 22, 38 get aHelpDefs[_HELP_TITLE]
   @ 23, 38 get aHelpDefs[_HELP_FOOTER]
   setcursor(2)
   read
   setcursor(0)
   ByeByeBox(buffer2)
   @ aHelpDefs[_HELP_TOP], aHelpDefs[_HELP_LEFT], aHelpDefs[_HELP_BOTTOM], aHelpDefs[_HELP_RIGHT] ;
             box substr(BOXFRAMES[aHelpDefs[_HELP_OUTLINE]], 1, 8) color Color_N2S(aHelpDefs[_HELP_BOXCOLOR])
   Draw_FtHd()

   // draw color palette based upon location of help window
   palettetop := if(aHelpDefs[_HELP_TOP] > 8, 0, 14)
   if empty(palette)
      for xx := 0 to 127
         palette += " " + chr(xx) + chr(4) + chr(xx) + " " + chr(xx)
      next
   endif
   restscreen(palettetop, 16, palettetop + 7, 63, palette)
   setcolor('w/n')
   scroll(palettetop + 8, 16, palettetop + 8, 63, 0)

   aHelpDefs[_HELP_BOXCOLOR]    := PickAColor('box',    aHelpDefs[_HELP_BOXCOLOR],    palettetop, palette)
   aHelpDefs[_HELP_TITLECOLOR]  := PickAColor('title',  aHelpDefs[_HELP_TITLECOLOR],  palettetop, palette)
   aHelpDefs[_HELP_FOOTERCOLOR] := PickAColor('footer', aHelpDefs[_HELP_FOOTERCOLOR], palettetop, palette)
   aHelpDefs[_HELP_TEXTCOLOR]   := PickAColor('text',   aHelpDefs[_HELP_TEXTCOLOR],   palettetop, palette)

   // okay, show 'em the beast and ask for confirmation
   restscreen(,,,, cMainScreen)
   shadowbox(aHelpDefs[_HELP_TOP], aHelpDefs[_HELP_LEFT], aHelpDefs[_HELP_BOTTOM], aHelpDefs[_HELP_RIGHT], aHelpDefs[_HELP_OUTLINE],,, ;
             Color_N2S(aHelpDefs[_HELP_BOXCOLOR]))
   Draw_FtHd()
   setcolor(Color_N2S(aHelpDefs[_HELP_TEXTCOLOR]))
   memoedit(helptext, aHelpDefs[_HELP_TOP]+1, aHelpDefs[_HELP_LEFT]+1, aHelpDefs[_HELP_BOTTOM]-1, aHelpDefs[_HELP_RIGHT]-1, .f., .f.)
   mainloop := ! yes_no('Is this satisfactory')
enddo

if lIndexentry
   aHelpDefs[_HELP_INDEXENTRY] := yes_no("Should this screen be included in the help index")
endif

if rlock()
   (HELPFILE)->toprow   := aHelpDefs[_HELP_TOP]
   (HELPFILE)->botrow   := aHelpDefs[_HELP_BOTTOM]
   (HELPFILE)->lt_col   := aHelpDefs[_HELP_LEFT]
   (HELPFILE)->rt_col   := aHelpDefs[_HELP_RIGHT]
   (HELPFILE)->title    := aHelpDefs[_HELP_TITLE]
   (HELPFILE)->footer   := aHelpDefs[_HELP_FOOTER]
   (HELPFILE)->boxcolor := aHelpDefs[_HELP_BOXCOLOR]
   (HELPFILE)->titcolor := aHelpDefs[_HELP_TITLECOLOR]
   (HELPFILE)->ftcolor  := aHelpDefs[_HELP_FOOTERCOLOR]
   (HELPFILE)->txtcolor := aHelpDefs[_HELP_TEXTCOLOR]
   (HELPFILE)->boxno    := aHelpDefs[_HELP_OUTLINE]
   if lIndexentry
      (HELPFILE)->indexentry := aHelpDefs[_HELP_INDEXENTRY]
   endif
   unlock
else
   err_msg( { NETERR_MSG } )
endif
return NIL

* end function HelpScreen()
*--------------------------------------------------------------------*


/*
   Function: PickAColor()
   allow user to select box coordinates, box type, title,
   and colors for this help screen
*/
static function PickAColor(msg, colortype, ntop, palette)
local mrow := ntop + int(colortype / 16), firstloop := .t., ;
      mcol := 17 + colortype % 16 * 3, key := 0, buffer
ColorSet(C_MESSAGE)
buffer := ShadowBox(maxrow()-2, 6, maxrow(), 73, 2)
@ maxrow()-1, 8 ssay padr('Use arrow keys to change ' + msg + ' color ' + ;
                         '- press Enter when finished', 64)
do while key <> K_ENTER
   @ mrow, mcol ssay chr(4) color '*' + Color_N2S(colortype)
   key := inkey(0)
   if firstloop
      ByeByeBox(buffer)
      firstloop := .f.
   endif
   do case

      case key == K_DOWN .and. mrow < ntop + 7
         mrow++
         colortype += 16

      case key == K_UP .and. mrow > ntop
         mrow--
         colortype -= 16

      case key == K_RIGHT .and. mcol < 62
         colortype++
         mcol += 3

      case key == K_LEFT .and. mcol > 17
         colortype--
         mcol -= 3

   endcase
   do case

      case msg == 'box'         // adjusting the box color
         @ aHelpDefs[_HELP_TOP], aHelpDefs[_HELP_LEFT], aHelpDefs[_HELP_BOTTOM], aHelpDefs[_HELP_RIGHT] ;
           box substr(BOXFRAMES[aHelpDefs[_HELP_OUTLINE]], 1, 8) color Color_N2S(colortype)
         Draw_FtHd()

      case msg == 'title'       // adjusting the title color
         @ aHelpDefs[_HELP_TOP], aHelpDefs[_HELP_LEFT] + (int(aHelpDefs[_HELP_RIGHT] - aHelpDefs[_HELP_LEFT]) / 2) - ;
           int(len(trim(aHelpDefs[_HELP_TITLE]))/2) ssay trim(aHelpDefs[_HELP_TITLE]) color Color_N2S(colortype)

      case msg == 'footer'      // adjusting the footer color
         @ aHelpDefs[_HELP_BOTTOM], aHelpDefs[_HELP_LEFT] + (int(aHelpDefs[_HELP_RIGHT] - aHelpDefs[_HELP_LEFT]) / 2) - ;
           int(len(trim(aHelpDefs[_HELP_FOOTER]))/2) ssay trim(aHelpDefs[_HELP_FOOTER]) color Color_N2S(colortype)

      case msg == 'text'        // adjusting the text color
         setcolor(Color_N2S(colortype))
         memoedit(helptext, aHelpDefs[_HELP_TOP]+1, aHelpDefs[_HELP_LEFT]+1, aHelpDefs[_HELP_BOTTOM]-1, ;
                  aHelpDefs[_HELP_RIGHT]-1, .f., .f.)
   endcase
   restscreen(ntop, 16, ntop + 7, 63, palette)
enddo
return colortype

* end static function PickAColor()
*--------------------------------------------------------------------*


/*
     function: PickABox()
*/
function PickABox(stat, curr_elem, rel_row)
local ret_val := 2, mrow := row(), mcol := col(), key := lastkey(), buffer
do case
   case stat == AC_IDLE                  // arrow key - redraw box
      dispbegin()
      buffer := savescreen(10, 35, 18, 46)
      @ aHelpDefs[_HELP_TOP], aHelpDefs[_HELP_LEFT], aHelpDefs[_HELP_BOTTOM], aHelpDefs[_HELP_RIGHT] box ;
                substr(BOXFRAMES[curr_elem], 1, 8) color Color_N2S(aHelpDefs[_HELP_BOXCOLOR])
      Draw_FtHd()
      restscreen(10, 35, 18, 46, buffer)
      dispend()

   case stat == AC_EXCEPT

      if key == K_ESC
         ret_val := AC_ABORT
      elseif key == K_ENTER
         ret_val := AC_SELECT
      endif

endcase
setpos(mrow, mcol)           // restore proper screen coordinates
return ret_val

* end function PickABox()
*--------------------------------------------------------------------*


/*

     function: Draw_FfHd() -- redraw header and footer

*/
static function draw_fthd
local mid := aHelpDefs[_HELP_LEFT] + (int(aHelpDefs[_HELP_RIGHT] - aHelpDefs[_HELP_LEFT]) / 2)
@ aHelpDefs[_HELP_TOP], mid - int(len(trim(aHelpDefs[_HELP_TITLE])) / 2) ssay trim(aHelpDefs[_HELP_TITLE]) ;
          color Color_N2S(aHelpDefs[_HELP_TITLECOLOR])
@ aHelpDefs[_HELP_BOTTOM], mid - int(len(trim(aHelpDefs[_HELP_FOOTER])) / 2) ssay trim(aHelpDefs[_HELP_FOOTER]) ;
          color Color_N2S(aHelpDefs[_HELP_FOOTERCOLOR])
return nil

* end static function Draw_FtHd()
*--------------------------------------------------------------------*


/*

    function: EditHelp() -- UDF for MEMOEDIT() above

*/
function EditHelp(mstat, mline, mcol)
local key := lastkey()
local linewidth
local lines
local currline
local ret_val := 0
if key == K_ALT_E      // edit existing help text
   setcursor(2)
   if rlock()
      (HELPFILE)->text := helptext := ;
            memoedit((HELPFILE)->text, (HELPFILE)->toprow + 1, ;
                     (HELPFILE)->lt_col + 1, (HELPFILE)->botrow - 1, ;
                     (HELPFILE)->rt_col - 1, .t.)
      setcursor(0)
      HelpScreen("E")  // edit the rest of the screen
      unlock
      ret_val := 23
   else
      err_msg( { NETERR_MSG } )
   endif
elseif key == K_ALT_P    // print this help screen
   if PrintOK()
      waiton('Now printing help text, press Esc to abort')
      set device to printer
      currline := 1
      linewidth := (HELPFILE)->rt_col - (HELPFILE)->lt_col + 1
      lines := mlcount((HELPFILE)->text, linewidth)
      HelpHead()
      do while currline <= lines .and. inkey() <> K_ESC
         @ prow()+1, (HELPFILE)->lt_col say ;
                     trim(memoline((HELPFILE)->text, linewidth, currline++))
         if prow() >= 59
            HelpHead()
         endif
      enddo
      eject
      set device to screen
      waitoff()
   endif
endif
return ret_val

* end function EditHelp()
*--------------------------------------------------------------------*


/*
    function: HelpHead() -- heading when printing help text
*/
static function HelpHead
static page := 1
devpos(0, 0)
@ 1, 1 say upper((HELPFILE)->title)
@ 1,72 say 'Page ' + ltrim(str(page++))      // increment page counter
devpos(3, 0)
return NIL

* end static function HelpHead()
*--------------------------------------------------------------------*


/*
    Function: HelpDevSet()
    Purpose:  Allows developer to change any of the help defaults
    Notes:    See manifest constants contained in GRUMPH.CH
    Returns:  Previous setting
    Examples:

     #include "grumph.ch"

     helpdevset(_HELP_FILENAME, "custhelp")  // use CUSTHELP.DBF, not HELP.DBF
     helpdevset(_HELP_TOP, 10)               // use 10 as top row for window
*/
function HelpdevSet(nsetting, value)
local ret_val := aHelpDefs[nsetting]
aHelpDefs[nsetting] := value
return ret_val

* end function HelpDevSet()
*--------------------------------------------------------------------*

* eof helpdev.prg
