/*
   Program: HELP.PRG
   System: GRUMPFISH LIBRARY
   Author: Greg Lief
   Copyright (c) 1988-93, Greg Lief
   Clipper 5.x Version
   Compile instructions: clipper help /n /w

   Displays user-defined help screens

   Procs & Fncts: HELP
                : SHOWHELP()
                : PRINTHELP()
                : HELPHEAD()
*/

#include "apick.ch"
#include "grump.ch"
#include "inkey.ch"
#include "achoice.ch"

static cMainScreen                 // saved entry screen
static helpfile := "help"     // default help file

function help(cProc, nLine, cVar)
local nWorkArea := select()
local bOldF1
local nHotKey
local bOldBlock
memvar helpcode        // would be declared in calling program if it exists
GFSaveEnv( , 0)        // shut off cursor
cMainScreen := savescreen()

/*
   usually if HELPDEV() was called from anything other than GINKEY(),
   cProc will be a null string.  We would then have to use the procedure
   name three levels deep.  However, in the event that the procedure
   name passed to HELPDEV() is an internal Clipper function (beginning
   with double underscore "__"), then we must use the procedure name
   four levels deep.
*/
if empty(cProc)
   cProc := procname(3)
elseif left(cProc, 2) = '__'
   cProc := procname(4)
endif
bOldBlock := setkey(nHotKey := lastkey(), NIL)

// set F1 key to show help index
bOldF1 := setkey(K_F1, {|p, l, v| showindex(p, l, v)} )

// make sure everything is proper before proceeding
if ! file(helpfile + '.dbf') .or. ! file(helpfile + '.dbt')
   err_msg( { 'Help files missing' } )
else
   use (helpfile) new
   // rebuild index if necessary
   if ! file(helpfile + indexext())
      if flock()
         waiton("Indexing help database")
         index on _field->theproc + _field->var to (helpfile)
         unlock
      else
         waiton("Database is being reindexed... please wait")
         do while ! flock()
         enddo
      endif
      waitoff()
   else
      set index to (helpfile)
   endif
   // seek on helpcode first if such a variable is defined
   if dbseek( if(type('helpcode') <> "U", helpcode, padr(cProc, 10) + cVar) )
      showhelp()
   else
      showindex()
   endif
   use
endif
GFRestEnv()
setkey(K_F1, bOldF1)          // restore F1 hot-key
setkey(nHotKey, bOldBlock)    // restore hot-key
restscreen(,,,, cMainScreen)
select(nWorkArea)
return NIL

* end of function Help() (i need somebody...)
*--------------------------------------------------------------------


/*
   Function: ShowHelp()  --> display help screen
*/
static function ShowHelp
local nMid
shadowbox((helpfile)->toprow, (helpfile)->lt_col, (helpfile)->botrow, ;
     (helpfile)->rt_col, (helpfile)->boxno,,, color_n2s((helpfile)->boxcolor))
nMid := (helpfile)->lt_col + (int((helpfile)->rt_col - (helpfile)->lt_col) / 2)
@ (helpfile)->botrow, nMid - int(len(trim((helpfile)->footer)) / 2) ;
       ssay trim((helpfile)->footer) color color_n2s((helpfile)->ftcolor)
@ (helpfile)->toprow, nMid - int(len(trim((helpfile)->title)) / 2) ;
       ssay trim((helpfile)->title) color color_n2s((helpfile)->titcolor)
setcolor(color_n2s((helpfile)->txtcolor))
memoedit((helpfile)->text, (helpfile)->toprow + 1, (helpfile)->lt_col + 1, ;
         (helpfile)->botrow - 1, (helpfile)->rt_col - 1, .f., 'PrintHelp')
return NIL

* end of static function ShowHelp() (not just anybody...)
*--------------------------------------------------------------------


/*
   Function: ShowIndex()  --> display help screen index
*/
static function ShowIndex(cProc, nLine, cVar)
static aHelpIndex := {}    // array to hold help screen titles
static aHelpRecs           // array to hold help screen record numbers
static lIndexentry
local nMarker := recno()
local cBuffer
local oldcolor
local ele := 1
local nHotKey

// if we got here from a hot-key, restore the original screen first!
if cProc <> NIL
   setkey(nHotKey := lastkey(), NIL)
   cBuffer := savescreen()
   restscreen(,,,, cMainScreen)    // original screen
endif
go top

// load lIndexentry flag and help index arrays on first pass only...
// NOTE: lIndexentry is for downward compatibility only because
// older versions of HELP.DBF will not have the INDEXENTRY field
if lIndexentry == NIL
   lIndexentry := ( (HelpFile)->( fieldpos("indexentry") ) > 0 )
   waiton( { "Creating help index" } )
   do while ! eof()
      if ! lIndexentry .or. _field->indexentry
         aadd( aHelpIndex, { _field->title, recno() } )
      endif
      skip
   enddo
   if ! empty(aHelpIndex)
      // sort this based on the title
      asort(aHelpIndex,,, { | a, b | a[1] < b[1] } )
      aHelpRecs := array(len(aHelpIndex))

      // make this array single-dimension and create parallel
      // array of record numbers
      aeval(aHelpIndex, { | ele, num | aHelpRecs[num] := ele[2], ;
                                       aHelpIndex[num] := ele[1] } )
   endif
   waitoff()
endif
if empty(aHelpIndex)
   err_msg({ "No help index available" })
else
   do while ele > 0
      apick ele array aHelpIndex initial ele title "Help Index"
      if ele > 0
         go aHelpRecs[ele]
         restscreen(,,,, cMainScreen)    // original screen
         ShowHelp()
         restscreen(,,,, cMainScreen)    // original screen
      endif
   enddo
endif
go nMarker

// restore screen and hot key if we got here from a hot key
if cProc <> NIL
   restscreen(,,,, cBuffer)
   setkey(nHotKey, {|p, l, v| showindex(p, l, v)} )
endif
setcolor(oldcolor)
return NIL

* end of static function ShowIndex() (ya know I need someone...)
*--------------------------------------------------------------------*


/*
  function: PrintHelp() -- UDF for MEMOEDIT() above
*/
function PrintHelp(mstat, nLine, mcol)
local key := lastkey()
local linewidth
local lines
local currline
if key == K_ALT_P    // print this help screen
   if PrintOK()
      waiton('Now printing help text, press Esc to abort')
      set device to printer
      currline := 1
      linewidth := (helpfile)->rt_col - (helpfile)->lt_col + 1
      lines := mlcount((helpfile)->text, linewidth)
      HelpHead(.t.)
      do while currline <= lines .and. inkey() <> K_ESC
         @ prow()+1,(helpfile)->lt_col say ;
                     trim(memoline((helpfile)->text, linewidth, currline++))
         if prow() >= 59
            HelpHead(.f.)
         endif
      enddo
      eject
      set device to screen
      waitoff()
   endif
endif
return 0

* end function PrintHelp()
*--------------------------------------------------------------------*


/*
  function: HelpHead() -- heading when printing help text
*/
static function HelpHead(lReset)
static nPage := 1
if lReset
   nPage := 1
endif
@ 0,0 say ''
@ 1,1 say upper((helpfile)->title)
@ 1,72 say 'Page ' + ltrim(str(nPage++))
@ 3, 0 say ''
return NIL

* end static function HelpHead()
*--------------------------------------------------------------------*


/*
  Function: HelpSet() -- change name of help database file (default HELP.DBF)
  Syntax:   HelpSet(<cNewfile>)
  Returns:  Previous help file name
*/
function HelpSet(cNewFile)
local cOldFile := helpfile
helpfile := cNewFile
return cOldFile

* end function HelpSet()
*--------------------------------------------------------------------*

* end of file HELP.PRG
