/*
    Function: CutNPaste()
    System: GRUMPFISH LIBRARY
    Author: Greg Lief
    Copyright (c) 1988-90, Greg Lief
    Clipper 5.x Version
    Compile instructions: clipper cutpaste /n/w/a

    Allows user to cut-and-paste from underlying screen into the
    Grumpfish notepad (POPNOTE)

    Syntax: CutNPaste(@<screen> [, <hicolor>, <locolor>] ;
                                [, <top>, <left>, <bottom>, <right> ;
                                [, <row>, <column>]])

    Parameters: <screen> is a character string representing the name
                of the previously saved underlying screen from which
                to cut and paste.  For performance considerations,
                you must pass this by reference (i.e., do not surround
                it with quotes and DO precede it with "@")

                Optional parameter <hicolor> is a numeric representing
                which color to use for highlighting.  The default is
                112 (inverse, or black on white).

                Optional parameter <locolor> is a numeric representing
                which color to use for the underlying screen.  The
                default is 7 (white on black).

                Optional parameters <top>, <left>, <bottom>, and <right>
                are numerics which delimit the screen area that can be
                accessed.

                Optional parameters <row> and <column> are numerics
                indicating initial placement of the cursor on the screen.
                NOTE: if these two parameters are passed, highlighting will
                begin IMMEDIATELY (rather than the user having to position
                the cursor).

    Methodology notes: this routine is used either to cut-and-paste from
                       an original underlying screen, or to highlight a
                       section on the current screen.  Therefore, you will
                       either pass the screen variable, or the four
                       coordinates, but not both!!!!  (i.e., if you pass
                       the coordinates, just pass a null string as the
                       screen -- example: CutNPaste('', 0, 0, 24, 79)

    Returns: A character string suitable for KEYBOARDing into a memo field

    Calls: GFATTR()     (Assembler function in GRUMPATT.ASM)

*/

// begin preprocessor directives

#include "grump.ch"
#include "inkey.ch"

// end preprocessor directives


function cutnpaste(origscrn, hicolor, locolor, ntop, nleft, ;
                   nbottom, nright, mrow, mcol)
local oldscrn, t_anchor, l_anchor, b_anchor, r_anchor, key := 0, ;
      xx, yy, shading, cutbuff, row_len, buffer

// establish colors if not passed as parameters
default hicolor to 112
default locolor to 7

// establish box coordinates if not passed as parameters
default ntop to 0
default nleft to 0
default nbottom to maxrow()
default nright to maxcol()

// establish initial screen location, anchor variables, and whether or not
// to begin highlighting immediately
if pcount() < 8
   shading := .f.   // flag to indicate whether we are highlighting or not
   t_anchor := b_anchor := mrow := ntop
   l_anchor := r_anchor := mcol := nleft
else
   t_anchor := b_anchor := mrow
   l_anchor := r_anchor := mcol
   shading := .t.   // start highlighting immediately
endif
oldscrn := savescreen(ntop, nleft, nbottom, nright)

// if the original screen parameter was passed, user restore the original
// underlying screen and then make it all the same color (white on black)
if pcount() = 1
   restscreen(0, 0, maxrow(), maxcol(), origscrn)
   gfattr(0, 0, maxrow(), maxcol(), locolor)
endif

cutbuff := []    && this will hold the buffer to be returned back
do while .t.
   if shading
      gfattr(t_anchor, l_anchor, b_anchor, r_anchor, hicolor)
   endif
   devpos(mrow, mcol)
   key := ginkey(0)
   do case
      case key == K_DOWN .and. mrow < nbottom
         if shading
            if mrow < b_anchor
               gfattr(mrow, l_anchor, mrow, r_anchor, locolor)
               t_anchor := ++mrow
            else
               b_anchor := ++mrow
            endif
         else
            mrow++
         endif

      case key == K_UP .and. mrow > ntop
         if shading
            if mrow > t_anchor
               gfattr(mrow, l_anchor, mrow, r_anchor, locolor)
               b_anchor := --mrow
            else
               t_anchor := --mrow
            endif
         else
            mrow--
         endif

      case key == K_RIGHT .and. mcol < nright
         if shading
            if mcol < r_anchor
               gfattr(t_anchor, mcol, b_anchor, mcol, locolor)
               l_anchor := ++mcol
            else
               r_anchor := ++mcol
            endif
         else
            mcol++
         endif

      case key == K_LEFT .and. mcol > nleft
         if shading
            if mcol > l_anchor
               gfattr(t_anchor, mcol, b_anchor, mcol, locolor)
               r_anchor := --mcol
            else
               l_anchor := --mcol
            endif
         else
            mcol--
         endif

      case key == K_ENTER
         if ! shading
            shading := .t.
            t_anchor := b_anchor := mrow
            l_anchor := r_anchor := mcol
         else
            exit
         endif

      case key == K_ESC
         exit

   endcase
enddo
if key == K_ENTER
   // now to parse out all the color attributes
   buffer := savescreen(t_anchor, l_anchor, b_anchor, r_anchor)
   // determine width of one row so that we can stick in cr/lf
   row_len := (r_anchor - l_anchor + 1) * 2
   for xx = 1 to (b_anchor - t_anchor + 1)
      for yy = 1 to row_len step 2
         cutbuff += substr(buffer, (xx - 1) * row_len + yy, 1)
      next
      cutbuff += CRLF
   next
endif
restscreen(ntop, nleft, nbottom, nright, oldscrn)
return cutbuff

* end function CutNPaste()
*--------------------------------------------------------------------*

*eof cutpaste.prg
