/*
     Program: COLORSET()
     System: GRUMPFISH LIBRARY
     Author: Greg Lief
     Copyright (c) 1988-90, Greg Lief
     Clipper 5.x Version
     Compile instructions: clipper colorset /n/w/a
     Complete color management for the Grumpfish Library.

     Note: if you don't plan to let users modify colors, you can
           compile this with the /dCANNOT_MODIFY_COLORS switch to
           strip out the ColorMod() code.
*/

// begin preprocessor directives

#include "grump.ch"
#include "inkey.ch"

#define COLOR_CNT 34            // total # of colors in array below

// end preprocessor directives

// begin global declarations

static iscolor := .T.           // global flag to indicate color vs. mono

static colorfile := "COLORS.GF" // default filename for color configuration

static gfcolors := { { "+W/N",   "N/W",   "Current Entry Fields" }     , ;
                     { "+W/BR",  "+W/N",  "Message Boxes" }            , ;
                     { "+W*/BR", "*N/W",  "Blinking Messages" }        , ;
                     { "+GR/N",  "+W/N",  "Wait Messages" }            , ;
                     { "+GR*/N", "+W*/N", "Blinking Wait Messages" }   , ;
                     { "+W/BR",  "+W/N",  "Yes/No Messages" }          , ;
                     { "W/R",    "N/W",   "Error Box Outline" }        , ;
                     { "+W/R",   "N/W",   "Error Messages" }           , ;
                     { "+W/R",   "W/N",   "Memoedit Window Outline " } , ;
                     { "+W/N",   "+W/N",  "Memoedit Window Interior" } , ;
                     { "W/B",    "W/N",   "Stopwatch Box" }            , ;
                     { "+W/N",   "N/W",   "Stopwatch Window " }        , ;
                     { "W/BR",   "W/N",   "Calculator Box" }           , ;
                     { "+W/N",   "N/W",   "Calculator Window " }       , ;
                     { "+W/B",   "W/N",   "Notepad Box" }              , ;
                     { "+W/N",   "W/N",   "Notepad Window " }          , ;
                     { "+W/R",   "W/N",   "Phonebook Main Window" }    , ;
                     { "+W/B",   "W/N",   "Phonebook Edit Window" }    , ;
                     { "W/B",    "W/N",   "Calendar Window" }          , ;
                     { "+W/BG",  "+W/N",  "Appointment Window" }       , ;
                     { "+W/B",   "+W/N",  "Picklist Box Outline" }     , ;
                     { "W/N",    "W/N",   "Picklist Status Bar" }      , ;
                     { "+GR/N",  "N/W",   "Picklist Indicator " }      , ;
                     { "W/B",    "W/N",   "Picklist Unselected Items" }, ;
                     { "N/W",    "N/W",   "Picklist Highlighted Item" }, ;
                     { "+GR/B",  "+W/N",  "Picklist Tagged Items" }    , ;
                     { "+W/R",   "N/W",   "Picklist Tagged/Current" }  , ;
                     { "R/B",    "+W/N",  "Picklist Unavailable Items" }   , ;
                     { "N/B",    "N/W",   "Picklist Unavailable/Current" } , ;
                     { "W/B",    "W/N",   "Unselected Menu Options " } , ;
                     { "+GR/R",  "N/W",   "Selected Menu Options" }    , ;
                     { "W/B",    "W/N",   "DataBrowse Box" }           , ;
                     { "+W/B",   "+W/N",  "DataBrowse Text" }          , ;
                     { "+W/N",   "N/W",   "DataBrowse Highlight" } }

// end global declarations

function ColorSet(mcolor, newcolor)
local ret_val
// if parameter was a character string, that means they are processing
// an "on-the-fly" color rather than one out of the static array
if valtype(mcolor) == "C"
   ret_val := setcolor(mcolor)
else
   do case

      // set color to the default and return the current setting
      case newcolor == NIL
         ret_val := setcolor(gfcolors[mcolor][IF(iscolor, 1, 2)])

      // change this color setting to the color string you just passed
      // and return the current setting
      case valtype(newcolor) == "C"
         ret_val := setcolor(gfcolors[mcolor][IF(iscolor, 1, 2)] := newcolor)

      // do not change color at this time... only return default setting
      otherwise
         ret_val := gfcolors[mcolor][IF(iscolor, 1, 2)]
   endcase
endif
return ret_val

* end function ColorSet()
*--------------------------------------------------------------------*


/*

    Function: ColorInit()
    Purpose:  Determine whether to use color or monochrome settings
    Syntax:   ColorInit(<mono/file>)
              Three possibilities here:

              1) If you do not pass a parameter, ColorInit() will
              go with whatever ISCOLOR() returns.

              2) Pass a logical, and ColorInit() will use COLOR
              if True, or MONO if False.

              3) Pass a filename and ColorInit() will load
              previously saved color settings from that file and
              use those instead of anything else.

              If you want something more flexible than this, go
              hire yourself a contortionist!

*/
function colorinit(mvar)
local nhandle, xx, ele, buffer, bytes
do case
   case mvar == NIL
      iscolor := iscolor()
   case valtype(mvar) == 'L'
      iscolor := mvar
   otherwise
      // save the name of this file to use as default
      // if you later change the colors via COLORMOD()
      colorfile := mvar
      if ( nhandle := fopen(mvar) ) != -1
         buffer := space(COLOR_CNT * 2 + 1)
         bytes := fread(nhandle, @buffer, COLOR_CNT * 2 + 1)
         fclose(nhandle)
         /*
            The first FOR..NEXT loop below is provided strictly for
            downward compatibility with GrumpLib 3.0.  This is necessary
            because color configuration files created with ColorMod()
            3.0 will contain color *or* monochrome settings, but not both.
            On the other hand, Color.cfg files created with 3.1 will
            contain both color and mono settings, along with an additional
            start byte signifying whether to use color or mono.

            If you are reasonably certain that you will only be using
            3.1 files, then by all means feel free to delete the first
            FOR..NEXT loop.
         */
         if bytes == COLOR_CNT
            for xx = 1 to bytes
               gfcolors[xx, 1] := Color_N2S(bin2i(substr(buffer, xx, 1)))
            next
         else
            iscolor := (substr(buffer, 1, 1) == "C")
            ele := 1
            for xx = 2 to bytes step 2
               gfcolors[ele, 1] := Color_N2S(bin2i(substr(buffer, xx, 1)))
               gfcolors[ele++, 2] := Color_N2S(bin2i(substr(buffer, xx + 1, 1)))
            next
         endif
      endif
endcase
return nil

* end function ColorInit()
*--------------------------------------------------------------------*


/*

   Function: ColorMod()
   Purpose:  View/Modify all Grumpfish global color settings
   Syntax:   ColorMod( <file> )
   Parameter: <file> is the filename & path to save colors to
              If not passed, the user will be allowed to enter
              a filename
*/
function ColorMod( savefile )

#ifndef CANNOT_MODIFY_COLORS

local key := 0
local keepgoing
local ntop := int(maxrow() / 2 - COLOR_CNT / 4) + 1
local nbottom := ntop + int( COLOR_CNT / 2 ) + COLOR_CNT % 2 - 1
local nhandle
local arrow := chr(196) + chr(16)
local buffer
local newcolor
local changed := .f.
local curr_color := 1
local oldscore
local maincolor := colorset(C_MESSAGE, .T.)
local oldf10 := setkey(K_F10, NIL)       // save F10 setting
local midcol := int((maxcol() + 1) / 2)
local midrow := int((maxrow() + 1) / 2)
local mcol   := midcol - 9
local mrow   := ntop
local getlist := {}
GFSaveEnv(.t., 0)                        // shut off cursor
ShowColors(ntop, nbottom, maincolor, midcol)
do while key != K_ESC
   @ mrow, mcol ssay chr(16)
   @ mrow, mcol+7 ssay chr(17)
   key := ginkey(0)
   do case

      case key == K_F10                  // toggle color/mono
         iscolor := ! iscolor
         changed := .t.
         maincolor := colorset(C_MESSAGE, .T.)
         ShowColors(ntop, nbottom, maincolor, midcol)

      case key == K_ENTER                // change this color
         newcolor := colorpal(ColorSet(curr_color, .T.))
         if lastkey() != K_ESC
            changed := .t.
            colorset(curr_color, newcolor)
            @ mrow, mcol + 1 ssay "SAMPLE"
            setcolor(maincolor)
         endif

      case key == K_UP
         @ mrow, mcol ssay chr(32)
         @ mrow, mcol+7 ssay chr(32)
         if mrow == ntop
            mrow := nbottom
            curr_color += COLOR_CNT - 2
         else
            mrow--
            curr_color -= 2
         endif

      case key == K_DOWN
         @ mrow, mcol ssay chr(32)
         @ mrow, mcol+7 ssay chr(32)
         if mrow == nbottom
            mrow := ntop
            curr_color -= (COLOR_CNT - 2)
         else
            mrow++
            curr_color += 2
         endif

      case key == K_PGUP
         @ mrow, mcol ssay chr(32)
         @ mrow, mcol+7 ssay chr(32)
         mrow := ntop
         curr_color := if(mcol == midcol - 9, 1, 2)

      case key == K_PGDN
         @ mrow, mcol ssay chr(32)
         @ mrow, mcol+7 ssay chr(32)
         mrow := nbottom
         curr_color := if(mcol == midcol + 30, COLOR_CNT, COLOR_CNT - 1)

      case key == K_HOME
         @ mrow, mcol ssay chr(32)
         @ mrow, mcol+7 ssay chr(32)
         mrow := ntop
         mcol := midcol - 9
         curr_color := 1

      case key == K_END
         @ mrow, mcol ssay chr(32)
         @ mrow, mcol+7 ssay chr(32)
         mrow := nbottom
         mcol := midcol + 30
         curr_color := COLOR_CNT

      case key == K_LEFT .or. key == K_RIGHT
         @ mrow, mcol ssay chr(32)
         @ mrow, mcol+7 ssay chr(32)
         if mcol == midcol - 9
            mcol := midcol + 30
            curr_color++
         else
            mcol := midcol - 9
            curr_color--
         endif

   endcase
enddo

// if they changed one or more settings, ask if they want to save 'em
if changed
   if yes_no2( { "Would you like to save these settings?" }, ;
                 midrow, " Yes ", " No ")
      // if filename was passed as parameter to ColorMod(), do
      // not allow user to pick a file
      if savefile == NIL
         oldscore := set(_SET_SCOREBOARD, .f.)
         colorfile := padr(colorfile, 12)
         ColorSet(C_MESSAGE)
         buffer := ShadowBox(midrow - 1, midcol - 22, midrow + 1, midcol + 22, 2)
         @ midrow, midcol - 20 ssay "Enter file name to save to:"
         @ row(), col() + 1 get colorfile picture '@!'
         setcursor(1)
         read
         setcursor(0)
         ByeByeBox(buffer)
         set(_SET_SCOREBOARD, oldscore)
      else
         colorfile := savefile
      endif
      if lastkey() != K_ESC .and. ! empty(colorfile)
         colorfile := ltrim(trim(colorfile))
         // note: we don't need confirmation to overwrite the file
         // if you passed it as a parameter to ColorMod()
         keepgoing := .t.
         if file(colorfile) .and. savefile == NIL
            keepgoing := yes_no2({ colorfile + ' already exists!', ;
                                 'Do you wish to overwrite it?' }, midrow, ;
                                 ' Overwrite ', ' Cancel ')
         endif
         if keepgoing
            if ( nhandle := fcreate(colorfile) ) != -1
               buffer := space(COLOR_CNT)
               // write color/mono setting as first byte
               fwrite(nhandle, if(iscolor, "C", "M"))
               // now loop through colors array, write color and
               // monochrome settings to the file
               aeval(gfcolors, { | a | fwrite(nhandle, chr(Color_S2N(a[1])) + ;
                                       chr(Color_S2N(a[2]))) } )
               fclose(nhandle)
            endif
         endif
      endif
   endif
endif
GFRestEnv()
setkey(K_F10, oldf10)         // restore F10 setting

#endif // CANNOT_MODIFY_COLORS

return NIL

* end function ColorMod()
*--------------------------------------------------------------------*


/*
   Function: ShowColors()
   Purpose:  Show samples for entire colors array
*/
static function showcolors(ntop, nbottom, maincolor, midcol)
local xx
dispbegin()
setcolor(maincolor)
SINGLEBOX(ntop - 1, 0, nbottom + 2, maxcol())
SCRNCENTER(nbottom + 1, chr(24) + chr(25) + chr(27) + chr(26) + "PgUp "   + ;
           "PgDn to move" + space(4) + "Enter to change color" + space(4) + ;
           "F10 switches to " + if(iscolor, "monochrome", "color"))
setpos(ntop, 0)
for xx := 1 to COLOR_CNT
   @ row(), if(xx % 2 == 1, midcol - 38, midcol + 2) ssay gfcolors[xx][3]
   colorset(xx)
   @ row(), if(xx % 2 == 1, midcol - 8, midcol + 31) ssay "SAMPLE"
   setcolor(maincolor)
   if xx % 2 == 0
      setpos(row()+1, 0)
   endif
next
dispend()
return nil

* end static function ShowColors()
*--------------------------------------------------------------------*

* eof colorset.prg
