/*
     Program: COLORPAL()
     System: GRUMPFISH LIBRARY
     Author: Greg Lief
     Copyright (c) 1988-89, Greg Lief
     Clipper 5.x Version
     Compile instructions: clipper colorpal /n/w/a
     Allows user to select color from interactive palette
     Procs & Fncts: COLORPAL()
     Calls: COLOR_N2S()   (function in COLORS.PRG)
*/

#include "inkey.ch"
#include "grump.ch"

// character to use as indicator when navigating thru palette
#define INDICATOR 10
#define NUM_COLORS  256


function colorpal(curr_color, pal_top, pal_left, redrawfunc)
static palette
local nrow
local ncol
local keypress := 0
local xx
local oldcolor := setcolor()
local use_sample := .f.
local num_colors
local palbuffer
local pal_bot

GFSaveEnv(.t., 0)           // shut off cursor

// establish top row and leftmost column for palette if not passed
if pal_top == NIL .or. pal_top > maxrow() - 16
   pal_top := maxrow() - 16
endif
pal_bot  := pal_top + 16
if pal_left == NIL .or. pal_left > maxcol() - 15
   pal_left := maxcol() - 15
endif

/*
   establish color sample UDF.  there are three options:
   a) user did not pass fourth parameter --- use default
   b) user passed a code block as fourth parameter - use that unchanged
   c) user passed character string as parameter (such as "whatever()") --
      compile it to a code block
*/
if valtype(redrawfunc) == 'C'
   redrawfunc := MakeBlock(redrawfunc)
elseif valtype(redrawfunc) != 'B'
   redrawfunc := { | a, b | colorsample(a, b) }
   use_sample := .T.
   keyboard chr(255)    // force color sample to be drawn immediately
endif

// set start-up color to current color if no parameter passed
curr_color := if(curr_color == NIL, setcolor(), upper(curr_color))

// convert the string to a number
curr_color := color_s2n(curr_color)

// create character string of text/color attributes for color palette
// if it was not already created on a previous visit to COLORPAL()
if palette == NIL
   palette := []
   for xx = 0 to NUM_COLORS
      palette += chr(4) + chr(xx)
   next
endif
restscreen(pal_top, pal_left, pal_bot - 1, pal_left+15, palette)
ColorSet(C_MESSAGE)
scroll(pal_bot, pal_left, pal_bot, pal_left+15, 0)
@ pal_bot, pal_left      ssay chr(24)+chr(25)+chr(27)+chr(26)
@ pal_bot, pal_left + 7  ssay chr(17) + chr(217)
@ pal_bot, pal_left + 13 ssay 'Esc'

// determine starting row and column within palette
nrow := pal_top + int(curr_color / 16)
ncol := pal_left + curr_color % 16

// commence main keypress loop
do while keypress != K_ESC .and. keypress != K_ENTER

   // change to currently selected color, but if it is a blinking
   // color, use the non-blinking component (will reset below)
   if curr_color > 127 .and. setblink()
      setcolor(Color_N2S(curr_color - 128))
   else
      setcolor(Color_N2S(curr_color))
   endif

   // draw indicator to mark current color
   @ nrow,ncol ssay chr(INDICATOR)

   // if currently selected color is blinking, must reset it (see above)
   if curr_color > 127 .and. setblink()
      setcolor(Color_N2S(curr_color))
   endif

   keypress := ginkey(0)

   // clear indicator
   @ nrow,ncol ssay chr(4)

   // process keystroke
   do case
      case keypress == K_DOWN
         // if we are at the bottom, jump to the top
         IF nrow == pal_bot - 1
            curr_color -= (pal_bot - 1 - (nrow := pal_top) ) * 16
         else
            nrow++
            curr_color += 16
         endif

      case keypress == K_UP
         // if we are at the top, jump to the bottom
         if nrow == pal_top
            curr_color += ( (nrow := pal_bot - 1) - pal_top) * 16
         else
            nrow--
            curr_color -= 16
         endif

      case keypress == K_RIGHT
         if ncol < pal_left + 15
            ncol++
            curr_color++
         else
            ncol := pal_left
            curr_color -= 15
         endif

      case keypress == K_LEFT
         if ncol > pal_left
            ncol--
            curr_color--
         else
            ncol := pal_left + 15
            curr_color += 15
         endif

      case keypress == K_PGDN
         curr_color += (pal_bot - 1 - nrow) * 16
         nrow := pal_bot - 1

      case keypress == K_PGUP
         curr_color -= (nrow - pal_top) * 16
         nrow := pal_top

   endcase
   if keypress != K_ENTER .and. keypress != K_ESC
      setcolor(color_n2s(curr_color))
      dispbegin()
      if ! use_sample
         palbuffer := savescreen(pal_top, pal_left, pal_bot, pal_left+15)
      endif
      eval(redrawfunc, pal_top, pal_left)
      if ! use_sample
         restscreen(pal_top, pal_left, pal_bot, pal_left+15, palbuffer)
      endif
      dispend()
   endif
enddo

// restore environment
GFRestEnv()

// if user pressed ESC to abort, return the old color
return (if(keypress == K_ESC, oldcolor, color_n2s(curr_color)))

* end function ColorPal()
*--------------------------------------------------------------------*


/*
   ColorSample(): draw sample box showing current selection
*/
static function ColorSample(pal_top, pal_left)
DOUBLEBOX(pal_top-6, pal_left, pal_top-3, pal_left+15)
@ pal_top-5, pal_left+3 ssay "Sample of"
@ pal_top-4, pal_left+1 ssay "current color"
return NIL

* end static function ColorSample()
*--------------------------------------------------------------------*

* eof colorpal.prg
