/*
    Program: GRUMPBROW()
    System: GRUMPFISH LIBRARY
    Author: Greg Lief
    Copyright (c) 1988-90, Greg Lief
    Clipper 5.01 Version
    Compile instructions: clipper brow /n /w

    If you recompile this file, at a minimum you MUST use the /N
    command-line parameter.

    If you are using Clipper 5.2, you may wish to recompile BROW.PRG
    with the "/dCLIPPER52" command-line parameter to take advantage of
    the TBrowse:picture instance variable and the TBrowse:forceStable()
    method.

    If you plan to use the "additional options" menu, you must specify
    the "/dEXTRA_OPTIONS" command-line parameter when recompiling
    BROW.PRG.

    If you plan to use the screen painter, you must specify the
    "/dREMBRANDT" command-line parameter when recompiling BROW.PRG.

    If you plan to use the code generator, you must specify the
    "/dCODEGEN" command-line parameter when recompiling BROW.PRG.

    If you do not want to start with a basic field template in
    the screen painter, you should specify the "/dNO_TEMPLATE"
    command-line parameter when recompiling BROW.PRG.

    Syntax:  GRUMPBROW([<security>, <top>, <left>, <bottom>, <right>,    ;
             <box>, <fields>, <heads>, <pictures>, <alternates>,         ;
             <low value>, <high value>] [, <vertical movement block>]    ;
             [, <horizontal movement block>] [, <extra options array>]   ;
             [, <extra options key>] [, <extra options menu title> ]     ;
             [, <exit key> ] [, <index descriptions array> ]             ;
             [, <pre-validation array> ]  [, <post-validation array>]    ;
             [, <auto refresh time>] [, <# of columns to lock> ]         ;
             [, <carry flag> ] [, <memo width> ] [, <title for window> ] ;
             [, <memo height> ] [, <TBrowse colorspec> ]                 ;
             [, <colorBlock array> ] [, <initialization codeblock> ]     ;
             [, <gotop logical flag> ] )

    For more specifics on this dizzying parameter list, please refer to
    the Grumpfish Norton Guide file.

    GRUMPBROW() returns a character string that can contain none, some,
    or all of the following letters: "A" (indicates that a record was
    added), "E" (indicates that a record was edited), "D" (indicates
    that a record was deleted). Note that these apply only to the
    default add/edit/delete routines -- if you use your own alternate
    routines, you can keep track of that yourself.

*/

#include "apick.ch"
#include "boxget.ch"
#include "grump.ch"
#include "inkey.ch"
#include "dbstruct.ch"
#include "error.ch"
#include "setcurs.ch"

// the reason for the next manifest constant is that some people
// have reported cursor weirdness on EGA monitors... if you need
// to change the cursor size, you can do so right here
#define CURSOR_ON   SC_NORMAL

// manifest constants to delineate structure of security array
#define S_ADD         1
#define S_DELETE      2
#define S_EDIT        3
#define S_QUERY       4
#define S_SEARCH      5
#define S_VIEW        6
#define S_CELLEDIT    7
#define S_OUTPUT      8
#define S_LOCKCOLUMN  9
#define S_SUBSET     10
#define S_SWAPINDEX  11
#define S_PACK       12

#define S_ITEMS      12


// the following PLUS & MINUS definitions are used in the code
// that handles expanding and shrinking columns...
#define K_PLUS       43
#define K_MINUS      45
#ifndef K_ALT_MINUS
   #define K_ALT_MINUS 386     // Alt-Minus (Non-keypad)
#endif
#ifndef K_ALT_PLUS
   #define K_ALT_PLUS  387     // Alt-Plus (Non-keypad)
#endif

// manifest constants relevant to movement, which determine whether
// vBlock or hBlock should be evaluated after stabilization
#define NO_MOVEMENT     0
#define VERTICAL        1
#define HORIZONTAL      2

#xtranslate getArray       => cargo\[1]
#xtranslate boxArray       => cargo\[2]
#xtranslate indexBlock     => cargo\[3]
#xtranslate queryCondition => cargo\[4]
#xtranslate startValues    => cargo\[5]
#xtranslate endValues      => cargo\[6]
#xtranslate origSkipBlock  => cargo\[7]
#xtranslate memoWidth      => cargo\[8]
#xtranslate memoHeight     => cargo\[9]

#define CARGO_LENGTH                 10

#xtranslate TextAt(<r>, <c>) => substr(savescreen(<r>, <c>, <r>, <c>), 1, 1)
#xtranslate ColorAt(<r>, <c>) => substr(savescreen(<r>, <c>, <r>, <c>), 2, 1)

/*
   GETs will be shown on screen in inverse, so all we need to do is save
   a chunk of the screen and look at the color attribute if the color
   attribute is 112 (inverse), then we must be in a GET
*/
#xtranslate IsItAGet(<r>, <c>) => ;
        substr(savescreen(<r>, <c>, <r>, <c>), 2, 1) == chr(112)

// manifest constants for main browse window coordinates
#define TopRow      boxcoords[1]
#define LeftColumn  boxcoords[2]
#define BottomRow   boxcoords[3]
#define RightColumn boxcoords[4]

// manifest constants for memo window coordinates
#define MEMOTOP     memocoords_[1]
#define MEMOLEFT    memocoords_[2]
#define MEMOBOTTOM  memocoords_[3]
#define MEMORIGHT   memocoords_[4]

// bogus "methods" for get "objects" (kinda OOP kinda WOW)
#xtranslate GetRow(<xx>)       =>  browse:getArray\[<xx>, 1]
#xtranslate GetCol(<xx>)       =>  browse:getArray\[<xx>, 2]
#xtranslate GetLength(<xx>)    =>  browse:getArray\[<xx>, 3]
#xtranslate GetName(<xx>)      =>  browse:getArray\[<xx>, 4]
#xtranslate GetPicture(<xx>)   =>  browse:getArray\[<xx>, 5]

// bogus "methods" for box "objects" (kinda OOP kinda WOW)
#xtranslate BoxTop(<xx>)       =>  browse:boxArray\[<xx>, 1]
#xtranslate BoxLeft(<xx>)      =>  browse:boxArray\[<xx>, 2]
#xtranslate BoxBottom(<xx>)    =>  browse:boxArray\[<xx>, 3]
#xtranslate BoxRight(<xx>)     =>  browse:boxArray\[<xx>, 4]
#xtranslate BoxString(<xx>)    =>  browse:boxArray\[<xx>, 5]
#xtranslate BoxColor(<xx>)     =>  browse:boxArray\[<xx>, 6]
#xtranslate BoxFill(<xx>)      =>  browse:boxArray\[<xx>, 7]

// indentation of generated source code (my preference is three)
#define INDENT(x) space(x * 3)

#xtranslate DefaultKey( <n>, <c>, <msg> )     ;
                                           => ;
          if valtype( aKeys\[<n>\] ) <> "A"   ;
          ;   aKeys\[<n>\] := { <c>, <msg> }  ;
          ; end


function GrumpBrow(cSecurity, ntop, nleft, nbottom, nright, lBox, ;
                   aTFields, aHeads, aPics, aAlternate,           ;
                   starting, ending, vblock, hblock, aExtraopts,  ;
                   nExtrakey, cExtratitle, nExitkey, aIndexdesc,  ;
                   aValids, aWhens, nRefresh, nLockcols, lCarry,  ;
                   nMemowidth, cTitle, nMemoHeight, cColorspec,   ;
                   aColorBlocks, bInitial, lGoTop)
local cBoxcolor
local cArrowcolor
local security[S_ITEMS]
local xx
local nMenuTop
local a_
local options
local mfield
local buffer
local lreadexit
local mtype
local lPainter
local key
local subkey
local browse
local mget
local column
local marker
local lOldScore
local keepgoing
local mrow
local mcol
local searchstr
local fields_
local stru_
local scrnbuff
local lOldInsert
local nDirection := NO_MOVEMENT
local lDescend := .f.
local cKey := upper(indexkey(0))
local midrow
local midcol
local memocoords_
local bOldskip, bOldgotop, bOldgobott       // original movement blocks
local num_flds
local piclens_ := {}          // array holding lengths of PICTURE clauses
local lSetDeleted := ! set(_SET_DELETED)   // used for showing deleted status
local lConfigure := .f.       // used when shrinking/expanding columns
local aKeys := array(S_ITEMS)

// next three flags are for the return value
local lDelete := .f.
local lAdd    := .f.
local lEdit   := .f.

// nOrder holds the current index order.  My reason for storing this
// to a variable is to optimize performance of the Gilligan() skip
// function, which will be used in the event that you have a data subset.
local nOrder

#ifdef EXTRA_OPTIONS

local aOptions
local aActions

#endif

// lNocase is a flag to denote case-insensitive searches
local lNocase := "UPPER(" $ cKey

// no database active... ergo, no point in continuing
if empty(alias())
   return ''
endif

GFSaveEnv(.t., 0)                // shuts off cursor

#ifdef EXTRA_OPTIONS

// initialize options/blocks arrays if "extra options" array was passed
default nExtrakey to K_ALT_F10
if aExtraopts <> NIL
   buffer := len(aExtraopts)
   aOptions := array(buffer)
   aActions := array(buffer)
   for xx := 1 to buffer
      aOptions[xx] := aExtraopts[xx][1]
      aActions[xx] := aExtraopts[xx][2]
   next
endif

#endif

default nExitkey to K_ESC
default cSecurity to ''
default ntop to 0
default nleft to 0
default nbottom to maxrow() - 1
default nright to maxcol()
default lBox to .t.
default lCarry to .f.
default lGoTop to .f.
if aAlternate == NIL
   aAlternate := array(8)
elseif len(aAlternate) < 8
   asize(aAlternate, 8)
endif

nbottom := min(nbottom, maxrow() - 1)
midrow := ntop + int(nbottom - ntop + 1) / 2
midcol := nleft + int(nright - nleft + 1) / 2

if cColorspec == NIL
   cColorspec  := ColorSet(C_GRUMPBROW_SAY, .T.) + ',' + ;
                  ColorSet(C_GRUMPBROW_GET, .T.) + ',,,'+ ;
                  ColorSet(C_GRUMPBROW_SAY, .T.)
   cBoxcolor   := ColorSet(C_GRUMPBROW_BOX, .T.)
   cArrowcolor := ColorSet(C_GRUMPBROW_GET, .T.)
else
   xx          := at(',', cColorspec)
   cBoxcolor   := left(cColorspec, xx - 1)
   cArrowcolor := ltrim(substr(cColorspec, xx + 1))
endif
options := []
afill(security, .f.)
cSecurity := upper(cSecurity)
if "A" $ cSecurity
   security[S_ADD] := .t.
   DefaultKey(S_ADD, asc("A"), "[A]dd")
   options += aKeys[S_ADD][2] + ' '
endif
if "D" $ cSecurity
   security[S_DELETE] := .t.
   DefaultKey(S_DELETE, asc("D"), "[D]elete")
   options += aKeys[S_DELETE][2] + ' '
endif
if "E" $ cSecurity .and. ! ("C" $ cSecurity)
   security[S_EDIT] := .t.
   DefaultKey(S_EDIT, asc("E"), "[E]dit")
   options += aKeys[S_EDIT][2] + ' '
   /*
      Determine if user will be able to edit the cell by pressing Enter.
      By default, if you pass "E" so that they can edit, they will be able
      to edit an individual cell.  However, if you don't want them to be able
      to do this, pass an "N" as part of the security string.  (Note that
      you can also pass "C" to enable individual cell editing - see below)
   */
   security[S_CELLEDIT] := ! ("N" $ cSecurity)
endif
// passing "C" as part of the security parameter allows cell editing
// and disables full-screen editing
if "C" $ cSecurity
   security[S_CELLEDIT] := .t.
   security[S_EDIT] := .f.
endif
if "Q" $ cSecurity
   security[S_QUERY] := .t.
   DefaultKey(S_QUERY, asc("Q"), "[Q]uery")
   options += aKeys[S_QUERY][2] + ' '
endif
if "K" $ cSecurity
   security[S_PACK] := .t.
   DefaultKey(S_PACK, asc("K"), "pac[K]")
   options += aKeys[S_PACK][2] + ' '
endif
if "S" $ cSecurity .and. ! empty(cKey)
   xx := 0
   do while ! empty( buffer := indexkey(++xx) ) .and. ! security[S_SEARCH]
      // strip out the descend() function if it's part of the key
      if "DESCEND(" == left(buffer, 8)
         lDescend := .t.     // used when searching... see below
         buffer := strtran(buffer, 'DESCEND(', '')
         buffer := left(buffer, len(buffer) - 1)
      endif
      // determine if this is a character indexkey
      security[S_SEARCH] := ;
               (valtype(eval( &("{ || " + buffer + "}") )) == "C")
   enddo
   if security[S_SEARCH]
      DefaultKey(S_SEARCH, asc("S"), "[S]earch")
      options += aKeys[S_SEARCH][2] + ' '
   endif
endif
if "V" $ cSecurity
   security[S_VIEW] := .t.
   DefaultKey(S_VIEW, asc("V"), "[V]iew")
   options += aKeys[S_VIEW][2] + ' '
endif
if "L" $ cSecurity
   security[S_LOCKCOLUMN] := .t.
   DefaultKey(S_LOCKCOLUMN, asc("L"), "[L]ock")
   options += aKeys[S_LOCKCOLUMN][2] + ' '
endif
if "O" $ cSecurity
   security[S_OUTPUT] := .t.
   DefaultKey(S_OUTPUT, asc("O"), "[O]utput")
   options += aKeys[S_OUTPUT][2] + ' '
endif
if ! empty(cKey) .and. "F" $ cSecurity
   security[S_SUBSET] := .t.
   DefaultKey(S_SUBSET, K_ALT_S, "[Alt-S]ubset")
   if len(options) < nright - nleft - 26
      options += aKeys[S_SUBSET][2] + ' '
   endif
endif
if ! empty( indexkey(2) ) .and. "I" $ cSecurity
   security[S_SWAPINDEX] := .t.
   DefaultKey(S_SWAPINDEX, K_ALT_I, "[Alt-I]ndex")
   if len(options) < nright - nleft - 25
      options += aKeys[S_SWAPINDEX][2] + ' '
   endif
endif

// only show exit option if using default exit key
if nExitkey == K_ESC
   options += "[ESC]=quit"
endif

// must pass a "P" in the security parameter to allow screen painting
lPainter := ("P" $ cSecurity)

// create multi-dimensional FIELDS array based on the fields
// array you passed as a parameter
if aTFields <> NIL
   fields_ := aclone(aTFields)
   // create array aTFields containing all fields in this database
   stru_ := dbstruct()
   num_flds := len(fields_)
   for xx := 1 to num_flds

      // if it's a code block, no need to create a field subarray
      if valtype(fields_[xx]) == "B"
         // do nothing

      // find this field in the database to create field subarray
      elseif (mfield := ascan(stru_, { | a | ;
                        upper(a[DBS_NAME]) == upper(fields_[xx]) } ) )  > 0
         fields_[xx] := stru_[mfield]
      else    // you called for a field that's not in the .dbf... bye-bye
         GFRestEnv()
         return ''
      endif
   next
else
   fields_ := dbstruct()
   num_flds := len(fields_)
endif

// either initialize blank colorblock array or verify its length
if aColorBlocks == NIL
   aColorBlocks := array(num_flds)
elseif len(aColorBlocks) < num_flds
   asize(aColorBlocks, num_flds)
endif

// create array aHeads (column headings) if necessary
if aHeads == NIL
   // dump all field names from array FIELDS_ into aHeads
   aHeads := array(num_flds)
   aeval(fields_, { | a, b | aHeads[b] := if(valtype(a) == "A", a[1], '') } )
endif

// create array aPics (picture clauses) if necessary
if aPics == NIL
   /*
      okay - create the array and fill it with garbage hi bit ASCII
      why?  because we must go through the following FOR..NEXT to
      establish the array holding PICTURE lengths, regardless of
      whether or not they passed the aPics array.  I didn't want to
      duplicate the code because that is totally wasteful, so I resorted
      to CHR(254) as an indicator that we had to create the array here.
   */
   aPics := array(num_flds)
   afill(aPics, chr(254))
endif

// create PICLENS_ array to hold the length of each GET.
piclens_ := array(num_flds)
for xx := 1 to num_flds
   if valtype(fields_[xx]) == "A"
      a_ := makepicture(fields_[xx], aPics[xx])
      aPics[xx] := a_[1]
      piclens_[xx] := a_[2]
   else
      piclens_[xx] := 10   // default for non-fields
   endif
next

// create a browse object
browse := TBrowseDB(ntop + 1, nleft + 1, nbottom - 1, nright - 1)
browse:cargo := array(CARGO_LENGTH)
browse:getArray := {}
browse:boxArray := {}
browse:origSkipBlock := browse:skipBlock

// set memo width -- if passed as parameter, must be > 12
// in either case, cannot be larger than the entire browse window
if nMemowidth <> NIL
   browse:memoWidth := min( max(nMemowidth, 12), nright - nleft - 4 )
else
   browse:memoWidth := min( 40, nright - nleft - 4 )
endif

// set memo height -- if passed as parameter, must be > 4
// in either case, cannot be larger than the entire browse window
if nMemoHeight <> NIL
   browse:memoHeight := min( max(nMemoHeight, 4), nbottom - ntop )
else
   browse:memoHeight := min( 8, nbottom - ntop )
endif

// if there's an active index, store reference block in cargo
if ! empty(cKey)
   browse:indexBlock := &("{ || " + cKey + "}")
endif

// initialize coordinates for memo window, using coordinates of
// primary window as limiting factors
memocoords_ := { max(midrow - browse:memoHeight/2, ntop), ;
                 max(midcol - browse:memoWidth/2 - 1, nleft),  ;
                 min(midrow + browse:memoHeight/2, nbottom), ;
                 min(midcol + browse:memoWidth/2 + 1, nright) }

// initialize data subset arrays
xx := 0
do while ! empty(indexkey(++xx))
enddo
browse:startValues := array(max(xx - 1, 1))
browse:endValues   := array(max(xx - 1, 1))
default aIndexdesc to array(max(xx - 1, 1))

// initialize subset start/end for current index if they were specified
// note that nOrder is used in Gilligan() for data subsets
if ( nOrder := indexord() ) > 0
   browse:startValues[nOrder] := starting
   browse:endValues[nOrder]   := ending
endif

// save original movement blocks in case they need to be reset later
bOldgotop := browse:goTopBlock
bOldgobott := browse:goBottomBlock
bOldskip := browse:skipBlock

// set up movement blocks if hi/low values were passed as parameters
if ! empty(cKey) .and. ;
         (browse:startValues[nOrder] <> NIL .or. browse:endValues[nOrder] <> NIL)
   pseudofilt(browse, .f., bOldgotop, bOldgobott, bOldskip, nOrder)
endif
setcolor(cColorspec)

browse:colorSpec := cColorspec
browse:headSep := ""
browse:colSep  := "  "
for xx := 1 to num_flds

   // if there is a code block in the FIELDS_ array rather than a
   // subarray of field info, plug the block directly into the column
   if valtype(fields_[xx]) == "B"
      column := TBColumnNew(aHeads[xx], fields_[xx])
   else

      // memos must be treated differently than regular fields...
      // also note that character strings longer than 255 must be
      // treated as memos, because TBrowse chokes on them!
      if fields_[xx][DBS_TYPE] == "M" .or. fields_[xx][DBS_LEN] > 255
         column := TBColumnNew(aHeads[xx], &("{ || if(empty(" + ;
                   fields_[xx][DBS_NAME] + "), '<memo>', '<MEMO>') }"))
      else
         column := TBColumnNew(aHeads[xx])
         // handle picture clause for this column (if there is one)
         if aPics[xx] == chr(254)
            column:block := fieldblock(fields_[xx][DBS_NAME])
         else
            // if we are using Clipper 5.2, we can take advantage
            // of the new TBColumn:picture instance variable
            // instead of slugging it out with TRANSFORM()
            #ifdef CLIPPER52
               column:block   := fieldblock(fields_[xx][DBS_NAME])
               column:picture := aPics[xx]
            #else
               // use TRANSFORM() to simulate PICTURE.
               column:block := &("{ | | transform(" + fields_[xx][DBS_NAME] + ;
                                 ", '" + aPics[xx] + "') }")
              /*
                 Now load cargo with a two element array.  Element 1
                 contains the "un-transformed" retrieval code block
                 for this field, and element 2 contains the desired
                 PICTURE clause.  These are used by GETNEW() when
                 editing a cell directly (see below).
              */
              column:cargo := { &("{ | x | if( pcount() == 0, " + ;
                     alias() + "->" + fields_[xx][DBS_NAME] + ", " + ;
                     alias() + "->" + fields_[xx][DBS_NAME] + " := x) }"), ;
                     aPics[xx] }
            #endif
         endif
      endif
   endif

   // assign colorBlock if one was specified for this column
   if aColorblocks[xx] <> NIL
      column:colorBlock := aColorblocks[xx]
   endif

   browse:AddColumn( column )
   // initialize column width so that columns can shrink/expand
   column:width := browse:colWidth(xx)
next

// initialize alternate cell editing array to proper length if needed
if valtype(aAlternate[S_CELLEDIT]) <> "A"
   aAlternate[S_CELLEDIT] := array(num_flds)
elseif len(aAlternate[S_CELLEDIT]) < num_flds
   asize(aAlternate[S_CELLEDIT], num_flds)
endif

// lock the desired number of columns
if nLockcols <> NIL
   browse:freeze := nLockcols
endif

// draw box around browse window if desired
if lBox
   ShadowBox(ntop, nleft, nbottom, nright, 1, cTitle,, cBoxcolor)
endif
@ if(lBox, nbottom, maxrow()), midcol - int(len(options) / 2) ssay options ;
  color cBoxcolor

// initialize or resize column pre- and post-validation arrays
default aValids to array(browse:colCount)
if len(aValids) < browse:colCount
   asize(aValids, browse:colCount)
endif
default aWhens to array(browse:colCount)
if len(aWhens) < browse:colCount
   asize(aWhens, browse:colCount)
endif

// execute initialization codeblock if one was specified
if valtype(bInitial) == "B"
   eval(bInitial, browse)
endif

lOldScore := set(_SET_SCOREBOARD, .F.)    // I hate it!

// abandon all hope ye who enter here...
do while key <> nExitkey
   // do not allow cursor to move into frozen columns
   if browse:colPos <= browse:freeze
      browse:colPos := browse:freeze + 1
   endif
   dispbegin()
   do while (key := inkey()) == 0 .and. ! browse:stabilize()
   enddo
   dispend()
   if key == 0

      // if there is a box around the browse window, and SET DELETED
      // is currently off, show deleted status at top right corner
      mrow := row()
      mcol := col()
      if lBox .and. lSetDeleted
         @ ntop, nright - 11 say if(deleted(), "<deleted>", "") ;
                             color cBoxcolor
         setpos(mrow, mcol)
      endif

      // evaluate vertical or horizontal movement blocks
      // if applicable
      if vblock <> NIL .and. nDirection == VERTICAL
         eval(vblock, browse)
      elseif hblock <> NIL .and. nDirection == HORIZONTAL
         eval(hblock, browse)
      endif

      // draw arrows if data off to left or right
      // must take frozen columns into account
      if browse:leftvisible > browse:freeze + 1
         @ nbottom, nleft say chr(17) + chr(196) ;
                          color cArrowcolor
      else
         @ nbottom, nleft say if(lBox, chr(200)+chr(205), space(2)) ;
                          color cBoxcolor
      endif
      if browse:rightvisible < browse:colCount
         @ nbottom, nright - 1 say chr(196) + chr(16) ;
                          color cArrowcolor
      else
         @ nbottom, nright - 1 say if(lBox, chr(205)+chr(188), space(2)) ;
                          color cBoxcolor
      endif
      setpos(mrow, mcol)

      if browse:hitTop
         waiton( { "Beginning of data!" }, .f.)
         inkey(2)
         waitoff()
      elseif browse:hitBottom
         waiton( { "End of data!" }, .f.)
         inkey(2)
         waitoff()
      endif

      if nRefresh <> NIL
         xx := seconds()
         // if auto refresh time was specified, use another loop
         // so we can refresh while waiting for a keypress
         do while ( key := ginkey( , "KEY", browse) ) == 0
            if seconds() - xx > nRefresh
               xx := seconds()
               browse:refreshAll()
               dispbegin()
               #ifdef CLIPPER52
                  browse:forceStable()
               #else
                  do while ! browse:stabilize()
                  enddo
               #endif
               dispend()
            endif
         enddo
      else
         key := ginkey(0, "KEY", browse)
      endif
   endif

   // convert key to upper-case for testing purposes below
   if key > 96 .and. key < 123
      key -= 32
   endif

   do case

#ifdef EXTRA_OPTIONS

      case key == nExtrakey .and. aOptions <> NIL
         xx := 1
         nMenuTop := max(ntop, midrow - (len(aOptions) / 2) - 1 )
         do while ( xx := lite_menu(nMenuTop,,aOptions,,,,xx,cExtratitle) ) <> 0
            eval(aActions[xx], browse)
         enddo

#endif

      case key == K_LEFT
         browse:left()
         nDirection := HORIZONTAL

      case key == K_RIGHT
         browse:right()
         nDirection := HORIZONTAL

      case key == K_UP
         browse:up()
         nDirection := VERTICAL

      case key == K_DOWN
         browse:down()
         nDirection := VERTICAL

      case key == K_PGUP
         browse:pageUp()
         nDirection := VERTICAL

      case key == K_PGDN
         browse:pageDown()
         nDirection := VERTICAL

      case key == K_HOME .and. browse:indexBlock <> NIL
         JumpUp(browse, norder)
         nDirection := VERTICAL

      case key == K_END .and. browse:indexBlock <> NIL
         JumpDown(browse, norder)
         nDirection := VERTICAL

      case key == K_CTRL_HOME
         browse:panHome()
         nDirection := HORIZONTAL

      case key == K_CTRL_END
         browse:panEnd()
         nDirection := HORIZONTAL

      case key == K_CTRL_PGUP
         browse:goTop()
         nDirection := VERTICAL

      case key == K_CTRL_PGDN
         browse:goBottom()
         nDirection := VERTICAL

      case key == K_CTRL_LEFT
         browse:panLeft()
         nDirection := HORIZONTAL

      case key == K_CTRL_RIGHT
         browse:panRight()
         nDirection := HORIZONTAL

      // direct edit or view memos
      case key == K_ENTER .and. browse:stable .and.             ;
                  (security[S_CELLEDIT] .or.                    ;
                  (valtype(fields_[browse:colPos]) == "A" .and. ;
                  (fields_[browse:colPos][DBS_TYPE] == "M" .or. ;
                  fields_[browse:colPos][DBS_LEN] > 255) ) )

         // determine if alternate cell-editing block was specified
         if valtype(aAlternate[S_CELLEDIT][browse:colPos]) == "B"

            eval(aAlternate[S_CELLEDIT][browse:colPos], browse)

         else

            // set INS key to toggle shape of cursor as well as insert mode
            lOldInsert := setkey( K_INS, {|| setcursor( ;
                         if(readinsert(! readInsert()), SC_NORMAL, SC_SPECIAL1))} )

            // process memos differently
            if valtype(fields_[browse:colPos]) == "A" .and.           ;
                       ( fields_[browse:colPos][DBS_TYPE] == "M" .or. ;
                         fields_[browse:colPos][DBS_LEN] > 255 )

               buffer := ShadowBox(MEMOTOP, MEMOLEFT, MEMOBOTTOM, MEMORIGHT, ;
                                   2, aHeads[browse:colPos])

               // set initial cursor size based on current mode
               if security[S_CELLEDIT]
                  setcursor( if(readInsert(), SC_SPECIAL1, SC_NORMAL) )
               endif
               mfield := memoedit(fieldget(;
                                  fieldpos(fields_[browse:colPos][DBS_NAME])), ;
                                  MEMOTOP + 1, MEMOLEFT + 1, MEMOBOTTOM - 1, ;
                                  MEMORIGHT - 1, security[S_CELLEDIT])
               if security[S_CELLEDIT]
                  setcursor(SC_NONE)
                  if lastkey() <> K_ESC .and. rec_lock()
                     lEdit := .t.   // set flag for return value
                     fieldput(fieldpos(fields_[browse:colPos][DBS_NAME]),mfield)
                     unlock
                     browse:refreshCurrent()
                  endif
               endif
               ByeByeBox(buffer)

            // we need both Edit security access before proceeding
            elseif security[S_CELLEDIT]
               if Rec_Lock()
                  // retrieve current column object from browse object
                  column := browse:getColumn(browse:colPos)

                  // set initial cursor size based on current mode
                  setcursor( if(readInsert(), SC_SPECIAL1, SC_NORMAL) )

                  // enable up/down arrow keys to exit the read
                  lreadexit := readexit(.t.)
                  /*
                     Create corresponding GET object with GETNEW() and
                     read it now. Note the use of the TBcolumn cargo instance
                     variable.  Cargo is a two-element array.  The first
                     element contains the retrieval code block for this
                     data item.  The second contains the PICTURE clause.
                     This was initialized above.
                  */
                  if valtype(fields_[browse:colPos]) == "A"
                     #ifdef CLIPPER52    // retrieve column:picture
                        xx := getnew(Row(), Col(), column:block, ;
                              column:heading, column:picture, browse:colorSpec)
                     #else
                        xx := getnew(Row(), Col(), column:cargo[1], ;
                              column:heading, column:cargo[2], browse:colorSpec)
                     #endif
                  else
                     // "code block" column -- picture won't apply
                     xx := getnew(Row(), Col(), column:block, ;
                           column:heading, , browse:colorSpec)
                  endif
                  // initialize WHEN clause for this column if one exists
                  if valtype(aWhens[browse:colPos]) == "B"
                     xx:preBlock := aWhens[browse:colPos]
                  endif
                  // initialize VALID clause for this column if one exists
                  if valtype(aValids[browse:colPos]) == "B"
                     xx:postBlock := aValids[browse:colPos]
                  endif
                  readmodal( { xx } )
                  setcursor(SC_NONE)
                  readexit(lreadexit)
                  lEdit := ( lEdit .or. (lastkey() <> K_ESC) )
                  // check if this field is part of an active filter
                  if upper(column:heading) $ upper(dbfilter())
                     // if so, see if filter condition is still fulfilled
                     // if not, bounce record pointer so edited record
                     // will no longer be displayed
                     if ! eval( &("{ || " + dbfilter() + "}"))
                        skip
                        browse:refreshAll()
                     else
                        browse:refreshCurrent()
                     endif
                  endif
                  // if this field is part of the active index, repaint
                  if upper(column:heading) $ upper(indexkey(0))
                     browse:refreshAll()
                  else
                     browse:refreshCurrent()
                  endif
                  // if we exited with an arrow key, pass it through
                  // and start editing same field in the next record
                  xx := lastkey()
                  if xx == K_UP .or. xx == K_DOWN
                     if xx == K_UP
                        browse:up()
                     else
                        browse:down()
                     endif
                     #ifdef CLIPPER52
                        browse:forceStable()
                     #else
                        do while ! browse:stabilize()
                        enddo
                     #endif
                     keyboard chr(K_ENTER)
                  endif
                  unlock
               endif
            endif
            setkey(K_INS, lOldInsert)

         endif // valtype(aAlternate[S_CELLEDIT]) == "A" etcetera ad nauseum

      case security[S_OUTPUT] .and. key == aKeys[S_OUTPUT][1]
         if ! override(aAlternate[S_OUTPUT], browse) .and. ;
                         yes_no('Output to printer')
            printit(browse)
         endif

      case security[S_PACK] .and. key == aKeys[S_PACK][1]
         if yes_no("Removing deleted records may take several minutes", ;
                   "Are you sure you want to continue")
            // if current record is marked for deletion, move
            // backwards to find the closest undeleted record
            if deleted()
               do while deleted() .and. ! bof()
                  dbskip( -1 )
               enddo
            endif
            xx := recno()
            pack
            dbgoto( xx )
            browse:refreshAll()
         endif

      // note: locking only available if not all columns are visible
      case security[S_LOCKCOLUMN] .and. key == aKeys[S_LOCKCOLUMN][1]
         if browse:rightVisible == browse:colCount .and. ;
                    browse:leftVisible == 1 .and. browse:freeze == 0
            err_msg( { "All columns are currently visible" } )
         else
            xx := browse:freeze
            boxget xx prompt "Lock how many columns" picture '##'
            if lastkey() <> K_ESC
               browse:freeze := xx
               browse:invalidate()
            endif
         endif

      case security[S_SEARCH] .and. key == aKeys[S_SEARCH][1]
         if valtype(eval( &("{ || " + indexkey(0) + "}") )) == "C"
            if ! override(aAlternate[S_SEARCH], browse)
               // save affected portion of top row of box
               scrnbuff := savescreen(browse:nTop - 1, browse:nLeft, ;
                                      browse:nTop - 1, browse:nRight)
               @ browse:nTop-1, browse:nLeft + ;
                 (browse:nRight - browse:nLeft)/2 - 11 ssay "[" + space(20) + "]"
               subkey := ginkey(0, "KEY")
               searchstr := ''
               do while ( (subkey > 31 .and. subkey < 255) .or. subkey == K_BS )
                  marker := recno()
                  if subkey == K_BS
                     searchstr := substr(searchstr, 1, len(searchstr) - 1)
                     if len(searchstr) > 0
                        browse:goTop()
                        if lDescend
                           seek descend(searchstr)
                        else
                           seek searchstr
                        endif
                     else
                        searchstr := NIL
                        exit
                     endif
                  else
                     browse:goTop()
                     // force upper-case search if necessary (see above)
                     if lNocase .and. subkey > 96 .and. subkey < 123
                        subkey -= 32
                     endif
                     if lDescend
                        seek descend( searchstr + chr(subkey) )
                     else
                        seek searchstr + chr(subkey)
                     endif
                  endif
                  if eof()
                     go marker
                  else
                     if subkey <> K_BS
                        searchstr += chr(subkey)
                     endif
                     @ browse:nTop-1, browse:nLeft + ;
                       (browse:nRight - browse:nLeft)/2 - 10 ssay padc(searchstr, 20)

                     // if data subset is active, only display the
                     // found record if it is within the data subset
                     if browse:startValues[nOrder] <> NIL
                        if eval(browse:indexBlock) < browse:startValues[nOrder] .or. ;
                                eval(browse:indexBlock) > browse:endValues[nOrder]
                           go marker
                        endif
                     endif

                     // did record pointer move?
                     if recno() <> marker
                        // save current record number... see below
                        marker := recno()
                        // force a redisplay
                        dispbegin()
                        browse:refreshAll()
                        #ifdef CLIPPER52
                           browse:forceStable()
                        #else
                           do while ! browse:stabilize()
                           enddo
                        #endif
                        /*
                           Sometimes the refreshAll() method will not leave
                           us on the correct record.  If this was the case,
                           we'll move up until we get to the right record.
                        */
                        do while recno() <> marker
                           browse:up()
                           skip -1
                        enddo
                        dispend()
                        nDirection := VERTICAL
                     else
                        browse:hiLite()
                     endif
                  endif
                  subkey := ginkey(0, "KEY")
               enddo
               restscreen(browse:nTop - 1, browse:nLeft, ;
                          browse:nTop - 1, browse:nRight, scrnbuff)
               searchstr := NIL
            else
               browse:refreshAll()
            endif
         else
            err_msg( { "Search feature unavailable for current index" } )
         endif

      case security[S_ADD] .and. key == aKeys[S_ADD][1]
         if ! override(aAlternate[S_ADD], browse)
            lAdd := gfbrecview('A', fields_, aHeads, aPics, browse, ;
                               piclens_, aWhens, aValids, lCarry)
         endif
         browse:refreshAll()

      case security[S_DELETE] .and. key == aKeys[S_DELETE][1]
         if ! override(aAlternate[S_DELETE], browse)
            if yes_no('This record will be deleted from the file',;
                  'Do you want to do this')
               if rec_lock()
                  dbDelete()
                  eval(browse:skipBlock, -1)
                  browse:refreshall()
                  unlock
                  lDelete := .t.  // set flag for return value
               else
                  err_msg( { NETERR_MSG } )
               endif
            endif
         else
            browse:refreshall()
         endif
      
      case security[S_EDIT] .and. key == aKeys[S_EDIT][1]
         if ! override(aAlternate[S_EDIT], browse)
            if gfbrecview('E', fields_, aHeads, aPics, browse, ;
                               piclens_, aWhens, aValids)
               lEdit := .t.    // set flag for return value
               browse:refreshAll()
            endif
         else
            browse:refreshAll()
         endif
      
#ifdef REMBRANDT

      case (key == 112 .or. key == 80) .and. lPainter      // screen painter
         gfbrecview('S', fields_, aHeads, aPics, browse, piclens_)

#endif

      case security[S_QUERY] .and. key == aKeys[S_QUERY][1]
         if ! override(aAlternate[S_QUERY], browse)
            keepgoing := .t.
            if ! empty(browse:queryCondition)
               if yes_no("Use most recent query search criteria")
                  keepgoing := .f.
                  marker := recno()
                  // browse:hitBottom won't work in this situation
                  // which is why I am using temporary variable XX
                  do while ( xx := eval(browse:skipBlock, 1) <> 0 ) ;
                      .and. ! eval(browse:queryCondition) .and. checkescape()
                  enddo
                  if ! xx
                     err_msg( { "No more matches" } )
                     go marker
                  else
                     browse:refreshAll()
                     nDirection := VERTICAL
                  endif
               endif
            endif
            if keepgoing
               gfbrecview('Q', fields_, aHeads, aPics, browse, piclens_)
               browse:refreshAll()
               nDirection := VERTICAL
            endif
         else
            browse:refreshAll()
            nDirection := VERTICAL
         endif

      case security[S_VIEW] .and. key == aKeys[S_VIEW][1]
         if ! override(aAlternate[S_VIEW], browse)
            gfbrecview('V', fields_, aHeads, aPics, browse, piclens_)
         else
            browse:refreshAll()
         endif

      case security[S_SWAPINDEX] .and. key == aKeys[S_SWAPINDEX][1]
         if indexord() == len(browse:startValues)
            nOrder := 1
         else
            nOrder++
         endif
         dbsetorder(nOrder)
         cKey := indexkey(0)
         scrnbuff := savescreen(browse:nTop - 1, browse:nLeft, ;
                                browse:nTop - 1, maxcol())
         // display index description, using the one from the index
         // description array if it was passed as a parameter
         @ browse:nTop - 1, browse:nLeft say "[ " + ;
                 if(aIndexdesc[nOrder] <> NIL, ;
                    aIndexdesc[nOrder], ;
                    "Switching to " + cKey + " index") + " ]"

         // reset case-insensitive flag
         lNocase := "UPPER(" $ upper(cKey)

         // reset descend flag
         lDescend := ( "DESCEND(" == left(cKey, 8) )

         // reset the cargo block referring to the indexkey
         browse:indexBlock := &("{ || " + cKey + "}")

         // if there is a data subset previously established
         // for this index, set it up
         if browse:startValues[nOrder] <> NIL .or. ;
                   browse:endValues[nOrder] <> NIL
            pseudofilt(browse, .f., bOldgotop, bOldgobott, bOldskip, nOrder)
         else
            browse:goTopBlock    := bOldgotop
            browse:goBottomBlock := bOldgobott
            browse:skipBlock     := bOldskip
            // move record pointer to top of file if requested
            if lGoTop
               browse:goTop()
            endif
         endif
         browse:refreshAll()
         inkey(3)
         restscreen(browse:nTop - 1, browse:nLeft, ;
                    browse:nTop - 1, maxcol(), scrnbuff)

      case security[S_SUBSET] .and. key == aKeys[S_SUBSET][1]
         pseudofilt(browse, .t., bOldgotop, bOldgobott, bOldskip, nOrder)

      case (key == K_PLUS .or. key == K_MINUS .or. key == K_ALT_PLUS .or. ;
            key == K_ALT_MINUS)
         column := browse:getColumn(browse:colPos)
         do case
            case key == K_PLUS       // expand current column by one
               column:width++
               lConfigure := .t.
            case key == K_MINUS      // shrink current column by one
               if column:width > 1
                  lConfigure := .t.
                  column:width--
               endif
            case key == K_ALT_PLUS   // expand current column by five
               column:width += 5
               lConfigure := .t.
            case key == K_ALT_MINUS  // shrink current column by one
               if column:width > 5
                  lConfigure := .t.
                  column:width -= 5
               endif
         endcase
         if lConfigure
            browse:configure() // force browse to update this column's width
            // we must force a full stabilization here because if the
            // user is permitted to break the stabilize loop (as above)
            // unpredictable results will occur (same for shrinking below)
            dispbegin()
            #ifdef CLIPPER52
               browse:forceStable()
            #else
               do while ! browse:stabilize()
               enddo
            #endif
            dispend()
            lConfigure := .f.
         endif

   endcase
enddo
GFRestEnv()
set(_SET_SCOREBOARD, lOldScore)
return if(lAdd, 'A', '') + if(lEdit, 'E', '') + if(lDelete, 'D', '')

* end function GrumpBrow()
*--------------------------------------------------------------------*


/*
   Override(): See if a procedure/function name was passed to override
               the generic add/edit/delete/search routines -- if so,
               and if it exists in the application, run that instead
               of the generic routine
*/
static function Override(melement, browse)
local mproc, ret_val := .f., xx
if ! empty(melement)
   // first check if they specified a code block, and if so, evaluate
   // the block passing the browse object as a parameter
   if valtype(melement) == "B"
      ret_val := .t.
      eval(melement, browse)
   else
      if ! type(melement) $ 'UE'
         ret_val := .t.
         eval( &("{ | | " + melement + "}") )
      endif
   endif
endif
return ret_val

* end static function Override()
*--------------------------------------------------------------------*


/*
    GfbRecView(): The heart of the order, so to speak.  This is
                  where all SAYs and GETs are SAID and GOTTEN
*/
static function GfbRecView(mode, fields_, aHeads, aPics, browse, ;
                           piclens_, aWhens, aValids, lCarry)
local lOldInsert
local picstring
local mstring
local mfield
local mrow
local handle
local pic_len
local xx
local yy
local zz
local nleft := maxcol() - 38     // used to limit long char strings below
local marker
local buffer
local oldcolor
local is_memo
local mtype
local key
local oldscrn    := savescreen(0, 0, maxrow(), maxcol())
local picclause
local getlist    := {}
local mscrnfile  := alias() + '.gfs'
local mainloop   := .t.
local curr_fld
local bytes      := (maxrow() + 1) * (maxcol() + 1) * 2
local num_flds   := len(fields_)
local scatter    := array(num_flds)
local num_boxes  := 0
local firstfield := 1
local goingdown
local curr_get
local ret_val    := .f.   // necessary to determine if add/edit was finalized

setcolor(browse:colorSpec)
scroll()
if mode <> "S"
   @ 0, 2 ssay '[ ' + if(mode == 'Q', "Query database", ;
            if(mode == 'A', "Add" , if(mode == "E", "Edit", "View")) + ;
            "ing record") + ' ]'
endif
if mode $ "AE"
   scrncenter(maxrow(), 'Ctrl-W = save edits   Esc = exit without saving')
elseif mode == "Q"
   scrncenter(maxrow(), 'Ctrl-W = process query    Esc = exit without processing')
elseif mode == 'V'
   scrncenter(maxrow(), 'N = next record   P = prev record   any other key = next screen (or exit)')
endif

asize(browse:getArray, 0)

// if we have a pre-saved screen file, read it in now
if file(mscrnfile)
   if ( handle := fopen(mscrnfile) ) == -1
      err_msg( { "Could not open " + mscrnfile } )
   else
      buffer := space(bytes)
      // first paint the static text by restoring the saved screen
      fread(handle, @buffer, bytes)
      restscreen(0, 0, maxrow(), maxcol(), buffer)
      buffer := [ ]
      // then read next byte, which will tell us how many fields
      // are stored in this file
      fread(handle, @buffer, 1)
      num_flds := bin2i(buffer)
      // now read in that number of fields
      asize(browse:getArray, num_flds)
      for yy := 1 to num_flds
         gfreadline(@buffer, handle)
         browse:getArray[yy] := { bin2i(substr(buffer, 1, 1)), ;   // row pos
                           bin2i(substr(buffer, 2, 1)), ;   // column position
                           bin2i(substr(buffer, 3, 1)), ;   // row position
                                 substr(buffer, 4, 10), ;   // field name
                                 substr(buffer, 14)   }     // PICTURE clause
      next
      // okay, now read next byte, which will tell us how many boxes
      // are stored in this file  -- note that we are limited to 20
      buffer := [ ]
      fread(handle, @buffer, 1)
      num_boxes := min(bin2i(buffer), 20)
      buffer := space(14)
      // now read in that number of boxes
      asize(browse:boxArray, num_boxes)
      for xx := 1 to num_boxes
         fread(handle, @buffer, 14)
         browse:boxArray[xx] := { bin2i(substr(buffer, 1, 1)),  ;   // top row
                           bin2i(substr(buffer, 2, 1)),  ;   // left column
                           bin2i(substr(buffer, 3, 1)),  ;   // bottom row
                           bin2i(substr(buffer, 4, 1)),  ;   // right column
                                 substr(buffer, 5, 8) ,  ;   // box string
                           bin2i(substr(buffer, 13, 1)), ;   // box color
                                 substr(buffer, 14, 1) }     // fill box? Y/N
      next
      fclose(handle)
   endif
endif

// determine topmost row at which to paint SAYs and GETs
mrow := max(int( ( maxrow() - num_flds ) / 2 ), 1)
curr_fld := 1
dispbegin()
do while curr_fld <= num_flds .and. mainloop
   if file(mscrnfile)
      mfield := GetName(curr_fld)
      pic_len := GetLength(curr_fld)
      picclause := GetPicture(curr_fld)
   elseif valtype(fields_[curr_fld]) == "A"
      mfield := fields_[curr_fld][DBS_NAME]
      // limit width of long character strings for edit/display
      if fields_[curr_fld][DBS_TYPE] == "C" .and. fields_[curr_fld][DBS_LEN] > nleft
         pic_len   := nleft
         picclause := "@S" + ltrim(str(nleft))
      else
         pic_len := piclens_[curr_fld]
         picclause := aPics[curr_fld]
      endif
   else
      mfield := NIL
   endif

// the NO_TEMPLATE manifest constant means don't present user with
// any fields in paint mode
#ifndef NO_TEMPLATE
   if mfield <> NIL
#else
   if mfield <> NIL .and. mode <> "S"
#endif

      mtype := type(mfield)
      // determine initial value for each get field
      do case
         // if you went nuts and changed the structure on me, there might
         // be a field that is undefined.  Fortunately for you, Mr. Grump
         // is here to protect you from the deadly DOS drop (but cut out
         // your monkey business, because next time I might not be around)
         case mtype == 'U'
            err_msg( { trim(mfield) + " is undefined - please delete the " + ;
                       mscrnfile + " file" } )
            restscreen(0, 0, maxrow(), maxcol(), oldscrn)
            return .f.

         case mtype == 'D'
            scatter[curr_fld] := if(mode == "S", padr(mfield, 8),   ;
                  if((mode == "A" .and. ! lCarry) .or. mode == "Q", ;
                      ctod(''), fieldget(fieldpos(mfield))))
         case mtype == 'M'
            // embed an ascii 255 at the front of this -- we need this when
            // in the get loop down below so that we know this is a memo!
            scatter[curr_fld] := if(mode == "S", padr(mfield, 6), chr(255) + ;
                  if((mode == "A" .and. ! lCarry) .or. mode $ "MQ",          ;
                      [], fieldget(fieldpos(mfield))))
         case mtype == 'C'
            scatter[curr_fld] := if(mode == "S", padr(mfield, pic_len), ;
                  if((mode == "A" .and. ! lCarry) .or. mode == "Q", ;
                      space(len(fieldget(fieldpos(mfield)))), ;
                      fieldget(fieldpos(mfield))))
         case mtype == 'L'
            scatter[curr_fld] := if(mode == "S", substr(mfield, 1, 1), ;
                  if((mode == "A" .and. ! lCarry) .or. mode == "Q",    ;
                      .f., fieldget(fieldpos(mfield))))
         otherwise
            scatter[curr_fld] := if(mode == "S", padr(mfield, pic_len), ;
                  if((mode == "A" .and. ! lCarry) .or. mode == "Q",     ;
                      0, fieldget(fieldpos(mfield))))
      endcase

      // do generic says and gets only if we didn't use a pre-saved screen file
      if ! file(mscrnfile)
         @ curr_fld + mrow, 37 - len(aHeads[curr_fld]) say aHeads[curr_fld]
         if mode == "S"
            aadd(browse:getArray, { curr_fld + mrow, 39, pic_len, mfield, picclause } )
            setcolor("i")
         endif
         // if this is a memo field, we want to simply display the
         // word "<memo>" (defined above as picstring) rather than
         // showing the memo.  the memo will be displayed when they
         // cursor to it.
         if type(mfield) == "M"
            @ curr_fld + mrow, 39 say "<memo>"
         else
            // truncate character fields that would spill off the right side
            if type(mfield) == "C" .and. len(scatter[curr_fld]) > 40
               @ curr_fld + mrow, 39 say substr(scatter[curr_fld], 1, 40) ;
                                         picture picclause
            else
               if mode == "S"
                  @ curr_fld + mrow, 39 say scatter[curr_fld]
               else
                  @ curr_fld + mrow, 39 say scatter[curr_fld] picture picclause
               endif
            endif
         endif
         if mode == "S"
            setcolor(browse:colorSpec)
         endif
      else
         // gotta display the fields in inverse for maintenance mode
         if mode == "S"
            setcolor("i")
         endif
         // pull get coordinates and info out of gets array
         // note: if this is a memo field, we want to simply display the
         // word "<memo>" rather than the memo
         if type(mfield) == "M"
            @ GetRow(curr_fld), GetCol(curr_fld) say "<memo>"
         else
            @ GetRow(curr_fld), GetCol(curr_fld) say scatter[curr_fld] ;
                                picture picclause
         endif
         if mode == "S"
            setcolor(browse:colorSpec)
         endif
      endif

   endif

   // we have either reached the last field or the bottom of the screen
   // whichever came first -- time to stop for the pause that refreshes
   if row() == maxrow() - 1 .or. curr_fld == num_flds
      dispend()
      /*
         GOINGDOWN is a flag that indicates in which direction we are
         moving through the gets.  if it is true, then we are indeed
         moving downward.  if false, we are moving upward.  It is
         important only because when we escape from editing a memo,
         we will not know where to go unless we have such a flag.
         Trust me -- I went through all kinds of gyrations on this one.
         NOTE: its value gets changed via GFBACKAGET() and GFSKIPAGET().
      */
      goingdown := .t.
      do case

#ifdef REMBRANDT

         case mode == "S"
            gfbmaint(fields_, browse, aPics, piclens_, aHeads)
            // we must force an exit from the main loop or else
            // the sucker will bomb horribly
            mainloop := .f.
#endif

         case mode <> "V"
            setkey(K_UP, { || curr_get := gfbackaget(firstfield, curr_get, .t.), ;
                              goingdown := .f. } )
            setkey(K_DOWN, { || goingdown := gfskipaget() } )

            // set INS key to toggle shape of cursor as well as insert mode
            lOldInsert := setkey( K_INS, {|| setcursor( ;
                      if(readinsert(! readInsert()), SC_NORMAL, SC_SPECIAL1))} )

            // set initial cursor size based on current mode
            setcursor( if(readInsert(), SC_SPECIAL1, SC_NORMAL) )

            for curr_get = firstfield to curr_fld
               // skip any "non-fields"
               if scatter[curr_get] <> NIL
                  /*
                     now for the tricky part - figuring out if this is a memo.
                     memos will look like character strings, but since we were
                     thoughtful enough to embed the chr(255), that will assist
                     us in making the determination now
                  */
                  is_memo := if(valtype(scatter[curr_get]) <> "C", .f., ;
                             substr(scatter[curr_get], 1, 1) == chr(255))
                  if ! is_memo
                     if ! file(mscrnfile)
                        // truncate characters that would spill off right side
                        if valtype(scatter[curr_get]) == "C" .and. ;
                                          len(aPics[curr_get]) > 40
                           @ curr_get + mrow, 39 get scatter[curr_get] ;
                                         picture "@S40 " + aPics[curr_get]
                        else
                           @ curr_get + mrow, 39 get scatter[curr_get] ;
                                                   picture aPics[curr_get]
                        endif
                     else
                        @ GetRow(curr_get), GetCol(curr_get) get ;
                               scatter[curr_get] picture GetPicture(curr_get)
                     endif
                     // tack on any applicable WHEN or VALID clauses
                     if mode $ "AE"
                        if valtype(aWhens[curr_get]) == "B"
                           atail(getlist):preBlock := aWhens[curr_get]
                        endif
                        if valtype(aValids[curr_get]) == "B"
                           atail(getlist):postBlock := aValids[curr_get]
                        endif
                     endif
                     read

                     // check last keypress
                     key := lastkey()
                     do case

                        // proceed downward
                        case key == K_ENTER .and. goingdown
                           goingdown := .t.

                        // jump to end of this screen
                        case key == K_PGDN
                           curr_get := curr_fld

                        // back to first get
                        case key == K_PGUP
                           curr_get := firstfield - 1

                        // (save and then) fall out
                        case key == K_CTRL_W .or. key == K_ESC
                           mainloop := .f.          // force exit from main loop
                           curr_get := curr_fld     // force exit from this loop
                     endcase
                  else
                     yy := int( maxcol() / 2 )
                     zz := browse:memoWidth / 2
                     buffer := shadowbox(mrow + curr_get, ;
                                         yy - zz - 1, ;
                                         mrow + curr_get + 4, ;
                                         yy + zz + 1, ;
                                         2, aHeads[curr_get], .f., , ;
                                         "Ctrl-W to save, or Esc to exit")
                     // shut off up and down arrows for the memoedit
                     setkey(K_UP, NIL)
                     setkey(K_DOWN, NIL)
                     scatter[curr_get] := chr(255) + ;
                         memoedit(substr(scatter[curr_get], 2), ;
                                  mrow + curr_get + 1, yy - zz, ;
                                  mrow + curr_get + 3, yy + zz, .t.)
                     // okay, now define them again
                     setkey(K_UP, { || ;
                            curr_get := gfbackaget(firstfield, curr_get, .t.), ;
                            goingdown := .f. } )
                     setkey(K_DOWN, { || goingdown := gfskipaget() } )
                     if ! goingdown
                        curr_get := gfbackaget(firstfield, @curr_get)
                     endif
                     byebyebox(buffer)
                  endif
               endif
            next
            setcursor(SC_NONE)
            setkey(K_INS, lOldInsert)

            // undefine left and right arrows for the memoedit
            setkey(K_UP, NIL)
            setkey(K_DOWN, NIL)
         otherwise
            key := upper(chr(ginkey(0)))
            if key $ "NP"
               // reset field pointer so we continue in this loop
               curr_fld := 0
               if key == "N"            // jump to NEXT record
                  if eval(browse:skipBlock, 1) == 0
                     err_msg( { "You are at the end of the file" } )
                  endif
               else                     // jump to PREVIOUS record
                  if eval(browse:skipBlock, -1) == 0
                     err_msg( { "You are at the beginning of the file" } )
                  endif
               endif
            endif
      endcase
      if curr_fld > 0
         mrow -= (maxrow() - 2)
      elseif curr_fld == 0 .and. mrow < 0
         mrow := 1
      endif

      // if we still have another pageful of gets to get,
      // then by all means clear out the previous bunch
      if mainloop .and. curr_fld <> num_flds
         firstfield := curr_fld + 1
         dispbegin()
         // only clear screen if we don't have a presaved screen file!
         // otherwise, all additional text/boxes will be obliterated
         if ! file(mscrnfile)
            scroll(1, 0, maxrow() - 1, maxcol(), 0)
         endif
      endif
   endif
   curr_fld++
enddo
// note that if last field on the edit screen is a memo, user
// might have pressed ESC to exit... we nonetheless should let
// them proceed into the logic below to save their edits
if mode $ "AEQ" .and. (lastkey() <> K_ESC .or. is_memo)
   if mode == 'Q'                                 // query-by-example
      // loop through all fields and build the query string
      browse:queryCondition := []
      for xx = 1 to curr_fld - 1
         // skip "non-fields"
         if scatter[xx] <> NIL
            mfield := if(file(mscrnfile), GetName(xx), fields_[xx][DBS_NAME])
            mtype := type(mfield)
            // first get rid of the dratted chr(255) littering the front of a memo
            if ( mtype := type(mfield) ) == "M"
               scatter[xx] := substr(scatter[xx], 2)
            endif
            if ! empty(scatter[xx])
               do case
                  case mtype == "N"
                     picstring := mfield + " == " + ltrim(str(scatter[xx]))
                  case mtype == "L"
                     picstring := if(! scatter[xx], "! ", "") + mfield
                  case mtype == "D"
                     picstring := mfield + " == ctod('" + dtoc(scatter[xx]) + "')"
                  otherwise
                     /*
                        if user specified "..whatever.." in character field,
                        then they want to find the first occurrence containing
                        wwhatever; consequently, we must first check for this
                        situation by looking for two periods
                     */
                     if substr(scatter[xx], 1, 2) == '..'
                        picstring := 'upper([' + ;
                                     trim(strtran(scatter[xx], '..', '')) + ;
                                     ']) $ upper(' + mfield + ')'
                     else
                        // note the case-insensitive search - feel free to
                        // change this if you need to by stripping out the
                        // references to "UPPER("
                        picstring := "upper(" + mfield + ") = [" + ;
                                      upper(trim(scatter[xx])) + "]"
                     endif
               endcase
               browse:queryCondition += ;
                   if(! empty(browse:queryCondition), " .and. ", "") + picstring
            endif
         endif
      next
      marker := recno()
      if ! empty(browse:queryCondition)

         // compile this string to a code block, which will speed up the
         // process a tad over the traditional macro expansion rigmarole
         browse:queryCondition := MakeBlock(browse:queryCondition)

         // browse:hitBottom won't work in this situation
         // which is why I am using the temporary variable YY
         do while ( yy := eval(browse:skipBlock, 1) <> 0 ) ;
             .and. ! eval(browse:queryCondition) .and. checkescape()
         enddo
         if ! yy
            err_msg( { "No matches found" } )
            browse:queryCondition := []  // reset for next time
            go marker
         endif
      endif
   else     && add/edit mode: gather memory variables back into fields
      if yes_no(if(mode == "A", "Add this new record", "Save your edits"))
         if if(mode == "A", Add_Rec(), Rec_Lock())
            ret_val := .t.
            for xx = 1 to curr_fld - 1
               // skip "non-fields"
               if scatter[xx] <> NIL
                  mfield := if(file(mscrnfile), GetName(xx), fields_[xx][DBS_NAME])
                  // note: memos will still have a chr(255) stuck on the front
                  // and we do not want to save that, so let's act accordingly
                  if type(mfield) == "M"
                     fieldput(fieldpos(mfield), substr(scatter[xx], 2))
                  else
                     fieldput(fieldpos(mfield), scatter[xx])
                  endif
               endif
            next
         endif
      endif
   endif
endif
unlock
restscreen(0, 0, maxrow(), maxcol(), oldscrn)
return ret_val

* end static function GFBRecView()
*--------------------------------------------------------------------*


/*
   GfBackAGet(): trap up arrow in the READ to
                 move back up by one GET
*/
static function gfbackaget(firstfield, curr_get, lhotkey)
/*
   we might have gotten here in one of two ways:
   (a) up arrow, in which case there would have
       been three parameters passed
   (b) direct call when escaping from a memofield
       and moving upwards through the get list
   we only need to stuff the keyboard in the first situation
*/
if lhotkey <> NIL
   keyboard chr(K_ENTER)
endif
return max(curr_get - 2, firstfield - 1)

* end static function GFBackAGet()
*--------------------------------------------------------------------*


/*
   GfSkipAGet(): trap down arrow in the READ to
                 move down one GET
*/
static function gfskipaget
keyboard chr(K_ENTER)
return .t.

* end static function GFSkipAGet()
*--------------------------------------------------------------------*


#ifdef REMBRANDT

/*
   GfbMaint(): Data Entry Screen Painter/Code Generator
*/
static function gfbmaint(fields_, browse, aPics, piclens_, aHeads)
local key := 0, mrow := 0, mcol := 0, mfile, buffer, xx, yy, zz, mstart
local nleft, nbottom, nright, oldf1, handle, mfield, mstring, mtype, ngets
local lOldInsert := readinsert(.f.), mcolor, oldprintf, oldconsole
local firstfield, bytes := (maxrow() + 1) * (maxcol() + 1) * 2
local getlist := {}, ccolor := browse:colorSpec
local num_boxes := len(browse:boxArray), fldnames_, memos_, num_memos

// disable f1 key so that they will only get my help screen!
oldf1 := setkey(K_F1, nil)
setcursor(CURSOR_ON)
do while key <> K_F10
   // show current screen position at top right corner
   @ 0, maxcol() - 6 ssay padr(ltrim(str(mrow, 2)) + ', ' + ;
                         ltrim(str(mcol, 3)), 7)
   setpos(mrow, mcol)
   key := ginkey(0)
   do case

      // help!
      case key == K_ALT_H .or. key == K_F1
         gfbhelpme()

      // insert field
      case key == K_ALT_F
         // load field names array first time only
         if fldnames_ == NIL
            yy := len(fields_)
            fldnames_ := {}
            for xx := 1 to yy
               // only use "actual" fields (no code blocks)
               if valtype(fields_[xx]) == "A"
                  aadd(fldnames_, fields_[xx][DBS_NAME])
               endif
            next
         endif
         apick xx array fldnames_
         if xx > 0
            yy := makepicture(fields_[xx], chr(254))
            zz := savescreen(mrow, mcol, mrow, mcol + yy[2] - 1)
            if empty(transform(zz, replicate('X ', len(zz) / 2) ))
               aadd(aPics, yy[1])
               aadd(piclens_, yy[2])
               aadd(browse:getArray, { mrow, mcol, yy[2], fldnames_[xx], yy[1]} )
               @ mrow, mcol say padr(fldnames_[xx], yy[2]) color 'I'
            else
               err_msg( { "GETs cannot be stacked" } )
            endif
         endif

      // alphanumeric or something printable
      case key > 31 .and. key < 255
         if isitaget(mrow, mcol)
            err_msg( { "You cannot type over a GET" } )
         else
            // if insert mode is on, push everything else on this row over
            if readinsert()
               /*
                  we don't want to move any gets, but how to do this?
                  simple... we jump merrily through every character on
                  this row and check for its color.  if its color is
                  inverse (112), then we know it must be a get and we
                  skip it.
               */
               mstart := 0
               for xx = 0 to maxcol() - mcol
                  if isitaget(mrow, mcol + xx)
                     restscreen(mrow, mcol + mstart + 1, mrow, mcol + xx - 1, ;
                          savescreen(mrow, mcol + mstart, mrow, mcol + xx - 2))
                     // now loop through until we find the next non-get
                     do while xx++ < maxcol() + 1 - mcol .and. ;
                                     isitaget(mrow, mcol+xx)
                     enddo
                     mstart := xx
                  endif
               next

               // if there were no gets, mstart will still be zero.
               if mstart == 0
                  restscreen(mrow, mcol + 1, mrow, maxcol(), ;
                             savescreen(mrow, mcol, mrow, maxcol() - 1))
               endif
            endif
            @ mrow, mcol ssay chr(key)
            if mcol == maxcol()
               /*
                   Because Clipper 5 doesn't assume a fixed screen of
                   25 rows x 80 columns, it allows you to keep typing
                   off the right side of the screen.  Therefore, we
                   must restrict this sort of nonsense ourselves by
                   forcing a wrap from the bottom row to the top.
               */
               if mrow == maxrow()
                  mrow := 0
               else
                  mrow++
               endif
               mcol := 0
            else
               mcol++
            endif
         endif

      // move a field or resize a box
      case key == K_ENTER
         if isitaget(mrow, mcol)
            if gfbmovefld(browse)
               mrow := row()
               mcol := col()
               // let's re-sort the gets array so that they will fall in the
               // order that they appear on the screen (rather than how they
               // appear in the .dbf)
               asort(browse:getArray,,, { | x, y | if(x[1] == y[1], ;
                                    x[2] < y[2], x[1] < y[1] ) } )
            endif
         else
            // are we on a box outline?? only the shadow knows
            if num_boxes > 0
               if ( xx := isitabox(browse) ) > 0
                  browse:boxArray[xx] := gfbsizebox(browse, xx)
               endif
            endif
         endif

      case key == K_BS .and. mcol > 0
         mcol--
         // wipe out this space if we are not on a get
         if ! isitaget(mrow, mcol)
            @ mrow, mcol ssay [ ]
         endif
         setpos(mrow, mcol)

      case key == K_DEL
         // first check if we are on a box outline
         if ( xx := isitabox(browse) ) > 0
            if yes_no("Are you sure you want to delete this box")
               // wipe it off the screen
               @ BoxTop(xx), BoxLeft(xx), BoxBottom(xx), BoxRight(xx) ;
                 box space(8)
               // then blast it from the array
               adel(browse:boxArray, xx)
               // finally, resize box array and decrement box counter
               asize(browse:boxArray, --num_boxes)
            endif
         // next check if we are on a get field
         elseif isitaget(mrow, mcol)
            if yes_no("Are you sure you want to delete this field")
               // first find this element
               xx := gfbwhichget(mrow, mcol, browse)
               if xx > 0
                  // wipe it off the screen
                  scroll(GetRow(xx), GetCol(xx), GetRow(xx), GetCol(xx) + ;
                         GetLength(xx) - 1, 0)
                  // blast it out of the array
                  akill(browse:getArray, xx)
               endif
            endif
         else
            /*
               note that we don't want to move any gets, so we must check
               each character on this row to see if it is part of a get.
               however, so that we do not have to go through this loop
               on lines without gets, we do a quick scan of the array
               to see if there is a get on this row
            */
            xx := ascan(browse:getArray, { | a | a[1] == mrow} )
            if xx > 0      // yes, there is at least one get on this row
               mstart := 0
               for xx = 0 to maxcol() - mcol
                  if isitaget(mrow, mcol + xx)
                     restscreen(mrow, mcol + mstart, mrow, mcol + xx - 2, ;
                       savescreen(mrow, mcol + mstart + 1, mrow, mcol + xx - 1))
                     @ mrow, mcol + xx - 1 ssay [ ]
                     // now loop through until we find the next non-get
                     do while xx++ < maxcol() + 1 - mcol .and. ;
                                     isitaget(mrow, mcol+xx)
                     enddo
                     mstart := xx
                  endif
               next
            else       // no gets on this row
               restscreen(mrow, mcol, mrow, maxcol() - 1, ;
                          savescreen(mrow, mcol + 1, mrow, maxcol()))
               @ mrow, maxcol() ssay [ ]
            endif
         endif

      // toggle insert mode
      case key == K_INS
         setcursor( if( readinsert(! readinsert()), 1, 3) )

      // color palette to change active color
      case key == K_ALT_P
         setcolor(colorpal())

      // draw box
      case key == K_ALT_B
         if gfbdrawbox(mrow, mcol, browse)
            num_boxes++
         endif


#ifdef CODEGEN

      // generate code
      case key == K_SH_F10
         mfile := upper(alias())
         boxget mfile prompt 'Enter filename for generated code:' picture '@K@!'
         mfile += ".PRG"
         if if(! file(mfile), .t., yes_no(mfile + " already exists", "Overwrite it"))
            buffer := savescreen(0, 0, maxrow(), maxcol())
            dispbegin()
            gfbcleanup(browse)
            oldprintf := set(_SET_PRINTFILE, mfile)
            set print on

            // first, get rid of position indicator so it doesn't get
            // stuck in the generated code (gasp!)
            scroll(0, maxcol() - 6, 0, maxcol(), 0)

            // next, let's get rid of the boxes so that they don't get
            // displayed as part of the text
            for xx = 1 to num_boxes
               @ BoxTop(xx), BoxLeft(xx), BoxBottom(xx), BoxRight(xx) box space(8)
            next
            oldconsole := set(_SET_CONSOLE, .F.)
            // next, get rid of
            // standard header -- change it as you see fit
            QOut("/*")
            QOut(INDENT(1) + "Program: " + mfile)
            QOut(INDENT(1) + "Date:    " + dtoc(date()))
            QOut(INDENT(1) + "Time:    " + time())
            QOut(INDENT(1) + "Dialect: Clipper 5.01")
            QOut(INDENT(1) + "Generated by GrumpBrow()")
            QOut(INDENT(1) + "Copyright (c) 1990 Greg Lief")
            QOut(INDENT(1) + "Notes: data entry screen for " + alias() + ".dbf")
            QOut(INDENT(1) + "expects to be passed a parameter (mode),")
            QOut(INDENT(1) + "which could be either [A]dd, [E]dit, [V]iew")
            QOut("*/")
            QOut()
            QOut('#include "inkey.ch"')
            QOut('#include "setcurs.ch"')
            QOut()
            // must maintain a counter separate from XX, because
            // if we skip any memos the counter value will be off
            // in the generated code, and chaos will ensue...
            yy := 0
            ngets := len(browse:getArray)
            for xx = 1 to ngets
               if type(GetName(xx)) <> "M"
                  QOut("#define m" + padr(GetName(xx), 12) + " scatter_[" + ;
                       ltrim(str(++yy)) + "]")
               endif
            next
            QOut()
            QOut("function addedit(mode)")
            QOut("local getlist := {}")
            QOut("local scatter_ := {}")
            QOut("local lContinue")
            QOut("local cAlias := '" + alias() + "'")
            QOut("local nOldArea")
            QOut("local marker")
            QOut("local lHad2Open := .f.")
            QOut("local lOldScore := set(_SET_SCOREBOARD, .F.)")
            QOut("local lOldInsert := setkey( K_INS, {|| setcursor( ;")
            QOut("      if(readinsert(! readInsert()), SC_NORMAL, SC_SPECIAL1))} )")
            QOut("local fieldnames_ := { ")
            firstfield := .t.
            memos_ := {}
            for xx := 1 to ngets
               if type(GetName(xx)) <> "M"
                  if firstfield
                     firstfield := .f.
                  else
                     QQOut(", ;")
                     QOut(space(23))
                  endif
                  QQOut("'" + trim(GetName(xx)) + "'")
               else
                  // save both actual name of memo field and
                  // column heading (for title of memoedit() box)
                  aadd(memos_, { getname(xx), aHeads[xx] } )
               endif
            next
            QQOut("}")
            num_memos := len(memos_)
            if num_memos > 0
               QOut()
               QOut("// initialize memo locals and hot keys for memo editing")
               for xx := 1 to num_memos
                  zz := ltrim(str(xx + 1))
                  QOut("local m" + memos_[xx][1])
                  QOut("local oldf" + zz + " := setkey(K_F" + upper(zz) + ", " + ;
                       "{ | bBlah | bBlah := setkey(K_F" + upper(zz) + ;
                       ", NIL), ;")
                  QOut(space(22) + "memedit(@m" + memos_[xx][1] + ",,,,,'" + ;
                       memos_[xx][2] + "'), setkey(K_F" + upper(zz) + ;
                       ", bBlah) } )" )
               next
            endif
            QOut()
            QOut("if select(cAlias) == 0")
            QOut(INDENT(1) + "nOldArea := select()")
            QOut(INDENT(1) + "use (cAlias) new")
            QOut(INDENT(1) + "lHad2Open := .t.")
            QOut("endif")
            QOut("gfsaveenv(.t.)")
            QOut("scroll()")
            QOut()
            QOut("// static text")
            // draw the boxes, dude (in Saudi Arabia, "dude" means "worm",
            // so that was not necessarily a term of endearment)
            for xx = 1 to num_boxes
               QOut("@ " + ltrim(str(BoxTop(xx)))  + ", " + ;
                           ltrim(str(BoxLeft(xx))) + ", " + ;
                           ltrim(str(BoxBottom(xx))) + ", " + ;
                           ltrim(str(BoxRight(xx))) + " box '" + BoxString(xx) + ;
                           if(BoxFill(xx) == "Y", " ", "") + ;
                           "' color '" + color_n2s(BoxColor(xx)) + "'")
            next
            // now draw all other text
            QOut("setcolor('" + ccolor + "')")
            for xx = 0 to maxrow()
               mstring := [ ]
               yy := 0
               do while yy <= maxcol() .and. asc(TextAt(xx, yy)) < 33
                  yy++
               enddo
               if yy <= maxcol()
                  do while yy <= maxcol()
                     mstring := TextAt(xx, yy)
                     mcolor := ColorAt(xx, yy)
                     zz := yy + 1
                     do while ColorAt(xx, zz) == mcolor .and. zz <= maxcol()
                        mstring += TextAt(xx, zz)
                        zz++
                     enddo
                     if ! empty(mstring)
                        if color_s2n(ccolor) <> bin2i(mcolor)
                           ccolor := color_n2s(mcolor)
                           QOut("setcolor('" + ccolor + "')")
                        endif
                        QOut("@ " + str(xx, 2) + ", " + str(yy, 2) + ;
                                  " say [" + trim(mstring) + "]")
                     endif
                     yy := zz
                  enddo
               endif
            next

            QOut()
            QOut("// use phantom record to grab initial values if adding")
            QOut("if mode == 'A'")
            QOut(INDENT(1) + "marker := recno()")
            QOut(INDENT(1) + "go 0")
            QOut("endif")
            QOut()
            QOut("// initialize field duplicates")
            QOut("aeval(fieldnames_, { | a | aadd(scatter_, " + ;
                 "(cAlias)->(fieldget((cAlias)->(fieldpos(a))))) } )")

            // memos must be handled as variables due to pass-by-reference
            if num_memos > 0
               QOut()
               QOut("// initialize memo duplicates")
               for xx := 1 to num_memos
                  QOut("m" + memos_[xx][1] + " := (cAlias)->" + memos_[xx][1])
               next
            endif

            /* do the gets */
            QOut()
            QOut("// go GET 'em")
            for xx = 1 to ngets
               if type(GetName(xx)) <> "M"
                  QOut("@ " + str(GetRow(xx), 2) + ", " + ;
                       str(GetCol(xx), 2) + " get m" + GetName(xx) + ;
                       " picture '" + GetPicture(xx) + "'")
               endif
            next

            /* basic stuff */
            QOut("if mode <> 'V'")
            QOut(INDENT(1)+"setcursor( if(readInsert(), SC_SPECIAL1, SC_NORMAL) )")
            QOut(INDENT(1) + "read")
            /* do the replaces */
            QOut(INDENT(1) + "// do the replaces if they didn't escape out")
            QOut(INDENT(1) + "if lastkey() <> K_ESC")
            QOut(INDENT(2) + "if mode == 'A'")
            QOut(INDENT(3) + "lContinue := (cAlias)->( add_rec() )")
            QOut(INDENT(2) + "else")
            QOut(INDENT(3) + "lContinue := (cAlias)->( rec_lock() )")
            QOut(INDENT(2) + "endif")
            QOut(INDENT(2) + "if lContinue")
            QOut(INDENT(3) + "aeval(fieldnames_, { |a,x| " + ;
                          "(cAlias)->(fieldput((cAlias)->(fieldpos(a)), scatter_[x])) } )")
            QOut(INDENT(3) + "(cAlias)->(dbunlock())")
            QOut(INDENT(2) + "endif")

            // memos must be handled as variables due to pass-by-reference
            if num_memos > 0
               QOut()
               QOut(INDENT(2) + "// replace all memos")
               for xx := 1 to num_memos
                  QOut(INDENT(2) + "(cAlias)->" + memos_[xx][1] + " := m" + ;
                                   memos_[xx][1])
               next
               QOut()
            endif
            QOut(INDENT(1) + "// if in add mode, reset record pointer")
            QOut(INDENT(1) + "elseif mode == 'A'")
            QOut(INDENT(2) + "(cAlias)->(dbgoto(marker))")
            QOut(INDENT(1) + "endif")
            QOut("else")
            QOut(INDENT(1) + "getlist := {}")
            QOut(INDENT(1) + "inkey(0)")
            QOut("endif")
            QOut("gfrestenv()")
            QOut("setkey(K_INS, lOldInsert)")
            QOut("set(_SET_SCOREBOARD, lOldScore)")
            if num_memos > 0
               QOut()
               QOut("// de-activate memo editing hot keys")
               for xx := 1 to num_memos
                  zz := ltrim(str(xx + 1))
                  QOut("setkey(K_F" + upper(zz) + ", oldf" + zz + ")")
               next
            endif
            QOut("if lHad2Open")
            QOut(INDENT(1)+"dbCloseArea()")
            QOut(INDENT(1)+"dbSelectArea(nOldArea)")
            QOut("endif")
            QOut("return NIL")
            QOut()
            QOut("* end of file " + mfile)

            set print off
            set(_SET_PRINTFILE, oldprintf)
            set(_SET_CONSOLE, oldconsole)
            restscreen(0, 0, maxrow(), maxcol(), buffer)
            dispend()
            waiton("Code successfully generated to " + mfile)
            inkey(1)
            waitoff()
         endif

#endif // CODEGEN

      case key == K_ESC
         if yes_no('Your changes will be lost','Are you sure you want to exit')
            exit
         endif

      otherwise
         gfarrowkey(key, maxcol(), @mrow, @mcol)

   endcase
enddo
setkey(K_F1, oldf1)     // re-enable f1 key to whatever it was before this
readinsert(lOldInsert)

if key <> K_ESC
   gfbcleanup(browse)
   mfile := alias() + ".gfs"
   if ( handle := fcreate(mfile) ) == -1
      err_msg( { "Could not create " + mfile } )
   else
      if fwrite(handle, ;
                savescreen(0, 0, maxrow(), maxcol()), bytes) <> bytes
         err_msg( { "Error writing to " + mfile } )
      endif
      // save number of fields
      ngets := len(browse:getArray)
      fwrite(handle, chr(ngets))
      for xx = 1 to ngets
         fwrite(handle, chr(GetRow(xx)) + chr(GetCol(xx)) + ;
                        chr(GetLength(xx)) + padr(GetName(xx), 10)  + ;
                        GetPicture(xx) + CRLF)
      next
      // save number of boxes
      fwrite(handle, chr(num_boxes))
      if num_boxes > 0
         for xx = 1 to num_boxes
            fwrite(handle, chr(BoxTop(xx)) + chr(BoxLeft(xx)) + ;
                    chr(BoxBottom(xx)) + chr(BoxRight(xx)) + ;
                    BoxString(xx) + chr(BoxColor(xx)) + BoxFill(xx) )
         next
      endif
      fclose(handle)
   endif
endif
setcursor(SC_NONE)
return NIL

* end static function GFBMaint()
*--------------------------------------------------------------------*


/*
   GfbCleanup(): clear GET fields in preparation for
                 saving .GFS file or generating code
*/
static function gfbcleanup(browse)
local xx, ngets := len(browse:getArray)
// we must go through the gets array and wipe all inverse
// gets off the screen -- if we leave them here, they will be
// restored when adding/editing only to be wiped out again,
// which creates an annoying little visual blip.
setcolor(browse:colorSpec)
for xx = 1 to ngets
   // if we have more than one screen of gets, we must add error-
   // trapping here so that the thing don't crash!!
   if valtype(browse:getArray[xx]) == "U"
      exit
   else
      scroll(GetRow(xx), GetCol(xx), GetRow(xx), GetCol(xx) + ;
             GetLength(xx) - 1, 0)
   endif
next
// get rid of the coordinates at the top right corner
scroll(0, maxcol() - 6, 0, maxcol(), 0)
return nil

* end static function GFBCleanUp()
*--------------------------------------------------------------------*


/*
   GfbHelpMe(): help screen for screen painting module
*/
static function gfbhelpme
gfsaveenv(.t., 0, '+w/b')          // shut off cursor & change color
@ 0, 0, maxrow(), maxcol() box BOXFRAMES[5] color "+gr/b"
@ 1, 19 ssay "GRUMPBROWSE() SCREEN PAINTER - ACTIVE KEYS" color "+gr/b"
@ 2, 16 ssay "insert field at current location"
@ 3, 16 ssay "draw box"
@ 4, 16 ssay "change current color"
@ 5, 16 ssay "toggle insert mode on/off"
@ 6, 16 ssay "destructive backspace"
@ 7, 16 ssay "delete from cursor position"
@ 8, 16 ssay "exit without saving changes"
@ 9, 16 ssay "save screen file (.gfs) and exit"
@10, 16 ssay "generate code for this screen"
@11, 16 ssay "move left one column"
@11, 54 ssay "move right one column"
@12, 16 ssay "move to top left"
@12, 54 ssay "move to bottom right"
@13, 16 ssay "move right five columns"
@13, 54 ssay "move left five columns"
@14, 16 ssay "move to top row"
@14, 54 ssay "move to bottom row"
@15, 16 ssay "move to left column"
@15, 54 ssay "move to right column"
@17, 4 ssay "Boxes may be filled or unfilled.  If you create a filled box, any text"
@18, 4 ssay "underneath it will be pulled in, and will be displayed in the color of"
@19, 4 ssay "the box.  To delete a box, place the cursor on it outline and press"
@20, 4 ssay "delete. to resize a box, place the cursor on it and press Enter.  To"
@21, 4 ssay "delete a GET, place the cursor on it and press Delete.  To move a get,"
@22, 4 ssay "place the cursor on it and press Enter."
@23, 20 ssay "press any key to return to screen painter"
setcolor("i")
@ 2,  4 ssay "Alt-F"
@ 3,  4 ssay "Alt-B"
@ 4,  4 ssay "Alt-P"
@ 5,  4 ssay "Insert"
@ 6,  4 ssay "Backspace"
@ 7,  4 ssay "Delete"
@ 8,  4 ssay "Esc"
@ 9,  4 ssay "F10"
@10,  4 ssay "Shift-F10"
@11,  4 ssay "LtArrow"
@11, 42 ssay "RtArrow"
@12,  4 ssay "Home"
@12, 42 ssay "End"
@13,  4 ssay "Tab"
@13, 42 ssay "Shift-Tab"
@14,  4 ssay "PgUp"
@14, 42 ssay "PgDn"
@15,  4 ssay "Ctrl-Left"
@15, 42 ssay "Ctrl-Right"
ginkey(0)
gfrestenv()
return nil

* end static function GFBHelpMe()
*--------------------------------------------------------------------*


/*
   GfbMoveFld(): logic to drag a field around on the screen
*/
static function gfbmovefld(browse)
local oldcolor := setcolor()
local xx := gfbwhichget(row(), col(), browse)
local mrow
local mcol
local mlength
local key
local yy
local buffer
local buffer2
local ret_val := .f.
if xx > 0
   mrow  := GetRow(xx)
   mcol  := GetCol(xx)
   mlength := GetLength(xx)
   key := 0
   // initialize a scrap buffer which will hold the underlying
   // screen area as we drag the field around.  if we did not
   // do this, then the field would act as a giant erase,
   // annihilating static text (and other fields) in its wake!
   buffer := []
   for yy = 2 to mcol + mlength
      buffer += chr(32) + chr(23)
   next
   buffer2 := savescreen(mrow, mcol, mrow, mcol + mlength - 1)
   do while key <> K_ENTER .and. key <> K_ESC
      @ 0, maxcol() - 6 ssay padr(ltrim(str(mrow, 2)) + ', ' + ;
                         ltrim(str(mcol, 3)), 7)
      setpos(mrow, mcol)
      key := ginkey(0)
      if key <> K_ENTER .and. key <> K_ESC
         setcolor(oldcolor)
         dispbegin()
         restscreen(mrow, mcol, mrow, mcol + mlength - 1, buffer)
         gfarrowkey(key, maxcol() + 1 - mlength, @mrow, @mcol)
         buffer := savescreen(mrow, mcol, mrow, mcol + mlength - 1)
         restscreen(mrow, mcol, mrow, mcol + mlength - 1, buffer2)
         dispend()
         buffer2 := savescreen(mrow, mcol, mrow, mcol + mlength - 1)
      endif
   enddo
   dispbegin()
   restscreen(mrow, mcol, mrow, mcol + mlength - 1, buffer)
   if key == K_ENTER
      ret_val := .t.
      // make sure that we are not plopping this get down over
      // another one!
      buffer := savescreen(mrow, mcol, mrow, mcol + mlength - 1)
      if ! empty(transform(buffer, replicate('X ', len(buffer) / 2) ))
         dispend()
         err_msg( { "GETs cannot be stacked" } )
         ret_val := .f.
      endif
   endif
   if ret_val
      // change this array element to reflect the new position
      GetRow(xx) := mrow
      GetCol(xx) := mcol
      restscreen(mrow, mcol, mrow, mcol + mlength - 1, buffer2)
      dispend()
   else
      // either they escaped, or they tried to place this get
      // on top of another one -- in either case we must now
      // redraw the original get
      @ GetRow(xx), GetCol(xx) ssay padr(GetName(xx), GetLength(xx)) color 'I'
   endif
endif
setcolor(oldcolor)
return ret_val

* end static function GFBMoveFld()
*--------------------------------------------------------------------*



/*
   IsItABox(): determine whether or not we are on a box outline
               useful for resizing and deleting boxes
*/
static function isitabox(browse)
local ret_val := 0, xx, temprow := row(), tempcol := col()
local num_boxes := len(browse:boxArray)
for xx = 1 to num_boxes
   if ((BoxTop(xx) == temprow .or. BoxBottom(xx) == temprow) .and.   ;
        BoxLeft(xx) <= tempcol .and. BoxRight(xx) >= tempcol) .or.   ;
        ((BoxLeft(xx) == tempcol .or. BoxRight(xx) == tempcol) .and. ;
           BoxTop(xx) <= temprow .and. BoxBottom(xx) >= temprow)
      ret_val := xx
      exit
   endif
next
return ret_val

* end static function IsItABox()
*--------------------------------------------------------------------*


/*
   GFBWhichGet(): locate current get in get array
*/
static function gfbwhichget(mrow, mcol, browse)
local mget := 0, xx
// figure out which field it is by scanning the array
// for the value of the row we are currently on
xx := ascan(browse:getArray, { | a | a[1] == mrow} )
do while xx > 0 .and. mget == 0
   // now check the length of this field (character 3 in
   // the array element) to see if we are actually on it
   if xx > 0
      if mcol < GetCol(xx) + GetLength(xx)
         mget := xx
      else
         xx := ascan(browse:getArray, { | a | a[1] == mrow }, xx + 1)
      endif
   endif
enddo
return mget

* end static function GFBWhichGet()
*--------------------------------------------------------------------*


/*
   GfbDrawBox(): logic to drag a field around on the screen
*/
static function gfbdrawbox(mrow, mcol, browse)
local buffer, mbox, key, ntop, nleft, oldscrn, mfill, mwidth, fillbuff, xx
local scrnbuff := savescreen(0, maxcol() - 16, 0, maxcol())
local ret_val := .f.
static boxtypes
// only load boxtypes array if it is empty (i.e., first time through)
if boxtypes == NIL
   boxtypes := array(5)
   for xx = 1 to 5
      boxtypes[xx] := substr(BOXFRAMES[xx], 1, 8)
   next
endif

/*
   even though it seems wasteful, we must make a safety copy of
   the entire screen because you might decide not to keep the
   box, right?  remember the old maxim: users change their mind
   as often as Cher changes costumes
*/
oldscrn := savescreen(0, 0, maxrow(), maxcol())
setpos(mrow, mcol)
buffer := shadowbox(9, 35, 15, 44, 2, "Boxes")
mbox := achoice(10, 36, 14, 43, boxtypes)
byebyebox(buffer)
if mbox > 0
   mfill := yes_no("Would you like this box to be filled")
   key := 0
   ntop := mrow
   nleft := mcol
   buffer := savescreen(ntop, nleft, mrow, mcol)
   do while key <> K_ENTER .and. key <> K_ESC
      // display current box coordinates at top right corner
      // but only if the box is not in the top right corner!!
      if ntop > 0 .or. mcol < maxcol() - 16
         @ 0, maxcol() - 16 ssay padr(ltrim(str(ntop, 2)) + ', '  + ;
                                      ltrim(str(nleft, 3)) + ', ' + ;
                                      ltrim(str(mrow, 2)) + ', '  + ;
                                      ltrim(str(mcol, 3)), 16)
      endif
      setpos(mrow, mcol)
      key := ginkey(0)
      if ! ((mrow == maxrow() .and. key == K_DOWN) .or. ;
            (mrow == ntop .and. key == K_UP) .or. ;
            (mcol == nleft .and. key == K_LEFT) .or. ;
            (mcol > maxcol() - 1 .and. key == K_RIGHT))
         dispbegin()
         restscreen(ntop, nleft, mrow, mcol, buffer)
         gfarrowkey(key, maxcol(), @mrow, @mcol)
         buffer := savescreen(ntop, nleft, mrow, mcol)
         @ ntop, nleft, mrow, mcol box boxtypes[mbox] + if(mfill, " ", "")
         dispend()
      endif
   enddo
   if key == K_ESC
      restscreen(0, 0, maxrow(), maxcol(), oldscrn)
      // restore original cursor position too
      setpos(ntop, nleft)
   else
      // restore screen contents under the coordinates area (top right)
      restscreen(0, maxcol() - 16, 0, maxcol(), scrnbuff)
      // only fill the box if there is at least one row inside it!
      if mfill .and. mrow > ntop + 1
         gfbfillbox(ntop, nleft, mrow, mcol, buffer)
      endif
      ret_val := .t.
      aadd(browse:boxArray, { ntop, nleft, mrow, mcol, boxtypes[mbox], ;
                     color_s2n(), if(mfill, "Y", "N") } )
   endif
endif
return ret_val

* end static function GFBDrawBox()
*--------------------------------------------------------------------*


/*
   GfbSizeBox(): logic to resize a box
*/
static function gfbsizebox(browse, xx)
local oldscrn
local key := 0
local buffer
local oldcolor
local ret_val
local ntop    := BoxTop(xx)
local nleft   := BoxLeft(xx)
local nbottom := BoxBottom(xx)
local nright  := BoxRight(xx)
// once again, we must make a safety copy of the entire screen
// because you might decide to leave the box size as is.  picky users!
oldscrn := savescreen(0, 0, maxrow(), maxcol())
// get rid of the box for a split second and save the underlying screen
// otherwise, we are going to have a real mess on our hands!
@ ntop, nleft, nbottom, nright box space(8)
buffer := savescreen(ntop, nleft, nbottom, nright)
oldcolor := setcolor(color_n2s(BoxColor(xx)))
@ ntop, nleft, nbottom, nright box BoxString(xx) + ;
                       if(BoxFill(xx) == "Y", " ", "")
do while key <> K_ENTER .and. key <> K_ESC
   setpos(nbottom, nright)
   key := ginkey(0)
   if ! ((nbottom == maxrow() .and. key == K_DOWN) .or. ;
         (nbottom == ntop .and. key == K_UP) .or. ;
         (nright == nleft .and. key == K_LEFT) .or. ;
         (nright > maxcol() - 1 .and. key == K_RIGHT))
      dispbegin()
      restscreen(ntop, nleft, nbottom, nright, buffer)
      gfarrowkey(key, maxcol(), @nbottom, @nright)
      buffer := savescreen(ntop, nleft, nbottom, nright)
      @ ntop, nleft, nbottom, nright box BoxString(xx) + ;
                     if(BoxFill(xx) == "Y", " ", "")
      dispend()
   endif
enddo
if key == K_ESC
   restscreen(0, 0, maxrow(), maxcol(), oldscrn)
   ret_val := browse:boxArray[xx]
else
   if BoxFill(xx) == "Y"
      gfbfillbox(ntop, nleft, nbottom, nright, buffer)
   endif
   ret_val := { ntop, nleft, nbottom, nright, BoxString(xx), BoxColor(xx), ;
                BoxFill(xx) }
endif
setcolor(oldcolor)
return ret_val

* end static function GFBSizeBox()
*--------------------------------------------------------------------*


/*
   GfbFillBox(): fill box with underlying screen
*/
static function gfbfillbox(ntop, nleft, nbottom, nright, buffer)
/*
   we have just drawn a filled box.  there is every likelihood
   that we have some relevant information underneath this box.
   therefore, we must pull the information from the underlying
   screen into the box.  but not so fast!  we must make this
   information conform to the color specification of the box.
   one more thing, leave the gets in inverse.  got all that?  good.
*/
local mwidth := (nright - nleft + 1) * 2, fillbuff := [], xx, yy, ;
      mcolor := chr(color_s2n())
buffer := substr(buffer, mwidth + 1, len(buffer) - mwidth * 2)
for xx = 0 to (nbottom - ntop - 2)
   for yy = 2 to nright - nleft
      fillbuff += substr(buffer, xx * mwidth + yy * 2 - 1, 1)
      if substr(buffer, xx * mwidth + yy * 2, 1) <> chr(112)
         fillbuff += mcolor
      else
         fillbuff += chr(112)
      endif
   next
next
restscreen(ntop + 1, nleft + 1, nbottom - 1, nright - 1, fillbuff)
return NIL

* end static function GFBFillBox()
*--------------------------------------------------------------------*


/*
   GfbArrowKey(): process arrow keypresses etcetera
*/
static function gfarrowkey(key, maxlength, mrow, mcol)
do case
   case key == K_UP
      if mrow == 0
         mrow := maxrow()
      else
         mrow--
      endif

   case key == K_DOWN
      if mrow == maxrow()
         mrow := 0
      else
         mrow++
      endif

   case key == K_LEFT .and. mcol > 0
      mcol--

   case key == K_RIGHT .and. mcol < maxlength
      mcol++

   case key == K_HOME
      mrow := mcol := 0

   case key == K_END
      mrow := maxrow()
      mcol := maxlength

   case key == K_PGUP
      mrow := 0

   case key == K_PGDN
      mrow := maxrow()

   case key == K_CTRL_LEFT
      mcol := 0

   case key == K_CTRL_RIGHT
      mcol := maxlength

   case key == K_TAB
      mcol := min(mcol + 5, maxlength)

   case key == K_SH_TAB
      mcol := max(mcol - 5, 0)

endcase
return NIL

* end static function GFBArrowKey()
*--------------------------------------------------------------------*

#endif // REMBRANDT


/*
   JumpDown(): jump to beginning of next subgroup (active index only!)
*/
static function jumpdown(browse, nOrder)
local val := eval(browse:indexBlock)
local marker := recno()
local searchval
if valtype(val) == "C"
   searchval := substr(val, 1, len(val) - 1) + chr( asc(right(val, 1)) + 1 )
elseif valtype(val) $ "DN"
   searchval := val + 1
endif

// only perform the jump if we are not passing the end of the data subset
if empty(browse:endValues[nOrder]) .or. browse:endValues[nOrder] >= searchval
   dbseek(searchval, .t.)
   if eof()
      go marker
   else
      browse:refreshAll()
   endif
endif
return nil

* end static function JumpDown()
*--------------------------------------------------------------------*


/*
   JumpUp(): jump to beginning of previous subgroup (active index only!)
*/
static function jumpup(browse, nOrder)
local val
// only perform the jump if we are not passing the start of the data subset
if empty(browse:startValues[nOrder]) .or. ;
               eval(browse:indexBlock) > browse:startValues[nOrder]
   dbseek(eval(browse:indexBlock), .t.)
   dbskip(-1)
   val := eval(browse:indexBlock)
   browse:goTop()
   dbseek(val, .t.)
   browse:refreshAll()
endif
return nil

* end static function JumpUp()
*--------------------------------------------------------------------*


// the next five functions are for viewing subsets with the
// three TBrowse navigation blocks...

static function pseudofilt(browse, lNeedInput, ;
                           bOldgotop, bOldgobott, bOldskip, nOrder)
local startval
local endval
local bOldHandler
local bNewHandler
local newblock
local getlist := {}
local oldscrn
local initval
local cpicture
local lReset := .t.
if lNeedInput
   initval := eval(browse:indexBlock)
   oldscrn := shadowbox(7, 9, 10, 70, 1, "View Subset")
   startval := if(empty(browse:startValues[nOrder]),initval,  ;
               if(valtype(browse:startValues[nOrder]) <> "C", ;
                          browse:startValues[nOrder],         ;
                          padr(browse:startValues[nOrder], 40)))
   endval   := if(empty(browse:endValues[nOrder]), initval,   ;
               if(valtype(browse:endValues[nOrder]) <> "C",   ;
                          browse:endValues[nOrder],           ;
                          padr(browse:endValues[nOrder], 40)))

   // check for long character fields... must limit to 40 spaces
   cpicture := "@K" + ;
               if(valtype(initval) == "C" .and. len(initval) > 40, "@S40", '')
   @ 8, 11 say "Enter low value:  " get startval picture cpicture ;
                                    valid changeendval(getlist)
   @ 9, 11 say "Enter high value: " get endval picture cpicture
   setcursor(CURSOR_ON)
   read
   setcursor(SC_NONE)
   byebyebox(oldscrn)

   if lastkey() <> K_ESC
      if valtype(initval) == "C"    // trim if character type
         browse:startValues[nOrder] := trim(startval)
         browse:endValues[nOrder]   := trim(endval)
      else
         browse:startValues[nOrder] := startval
         browse:endValues[nOrder]   := endval
      endif
   endif
endif

if lastkey() <> K_ESC .or. ! lNeedInput
   if ! empty(browse:startValues[nOrder]) .or. ! empty(browse:endValues[nOrder])
      bNewHandler = { | e | blockhead(e, bOldHandler) }
      bOldHandler = errorblock(bNewHandler)
      begin sequence
         if ! empty(browse:startValues[nOrder])
            browse:goTopBlock := &(" { | | dbseek(" + ;
                         convertit(browse:startValues[nOrder]) + ", .t.) }")
         else
            browse:goTopBlock := bOldgotop
         endif
         if ! empty(browse:endValues[nOrder])
            browse:goBottomBlock := &(" { | | dbseek(" +             ;
                                    convertit(browse:endValues[nOrder], .t.) + ;
                                    ", .t.) , dbskip(-1) }")
         else
            browse:goBottomBlock := bOldgobott
         endif
         browse:skipBlock := { | SkipCnt | Gilligan(SkipCnt, browse, nOrder) }
         browse:goTop()           // force "filter" to be set
         // verify that data is within range
         initval := eval(browse:indexBlock)
         lReset := ( initval < browse:startValues[nOrder] .or. ;
                     initval > browse:endValues[nOrder] )
         if lReset
            waiton("No data in specified range... viewing all data")
            inkey(2)
            waitoff()
         endif
      end
      errorblock(bOldHandler)            // reset previous error handler
   endif
   if lReset
      // reset original movement blocks
      browse:goTopBlock    := bOldgotop
      browse:goBottomBlock := bOldgobott
      browse:skipBlock     := bOldskip
      browse:goTop()           // force "filter" to be set
   endif
endif
return nil

* end static function PseudoFilt()
*--------------------------------------------------------------------*


/*
   Function:  ChangeEndVal()
   Purpose:   Change ending value for data subset to match starting value
*/
static function ChangeEndVal(getlist)
if getlist[1]:changed
   getlist[2]:varPut( getlist[1]:varGet() )
endif
return .t.

* end static function ChangeEndVal()
*--------------------------------------------------------------------*


/*
   Function: Gilligan()
   Purpose:  Skipblock function used for data subsets
*/
static function gilligan(skipcnt, browse, norder)
local movement := 0
local bIndex := browse:indexBlock
static lReset := .f.
do case
   // no movement
   case skipcnt == 0
      skip 0
   // moving forward
   case skipcnt > 0
      do while movement < skipcnt .and. eval(bIndex) <= browse:endValues[norder] ;
               .and. ! eof()
         skip 1
         movement++
      enddo
      // make sure that we are within range - if not, move backward
      do while (eval(bIndex) > browse:endValues[norder] .or. eof()) .and. ! bof()
         skip -1
         movement--
      enddo
      // ensure once again that data is within range
      if bof() .or. eval(bIndex) < browse:startValues[norder]
         lReset := .t.
      endif
   // moving backward
   case skipcnt < 0
      do while movement > skipcnt .and. eval(bIndex) >= browse:startValues[norder]
         skip -1
         if bof()
            exit
         endif
         movement--
      enddo
      // make sure that we are within range -- if not, move forward
      do while eval(bIndex) < browse:startValues[norder] .and. ! eof()
         skip
         movement++
      enddo
      // ensure once again that data is within range
      if eof() .or. eval(bIndex) > browse:endValues[norder]
         lReset := .t.
      endif
endcase
if lReset
   err_msg({ "No data within this subset... viewing all records" })
   browse:startValues[norder] := NIL
   browse:endValues[norder] := NIL
   browse:goTopBlock := { || dbgotop() }
   browse:goBottomBlock := { || dbgobottom() }
   browse:skipBlock := browse:origSkipBlock
   lReset := .f.
endif
return movement

* end static function Gilligan()
*--------------------------------------------------------------------*


/*
   do type conversion for character/numeric/date index keys
   in preparation for compiling goTopBlock and goBottomBlock
   2nd parameter passed to increment low value for SOFTSEEK
*/
static function convertit(val, lo)
local ret_val := valtype(val)
local cdelimiters
do case
   case ret_val == "D"
      ret_val := "CTOD('" + dtoc(val + if(lo <> NIL, 1, 0)) + "')"
   case ret_val == "N"
      ret_val := ltrim(str(val + if(lo <> NIL, 1, 0)))
   case ret_val == "C"
      // if 2nd parameter was passed, then increment the rightmost
      // character in the string by 1 ASCII value.  This is required
      // so that the record pointer will be properly positioned...
      cdelimiters := if("'" $ val, { "[", "]" }, { "'", "'" })
      if lo <> NIL
         ret_val := cdelimiter[1] + left(val, len(val) - 1) + ;
                    chr(asc(right(val, 1)) + 1) + cdelimiters[2]
      else
         ret_val := cdelimiters[1] + val + cdelimiters[2]
      endif
endcase
return ret_val

* end static function ConvertIt()
*--------------------------------------------------------------------*


/*
   BlockHead(): custom error handler for code block syntax bombs
*/
static function blockhead(e, bOldHandler)
if e:gencode() == EG_SYNTAX .or. e:gencode() == EG_ARG
   dispend()
   err_msg( { "Error in code block syntax" } )
   break
   return .t.
endif
return eval(bOldHandler, e)

* end static function BlockHead()
*--------------------------------------------------------------------*


/*
   MakePicture(): derive default picture and picture length for a field
*/
static function makepicture(field_, pic)
local piclen
local x
do case
   case field_[DBS_TYPE] == 'D'
      if pic == chr(254)
         pic := "@D"
      endif
      piclen := 8

   case field_[DBS_TYPE] == 'M'
      pic := "<memo>"
      piclen := 6

   case field_[DBS_TYPE] == 'C'
      if pic == chr(254)
         pic := replicate("X", field_[DBS_LEN])
      endif
      piclen := field_[DBS_LEN]

   case field_[DBS_TYPE] == 'L'
      if pic == chr(254)
         pic := "Y"
         piclen := 1
      else
         piclen := len(pic)
      endif

   otherwise
      if pic == chr(254)
         x := str(fieldget(fieldpos(field_[DBS_NAME])))
         if "." $ x
            pic := replicate('9', at(".", x) - 1) + "."
            pic += replicate('9', len(x) - len(pic))
         else
            pic := replicate('9', len(x))
         endif
      endif
      piclen := len(pic)
endcase
return { pic, piclen }

* end static function MakePicture()
*--------------------------------------------------------------------*

// the following manifest constant sets spacing between printed columns
#define COLUMN_SPACING   1

/*
   PrintIt(): print all or some records
*/
static function printit(obrowse)
local marker := recno()
local ncolumns := obrowse:colCount
local mainloop  := .t.
local keepgoing := .t.
local lFiltered := .f.
local x
if ! empty(obrowse:queryCondition)
   lFiltered := ! Yes_No2({ "Print all records or those matching query?" }, ;
                             maxrow()/2 - 2, " All ", " Filtered ")
endif
if printok()
   eval(obrowse:goTopBlock)
   if lFiltered
      waiton({ "Locating first matching record... press ESC to exit" } )
      do while ( mainloop := eval(obrowse:skipBlock, 1) <> 0 ) ;
         .and. ! eval(obrowse:queryCondition) .and. checkescape()
      enddo
      waitoff()
      if ! mainloop
         err_msg({ "No matching records found" })
         keepgoing := .f.
      elseif ! eval(obrowse:queryCondition)   // they escaped out
         keepgoing := .f.
      endif
   endif
   if keepgoing
      waiton({ "Printing... press ESC to exit" } )
      printheadings(obrowse)
      do while mainloop .and. checkescape()
         for x := 1 to ncolumns
            printitem( eval(obrowse:getColumn(x):block), ;
                       obrowse:colWidth(x) + COLUMN_SPACING)
         next
         if prow() > 57
            eject
            printheadings(obrowse)
         else
            devpos( prow()+1, 0 )
         endif
         if lFiltered
            do while ( mainloop := eval(obrowse:skipBlock, 1) <> 0 ) ;
                       .and. ! eval(obrowse:queryCondition)
            enddo
         else
            mainloop := ( eval(obrowse:skipBlock, 1) <> 0 )
         endif
      enddo
      if mainloop
         devout("*** interrupted ***")
      endif
      eject
      set(_SET_DEVICE, "SCREEN")
      waitoff()
   endif
   go marker
endif
return nil

* end static function Printit()
*--------------------------------------------------------------------*


/*
   CheckEscape(): test for ESC key pressed during output process
*/
static function CheckEscape
local ret_val := .t.
if inkey() == K_ESC
   // don't allow to user to ESC out of this -- force them to confirm
   ret_val := ! Yes_No2({ "Are you sure you want to cancel output?" }, ;
                           maxrow()/2 - 2, " Yes ", " No ", .f.)
endif
return ret_val

* end static function CheckEscape()
*--------------------------------------------------------------------*


/*
   PrintItem() - pad each item to column width
*/
static function printitem(data, width)
local ctype := valtype(data)
do case
   case ctype == "C"
      devout( padr(data, width) )
   case ctype == "L"
      devout( padr(if(data, "T", "F"), width) )
   case ctype == "N"
      devout( padl(ltrim(str(data)), width) )
   case ctype == "D"
      devout( padl(dtoc(data), width) )
endcase
return nil

* end static function PrintItem()
*--------------------------------------------------------------------*

/*
   PrintHeadings() - print column headings
*/
static function printheadings(obrowse)
local num_columns := obrowse:colCount
local x, cheading
for x := 1 to num_columns
   if ( cheading := obrowse:getColumn(x):heading ) <> NIL
      devout(padr(cheading, obrowse:colWidth(x) + COLUMN_SPACING))
   else
      devout(space(obrowse:colWidth(x) + COLUMN_SPACING))
   endif
next
devpos( prow() + 1, 0)
for x := 1 to num_columns
   devout(replicate('-', obrowse:colWidth(x)) + space(COLUMN_SPACING))
next
devpos( prow() + 1, 0 )
return nil

* eof brow.prg
