/*
   Program:   APICK.PRG
   System:    GRUMPFISH LIBRARY
   Author:    Greg Lief
   Copyright (c) 1988-93, Greg Lief
   Dialect:   Clipper 5.x
   Compile instructions: clipper apick /n /w

   If you recompile this file, at a minimum you MUST use the /N
   command-line parameter.

   If you are using Clipper 5.2, you may wish to recompile APICK.PRG
   with the "/dCLIPPER52" command-line parameter to take advantage of
   the TBrowse:forceStable() method.

   If you always want up/down arrows visible, recompile APICK.PRG with
   the "/dARROWS_ALWAYS" command-line parameter.

*/

// begin preprocessor directives

#include "inkey.ch"
#include "grump.ch"

#define ISCHAR(a)        valtype(a) == "C"

// manifest constants used for drawing arrows, search string,
// and restoring the box outline underneath those items
#define S_ROW          cScrnBuff[1]
#define S_LEFT         cScrnBuff[2]
#define S_RIGHT        cScrnBuff[3]
#define S_MID          cScrnBuff[4]
#define S_COLOR        cScrnBuff[5]
#define S_TOPARROW     cScrnBuff[6]
#define S_BOTTOMARROW  cScrnBuff[7]
#define S_BUFFER       cScrnBuff[8]

function apick(nTop, nLeft, nBottom, nRight, aSource, cBoxColor, ;
               cBarColor, cStatColor, nActualLen, aTags, cUnselected, ;
               cHighlight, ele, cSelected, cUnavailable, cTitle, lWrap, ;
               nBox, lMovedown, lDrawbar, lRestore)
local marray := aclone(aSource)
local browse
local column
local nKey
local xx
local nMaxEle
local cType
local temp
local aOldscrn
local nMaxWidth
local nChoice
local aAvailable
local nBarLine
local nOldRow
local nOldCol
local lTagging
local cSearchstr        // search string for additive searches
local cScrnBuff
local nGroupStart := 0  // for group marking w/ F7
local nGroupEnd         // likewise

GFSaveEnv(, 0)          // shut off cursor

/*
   establish colors for box, status bar, and indicator (see
   COLORSET.PRG for default settings).  Pay attention to the second
   parameter -- this tells ColorSet() to merely return the default
   value rather than actually changing the color
*/
default cBoxColor to ColorSet(C_APICK_BOXOUTLINE, .T.)
default cBarColor to ColorSet(C_APICK_STATUSBAR, .T.)
default cStatColor to ColorSet(C_APICK_INDICATOR, .T.)
default cUnselected to ColorSet(C_APICK_UNSELECTED, .T.)
default cHighlight to ColorSet(C_APICK_CURRENT, .T.)
default cSelected to ColorSet(C_APICK_TAGGED, .T.) + ',' + ;
                     ColorSet(C_APICK_TAGGEDCURRENT, .T.)
default cUnavailable to ColorSet(C_APICK_UNAVAILABLE, .T.) + ',' + ;
                        ColorSet(C_APICK_UNAVAILABLECURRENT, .T.)
default ele to 1          // this will serve as our placeholder in the array
default cTitle to ''      // title for APICK() box
default lWrap to .t.      // enable wraparound from top to bottom of array
default nBox to 1         // boxtype: default is double line
default lMovedown to .t.  // move down a row after tagging an item
default lRestore to .t.   // restore screen after APICK() is finished

// we will allow tagging only if an array was passed to hold the tags
lTagging := valtype(aTags) == "A"
cSearchstr := []         // reset search string for additive search

// determine # of last element by grabbing length of array,
// unless of course the actual length was passed as parameter #9
nMaxEle := if(valtype(nActualLen) <> "N", len(marray), nActualLen)

// initialize parallel array that will determine whether each element
// can be selected... will load this in the loop just below
aAvailable := array(nMaxEle)

/*
   Loop through array to determine maximum width for the column.  Starting
   point for width is length of box title + delimiters, or 10 (necessary
   for showing the additive search string on bottom row of box), whichever
   is greater.  We'll also initialize each element in the aAvailable array.
*/
nMaxWidth := max(len(cTitle) + 4, 10)
for xx := 1 to nMaxEle
   cType := valtype(marray[xx])
   if cType == "C"
      nMaxWidth := max(nMaxWidth, len(marray[xx]))
      aAvailable[xx] := (left(marray[xx], 1) <> "~")
   else
      aAvailable[xx] := .t.
      do case
         case cType == "D"
            nMaxWidth := max(nMaxWidth, 8)
         case cType == "N"
            nMaxWidth := max(nMaxWidth, 10)
      endcase
   endif
next

// if the maximum width was changed in the processing above, and
// if we are allowing tagging, we must increment it by one so
// that the checkmark will actually appear!
if lTagging .and. nMaxWidth > len(cTitle) + 4
   nMaxWidth++
endif

// establish coordinates if not passed as parameters
default nTop to if(nMaxEle > maxrow() - 2, 6, int((maxrow() - nMaxEle - 1) / 2))
default nLeft to int( maxcol() - nMaxWidth - 2) / 2
default nBottom to if(nMaxEle > maxrow() - 2, maxrow() - 6, nTop + nMaxEle + 1)
default nRight to nLeft + nMaxWidth + 1

if lTagging
   default aTags to {}   // just in case you didn't pass an array (for shame!)
   // if tags array was empty, initialize it now
   if empty(aTags)
      aTags := afill(array(nMaxEle), .f.)
   else
      // ensure that blank elements get initialized to false
      for xx := 1 to len(aTags)
         if aTags[xx] == NIL
            aTags[xx] := .f.
         endif
      next
   endif
endif

// Create tbrowse object set to the appropriate coordinates
browse := TBrowseNew(nTop + 1, nLeft + 1, nBottom - 1, nRight - 1)
browse:colorSpec := cUnselected + "," + cHighlight + "," + cSelected + ;
                    "," + cUnavailable
// establish navigation blocks for the tbrowse object
browse:skipBlock := { | SkipCnt | Gilligan(@ele, SkipCnt, nMaxEle) }
browse:goTopBlock := { || ele := 1 }
browse:goBottomBlock := { || ele := nMaxEle }

// look for horizontal lines -- any elements like so: "~"
for xx := 1 to nMaxEle
   if ISCHAR(marray[xx]) .and. substr(marray[xx], 1, 1) == '~'
      marray[xx] := substr(marray[xx], 2)

      // do they want to draw a horizontal line? if so, the
      // trimmed length of this array element will now be one.
      if len(trim(marray[xx])) == 1
         marray[xx] := replicate(trim(marray[xx]), nMaxWidth)

      // if the first and last characters match, that means we
      // are to center the rest of the text on a horizontal line
      // for example:  "======This is the Text======"
      elseif left(marray[xx], 1) == right(marray[xx], 1)
         marray[xx] := padc(substr(marray[xx], 2, len(marray[xx]) - 2), ;
                            nMaxWidth, left(marray[xx], 1))
      endif
   endif
next

// create column for the browse object
column := TBColumnNew("", { || marray[ele] } )
column:width = nMaxWidth - if(lTagging, 1, 0)

/*
    establish color setup - use the following pairs for these items:
       unselected/available options  --> {1, 2}
       tagged/available options      --> {3, 4}
       unavailable options           --> {5, 6}

   these numbers correspond to colors as follows:
   1) unselected                    2) hilight
   3) tagged (unselected)           4) tagged (selected)
   5) unavailable (unselected)      6) unavailable (selected)
*/
if lTagging
   column:colorBlock := {|| if(! aAvailable[ele], {5, 6}, ;
                            if(aTags[ele], {3, 4}, {1, 2} ))}
else
   column:colorBlock := {|| if(! aAvailable[ele], {5, 6}, {1, 2} ) }
endif

// add this column to the browse object and get ready for browsing fun
browse:AddColumn( column )

// if tagging, add a new column to hold everybody's fave checkmark
if lTagging
   column := TBColumnNew("", { || if(aTags[ele], chr(251), chr(32)) } )
   column:colorBlock := {||  if(! aAvailable[ele], {5, 6}, ;
                             if(aTags[ele], {3, 4}, {1, 2} ))}
   browse:addColumn(column)
   browse:colSep := ''   // we want the two columns to butt against each other
endif

// draw box and status bar, using box title if it was passed
aOldscrn := ShadowBox(nTop, nLeft, nBottom, nRight, nBox, cTitle, ,cBoxColor)

// save bottom row of box and necessary coordinates -- if you use
// the additive search feature, the ShowString() function will
// need to restore the bottom row of the box
cScrnBuff := { nBottom, nLeft, nRight, ;
     nLeft + int((nRight - nLeft) / 2) - 4, ;  // horizontal midpoint
     cBoxColor, ;
     left(savescreen(nTop+1,nRight,nTop+1,nRight), 1),       ;  // top arrow
     left(savescreen(nBottom-1,nRight,nBottom-1,nRight), 1), ;  // bottom arrow
     savescreen(nBottom, nLeft, nBottom, nRight) }

// draw status bar if not all of the array elements fit in the window
nBarLine := nTop + 3
default lDrawbar to (nMaxEle > nBottom - nTop - 1)
if lDrawbar
   #ifdef ARROWS_ALWAYS
      @ nTop + 1,    nRight say chr(24)
      @ nBottom - 1, nRight say chr(25)
   #endif
   @ nTop + 2, nRight, nBottom - 2, nRight box chr(176) color cBarColor
endif
browse:autoLite := .f.
do while nChoice == NIL

   // wait for the display to stabilize, which will
   // loop once for each row in the browse window.
   // allow a keypress to bust out of this loop
   dispbegin()
   do while (nKey := inkey()) == 0 .and. ! browse:stabilize()
   enddo
   dispend()

   if browse:stable

      // the next bit of logic ensures that we are not sitting on
      // an unavailable element (does not apply to top and bottom)
      if ! aAvailable[ele]
         xx := lastkey()
         // note: we must take into account the possibility that
         // the user pressed SPACEBAR to tag and thus automatically
         // moved down to an unavailable item (hence the 32 logic)
         if (xx == K_UP .and. (ele > 1 .or. lWrap) ) .or. ;
                          ((xx == K_DOWN .or. xx == 32) .and. ;
                           (ele < nMaxEle .or. lWrap) )
            keyboard chr(if(xx == 32, K_DOWN, xx))
         elseif ele == 1
            keyboard chr(K_DOWN)
         elseif ele == nMaxEle
            keyboard chr(K_UP)
         endif
      endif

      if lDrawbar
         nOldRow := row()
         nOldCol := col()
         #ifndef ARROWS_ALWAYS

         // draw arrows if there are elements beyond top or bottom of window
         // first the bottom
         @ nBottom - 1, nRight ssay if(nMaxEle - ele >= nBottom - nOldRow, ;
                                       chr(25), S_BOTTOMARROW) color cBoxColor
         // then the top
         @ nTop + 1,nRight ssay if(nOldRow - ele < nTop, chr(24), S_TOPARROW) ;
                           color cBoxColor

         #endif

         // if status bar position has changed...
         if nBarLine <> nTop + 2 + ;
                        int((ele / nMaxEle) * (nBottom - nTop - 4))
            // first, blank out previous status bar
            @ nBarLine, nRight ssay chr(176) color cBarColor
            // then recalculate position of status bar
            nBarLine := nTop + 2 + int((ele / nMaxEle) * ;
                                   (nBottom - nTop - 4))
            // finally, redraw it
            @ nBarLine, nRight ssay chr(219) color cStatColor
         endif
         setpos(nOldRow, nOldCol)
      endif
      if nextkey() == 0
         browse:hiLite()
      endif
      nKey := ginkey(0, str(ele))    // pass along array element
      browse:deHilite()
   endif

   // deal with the keypress
   do case

      case nKey == 32 .and. aAvailable[ele] .and. lTagging  // tag 'em, Dan-O
         aTags[ele] = ! aTags[ele]
         if ! empty(cSearchstr)
            cSearchstr := []
            showstring(cSearchstr, cScrnBuff)
         endif
         // force redrawing this item to change its color
         browse:refreshCurrent()
         // move down a row if user requested it
         if lMovedown
            browse:down()
         endif

      case nKey == K_F7 .and. lTagging     // group tag
         if nGroupStart == 0
            nGroupStart := ele
         else
            nGroupEnd := ele
            if nGroupEnd < nGroupStart
               nGroupEnd   := nGroupStart
               nGroupStart := ele
            endif
            for xx := nGroupStart to nGroupEnd
               if aAvailable[xx]
                  aTags[xx] := .t.
               endif
            next
            // force redrawing entire window to change color of all items
            browse:refreshAll()
            nGroupStart := nGroupEnd := 0
         endif

      case nKey == K_F8 .and. lTagging     // tag 'em all, Dan-O
         for xx := 1 to nMaxEle
            if aAvailable[xx]
               aTags[xx] := .t.
            endif
         next
         // force redrawing entire window to change color of all items
         browse:refreshAll()

      case nKey == K_F9 .and. lTagging     // set 'em free, Dan-O
         afill(aTags, .f.)
         // force redrawing entire window to change color of all items
         browse:refreshAll()

      case nKey == K_F10 .and. lTagging   // switch all tags (Chinese Fire Drill)
         for xx := 1 to nMaxEle
            if aAvailable[xx]
               aTags[xx] := ! aTags[xx]
            endif
         next
         // force redrawing entire window to change color of all items
         browse:refreshAll()

      case nKey == K_UP          // up one row
         if ! empty(cSearchstr)
            // clear search string
            cSearchstr := []
            showstring(cSearchstr, cScrnBuff)
         endif
         // if at the top element, wrap to bottom if wrap is enabled
         if ele == 1 .and. lWrap
            browse:goBottom()
         else
            browse:up()
         endif

      case nKey == K_DOWN        // down one row
         if ! empty(cSearchstr)
            // clear search string
            cSearchstr := []
            showstring(cSearchstr, cScrnBuff)
         endif
         // if at the bottom element, wrap to top if wrap is enabled
         if ele == nMaxEle .and. lWrap
            browse:goTop()
         else
            browse:down()
         endif

      case nKey == K_CTRL_PGUP  // take it to the top, Jerome!
         if ! empty(cSearchstr)
            // clear search string
            cSearchstr := []
            showstring(cSearchstr, cScrnBuff)
         endif
         browse:goTop()

      case nKey == K_CTRL_PGDN  // goin' down.... down.... down ........
         if ! empty(cSearchstr)
            // clear search string
            cSearchstr := []
            showstring(cSearchstr, cScrnBuff)
         endif
         browse:goBottom()

      case nKey == K_PGUP .or. nKey == K_HOME   // top o' window
         if ! empty(cSearchstr)
            // clear search string
            cSearchstr := []
            showstring(cSearchstr, cScrnBuff)
         endif
         browse:pageUp()

      case nKey == K_PGDN .or. nKey == K_END    // bottom o' window
         if ! empty(cSearchstr)
            // clear search string
            cSearchstr := []
            showstring(cSearchstr, cScrnBuff)
         endif
         browse:pageDown()

      case nKey == K_ESC                        // aloha, you quitter
         nChoice := 0

      case nKey > 31 .and. nKey < 255      // search 'em
         if (temp := Ascan2(marray, cSearchstr + chr(nKey), aAvailable)) > 0
            cSearchstr += chr(nKey)
            // if moving backwards (up) through the array,
            // we have to handle it manually
            if ele > temp
               for xx := 1 to ele - temp
                  browse:up()
               next
            elseif ele <> temp
               ele := temp
               browse:refreshAll()
               #ifdef CLIPPER52
                  browse:forceStable()
               #else
                  do while ! browse:stabilize()
                  enddo
               #endif
            endif
            showstring(cSearchstr, cScrnBuff)
         endif

      case nKey == K_BS .OR. nKey == K_LEFT  // truncate the search string
         if len(cSearchstr) > 0
            cSearchstr := substr(cSearchstr, 1, len(cSearchstr) - 1)
            if (temp := Ascan2(marray, cSearchstr, aAvailable)) > 0
               ele := temp
               browse:refreshAll()
               #ifdef CLIPPER52
                  browse:forceStable()
               #else
                  do while ! browse:stabilize()
                  enddo
               #endif
            endif
            showstring(cSearchstr, cScrnBuff)
         endif

      case nKey == K_ENTER .and. aAvailable[ele]   // select if available
        nChoice := ele

   endcase
enddo
if lRestore
   ByeByeBox(aOldscrn)
else
   browse:hiLite()  // leave selected option highlighted for reference
endif
GFRestEnv()
return nChoice

* end function APick()
*--------------------------------------------------------------------*


/*
   Function: ShowString()
   Purpose:  Display the search string
*/

static function showstring(cSearchstr, cScrnBuff)
if len(cSearchstr) > 0
   @ S_ROW, S_MID ssay "[" + ;
            padr(substr(cSearchstr, max(1, len(cSearchstr) - 7)), 8) + "]" ;
            color S_COLOR
else
   restscreen(S_ROW, S_LEFT, S_ROW, S_RIGHT, S_BUFFER)
endif
return NIL

* end static function ShowString()
*--------------------------------------------------------------------*


/*
     Function: AScan2()
     Purpose:  Perform case-insensitive ASCAN()
*/
static function AScan2(array, value, aAvailable)
local counter  := 1
local lOldExact := set(_SET_EXACT, .f.)
local nRetval  := ascan(array, { |a| if(ISCHAR(a) .and. aAvailable[counter++], ;
                                  ltrim(upper(a)) = upper(value), .F.) }, 1)
set(_SET_EXACT, lOldExact)
return nRetval

* end static function AScan2()
*--------------------------------------------------------------------*

* eof apick.prg
