{$O+}

unit WcMsgDb;

interface

uses
  Dos,
  Crt,
  Desq,
  WcType,
  Filer,
  WcMisc,
  WcDb,
  WcUserDb,
  WcGlobal;
(*
  Strings,
  DOS,
  CR
{$IFDEF OPRO}
  OpInline,
  OpString,
  OpDate,
  OpCrt,
  OpDos,
{$ENDIF}
{$IFDEF TPRO}
  TpInline,
  TpString,
  TpDate,
  TpCrt,
  TpDos,
{$ENDIF}
  WcType,
  Filer,
  WcMisc,
  {TimeDate,
  WcNovell,}
  WcDb,
  Desq,
  WcUserDb,
  WcGlobal;
*)

const
  MagicHeaderActive   = $001A1A1B;
  MagicHeaderInactive = $011A1A1B;

type
  { This is an artifact of how Wildcat used to work and is not stored in }
  { a database anywhere.  It is used by TMsgDatabase::GetMsgStatus which }
  { creates one of these from information in the database.               }
  TMsgStatus = record
                 LowMsg,
                 HighMsg,
                 ActiveMsg : Word;
               end;

type
  PMsgDatabase = ^TMsgDatabase;
  TMsgDatabase = object
    constructor Init; {!!.U}
    function IsOpen(CheckConf : Word) : Boolean; {!!.U}
    function Open(AConf : Word; TSecondary : Boolean) : Boolean; {!!.U}
    procedure Done;
    procedure Lock;
    procedure Unlock;
    function DatabaseActive : Boolean;
    function AddMsg(var ref: Longint; var msg: TMsgHeader; msgtext: PMsgText) : Boolean;
    function FindMsg(msgnum: Word): Longint;
    function SearchMsg(msgnum: Word): Longint;
    procedure NextMsg(var ref: Longint);
    procedure PrevMsg(var ref: Longint);
    procedure GetMsgStatus(var msr: TMsgStatus);
    procedure GetMsgHeader(ref: Longint; var msg: TMsgHeader);
    procedure GetMsgHeaderAndText(ref: Longint; var msg: TMsgHeader; buffer: PMsgText; offset, len: Word);
    procedure UpdateMsgHeader(var NewMsg: TMsgHeader);
    procedure UpdateMsgHeaderAndText(var NewMsg: TMsgHeader; msgtext: PMsgText);
    procedure UpdateMsgText(ref: Longint; msgtext: PMsgText; offset, len: Word);
    procedure SetMsgFlagsNum(MsgNumber: Word; NewMsgFlags : Word);
    procedure MarkMsgRead(var MsgHdr : TMsgHeader);
    procedure UpdateMasterInfo;
  private
    AlreadyOpen : Boolean;
    Conf: Word;
    Secondary : Boolean;
    IndexFile: File;
    DataFile: File;
    LockCount: Integer;
    MsgCountDelta : LongInt;
    function IndexOffset(index: Word): Longint;
    function OffsetIndex(ofs: Longint): Word;
    procedure FatalDBError(const S : String);
    procedure LogDBError(const S : String);
    function FindMsgIndex(msgnum: Word; var msghdr: TMsgHeader): Word;
    procedure ReadIndexHeader(var header: TMsgIndexHeader);
    procedure WriteIndexHeader(var header: TMsgIndexHeader);
    function ReadMsgHeader(index: Word; var msghdr: TMsgHeader): Boolean;
    procedure WriteMsgHeader(index: Word; var msghdr: TMsgHeader);
    procedure UnhookMessage(index: Word; var msghdr: TMsgHeader);
    procedure UpdateNodeInfo(CurConf : LongInt);
    (*
    procedure IntegrityCheck(const where: String; userid: Longint);
    *)
  end;

var
  MsgDb : TMsgDatabase;

{const
  MsgDbOpen : Integer = 0;}

implementation

{uses WcConf, WcLog;}


  constructor TMsgDatabase.Init;
  begin
    AlreadyOpen := False;
  end;


  function TMsgDatabase.IsOpen(CheckConf : Word) : Boolean;
  begin
    IsOpen := (CheckConf = Conf) and AlreadyOpen;
  end;


  function TMsgDatabase.Open(AConf: Word; TSecondary : Boolean) : Boolean;
  var
    fn: String;
    header: TMsgIndexHeader;
    ie: TMsgIndexEntry;
    index: Word;
    cd: TConfDesc;

  begin
    LockCount := 0;
    IsamClearOk;
    Conf := AConf;
    Open := False;
    if AlreadyOpen then
      Done;
    Secondary := TSecondary;
    if ExistFile('MSGLOCK\'+Long2Str(Conf)+'.LCK') then
      Exit
    else
      UpdateNodeInfo(Conf);
    LoadConfDesc(cd, Conf);
    fn := AddBackSlash(cd.MsgPath) + 'MSG' + Long2Str(Conf);
    Assign(IndexFile, fn+'.IX');
    FileMode := $42;
    Reset(IndexFile, 1);
    if IoResult <> 0 then begin
      Rewrite(IndexFile, 1);
      if IoResult <> 0 then begin
        IsamOk := False;
        IsamError := 9903;
        UpdateNodeInfo(-1);
        Exit;
      end;
      FillChar(header, sizeof(header), 0);
      header.RecordSize := sizeof(TMsgIndexEntry);
      header.ActiveRecords := 0;
      header.NextMsgNumber := 1;
      BlockWrite(IndexFile, header, sizeof(header));
      if IoResult <> 0 then begin
        IsamOk := False;
        {IsamError := 9903;}
        LogError('Error creating conference '+Long2Str(Conf), 9903);
        UpdateNodeInfo(-1);
        Exit;
      end;
    end;
    ReadIndexHeader(header);
    if (header.RecordSize <> sizeof(TMsgIndexEntry)) then begin
      Close(IndexFile);
      IsamOk := False;
      LogError('Corrupted header on conference '+Long2Str(Conf), 10120);
      {IsamError := 10120;}
      UpdateNodeInfo(-1);
      Exit;
    end;
    index := OffsetIndex(FileSize(IndexFile));
    if index > 1 then begin
      Seek(IndexFile, IndexOffset(index-1));
      BlockRead(IndexFile, ie, sizeof(TMsgIndexEntry));
      if (IoResult <> 0) and (header.NextMsgNumber <= ie.MsgNumber) then begin
        header.NextMsgNumber := ie.MsgNumber + 1;
        WriteIndexHeader(header);
      end;
    end;
    Assign(DataFile, fn+'.DAT');
    FileMode := $42;
    Reset(DataFile, 1);
    if IoResult <> 0 then begin
      Rewrite(DataFile, 1);
      if IoResult <> 0 then begin
        Close(IndexFile);
        IsamOk := False;
        {IsamError := 9903;}
        LogError('Error creating DAT for conference '+Long2Str(Conf), 9903);
        UpdateNodeInfo(-1);
        Exit;
      end;
    end;
    Open := True;
    AlreadyOpen := True;
    MsgCountDelta := 0;
  end;


  procedure TMsgDatabase.Done;
  begin
    if not AlreadyOpen then
      Exit;
    AlreadyOpen := False;
    while LockCount > 0 do
      Unlock;
    if MsgCountDelta <> 0 then
      UpdateMasterInfo;
    Close(IndexFile);
    Close(DataFile);
    UpdateNodeInfo(-1);
  end;


  function TMsgDataBase.DatabaseActive : Boolean;
  begin
    DataBaseActive := AlreadyOpen;
  end;


  procedure TMsgDatabase.Lock;
  var
    retries : Word;

  begin
    if LockCount = 0 then begin
      retries := 0;
      while not BTIsamLockRecord(0, 1, FileRec(IndexFile).Handle, 768, 64) do begin
        IsamDelay(Random(200));
        Inc(retries);
        if retries > 50 then
          LogFatalError('Error locking message database', IsamError);
        WriteTopRight('Lock retry #'+Long2Str(retries));
      end;
      if retries > 0 then
        WriteTopRight('              ');
    end;
    Inc(LockCount);
  end;


  procedure TMsgDatabase.Unlock;

  begin
    Dec(LockCount);
    if LockCount = 0 then begin
      if not BTIsamUnLockRecord(0, 1, FileRec(IndexFile).Handle) then
        LogFatalError('Error unlocking message database', IsamError);
    end;
  end;


  function TMsgDatabase.AddMsg(var ref : LongInt; var msg: TMsgHeader; msgtext: PMsgText) : Boolean;
  var
    header: TMsgIndexHeader;
    ie: TMsgIndexEntry;
    userref: Longint;
    userrec: TUserRec;
    tuserconf: PUserWrapper;
    firstunread: Word;
    tmsghdr, tmsghdr2: TMsgHeader;
    tindex, tindex2: Word;
    b: Byte;
    SendNotification : Boolean;

  begin
    AddMsg := False;
    SendNotification := False;
    IsamClearOk;
    Lock;
    ReadIndexHeader(header);
    if header.NextMsgNumber < $FFFF then begin
      msg.MagicNumber := MagicHeaderActive;
      msg.MsgNumber := header.NextMsgNumber;
      msg.NextUnread := 0;
      msg.PrevUnread := 0;

      if (msg.DestUserID > 0) and not FlagIsSet(msg.mFlags, mfReceived) then
        begin
          UserDb.Lock;
          if UserDb.FindKey(UserIDKey, userref, BuildUserIdKey(msg.DestUserId)) then begin
            SendNotification := True;
            SetFlag(msg.mFlags, mfReceiveable);
            UserDb.GetRec(userref, userrec);
            tuserconf := New(PUserWrapper, Init(userrec));
            firstunread := tuserconf^.GetFirstUnread(Conf);
            if firstunread = 0 then begin
              msg.PrevUnread := msg.MsgNumber;
              msg.NextUnread := msg.MsgNumber;
              tuserconf^.SetFirstUnread(Conf, msg.MsgNumber);
            end else begin
              tindex := FindMsgIndex(firstunread, tmsghdr);
              if IsamOk and (tmsghdr.destuserid = userrec.userid) then begin
                if tmsghdr.PrevUnread = firstunread then begin
                  tmsghdr.NextUnread := msg.MsgNumber;
                  tmsghdr.PrevUnread := msg.MsgNumber;
                  WriteMsgHeader(tindex, tmsghdr);
                  msg.NextUnread := tmsghdr.MsgNumber;
                  msg.PrevUnread := tmsghdr.MsgNumber;
                end else begin
                  msg.NextUnread := tmsghdr.MsgNumber;
                  tindex2 := FindMsgIndex(tmsghdr.PrevUnread, tmsghdr2);
                  if IsamOk then begin
                    msg.PrevUnread := tmsghdr2.MsgNumber;
                    tmsghdr.PrevUnread := msg.MsgNumber;
                    WriteMsgHeader(tindex, tmsghdr);
                    tmsghdr2.NextUnread := msg.MsgNumber;
                    WriteMsgHeader(tindex2, tmsghdr2);
                  end;
                end;
              end else begin
                msg.PrevUnread := msg.MsgNumber;
                msg.NextUnread := msg.MsgNumber;
                tuserconf^.SetFirstUnread(Conf, msg.MsgNumber);
              end;
            end;
            Dispose(tuserconf, Done);
          end;
          UserDb.Unlock;
        end;

      ie.MsgNumber := msg.MsgNumber;
      ie.HeaderOffset := FileSize(DataFile);
      Seek(IndexFile, FileSize(IndexFile));
      BlockWrite(IndexFile, ie, sizeof(TMsgIndexEntry));

      Seek(DataFile, ie.HeaderOffset);
      BlockWrite(DataFile, msg, sizeof(TMsgHeader));
      if msgtext <> nil then
        BlockWrite(DataFile, msgtext^, msg.MsgBytes)
      else if msg.MsgBytes > 0 then begin
        Seek(DataFile, FilePos(DataFile) + msg.MsgBytes-1);
        b := 0;
        BlockWrite(DataFile, b, 1);
      end;

      ref := OffsetIndex(FilePos(IndexFile)) - 1;
      Inc(header.NextMsgNumber);
      Inc(header.ActiveRecords);
      WriteIndexHeader(header);

      Inc(MsgCountDelta);
      AddMsg := True;
    end;
    Unlock;
  end;

  function TMsgDatabase.FindMsgIndex(msgnum: Word; var msghdr: TMsgHeader): Word;
  var
    left, right, mid, last: Longint;
    ie: TMsgIndexEntry;
    ref: Longint;
  begin
    IsamClearOk;
    left := 1;
    last := Longint(OffsetIndex(FileSize(IndexFile)));
    right := last - 1;
    ref := last;
    while left <= right do begin
      mid := (left + right) div 2;
      Seek(IndexFile, IndexOffset(mid));
      BlockRead(IndexFile, ie, sizeof(TMsgIndexEntry));
      if IoResult <> 0 then
        Break;
      if msgnum = ie.MsgNumber then begin
        ReadMsgHeader(mid, msghdr);
        FindMsgIndex := mid;
        Exit;
        end
      else if msgnum < ie.MsgNumber then begin
        ref := mid;
        right := mid - 1;
        end
      else
        left := mid + 1;
    end;
    IsamOk := False;
    if ref < last then
      IsamError := 10200
    else
      IsamError := 10210;
    FindMsgIndex := ref;
  end;

  function TMsgDatabase.FindMsg(msgnum: Word): Longint;
  var
    msghdr: TMsgHeader;
  begin
    FindMsg := FindMsgIndex(msgnum, msghdr);
  end;

  function TMsgDatabase.SearchMsg(msgnum: Word): Longint;

  begin
    IsamClearOk;
    SearchMsg := FindMsg(msgnum);
    if not IsamOk and (IsamError = 10200) then
      IsamOk := True;
  end;

  procedure TMsgDatabase.NextMsg(var ref: Longint);

  begin
    IsamClearOk;
    Inc(ref);
    IsamOk := IndexOffset(ref) < FileSize(IndexFile);
    if not IsamOk then
      IsamError := 10250;
  end;

  procedure TMsgDatabase.PrevMsg(var ref: Longint);

  begin
    IsamClearOk;
    Dec(ref);
    IsamOk := ref > 0;
    if not IsamOk then
      IsamError := 10260;
  end;


  procedure TMsgDatabase.GetMsgStatus(var msr: TMsgStatus);
  var
    header: TMsgIndexHeader;
    msghdr: TMsgHeader;
  begin
    IsamClearOk;
    ReadIndexHeader(header);
    FillChar(msr, sizeof(msr), 0);
    if ReadMsgHeader(1, msghdr) then
      msr.LowMsg := msghdr.MsgNumber;
    msr.HighMsg := header.NextMsgNumber - 1;
    msr.ActiveMsg := header.ActiveRecords;
  end;


  procedure TMsgDatabase.GetMsgHeader(ref: Longint; var msg: TMsgHeader);

  begin
    IsamClearOk;
    IsamOk := (ref > 0) and ReadMsgHeader(ref, msg);
    if not IsamOk then
      IsamError := 10070;
  end;

  procedure TMsgDatabase.GetMsgHeaderAndText(ref: Longint; var msg: TMsgHeader; buffer: PMsgText; offset, len: Word);
  var
    ofs: Longint;
    x: Word;

  begin
    IsamClearOk;
    if ref <= 0 then
      begin
        IsamOk := False;
        IsamError := 10131;
        Exit;
      end;
    if not ReadMsgHeader(ref, msg) then
      begin
        IsamOk := False;
        IsamError := 10070;
        Exit;
      end;
    if offset > 0 then
      Seek(DataFile, FilePos(DataFile)+offset);
    if offset < msg.MsgBytes then
      begin
        x := msg.MsgBytes - offset;
        if len > x then
          len := x;
        BlockRead(DataFile, buffer^, len);
        if IoResult <> 0 then
          begin
            IsamOk := False;
            IsamError := 10070;
          end;
      end;
  end;


  procedure TMsgDatabase.UpdateMsgHeader(var NewMsg: TMsgHeader);
  var
    iheader: TMsgIndexHeader;
    header: TMsgHeader;
    index: Word;
  begin
    Lock;
    ReadIndexHeader(iheader);
    index := FindMsgIndex(NewMsg.MsgNumber, header);
    if IsamOk then begin
      ClearFlag(NewMsg.mFlags, mfReceived);
      SetFlag(NewMsg.mFlags, header.mFlags and mfReceived);
      NewMsg.NextUnread := header.NextUnread;
      NewMsg.PrevUnread := header.PrevUnread;
      WriteMsgHeader(index, NewMsg);
      if header.mFlags and mfDeleted <> NewMsg.mFlags and mfDeleted then
        begin
          if NewMsg.mFlags and mfDeleted <> 0 then
            Dec(iheader.ActiveRecords)
          else
            Inc(iheader.ActiveRecords);
          WriteIndexHeader(iheader);
        end;
    end;
    Unlock;
  end;

  procedure TMsgDatabase.UpdateMsgHeaderAndText(var NewMsg: TMsgHeader; msgtext: PMsgText);
  var
    iheader: TMsgIndexHeader;
    header: TMsgHeader;
    index: Word;
    ofs: Longint;
    ie: TMsgIndexEntry;
    io: Integer;
    b : byte;

  begin
    Lock;
    ReadIndexHeader(iheader);
    index := FindMsgIndex(NewMsg.MsgNumber, header);
    if IsamOk and (NewMsg.MsgNumber = header.MsgNumber) then begin
      ClearFlag(NewMsg.mFlags, mfReceived);
      SetFlag(NewMsg.mFlags, header.mFlags and mfReceived);
      NewMsg.NextUnread := header.NextUnread;
      NewMsg.PrevUnread := header.PrevUnread;
      ReadMsgHeader(index, header);
      if NewMsg.MsgBytes <= header.MsgBytes then begin
        if msgtext <> nil then
          BlockWrite(DataFile, msgtext^, NewMsg.MsgBytes);
        WriteMsgHeader(index, NewMsg);
        end
      else begin
        header.MagicNumber := MagicHeaderInactive;
        WriteMsgHeader(index, header);
        ofs := FileSize(DataFile);
        Seek(DataFile, ofs);
        BlockWrite(DataFile, NewMsg, sizeof(TMsgHeader));
        if msgtext <> nil then
          BlockWrite(DataFile, msgtext^, NewMsg.MsgBytes)
        else if NewMsg.MsgBytes > 0 then begin
          Seek(DataFile, FilePos(DataFile) + NewMsg.MsgBytes-1);
          b := 0;
          BlockWrite(DataFile, b, 1);
        end;
        ie.MsgNumber := NewMsg.MsgNumber;
        ie.HeaderOffset := ofs;
        Seek(IndexFile, IndexOffset(index));
        BlockWrite(IndexFile, ie, sizeof(TMsgIndexEntry));
        io := IoResult;
        if io <> 0 then begin
          IsamError := 9500+io;
          FatalDbError('Error writing index entry');
        end;
      end;
      if header.mFlags and mfDeleted <> NewMsg.mFlags and mfDeleted then
        begin
          if NewMsg.mFlags and mfDeleted <> 0 then
            Dec(iheader.ActiveRecords)
          else
            Inc(iheader.ActiveRecords);
          WriteIndexHeader(iheader);
        end;
    end;
    Unlock;
  end;

  procedure TMsgDatabase.UpdateMsgText(ref: Longint; msgtext: PMsgText; offset, len: Word);

  var hdr: TMsgHeader;

  begin
    Lock;
    if not ReadMsgHeader(ref, hdr) then
      Exit;
    if offset >= hdr.MsgBytes then
      Exit;
    if offset+len > hdr.MsgBytes then
      len := hdr.MsgBytes - offset;
    { this now relies on the fact that ReadMsgHeader leaves the file pointer
      for the .dat file at the start of the message text }
    Seek(DataFile, FilePos(DataFile) + offset);
    BlockWrite(DataFile, msgtext^, len);
    Unlock;
  end;

  procedure TMsgDatabase.SetMsgFlagsNum(MsgNumber: Word; NewMsgFlags : Word);
  var
    header: TMsgHeader;
    index: Word;
  begin
    index := FindMsgIndex(MsgNumber, header);
    if IsamOk then begin
      if FlagIsSet(NewMsgFlags, mfDeleted) and not FlagIsSet(header.mFlags, mfDeleted) then
        Dec(MsgCountDelta);
      SetFlag(header.mFlags, NewMsgFlags);
      WriteMsgHeader(index, header);
    end;
  end;


  procedure TMsgDatabase.MarkMsgRead(var MsgHdr : TMsgHeader);
  var
    index: Word;
  begin
    MsgDB.Lock;
    index := FindMsgIndex(msghdr.MsgNumber, msghdr);
    if IsamOk then begin
      SetDateTime(msghdr.ReadTime);
      SetFlag(msghdr.mFlags, mfReceived);
      UnhookMessage(index, msghdr);
    end;
    MsgDB.Unlock;
  end;

  procedure TMsgDatabase.UpdateNodeInfo(CurConf : LongInt);
  var
    f : file of TNodeInfo;
    NI : TNodeInfo;
    Io : Word;
    Retries: Word;

    procedure UnLockNode;
    begin
      if not BTIsamUnLockRecord(0, SizeOf(TNodeInfo), FileRec(f).Handle) then
        FatalDBError('Error unlocking NODEINFO.DAT');
    end;


  begin
    Assign(f, MwConfig.NodeInfoPath + 'NODEINFO.DAT');
    Filemode := $42;
    Reset(f);
    Io := IoResult;
    if Io <> 0 then
      FatalDBError('Unable to open NODEINFO.DAT.');
    Retries := 0;
    while not BTIsamLockRecord(0, SizeOf(TMasterInfo), FileRec(f).Handle, 768, 64) do
      begin
        Inc(Retries);
        if Retries = 50 then
          FatalDBError('Unable to lock NODEINFO.DAT.');
        WcDelay(5 + Random(10));
      end;

    Seek(f, MwConfig.NodeId+1);
    Read(f, NI);

    Io := IoResult;
    if Io <> 0 then
      begin
        UnLockNode;
        FatalDBError('Unable to read from NODEINFO.DAT.');
      end;

    if Secondary then
      NI.LockConf2 := CurConf
    else
      NI.LockConf1 := CurConf;

    Seek(f, MwConfig.NodeId+1);
    Write(f, NI);
    UnLockNode;
    Io := IoResult;
    if Io <> 0 then
      FatalDBError('Unable to write to NODEINFO.DAT.');
    Close(f);
    if IoResult <> 0 then ;
  end;


  procedure TMsgDatabase.UpdateMasterInfo;

  var
    f : file of TMasterInfo;
    MI : TMasterInfo;
    Io : Word;
    Retries: Word;

    procedure UnLockNode;
    begin
      if not BTIsamUnLockRecord(0, SizeOf(TNodeInfo), FileRec(f).Handle) then
        FatalDBError('Error unlocking NODEINFO.DAT');
    end;


  begin
    Assign(f, MwConfig.NodeInfoPath + 'NODEINFO.DAT');
    Filemode := $42;
    Reset(f);
    Io := IoResult;
    if Io <> 0 then
      FatalDBError('Unable to open NODEINFO.DAT.');
    Retries := 0;
    while not BTIsamLockRecord(0, SizeOf(TMasterInfo), FileRec(f).Handle, 768, 64) do
      begin
        Inc(Retries);
        if Retries = 50 then
          FatalDBError('Unable to lock NODEINFO.DAT.');
        WcDelay(5 + Random(10));
      end;

    Read(f, MI);
    Io := IoResult;
    if Io <> 0 then
      begin
        UnLockNode;
        FatalDBError('Unable to read from NODEINFO.DAT.');
      end;
    Inc(MI.TotalMessages, MsgCountDelta);
    if (MsgCountDelta > 0) then              {Do not Subtract From Dailys}
      Inc(MI.TempMsgs, MsgCountDelta);
    Seek(f, 0);
    Write(f, MI);
    UnLockNode;

    Io := IoResult;
    if Io <> 0 then
      FatalDBError('Unable to write to NODEINFO.DAT.');
    Close(f);
    if IoResult <> 0 then ;
    MsgCountDelta := 0;
  end;

  procedure TMsgDatabase.FatalDBError(const S : String);
  begin
    LogFatalError('MESSAGE DATABASE (Conference '+Long2Str(Conf)+') : '+ S, IsamError);
  end;

  procedure TMsgDatabase.LogDBError(const S : String);
  begin
    LogError('MESSAGE DATABASE (Conference '+Long2Str(Conf)+') : ' + S, IsamError);
  end;

  function TMsgDatabase.IndexOffset(index: Word): Longint;

  begin
    IndexOffset := Longint(index)*sizeof(TMsgIndexEntry);
  end;

  function TMsgDatabase.OffsetIndex(ofs: Longint): Word;

  begin
    OffsetIndex := ofs div sizeof(TMsgIndexEntry);
  end;

  procedure TMsgDatabase.ReadIndexHeader(var header: TMsgIndexHeader);
  var
    nr, io: Word;
  begin
    Lock;
    Seek(IndexFile, 0);
    BlockRead(IndexFile, header, sizeof(TMsgIndexHeader), nr);
    io := IoResult;
    if (io <> 0) or (nr <> sizeof(TMsgIndexHeader)) then begin
      if io <> 0 then
        IsamError := 9500+io
      else
        IsamError := 10070;
      FatalDbError('Error reading index header');
    end;
    Unlock;
  end;

  procedure TMsgDatabase.WriteIndexHeader(var header: TMsgIndexHeader);
  var
    io: Word;
  begin
    Seek(IndexFile, 0);
    BlockWrite(IndexFile, header, sizeof(TMsgIndexHeader));
    io := IoResult;
    if io <> 0 then begin
      IsamError := 9500+io;
      FatalDbError('Error writing index header');
    end;
  end;

  function TMsgDatabase.ReadMsgHeader(index: Word; var msghdr: TMsgHeader): Boolean;
  var
    ie: TMsgIndexEntry;
    nr: Word;
  begin
    ReadMsgHeader := False;
    Seek(IndexFile, IndexOffset(index));
    BlockRead(IndexFile, ie, sizeof(TMsgIndexEntry), nr);
    if (IoResult <> 0) or (nr <> sizeof(TMsgIndexEntry)) then begin
      IsamOk := False;
      IsamError := 10070;
      Exit;
    end;
    Seek(DataFile, ie.HeaderOffset);
    BlockRead(DataFile, msghdr, sizeof(TMsgHeader), nr);
    if IoResult <> 0 then begin
      IsamOk := False;
      IsamError := 10070;
      Exit;
    end;
    ReadMsgHeader := nr = sizeof(TMsgHeader);
  end;

  procedure TMsgDatabase.WriteMsgHeader(index: Word; var msghdr: TMsgHeader);
  var
    ie: TMsgIndexEntry;
    nr: Word;
  begin
    Seek(IndexFile, IndexOffset(index));
    BlockRead(IndexFile, ie, sizeof(TMsgIndexEntry), nr);
    if (IoResult <> 0) or (nr <> sizeof(TMsgIndexEntry)) then begin
      IsamOk := False;
      IsamError := 10070;
      Exit;
    end;
    if ie.MsgNumber <> msghdr.MsgNumber then
      {LogFatalError('Message database error - run wcREPAIR', 10010);}
      LogFatalError('MsgNumber mismatch, conf='+Long2Str(conf)+
                    ' ie='+Long2Str(ie.MsgNumber)+
                    ' hdr='+Long2Str(msghdr.MsgNumber), 0);
    Seek(DataFile, ie.HeaderOffset);
    BlockWrite(DataFile, msghdr, sizeof(TMsgHeader));
    if IoResult <> 0 then begin
      IsamError := 10070;
      FatalDbError('Error writing message header');
    end;
  end;

  procedure TMsgDatabase.UnhookMessage(index: Word; var msghdr: TMsgHeader);
  var
    tmsghdr: TMsgHeader;
    tindex: Word;
    tuserconf: PUserWrapper;
    userref: Longint;
    userrec: PUserRec;
  begin
    (*
    IntegrityCheck('UnhookMessage start', msghdr.DestUserId);
    *)
    if (msghdr.PrevUnread = msghdr.MsgNumber) and (msghdr.DestUserId > 0) then begin
      UserDb.Lock;
      if UserDb.FindKey(UserIDKey, userref, BuildUserIdKey(msghdr.DestUserId)) then begin
        New(userrec);
        UserDb.GetRec(userref, userrec^);
        tuserconf := New(PUserWrapper, Init(userrec^));
        tuserconf^.SetFirstUnread(Conf, 0);
        Dispose(tuserconf, Done);
        Dispose(userrec);
      end;
      UserDb.Unlock;
      end
    else begin
      if (msghdr.PrevUnread > msghdr.MsgNumber) and (msghdr.DestUserId > 0) then begin
        UserDb.Lock;
        if UserDb.FindKey(UserIdKey, userref, BuildUserIdKey(msghdr.DestUserId)) then begin
          New(userrec);
          UserDb.GetRec(userref, userrec^);
          tuserconf := New(PUserWrapper, Init(userrec^));
          tuserconf^.SetFirstUnread(Conf, msghdr.NextUnread);
          Dispose(tuserconf, Done);
          Dispose(userrec);
        end;
        UserDb.Unlock;
      end;
      tindex := FindMsgIndex(msghdr.PrevUnread, tmsghdr);
      if IsamOk and (tmsghdr.NextUnread = msghdr.MsgNumber) then begin
        tmsghdr.NextUnread := msghdr.NextUnread;
        WriteMsgHeader(tindex, tmsghdr);
      end;
      tindex := FindMsgIndex(msghdr.NextUnread, tmsghdr);
      if IsamOk and (tmsghdr.PrevUnread = msghdr.MsgNumber) then begin
        tmsghdr.PrevUnread := msghdr.PrevUnread;
        WriteMsgHeader(tindex, tmsghdr);
      end;
    end;
    msghdr.PrevUnread := 0;
    msghdr.NextUnread := 0;
    WriteMsgHeader(index, msghdr);
  end;

end.
