#include <windows.h> 
#include <stdlib.h>
#include <string.h>
#include <dir.h>
#include "wpddefs.h"
#include "sammenu.h"
#include "menus.h"
#include "iconstub.h"

#define WIDTHBYTES(i)	((i+31)/32*4)		/* bitmap rounding equation */

// Function prototypes for all local routines.
HANDLE PASCAL CreateBorderLessDIB(HBITMAP);
HANDLE PASCAL DIBFromBitmap(HBITMAP,DWORD,WORD);
WORD NEAR PASCAL DIBPaletteSize(void FAR *);
WORD NEAR PASCAL DIBNumColors(void FAR *);
void NEAR PASCAL DrawActualBitmap(EDITCUSTOMICON *, HDC);
void NEAR PASCAL DrawZoomedBitmap(EDITCUSTOMICON *, HDC);
void NEAR PASCAL DrawIconFrame(HDC,POINT,int,int,int);
void NEAR PASCAL DrawMouseBitmap(HDC);
void NEAR PASCAL DrawMouseButtons(EDITCUSTOMICON *, HDC);
BYTE NEAR PASCAL GetColorIndex(LPBITMAPINFO, DWORD, DWORD *, WORD, BOOL *);
void NEAR PASCAL SetColorIndex(LPBITMAPINFO, DWORD, int, DWORD *, BOOL *);
DWORD * NEAR PASCAL GetIconColors(LPBITMAPINFO, WORD);
void NEAR PASCAL SetIconColors(LPBITMAPINFO, int, DWORD);
void NEAR PASCAL SetByte(LPBITMAPINFO, DWORD *, DWORD, WORD, int, int, BOOL *);
void NEAR PASCAL SetNibble(LPBITMAPINFO, DWORD *, DWORD, WORD, int, int, BOOL *);
void NEAR PASCAL SetBit(LPBITMAPINFO, DWORD *, DWORD, WORD, int, int);
void NEAR PASCAL SetDWord(LPBITMAPINFO, DWORD, int, int);
LONG FAR PASCAL IconControlProc(HWND, unsigned int, unsigned int, LONG);
int NEAR PASCAL WhichPixelClicked(EDITCUSTOMICON *, HWND, int, int, WORD);
void NEAR PASCAL IconEditDestroy(EDITCUSTOMICON *);
void NEAR PASCAL IconEditInit(EDITCUSTOMICON *TmpPtr,HWND hDlg, HANDLE hDIB);

/***************************************************************************
***************************************************************************/
void NEAR PASCAL
IconEditInit(EDITCUSTOMICON *TmpPtr,HWND hDlg, HANDLE hDIB)
{
	HWND hCtl;
	int BoxWidth, BoxHeight, BmpWidth, BmpHeight;
	LPBITMAPINFO lpbi;
	RECT Rect;

	if (hDIB == NULL)
		return;

	TmpPtr->CurrentDIBChanged = FALSE;

	TmpPtr->hDIB = hDIB;

	lpbi = (LPBITMAPINFO)GlobalLock(TmpPtr->hDIB);

	// Center the actual size bitmap in the box.
	hCtl = GetDlgItem(hDlg, ID_STATIC3);
	GetWindowRect(hCtl, &Rect);
	TmpPtr->ActualOffset.x = ((Rect.right - Rect.left 
							- (int)lpbi->bmiHeader.biWidth) / 2) + 1;
	TmpPtr->ActualOffset.y = 0;

	// Get the size of the window for the fat bits bitmap.
	hCtl = GetDlgItem(hDlg, ID_STATIC2);
	GetWindowRect(hCtl, &Rect);
	
	// Calculate the scale for the fat bits bitmap.
	BoxWidth = Rect.right - Rect.left;
	BoxHeight = Rect.bottom - Rect.top;
	BmpWidth = (int)lpbi->bmiHeader.biWidth;
	BmpHeight = (int)lpbi->bmiHeader.biHeight;
	
	TmpPtr->Scale = min((BoxWidth-3) / (BmpWidth + 3), 
								(BoxHeight-3) / (BmpHeight + 3));
	TmpPtr->ZoomedSize.x = (WORD)BmpWidth * TmpPtr->Scale;
	TmpPtr->ZoomedSize.y = (WORD)BmpHeight * TmpPtr->Scale;

	// Center the fat bits bitmap in the box.
	TmpPtr->ZoomedOffset.x = (BoxWidth - TmpPtr->ZoomedSize.x - TmpPtr->Scale) 
									/ 2;
	TmpPtr->ZoomedOffset.y = (BoxHeight - TmpPtr->ZoomedSize.y - TmpPtr->Scale) 
									/ 2;

	// Get the number of colors.  Put them in a list.  
	TmpPtr->NumColors = DIBNumColors(lpbi);
	if (TmpPtr->NumColors)
	{
		TmpPtr->IconColors = GetIconColors(lpbi, TmpPtr->NumColors);
	}

	TmpPtr->MButtonsSwapped = GetSystemMetrics(SM_SWAPBUTTON);
	if (TmpPtr->MButtonsSwapped)
	{
		TmpPtr->LColor = TmpPtr->IconColors[1];
		TmpPtr->RColor = TmpPtr->IconColors[0];
	}
	else
	{
		TmpPtr->LColor = TmpPtr->IconColors[0];
		TmpPtr->RColor = TmpPtr->IconColors[1];
	}

	GlobalUnlock(TmpPtr->hDIB);
}

/***************************************************************************
***************************************************************************/
void NEAR PASCAL
IconEditDestroy(EDITCUSTOMICON *TmpPtr)
{
	if (TmpPtr)
	{
		if (TmpPtr->IconColors)
			LocalFree((HANDLE)TmpPtr->IconColors);
		if (TmpPtr->ColorsUsed)
			LocalFree((HANDLE)TmpPtr->ColorsUsed);
		if (TmpPtr->hDIB)
			GlobalFree(TmpPtr->hDIB);
		TmpPtr->hDIB = NULL;
		TmpPtr->IconColors = NULL;
		TmpPtr->ColorsUsed = NULL;
	}
}

/***************************************************************************
***************************************************************************/
void NEAR PASCAL 
DrawActualBitmap(EDITCUSTOMICON *TmpPtr, HDC hdc)
{
	LPBITMAPINFO lpbi;
	
	if (TmpPtr->hDIB == 0)
		return;

	lpbi = (LPBITMAPINFO)GlobalLock(TmpPtr->hDIB);

	// Paint it.
	StretchDIBits(hdc, 
		TmpPtr->ActualOffset.x, TmpPtr->ActualOffset.y,
		(WORD)lpbi->bmiHeader.biWidth, (WORD)lpbi->bmiHeader.biHeight,
		0, 0, 
		(WORD)lpbi->bmiHeader.biWidth, (WORD)lpbi->bmiHeader.biHeight,
		(LPSTR)lpbi + (WORD)lpbi->bmiHeader.biSize + 
		DIBPaletteSize(lpbi),
		lpbi, DIB_RGB_COLORS, SRCCOPY);

	GlobalUnlock(TmpPtr->hDIB);

	DrawIconFrame(hdc,TmpPtr->ActualOffset,(int)lpbi->bmiHeader.biWidth,
						(int)lpbi->bmiHeader.biHeight,1);

}

/***************************************************************************
***************************************************************************/
void NEAR PASCAL 
DrawZoomedBitmap(EDITCUSTOMICON *TmpPtr,HDC hdc)
{
	HANDLE hPen, hOldPen;
	LPBITMAPINFO lpbi;
	RECT LogRect;
	
	if (TmpPtr->hDIB == 0)
		return;

	lpbi = (LPBITMAPINFO) GlobalLock(TmpPtr->hDIB);

	// First - Paint a black box as the boundary.
	hPen = CreatePen(PS_SOLID, 3, RGB(0,0,0));
	hOldPen = SelectObject(hdc, hPen);

	LogRect.left = TmpPtr->ZoomedOffset.x - TmpPtr->Scale;
	LogRect.right = LogRect.left + TmpPtr->ZoomedSize.x + (TmpPtr->Scale * 3);
	LogRect.top = TmpPtr->ZoomedOffset.y - TmpPtr->Scale;
	LogRect.bottom = LogRect.top + TmpPtr->ZoomedSize.y + (TmpPtr->Scale * 3);

	Rectangle(hdc, LogRect.left - 2, LogRect.top - 2, 
				LogRect.right + 2, LogRect.bottom + 2);

	SelectObject(hdc, hOldPen);
	DeleteObject(hPen);

	// Paint it.
	StretchDIBits(hdc, 
		TmpPtr->ZoomedOffset.x, TmpPtr->ZoomedOffset.y, 
		(WORD)TmpPtr->ZoomedSize.x, (WORD)TmpPtr->ZoomedSize.y,
		0, 0, 
		(WORD)lpbi->bmiHeader.biWidth, (WORD)lpbi->bmiHeader.biHeight,
		(LPSTR)lpbi + (WORD)lpbi->bmiHeader.biSize + 
		DIBPaletteSize(lpbi),
		lpbi, DIB_RGB_COLORS, SRCCOPY);

	GlobalUnlock(TmpPtr->hDIB);

	DrawIconFrame(hdc,TmpPtr->ZoomedOffset,TmpPtr->ZoomedSize.x,
						TmpPtr->ZoomedSize.y,TmpPtr->Scale);

}

/***************************************************************************
***************************************************************************/
WORD NEAR PASCAL DIBNumColors(void far *pv)
{
	int 		bits;
	LPBITMAPINFOHEADER	lpbi;
	LPBITMAPCOREHEADER	lpbc;

	lpbi = ((LPBITMAPINFOHEADER)pv);
	lpbc = ((LPBITMAPCOREHEADER)pv);

	/*	With the BITMAPINFO format headers, the size of the palette
	 *	is in biClrUsed, whereas in the BITMAPCORE - style headers, it
	 *	is dependent on the bits per pixel ( = 2 raised to the power of
	 *	bits/pixel).
	 */

	if(lpbi->biSize != sizeof(BITMAPCOREHEADER))
	{
		if (lpbi->biClrUsed != 0)
			return (WORD)lpbi->biClrUsed;
		bits = lpbi->biBitCount;
	}
	else
		bits = lpbc->bcBitCount;

	switch(bits)
	{
		case 1:
			return 2;
		case 4:
			return 16;
		case 8:
			return 256;
		default:
			/* A 24 bitcount DIB has no color table */
			return 0;
	}
}

/***************************************************************************
***************************************************************************/
/* get size of palette entries */
WORD NEAR PASCAL DIBPaletteSize(void far *pv)
{
	LPBITMAPINFOHEADER lpbi;
	WORD	       NumColors;

	lpbi      = (LPBITMAPINFOHEADER)pv;
	NumColors = DIBNumColors(lpbi);

	if(lpbi->biSize == sizeof(BITMAPCOREHEADER))
		return NumColors * sizeof(RGBTRIPLE);
	else
		return NumColors * sizeof(RGBQUAD);
}

/***************************************************************************
***************************************************************************/
HANDLE PASCAL 
DIBFromBitmap(HBITMAP hbm,DWORD biStyle,WORD biBits)
{
	BITMAP               bm;
	BITMAPINFOHEADER     bi;
	BITMAPINFOHEADER FAR *lpbi;
	DWORD                dwLen;
	HANDLE               hdib;
	HANDLE               h;
	HDC                  hdc;
	HPALETTE	hpal;

	if(!hbm)
		return NULL;

	hpal = GetStockObject(DEFAULT_PALETTE);

	GetObject(hbm,sizeof(bm),(LPSTR)&bm);

	if(biBits == 0)
		biBits =  bm.bmPlanes * bm.bmBitsPixel;

	bi.biSize = sizeof(BITMAPINFOHEADER);
	bi.biWidth = bm.bmWidth;
	bi.biHeight = bm.bmHeight;
	bi.biPlanes = 1;
	bi.biBitCount = biBits;
	bi.biCompression = biStyle;
	bi.biSizeImage = 0;
	bi.biXPelsPerMeter = 0;
	bi.biYPelsPerMeter = 0;
	bi.biClrUsed = 0;
	bi.biClrImportant = 0;

	dwLen  = bi.biSize + DIBPaletteSize(&bi);

	hdc = GetDC(NULL);
	hpal = SelectPalette(hdc,hpal,FALSE);
	RealizePalette(hdc);

	hdib = GlobalAlloc(GHND,dwLen);

	if(!hdib)
	{
		SelectPalette(hdc,hpal,FALSE);
		ReleaseDC(NULL,hdc);
		return NULL;
	}

	lpbi = (LPBITMAPINFOHEADER)GlobalLock(hdib);

	*lpbi = bi;

	/* call GetDIBits with a NULL lpBits param, so it will calculate the
	*  biSizeImage field for us
	*/

	GetDIBits(hdc,hbm,0,(WORD)bi.biHeight,NULL,
		(LPBITMAPINFO)lpbi, DIB_RGB_COLORS);

	bi = *lpbi;
	GlobalUnlock(hdib);

	/* If the driver did not fill in the biSizeImage field, make one up */

	if(bi.biSizeImage == 0)
	{
		bi.biSizeImage = WIDTHBYTES((DWORD)bm.bmWidth * biBits) * bm.bmHeight;

		if(biStyle != BI_RGB)
			bi.biSizeImage = (bi.biSizeImage * 3) / 2;
	}

	/* realloc the buffer big enough to hold all the bits */

	dwLen = bi.biSize + DIBPaletteSize(&bi) + bi.biSizeImage;
	if(h = GlobalReAlloc(hdib,dwLen,0))
		hdib = h;
	else
	{
		GlobalFree(hdib);
		hdib = NULL;

		SelectPalette(hdc,hpal,FALSE);
		ReleaseDC(NULL,hdc);
		return hdib;
	}

	/*	call GetDIBits with a NON-NULL lpBits param, and actualy get the
	*  bits this time
	*/

	lpbi = (LPBITMAPINFOHEADER)GlobalLock(hdib);

	if(GetDIBits(hdc,hbm,0,(WORD)bi.biHeight,
		(LPSTR)lpbi + (WORD)lpbi->biSize + DIBPaletteSize(lpbi),
		(LPBITMAPINFO)lpbi,DIB_RGB_COLORS) == 0)
	{
		GlobalUnlock(hdib);
		hdib = NULL;
		SelectPalette(hdc,hpal,FALSE);
		ReleaseDC(NULL,hdc);
		return NULL;
	}

	/* If the driver did not fill in the biSizeImage field, make one up */

	if(lpbi->biSizeImage == 0)
	{
		lpbi->biSizeImage = WIDTHBYTES((DWORD)lpbi->biWidth * 
					biBits) * lpbi->biHeight;

		if(biStyle != BI_RGB)
			lpbi->biSizeImage = (lpbi->biSizeImage * 3) / 2;
	}
	GlobalUnlock(hdib);

	SelectPalette(hdc,hpal,FALSE);
	ReleaseDC(NULL,hdc);
	return hdib;
}


/***************************************************************************
 * Allocate enough memory to hold all of the icon colors.   Keep them in
 * a list for easy retrieval.
 ***************************************************************************/
DWORD * NEAR PASCAL
GetIconColors(LPBITMAPINFO lpbi, WORD NumColors)
{
	DWORD *IconColors;
	DWORD *dwp;
	int i;
	RGBQUAD FAR * pQuad;

	IconColors = dwp = (DWORD *)LocalAlloc(LMEM_FIXED,
							NumColors * sizeof(DWORD));

	pQuad = (RGBQUAD FAR *)((BYTE FAR *)lpbi + lpbi->bmiHeader.biSize);
	for (i = 0; i < (int)NumColors; i++)
	{
		*dwp = RGB(pQuad->rgbRed, pQuad->rgbGreen, pQuad->rgbBlue);
		dwp++;
		pQuad++;
	}

	return IconColors;
}

/***************************************************************************
 * Change the bitmap to reflect the new color that we have added.
 ***************************************************************************/
void NEAR PASCAL
SetIconColors(LPBITMAPINFO lpbi, int index, DWORD color)
{
	RGBQUAD FAR * pQuad;

	pQuad = (RGBQUAD FAR *)((BYTE FAR *)lpbi + lpbi->bmiHeader.biSize);
	pQuad += index;
	pQuad->rgbBlue = (BYTE)((color & 0x00FF0000) >> 16);
	pQuad->rgbGreen = (BYTE)((color & 0x0000FF00) >> 8);
	pQuad->rgbRed = (BYTE)(color & 0x000000FF);
}


/***************************************************************************
 * Search through the Icon Colors to find one that matches the color
 * passed in. Return the index into the IconColors array.
 ***************************************************************************/
BYTE NEAR PASCAL
GetColorIndex(LPBITMAPINFO lpbi, DWORD color, DWORD *IconColors, WORD NumColors,
				BOOL *ColorsUsed)
{
	BOOL found = FALSE;
	DWORD FAR *dwp;
	int i = 0;

	dwp = IconColors;
	while ((i < (int)NumColors) && !found)
	{
		if (color == *dwp)
			found = TRUE;

		dwp++;
		i++;
	}

	// If the color is not found, lets find an unused space and set it.
	if (!found && ColorsUsed)
		SetColorIndex(lpbi, color, NumColors, IconColors, ColorsUsed);

	return (BYTE)(i - 1);
}

/***************************************************************************
 * We did not find the color we were looking for.  Let's see if there is
 * room to add it.
 ***************************************************************************/
void NEAR PASCAL
SetColorIndex(LPBITMAPINFO lpbi, DWORD color, int NumColors, DWORD *IconColors, 
					BOOL *ColorsUsed)
{
	BOOL found;
	int i;

	// look for a black one.
	i = 0;
	found = FALSE;
	while (!found && i < NumColors)
	{
		if (IconColors[i] == 0)
			found = TRUE;
		i++;
	}

	// look for a duplicate black one.
	found = FALSE;
	while (!found && i < NumColors)
	{
		if ((IconColors[i] == 0) && !ColorsUsed[i])
			found = TRUE;
		else 
			i++;
	}

	if (found)
	{
		// We found a space to put the new color, reflect it in our array.
		IconColors[i] = color;

		// Write the change to the bitmap.
		SetIconColors(lpbi, i, color);
	}
}

/***************************************************************************
 * This is for monochrome bitmaps.
 ***************************************************************************/
void NEAR PASCAL
SetBit(LPBITMAPINFO lpbi, DWORD *IconColors, DWORD color, WORD NumColors, 
			int x, int y)
{
	int bite, pixel, width;
	BYTE mask, bit;
	BYTE FAR *lpBits;
	BYTE FAR *pBite;
	BYTE ColorIndex;

	// The bits are stored differently in lpbi.  The lower left corner is
	// (0, 0).  So adjust y before the calculations.
	y = (int)lpbi->bmiHeader.biHeight - y - 1;

	// Calculate the real width in pixels.
	width = 
		((int)lpbi->bmiHeader.biSizeImage / (int)lpbi->bmiHeader.biHeight) * 8;

	pixel = y * width + x;		// which pixel
	bite = pixel / 8;			// which byte
	bit = (BYTE) (pixel % 8);	// which bit

	// Now that we have figured out which nibble was affected.  Change it.
	// Find the index into the color table that will go in the nibble.
	ColorIndex = GetColorIndex(lpbi, color, IconColors, NumColors, 
							(BOOL *)NULL);

	// First get a pointer into the DIB.
	lpBits = (BYTE FAR *)lpbi + (WORD)lpbi->bmiHeader.biSize 
										+ DIBPaletteSize(lpbi);
	pBite = lpBits + bite;

	switch (bit)
	{
		case 0:	// the first bit in the byte
			mask = 0x7F;
			break;
		case 1:	// the second bit in the byte
			mask = 0xBF;
			break;
		case 2:
			mask = 0xDF;
			break;
		case 3:
			mask = 0xEF;
			break;
		case 4:
			mask = 0xF7;
			break;
		case 5:
			mask = 0xFB;
			break;
		case 6:
			mask = 0xFD;
			break;
		case 7:
			mask = 0xFE;
			break;
	}
	*pBite = (BYTE)(*pBite & mask) + (BYTE)(ColorIndex << (7 - bit));
	return;
}

/***************************************************************************
 * This is for 4 bit per pixel bitmaps.  That is 16 colors.
 ***************************************************************************/
void NEAR PASCAL
SetNibble(LPBITMAPINFO lpbi, DWORD *IconColors, DWORD color, WORD NumColors, 
				int x, int y, BOOL *ColorsUsed)
{
	int bite, nibble, pixel, width;
	BYTE FAR *lpBits;
	BYTE FAR *pBite;
	BYTE ColorIndex;

	// The bits are stored differently in lpbi.  The lower left corner is
	// (0, 0).  So adjust y before the calculations.
	y = (int)lpbi->bmiHeader.biHeight - y - 1;

	// Calculate the real width in pixels.
	width = 
		((int)lpbi->bmiHeader.biSizeImage / (int)lpbi->bmiHeader.biHeight) * 2;

	pixel = y * width + x;	// which pixel
	bite = pixel / 2;								// which byte
	nibble = pixel % 2;								// which nibble

	// Now that we have figured out which nibble was affected.  Change it.
	// Find the index into the color table that will go in the nibble.
	ColorIndex = GetColorIndex(lpbi, color, IconColors, NumColors, ColorsUsed);

	// First get a pointer into the DIB.
	lpBits = (BYTE FAR *)lpbi + (WORD)lpbi->bmiHeader.biSize 
										+ DIBPaletteSize(lpbi);
	pBite = lpBits + bite;
	if (nibble == 0)
		*pBite = (BYTE)(*pBite & 0x0F) + (BYTE)(ColorIndex << 4);
	else
		*pBite = (BYTE)(*pBite & 0xF0) + (BYTE)ColorIndex;

	return;
}

/***************************************************************************
 * This is for 8 bits per pixel bitmaps.  That is 256 colors.
 ***************************************************************************/
void NEAR PASCAL
SetByte(LPBITMAPINFO lpbi, DWORD *IconColors, DWORD color, WORD NumColors, 
			int x, int y, BOOL *ColorsUsed)
{
	int bite, pixel, width;
	BYTE FAR *lpBits;
	BYTE FAR *pBite;
	BYTE ColorIndex;

	// The bits are stored differently in lpbi.  The lower left corner is
	// (0, 0).  So adjust y before the calculations.
	y = (int)lpbi->bmiHeader.biHeight - y - 1;

	// Calculate the real width in pixels.
	width = 
		((int)lpbi->bmiHeader.biSizeImage / (int)lpbi->bmiHeader.biHeight);

	bite = pixel = y * width + x;	// which pixel

	// Now that we have figured out which byte was affected.  Change it.
	// Find the index into the color table that will go in the nibble.
	ColorIndex = GetColorIndex(lpbi, color, IconColors, NumColors, ColorsUsed);

	// First get a pointer into the DIB.
	lpBits = (BYTE FAR *)lpbi + (WORD)lpbi->bmiHeader.biSize 
										+ DIBPaletteSize(lpbi);
	pBite = lpBits + bite;
	*pBite = (BYTE)ColorIndex;

	return;
}

/***************************************************************************
 * This is for 24 bit per pixel bitmaps.
 ***************************************************************************/
void NEAR PASCAL
SetDWord(LPBITMAPINFO lpbi, DWORD color, int x, int y)
{
	int bite, pixel, width;
	BYTE FAR *lpBits;
	BYTE FAR *pBite;

	// The bits are stored differently in lpbi.  The lower left corner is
	// (0, 0).  So adjust y before the calculations.
	y = (int)lpbi->bmiHeader.biHeight - y - 1;

	// Calculate the real width in pixels.
	width = 
		((int)lpbi->bmiHeader.biSizeImage / (int)lpbi->bmiHeader.biHeight);

	bite = pixel = (y * width) + (x * 3);	// which pixel

	// First get a pointer into the DIB.
	lpBits = (BYTE FAR *)lpbi + (WORD)lpbi->bmiHeader.biSize 
										+ DIBPaletteSize(lpbi);
	pBite = lpBits + bite;
	*pBite = (BYTE)((color & 0x00FF0000) >> 16);
	*(pBite+1) = (BYTE)((color & 0x0000FF00) >> 8);
	*(pBite+2) = (BYTE)(color & 0x000000FF);

	return;
}

/***************************************************************************
 ***************************************************************************/
BOOL NEAR PASCAL
WhichPixelClicked(EDITCUSTOMICON *TmpPtr, HWND hWnd, int x, int y, 
						WORD WhichButton)
{
	HDC hDC;
	int i, j;
	LPBITMAPINFO lpbi;
	DWORD Color;

	if (TmpPtr->hDIB == 0)
		return FALSE;

	if ((x < TmpPtr->ZoomedOffset.x) 
		|| ( x >= (TmpPtr->ZoomedOffset.x + TmpPtr->ZoomedSize.x))
		|| (y < TmpPtr->ZoomedOffset.y) 
		|| ( y >= (TmpPtr->ZoomedOffset.y + TmpPtr->ZoomedSize.y)))
	{
		// Mouse was not in the editable part of the window.
		// Notice they are not able to edit the first row or column or
		// the last two rows or columns.
		return FALSE;
	}

	// The click was in the editable part of the bitmap. 
	// Figure out which "pixel" was clicked.

	// Subtract the offset from x and y to make them map to the top of the icon
	x = x - TmpPtr->ZoomedOffset.x;
	y = y - TmpPtr->ZoomedOffset.y;

	// Dived by the scale to get the pixel.
	x = x / TmpPtr->Scale;
	y = y / TmpPtr->Scale;

	if (TmpPtr->MButtonsSwapped)
		WhichButton = !WhichButton;

	if (WhichButton)
		Color = TmpPtr->RColor;
	else
		Color = TmpPtr->LColor;

	// Update the DIB in memory.
	lpbi = (LPBITMAPINFO)GlobalLock(TmpPtr->hDIB);
	switch (lpbi->bmiHeader.biBitCount)
	{
		case 1:
			SetBit(lpbi, TmpPtr->IconColors, Color, TmpPtr->NumColors, x, y);
			break;
		case 4:
			SetNibble(lpbi, TmpPtr->IconColors, Color, TmpPtr->NumColors, x, y,
					 TmpPtr->ColorsUsed);
			break;
		case 8:
			SetByte(lpbi, TmpPtr->IconColors, Color, TmpPtr->NumColors, x, y, 
					TmpPtr->ColorsUsed);
			break;
		case 24:
			SetDWord(lpbi, Color, x, y);
			break;
	}
	GlobalUnlock(TmpPtr->hDIB);

	// Update the picture on the screen.
	hDC = GetDC(hWnd);
	x = x * TmpPtr->Scale + TmpPtr->ZoomedOffset.x;
	y = y * TmpPtr->Scale + TmpPtr->ZoomedOffset.y;
	for (i = x; i < x + TmpPtr->Scale; i++)
	{
		for (j = y; j < y + TmpPtr->Scale; j++)
			SetPixel(hDC, i, j, Color);
	}

	ReleaseDC(hWnd, hDC);

	TmpPtr->CurrentDIBChanged = TRUE;

	return TRUE;
}

/***************************************************************************
***************************************************************************/
LONG FAR PASCAL 
IconControlProc(HWND hwnd, unsigned int message, unsigned int wParam, LONG lparam)
{
	HDC hdc;
	HWND hCtl;
	PAINTSTRUCT ps;
	WORD id;
	MainWindow *pWindow;

	switch (message) 
	{
		case WM_PAINT:
			pWindow = (MainWindow *)GetPointer(GetParent(GetParent(hwnd)));
			hdc = BeginPaint(hwnd, (LPPAINTSTRUCT)&ps);
			id = GetWindowWord(hwnd, GWW_ID);
			if (id == ID_STATIC2)
				DrawZoomedBitmap(pWindow->GetTmpPtr(), hdc);
			else if (id == ID_STATIC3)
				DrawActualBitmap(pWindow->GetTmpPtr(), hdc);
			else if (id == ID_STATIC4)
			{
				DrawMouseBitmap(hdc);
				DrawMouseButtons(pWindow->GetTmpPtr(), hdc);
			}
			EndPaint(hwnd, (LPPAINTSTRUCT)&ps);
			break;
		case WM_LBUTTONDOWN:
		case WM_RBUTTONDOWN:
			SetCapture(hwnd);	
			id = GetWindowWord(hwnd, GWW_ID);
			if (id == ID_STATIC2)
			{
				pWindow = (MainWindow *)GetPointer(GetParent(GetParent(hwnd)));
				if (WhichPixelClicked(pWindow->GetTmpPtr(), hwnd,LOWORD(lparam),
						HIWORD(lparam), message == WM_LBUTTONDOWN ? 0 : 1))
				{
					// Update the little one
					hCtl = GetDlgItem(GetParent(hwnd), ID_STATIC3);
					InvalidateRect(hCtl, NULL, FALSE);
				}
			}
			break;

		case WM_LBUTTONUP:
		case WM_RBUTTONUP:
			ReleaseCapture();	
			break;

		case WM_MOUSEMOVE:
			if ((GetCapture() == hwnd) 
				&& (wParam & MK_LBUTTON || wParam & MK_RBUTTON))
			{
				id = GetWindowWord(hwnd, GWW_ID);
				if (id == ID_STATIC2)
				{
					pWindow = (MainWindow *)GetPointer(GetParent(
										GetParent(hwnd)));
					if (WhichPixelClicked(pWindow->GetTmpPtr(), hwnd, 
							LOWORD(lparam), HIWORD(lparam), 
							wParam == MK_LBUTTON ? 0 : 1))
					{
						// Update the little one
						hCtl = GetDlgItem(GetParent(hwnd), ID_STATIC3);
						InvalidateRect(hCtl, NULL, FALSE);
					}
				}
			}
		default:
			return DefWindowProc(hwnd,message,wParam,lparam);
	}
	return (LONG)0;
}

/***************************************************************************
***************************************************************************/
void NEAR PASCAL
DrawIconFrame(HDC hdc,POINT Offset,int Width,int Height,int Scale)
{
	RECT rect;
	HPEN GrayPen,WhitePen,OldPen;
	int i;

	rect.left = Offset.x;
	rect.right = Offset.x + Width ;
	rect.top = Offset.y;
	rect.bottom = Offset.y + Height ;

	GrayPen = CreatePen(PS_SOLID,0,RGB(128,128,128));
	OldPen = SelectObject(hdc,GrayPen);
	for (i = 0; i < Scale; i++)
	{
		MoveTo(hdc,rect.right + i, rect.top);
		LineTo(hdc,rect.right + i,rect.bottom + i);
		LineTo(hdc,rect.left - 1,rect.bottom + i);
	}

	for (i = 0; i < Scale; i++)
	{
		MoveTo(hdc,rect.right + Scale + i,rect.top - Scale);
		LineTo(hdc,rect.right + Scale + i,rect.bottom + Scale + i);
		LineTo(hdc,rect.left - Scale - 1,rect.bottom + Scale + i);
	}

	WhitePen = GetStockObject(WHITE_PEN);
	SelectObject(hdc,WhitePen);
	DeleteObject(GrayPen);

	for (i = 0; i < Scale; i++)
	{
		MoveTo(hdc,rect.left - Scale + i,rect.bottom);
		LineTo(hdc,rect.left - Scale + i,rect.top - Scale + i);
		LineTo(hdc,rect.right + 1,rect.top - Scale + i);
	}

	SelectObject(hdc,OldPen);
}

/***************************************************************************
***************************************************************************/
void NEAR PASCAL
DrawMouseBitmap(HDC hDC)
{
	BITMAP Bitmap;
	HANDLE hBitmap, hOldBitmap;
	HDC hMemDC;

	hBitmap = LoadBitmap(Main::hInstance, "RS_BMP_MOUSE");
	if (!hBitmap)
		return;

	hMemDC = CreateCompatibleDC(hDC); 		// Create compatible memory dc.
	hOldBitmap = SelectObject(hMemDC, hBitmap); // Select the bitmap into dc
	
	GetObject(hBitmap, sizeof(BITMAP), (LPSTR)&Bitmap);	// Get bitmap info.

	// Paint it.
	BitBlt(hDC, 0, 0, Bitmap.bmWidth, Bitmap.bmHeight, hMemDC, 0, 0, SRCCOPY);

	SelectObject(hMemDC, hOldBitmap);	// restore the old bitmap
	DeleteObject(hBitmap);					// delete the bitmap
	DeleteDC(hMemDC);					// delete the dc
}

/***************************************************************************
***************************************************************************/
void NEAR PASCAL
DrawMouseButtons(EDITCUSTOMICON *TmpPtr, HDC hDC)
{
	int i, j;

	if (TmpPtr->hDIB == 0)
		return;

	// Show mouse left button with the left color
	for (i = 8; i < 19; i++)
		for (j = 11; j < 19; j++)
			SetPixel(hDC, i, j, TmpPtr->LColor);

	// Show mouse right button with the right color
	for (i = 20; i < 30; i++)
		for (j = 11; j < 19; j++)
			SetPixel(hDC, i, j, TmpPtr->RColor);
}

/***************************************************************************
***************************************************************************/
HANDLE PASCAL
CreateBorderLessDIB(HBITMAP hBitmap)
{
	HBITMAP NoBorderBitmap,Old1,Old2;
	HDC MemDC1,MemDC2,ScreenDC;
	BITMAP bm;
	HANDLE hDIB;

	GetObject(hBitmap,sizeof(BITMAP),(LPSTR)&bm);
	if (!(ScreenDC = GetDC(NULL)))
		return NULL;
	if (!(MemDC1 = CreateCompatibleDC(ScreenDC)))
	{
		ReleaseDC(NULL,ScreenDC);
		return NULL;
	}
	if (!(NoBorderBitmap = CreateCompatibleBitmap(ScreenDC,bm.bmWidth - 3, 
							bm.bmHeight - 3)))
	{
		ReleaseDC(NULL,ScreenDC);
		DeleteDC(MemDC1);
		return NULL;
	}
	ReleaseDC(NULL,ScreenDC);
	if (!(Old1 = SelectObject(MemDC1,NoBorderBitmap)))
	{
		DeleteDC(MemDC1);
		return NULL;
	}
	if (!(MemDC2 = CreateCompatibleDC(MemDC1)))
	{
		SelectObject(MemDC1,Old1);
		DeleteDC(MemDC1);
		return NULL;
	}
	if (!(Old2 = SelectObject(MemDC2,hBitmap)))
	{
		SelectObject(MemDC1,Old1);
		DeleteDC(MemDC1);
		DeleteDC(MemDC2);
		return NULL;
	}
	BitBlt(MemDC1,0,0,bm.bmWidth - 3,bm.bmHeight - 3,MemDC2,1,1,SRCCOPY);
	SelectObject(MemDC2,Old2);
	DeleteDC(MemDC2);
	SelectObject(MemDC1,Old1);
	DeleteDC(MemDC1);

	// Convert the bitmap into a Device Independent Bitmap.
	hDIB = DIBFromBitmap(NoBorderBitmap, BI_RGB, 0);
	DeleteObject(NoBorderBitmap);
	return hDIB;
}

