
/* Standard compiler headers */

#include <stdio.h>
#include <math.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

/* AXP header included with the IPAS toolkit */
#include "axp.h"

/* SHIELD header file */
#include "shield.h"

/* Defines for the dialog box variables */

#define CENX   1
#define CENY   2
#define CENZ   3
#define FR1    4
#define FR2    5
#define FR3    6
#define FR4    7
#define DN1    8
#define DN2    9
#define DN3    10
#define DN4    11

/* The dialog box for the energy shield settings
   This sets up the whole dialog box with the layout being done
   by 3D Studio. See the IPAS Toolkit guide for the syntax. */

DlgEntry cdialog[]={
   0,"TITLE=\"3D Studio Special Effects\"",
   0,"TITLE=\"Keith A. Seifert\"",
   0,"TITLE=\"Energy Shield\"",
   0,"TITLE=\"Version 1.0\"",
   0,"TITLE=\"\"",
   CENX,"LFLOAT=\"Center X:\",-1.0,1.0,9",
   CENY,"LFLOAT=\"       Y:\",-1.0,1.0,9",
   CENZ,"LFLOAT=\"       Z:\",-1.0,1.0,9",
   0,"TITLE=\"\"",
   0,"TITLE=\"        Point 1    Point 2    Point 3    Point 4\"",
   FR1,"LINT=\" Frames:\",0,999999,7",
   FR2,"-LINT=\"\",0,999999,7",
   FR3,"-LINT=\"\",0,999999,7",
   FR4,"-LINT=\"\",0,999999,7",
   DN1,"LFLOAT=\"Density:\",0.0,2.0,7",
   DN2,"-LFLOAT=\"\",0.0,1.0,7",
   DN3,"-LFLOAT=\"\",0.0,1.0,7",
   DN4,"-LFLOAT=\"\",0.0,1.0,7",
   0,"TITLE=\"\"",
   0,NULL
};

/* Each IPAS process needs a unique version number */

#define VERSION 0xAB02

/* The state structure contains all the variables which will be
   saved and restored by 3D Studio */

typedef struct {
   ulong version;
   float cenx,ceny,cenz;
   int fr1,fr2,fr3,fr4;
   float dn1,dn2,dn3,dn4;
} State;

/* The default settings for the state structure */

static State init_state = { VERSION,0.5,0.5,0.5,0,20,40,60,0.0,
   0.5,0.75,0.0 };
static State state = { VERSION,0.5,0.5,0.5,0,20,40,60,0.0,0.5,
   0.75,0.0 };


/* The standard function for setting the variables from the
   dialog box. All AXP routines need this function */

void ClientSetStateVar(int id, void *ptr) {
   OVL o;
   ulong *ul;

   ul=(ulong *)ptr;
   o.ul = *ul;
   switch(id) {
      case CENX:
         state.cenx = o.f;
         break;
      case CENY:
         state.ceny = o.f;
         break;
      case CENZ:
         state.cenz = o.f;
         break;
      case FR1:
         state.fr1 = o.i;
         break;
      case FR2:
         state.fr2 = o.i;
         break;
      case FR3:
         state.fr3 = o.i;
         break;
      case FR4:
         state.fr4 = o.i;
         break;
      case DN1:
         state.dn1 = o.f;
         break;
      case DN2:
         state.dn2 = o.f;
         break;
      case DN3:
         state.dn3 = o.f;
         break;
      case DN4:
         state.dn4 = o.f;
         break;
   }
}


/* The standard function for getting the variables for the
   dialog box. All AXP routines need this function */

ulong ClientGetStateVar(int id) {
   OVL o;
   switch(id) {
      case CENX:
         o.f = state.cenx;
         break;
      case CENY:
         o.f = state.ceny;
         break;
      case CENZ:
         o.f = state.cenz;
         break;
      case FR1:
         o.i = state.fr1;
         break;
      case FR2:
         o.i = state.fr2;
         break;
      case FR3:
         o.i = state.fr3;
         break;
      case FR4:
         o.i = state.fr4;
         break;
      case DN1:
         o.f = state.dn1;
         break;
      case DN2:
         o.f = state.dn2;
         break;
      case DN3:
         o.f = state.dn3;
         break;
      case DN4:
         o.f = state.dn4;
         break;
   }
   return(o.ul);
}


/* The standard function for setting the variable size.
   All AXP routines need this function */

int ClientVarSize(int id) {
   return(1);
}


/* The standard function for getting the size of the state
   structure. All AXP routines need this function */

char  *ClientGetState(int *size) {
   *size = sizeof(State);
   return((char *)&state);
}


/* The standard function for resetting the state structure
   with the default setup. All AXP routines need this function */

void ClientResetState() {
   state = init_state;
}


/* The standard function for starting the external process.
   At this time all the global setup should be done. At this
   point the external process begins to control the flow of
   the process. All AXP routines need this function */

void ClientStartup(EXPbuf *buf) {

      /* Get the current 3D Studio version */

   version3d = studio_version();

      /* Get the current frame */

   if( buf->data.object.curfield & 1 )
      frame = (buf->data.object.curfield-1) / 2;
   else
      frame = (buf->data.object.curfield) / 2;

      /* Setup the frame points */

   if( state.fr2 <= state.fr1 )
      state.fr2 = state.fr1 + 1;
   if( state.fr3 <= state.fr2 )
      state.fr3 = state.fr2 + 1;
   if( state.fr4 <= state.fr3 )
      state.fr4 = state.fr3 + 1;

      /* Get the number vertices in the current object */

   obj.nvertex = buf->data.object.verts;
   obj.nface = buf->data.object.faces;

      /* Allocate space to store the object */

   obj.vt = (VData*)malloc( obj.nvertex * sizeof( VData ) );
   obj.fc = (FData*)malloc( obj.nface * sizeof( FData ) );

   if( !obj.vt || !obj.fc ) {
      /* Not enough memory */

      ClientTerminate();
      buf->opcode = buf->usercode = EXP_TERMINATE;
      buf->status = 0;
   }
   else {
      /* Initialize global variables */

      first = 1;
      ix = 0;

      /* Setup the next event */

      buf->data.vert.vertex = ix;
      buf->opcode = EXP_GET_VERTEX;
      buf->usercode = 0x0200;
      buf->status = 1;
   }
}


/* The standard function for the whole external process.
   The flow between 3D Studio and the external process all
   happens within this structure.
   All AXP routines need this function */

void ClientUserCode(EXPbuf *buf) {

   /* The usercode value controls what happens next. It is set
      by the external process on the previous call to
      3D studio */

   switch(buf->usercode) {
      case 0x0200:

         /* For each vertex find the min and max at each axis */

         if( first ) {
            minx = maxx = buf->data.vert.x;
            miny = maxy = buf->data.vert.y;
            minz = maxz = buf->data.vert.z;
            first = 0;
         }
         else {
            if( buf->data.vert.x < minx )
               minx = buf->data.vert.x;
            else if( buf->data.vert.x > maxx )
               maxx = buf->data.vert.x;
            if( buf->data.vert.y < miny )
               miny = buf->data.vert.y;
            else if( buf->data.vert.y > maxy )
               maxy = buf->data.vert.y;
            if( buf->data.vert.z < minz )
               minz = buf->data.vert.z;
            else if( buf->data.vert.z > maxz )
               maxz = buf->data.vert.z;
         }

         /* Save the vertex */

         obj.vt[ix].x = buf->data.vert.x;
         obj.vt[ix].y = buf->data.vert.y;
         obj.vt[ix].z = buf->data.vert.z;

         /* Do not need the texture coordinates */

         obj.vt[ix].u = 0.0;
         obj.vt[ix].v = 0.0;

         /* Increament vertex count */

         ix++;
         if( ix < obj.nvertex ) {

            /* Get next vertex */

            buf->data.vert.vertex = ix;
            buf->opcode = EXP_GET_VERTEX;
            buf->usercode = 0x0200;
            buf->status = 1;
            break;
         }

         if( buf->status == 0 ) {
            /* Some kind of problem in 3D Studio */

            terminate:
            buf->opcode=buf->usercode = EXP_TERMINATE;
            break;
         }

         /* Find X axis range */

         dx = maxx - minx;

         /* Now get faces */

         ix = 0;
         buf->opcode = EXP_GET_FACE;
         buf->usercode = 0x0210;
         buf->data.face.face = ix;
         break;

      case 0x0210:
         /* Pick up material to apply later */

         if( !ix )
            material = buf->data.face.material;

         /* Save the face data */

         obj.fc[ix].a = buf->data.face.a;
         obj.fc[ix].b = buf->data.face.b;
         obj.fc[ix].c = buf->data.face.c;
         obj.fc[ix].material = material;
         obj.fc[ix].flags = FC_CALINE | FC_ABLINE | FC_BCLINE;
         obj.fc[ix].sm_group = buf->data.face.sm_group;

         /* Increament face count */

         ix++;
         if( ix < obj.nface ) {

            /* Get next face */

            buf->data.face.face = ix;
            buf->opcode = EXP_GET_FACE;
            buf->usercode = 0x0210;
            buf->status = 1;
            break;
         }

         /* Now create the energy shield effect */

         if( ShieldShift() ) {

            /* Error returned */

            ClientTerminate();
            buf->opcode = buf->usercode = EXP_TERMINATE;
            buf->status = 1;
            break;
         }

         buf->data.object.verts = obj.nvertex;
         buf->data.object.faces = obj.nface;
         buf->data.object.tverts = obj.nvertex;

         /* Ready Object to be created */

         buf->opcode = EXP_READY_OBJ;

         /* Use single mode if version 2.0
            Use burst mode if version 2.01 or higher */

         if( version3d > 200 )
            buf->usercode = 0x0260;
         else
            buf->usercode = 0x0240;
         buf->status = 1;
         break;

      case 0x0240:

         /* Single vertex mode */

         vertex = 0;
         if( buf->status == 0 )
            goto terminate;

      case 0x0245:

         /* For each new vertex send the vertex data to Studio */

         vertex_loop:
         if( vertex >= obj.nvertex )
            goto face_loop;
         buf->opcode = EXP_PUT_VERTEX;
         buf->usercode = 0x0245;
         buf->data.vert.x = obj.vt[vertex].x;
         buf->data.vert.y = obj.vt[vertex].y;
         buf->data.vert.z = obj.vt[vertex].z;
         buf->data.vert.u = obj.vt[vertex].u;
         buf->data.vert.v = obj.vt[vertex].v;
         buf->data.vert.vertex = vertex++;
         break;

      case 0x0250:

         /* Single face mode */

         face_loop:
         face = 0;

      case 0x0255:
         if( face >= obj.nface ) {

            /* Finished with faces now create object */

            buf->opcode = EXP_CREATE_OBJ;
            buf->usercode = EXP_TERMINATE;
            ClientTerminate();
         }
         else {

            /* For each face send Studio the face data */

            buf->opcode = EXP_PUT_FACE;
            buf->data.face.sm_group = obj.fc[face].sm_group;
            buf->data.face.material = material;
            buf->data.face.a = obj.fc[face].a;
            buf->data.face.b = obj.fc[face].b;
            buf->data.face.c = obj.fc[face].c;
            buf->data.face.face = face++;
            buf->data.face.flags =
               FC_CALINE | FC_ABLINE | FC_BCLINE;

            buf->usercode = 0x0255;
         }
         buf->status = 1;
         break;

      case 0x0260:

         /* Burst Mode sends the whole vertex list */

         buf->opcode = EXP_PUT_VLIST;
         buf->usercode = 0x0265;
         buf->data.vlist.start = 0;
         buf->data.vlist.count = obj.nvertex;
         buf->data.vlist.data = (_far VData *)obj.vt;
         buf->status = 1;
         break;

      case 0x0265:

         /* Burst mode sends the whole face list */

         buf->opcode = EXP_PUT_FLIST;
         buf->usercode = 0x0270;
         buf->data.flist.start = 0;
         buf->data.flist.count = obj.nface;
         buf->data.flist.data = (_far FData *)obj.fc;
         buf->status = 1;
         break;

      case 0x0270:

         /* Create the object in Studio */

         buf->data.object.flags = (uchar)0;
         buf->opcode = EXP_CREATE_OBJ;
         buf->usercode = EXP_TERMINATE;
         buf->status = 1;
         break;

         default:
         error("Invalid user opcode",NULL,NULL,NULL,NULL);
         buf->opcode = buf->usercode = EXP_TERMINATE;
         buf->status = 0;
   }
}


/* The standard function for terminating. The external function
   should free all allocated data here. 3D Studio will call this
   function when unloading the process. Version 3.0 will not
   unload the process until 3D Studio is exited so memory
   may be need to be freed prior to unloading.
   All AXP routines need this function */

void ClientTerminate(void) {
   if( obj.vt ) { free( obj.vt ); obj.vt = NULL; }
   if( obj.fc ) { free( obj.fc ); obj.fc = NULL; }
}


/* The standard function for returning the number lines in the
   dialog box. All AXP routines need this function */

DlgEntry *ClientDialog(int n) {
   return(&cdialog[n]);
}


/* The standard function for returning the name of process.
   All AXP routines need this function */

char *ClientName(void) {
   return("SHIELD.AXP");
}


/* The external process code itself */

/* The energy shield code changes the texture coordinates over
   the animation to animate the material on an object */

int ShieldShift(void) {
   double hmap,d1,az,ax,px,py,pz;
   int i;
   matx rmat;

   /* Find the maximum mapped value */

   if( frame < state.fr1 )
      hmap = (double)state.dn1;
   else if( frame < state.fr2 )
      hmap = (double)(frame - state.fr1) /
         (double)(state.fr2 - state.fr1) *
         (state.dn2 - state.dn1) + state.dn1;
   else if( frame < state.fr3 )
      hmap = (double)(frame - state.fr2) /
         (double)(state.fr3 - state.fr2) *
         (state.dn3 - state.dn2) + state.dn2;
   else if( frame < state.fr4 )
      hmap = (double)(frame - state.fr3) /
         (double)(state.fr4 - state.fr3) *
         (state.dn4 - state.dn3) + state.dn3;
   else
      hmap = (double)state.dn4;

   /* If maximum mapped value is 0 then no object is created */

   if( !hmap )
      return 1;

   /* Setup rotations to bring the start point to the X axis */

   d1 = sqrt((double)state.cenx*(double)state.cenx+
      (double)state.ceny*(double)state.ceny);
   if( !d1 ) {
      if( state.cenz > 0.0 )
         az = -RD_90;
      else if( state.cenz < 0.0 )
         az = RD_90;
      else
         az = 0.0;
   }
   else
      az = atan((double)-state.cenz/d1);
   if( !state.cenx ) {
      if( state.ceny > 0.0 )
         ax = RD_90;
      else if( state.ceny < 0.0 )
         ax = -RD_90;
      else
         ax = 0.0;
   }
   else
      ax = atan((double)state.ceny/(double)state.cenx);

   /* Setup the matrix */

   memmove(rmat,idMatx,12*sizeof(double));

   /* Apply the rotations to the matrix */

   if( ax )
      Zrotation(rmat,-ax);
   if( az )
      Yrotation(rmat,-az);

   /* For each point transform the point to along the X axis.
      Calculate the mapping coordinate based on location
      along the X axis. */

   for( i=0;i<obj.nvertex;i++ ) {
      TransformPoint( rmat,(double)(obj.vt[i].x),
         (double)(obj.vt[i].y),
         (double)(obj.vt[i].z),&px,&py,&pz );
      px -= minx;
      if( px / dx < hmap ) {
         obj.vt[i].u  = 1.0 - (px / (dx * hmap));
         obj.vt[i].v = fabs(py * 2.0);
      }
      else
         obj.vt[i].u = 0.0;
   }

   /* Remove all faces which have all vertices at or less then
      0. */

   for( i=0;i<obj.nface;i++ ) {
      if( obj.vt[obj.fc[i].a].u == 0.0 &&
          obj.vt[obj.fc[i].b].u == 0.0 &&
          obj.vt[obj.fc[i].c].u == 0.0 ) {
         if( i < obj.nface - 1 ) {
            obj.fc[i].a = obj.fc[obj.nface-1].a;
            obj.fc[i].b = obj.fc[obj.nface-1].b;
            obj.fc[i].c = obj.fc[obj.nface-1].c;
            obj.fc[i].material = obj.fc[obj.nface-1].material;
            obj.fc[i].flags = obj.fc[obj.nface-1].flags;
            obj.fc[i].sm_group = obj.fc[obj.nface-1].sm_group;
         }
         obj.nface--;
         i--;
      }
   }

   /* Normalize texture coordinates to 0.02 - 0.98
      This allows for filtering of the material. */

   for( i=0;i<obj.nvertex;i++ )
      obj.vt[i].u = obj.vt[i].u * 0.96 + 0.02;

   return 0;
}


/* Standard function to add rotation around the Z axis to a
   matrix */

void Zrotation( matx tmat,double angle ) {
   double tmp,s,c,*t;
   int i;

   s = sin(angle);
   c = cos(angle);
   for( i=0;i<4;i++ ) {
      t = tmat[i];
      tmp = t[0]*c - t[1]*s;
      t[1] = t[0]*s + t[1]*c;
      t[0] = tmp;
   }
}

/* Standard function to add rotation around the Y axis to a
   matrix */

void Yrotation( matx tmat,double angle ) {
   double tmp,s,c,*t;
   int i;

   s = sin(angle);
   c = cos(angle);
   for( i=0;i<4;i++ ) {
      t = tmat[i];
      tmp = t[0]*c - t[2]*s;
      t[2] = t[0]*s + t[2]*c;
      t[0] = tmp;
   }
}


/* Standard function to use the matrix to tranform a point */

void TransformPoint( matx tm,double ix,double iy,double iz,
      double *ox,double *oy,double *oz ) {
   *ox = ix*tm[0][0] + iy*tm[1][0] + iz*tm[2][0] + tm[3][0];
   *oy = ix*tm[0][1] + iy*tm[1][1] + iz*tm[2][1] + tm[3][1];
   *oz = ix*tm[0][2] + iy*tm[1][2] + iz*tm[2][2] + tm[3][2];
}

