/* utils.c -- Miscellaneous utility functions
   Copyright (c) 1994 by Eberhard Mattes

This file is part of emx.

emx is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

emx is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with emx; see the file COPYING.  If not, write to
the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

As special exception, emx.dll can be distributed without source code
unless it has been changed.  If you modify emx.dll, this exception
no longer applies and you must remove this paragraph from all source
files for emx.dll.  */


#include <stdarg.h>
#define INCL_DOSSEMAPHORES
#define INCL_DOSMISC
#define INCL_DOSERRORS
#include <os2emx.h>
#include "emxdll.h"

#define NULL ((void *)0)


/* Mutex semaphore for protecting data in the COMMON32 segment. */

static char const common_mutex_name[] = "/sem32/emx/common.mtx";
HMTX common_mutex;
static BYTE common_open;


/* Convert the unsigned number X to a hexadecimal string of DIGITS
   digits at DST. */

static void hexn (char *dst, ULONG x, int digits)
{
  int i;

  for (i = digits - 1; i >= 0; --i)
    {
      dst[i] = "0123456789abcdef"[x & 0x0f];
      x >>= 4;
    }
}


/* Display a string on the error output. */

void otext (const char *msg)
{
  ULONG written;

  DosWrite (errout_handle, msg, strlen (msg), &written);
}


/* Handle an unexpected OS/2 error code by beeping and displaying a
   message which shows the function name (a string pointed to by MSG)
   and the return code (RC). */

void error (ULONG rc, const char *msg)
{
  oprintf ("%s: 0x%.8x ", msg, (unsigned)rc);
  DosBeep (698, 500);
}


/* Create an unnamed Mutex semaphore and store the handle to *PSEM.
   Return the OS/2 error code. */

ULONG create_mutex_sem (HMTX *psem)
{
  ULONG rc;

  rc = DosCreateMutexSem (NULL, psem, 0, 0);
  if (rc != 0)
    error (rc, "DosCreateMutexSem");
  return (rc);
}


/* Request the Mutex semaphore SEM. */

void request_mutex (HMTX sem)
{
  ULONG rc;

  do
    {
      rc = DosRequestMutexSem (sem, -1);
    } while (rc == ERROR_SEM_OWNER_DIED || rc == ERROR_INTERRUPT);
}


/* Create an unnamed event semaphore and store the semaphore handle to
   *PSEM.  Attribute bits (such as DC_SEM_SHARED) are given in
   ATTR.  Return the OS/2 error code. */

ULONG create_event_sem (HEV *psem, ULONG attr)
{
  ULONG rc;

  rc = DosCreateEventSem (NULL, psem, attr, FALSE);
  if (rc != 0)
    error (rc, "DosCreateEventSem");
  return (rc);
}


/* Reset the event semaphore SEM.  It's no error if the semaphore is
   already reset.  Return the OS/2 error code. */

ULONG reset_event_sem (HEV sem)
{
  ULONG rc, post_count;

  rc = DosResetEventSem (sem, &post_count);
  if (rc == ERROR_ALREADY_RESET)
    rc = 0;
  if (rc != 0)
    error (rc, "DosResetEventSem");
  return (rc);
}


/* Lock the common data area.  This creates or opens the semaphore if
   not already done. */

void do_lock_common (void)
{
  ULONG rc;

  if (!common_open)
    {
      common_open = TRUE;
      rc = DosCreateMutexSem (common_mutex_name, &common_mutex,
                              DC_SEM_SHARED, 0);
      if (rc == ERROR_DUPLICATE_NAME)
        rc = DosOpenMutexSem (common_mutex_name, &common_mutex);
    }
  request_mutex (common_mutex);
}


/* Terminate the process, returning RC. */

void quit (ULONG rc)
{
  kbd_stop ();
  DosExit (EXIT_PROCESS, rc);
}


/* Return the value of a static system variable. */

ULONG querysysinfo (ULONG index)
{
  ULONG value;

  DosQuerySysInfo (index, index, &value, sizeof (value));
  return (value);
}


/* This function implements a subset of ANSI vsprintf().  Supported
   format specifications are %%, %s and %x, with optional precision.
   The default precision for %x is currently 8 (no caller depends on
   this). */

int vsprintf (char *dst, const char *fmt, va_list arg_ptr)
{
  char *d;
  int prec;
  unsigned u;
  const char *s;

  d = dst;
  while (*fmt != 0)
    if (*fmt != '%')
      *d++ = *fmt++;
    else if (fmt[1] == '%')
      *d++ = '%', fmt += 2;
    else
      {
        ++fmt; prec = -1;
        if (*fmt == '.')
          {
            ++fmt; prec = 0;
            while (*fmt >= '0' && *fmt <= '9')
              prec = prec * 10 + (*fmt++ - '0');
          }
        switch (*fmt)
          {
          case 's':
            s = va_arg (arg_ptr, char *);
            if (prec == -1) prec = 1 << 30;
            while (*s != 0 && prec > 0)
              *d++ = *s++, --prec;
            break;
          case 'x':
            u = va_arg (arg_ptr, unsigned);
            if (prec == -1) prec = 8;
            hexn (d, u, prec);
            d += prec;
            break;
          }
        ++fmt;
      }
  *d = 0;
  return (d - dst);
}


/* Simple version of sprintf().  See vsprintf() for supported format
   specifications. */

int sprintf (char *dst, const char *fmt, ...)
{
  int result;
  va_list arg_ptr;

  va_start (arg_ptr, fmt);
  result = vsprintf (dst, fmt, arg_ptr);
  va_end (arg_ptr);
  return (result);
}


/* Simple version of printf() with output to error output.  See
   vsprintf() for supported format specifications. */

void oprintf (const char *fmt, ...)
{
  va_list arg_ptr;
  char buf[1024];
  int len;
  ULONG written;

  va_start (arg_ptr, fmt);
  len = vsprintf (buf, fmt, arg_ptr);
  DosWrite (errout_handle, buf, len, &written);
  va_end (arg_ptr);
}
