#ifndef _ASECLASS_H
#define _ASECLASS_H
/*****************************************************************************
    Name: aseclass.h                                       

    Description:  
            This file describes the ASE Program Interface
            classes for C++ AutoCAD applications.

    Author: Volodya Sirotinin
            Autodesk, Inc.
            Moscow, Russia.

   Copyright (C) 1992, 1993, 1994 by Autodesk, Inc.
 
   Permission to use, copy, modify, and distribute this software in 
   object code form for any purpose and without fee is hereby granted, 
   provided that the above copyright notice appears in all copies and 
   that both that copyright notice and the limited warranty and 
   restricted rights notice below appear in all supporting 
   documentation.
 
   AUTODESK PROVIDES THIS PROGRAM "AS IS" AND WITH ALL FAULTS.  
   AUTODESK SPECIFICALLY DISCLAIMS ANY IMPLIED WARRANTY OF 
   MERCHANTABILITY OR FITNESS FOR A PARTICULAR USE.  AUTODESK, INC.
   DOES NOT WARRANT THAT THE OPERATION OF THE PROGRAM WILL BE 
   UNINTERRUPTED OR ERROR FREE.
 
   Use, duplication, or disclosure by the U.S. Government is subject to 
   restrictions set forth in FAR 52.227-19 (Commercial Computer 
   Software - Restricted Rights) and DFAR 252.227-7013(c)(1)(ii) 
   (Rights in Technical Data and Computer Software), as applicable.

    Entry Points:

    Modification History:
    11 Oct 1993 - volodyas - original written

    Bugs and restriction on use:

    Notes:

*****************************************************************************/

/***************************************************************************/
/* INCLUDES */
/***************************************************************************/
#include <asiappl.h>
#include <aseconst.h>

/***************************************************************************/
/* DEFINES */
/***************************************************************************/
#define CASEAPPL(x)       ((CAseAppl*)x)
#define CASELINKPATH(x)   ((CAseLinkPath*)x)
#define CASELINK(x)       ((CAseLink*)x)
#define CASELINKSEL(x)    ((CAseLinkSel*)x)

// Can be used to throw ASI exception with the 
// diagnostic parameters, containing in the specified 
// CAseApiObj-derived object
#define ASE_THROW(x)  {((CAseApiObj*)x)->appThrow() ;}

/***************************************************************************/
/* TYPEDEFS */
/***************************************************************************/

/***************************************************************************/
/* CLASS DESCRIPTION */
/***************************************************************************/
#if defined(__cplusplus)

class CAseApiErr ;
class CAseApiObj ;
  class CAseAppl ;
  class CAseLinkPath ;
  class CAseLinkSel ;

class CAseApiObj 
//
//  This class defines the properties of the
//  all ASE classes and has to be inherited by the all
//  other concrete ASE API classes.
//
{
  public:
                         CAseApiObj(CAseApiObj *pObj) ;
                         CAseApiObj(CAseApiObj &Obj) ;
  virtual                ~CAseApiObj() ;
  virtual EAsiBoolean    init() ;
  virtual EAseApiClassId isA() const = 0 ;
  virtual CAseApiObj    *clone() const = 0 ;
  virtual EAsiBoolean    copyFrom(const CAseApiObj *pObj) ;
  virtual CAseApiObj&    operator=(const CAseApiObj &Obj) ;
  virtual int            operator==(const CAseApiObj *pObj) const = 0 ;
  virtual int            operator==(const CAseApiObj &Obj) const = 0 ;
  virtual EAsiBoolean    isInit() const ;
  const char            *version() const ;
  const CAsiException   *errGet() const ;
  int                    errQty() const ;
  EAseErrDsc             errDsc(int ErrNum) const ;
  int                    errCode(int ErrNum) const ;
  EAsiBoolean            errMsg(int ErrNum,
                                char *pBuf,
                                int BufLen) const;
  EAsiBoolean            errClear() ;
  void                   appThrow() const ;
  const char            *errDiagParNameCode(int ParNum,
                                            int *pParCode,
                                            int *pIsStr) const ;
  EAsiBoolean            errDiagPar(int ErrNum, 
                                    int ParCode,
                                    int *pIntValue) const ;
  EAsiBoolean            errDiagPar(int ErrNum, 
                                    int ParCode,
                                    char *pStrValue,
                                    int BufLen) const ;

  // Non-documented
  // Init the descriptor by the CAsiException 
  EAsiBoolean            errInitAsi(CAsiException *pErr) ;

  // Init the descriptor by the CAsiSQLObject
  EAsiBoolean            errInitAsi(CAsiSQLObject *pSQLObj) ;
  EAsiBoolean            errInitAsi(CAsiSQLObject &SQLObj) ;

  friend class CAseApiErr ;
  friend class CAseAppl ;
  protected:
  struct resbuf         *getLast(const struct resbuf *bufptr) const ;
  struct resbuf         *allocAttr(EAseLinkAttr attr, const char *pStr=NULL) const ;
  struct resbuf         *copyBuf(const struct resbuf *bufptr, 
                                 const struct resbuf *pBorder=NULL) const ;
  struct resbuf         *nextAttr(const struct resbuf *bufptr) const ;
  struct resbuf         *getAttrType(const struct resbuf *bufptr,
                                     EAseLinkAttr lattr) const ;
  const struct resbuf   *getAttrPtr(const struct resbuf *bufptr,
                                    EAseLinkAttr lattr) const ;
  EAsiBoolean            setAttrPtr(struct resbuf *bufptr,
                                    EAseLinkAttr LinkAttr,
                                    struct resbuf *pAttrValue) const ;
  struct resbuf         *delAttrPtr(struct resbuf *bufptr,
                                    struct resbuf *pAttr) const ;
  EAsiBoolean            parseLisp(struct resbuf *bufptr) ;
  CAseAppl              *getAppl() const ;
  void                   freeStr(struct resbuf *pBuf) const ;
  EAsiBoolean            isDiffChar() const ;

  void                  *mpErr ;       // The error descriptor pointer
} ;

class CAseAppl : public CAseApiObj 
//
//            This class is responsible for the interface
//            with the ASE main module.
//
{
  public:
                         CAseAppl(CAseApiObj *pObj=NULL) ;
                         CAseAppl(CAseApiObj &Obj) ;
  virtual                ~CAseAppl() ;
  virtual EAsiBoolean    init() ;
  virtual EAseApiClassId isA() const ;
  virtual CAseApiObj    *clone() const ;
  virtual EAsiBoolean    copyFrom(const CAseApiObj *pObj) ;
  virtual CAseApiObj&    operator=(const CAseApiObj &Obj) ;
  virtual int            operator==(const CAseApiObj *pObj) const ;
  virtual int            operator==(const CAseApiObj &Obj) const ;
  virtual EAsiBoolean    isInit() const ;
  void                   term() ;
  EAsiBoolean            getAseErr() ;

  protected:
    // ASE call
  struct resbuf *        aseiCallAse(int funCode,
                                     struct resbuf* pParms,
                                     void *pErr)  ;
    // ADS-RX, RX-RX communication
  struct resbuf *        ase_invoke(int funCode,
                                    struct resbuf* pParms,
                                    void *pErr)  ;
  struct resbuf *        ase_fromAse(struct resbuf *pBuf) ;
  EAsiBoolean            ase_toAse(struct resbuf *pBuf) ;
  EAsiBoolean            addObj(const CAseApiObj *pObj) ;
  EAsiBoolean            remObj(const CAseApiObj *pObj) ;

  EAsiBoolean    isADS() const ;

  friend class CAseApiObj ;
  friend class CAseLinkPath ;
  friend class CAseLink ;
  friend class CAseLinkSel ;
  friend class CAseApiErr ;
  char                  *mpVersion ;    // The ASE version
  void                  *mpXmf ;        // XMF file reference
  void                  *mpObjs ;       // Attached objects list
  EAsiBoolean            mIsDiffChar ;  // ASE & ASE appl have the
                                        // different character sets (kAsiTrue)
  char                  *mpDefChar ;    // ASE default character set
                                        // (NULL if the same as ASE has)
} ;


class CAseLinkPath : public CAseApiObj
//
//            This class is responsible for the manipulations
//            with the Link Paths. The class can refer to the
//            one Link Path is its name is set or provide the
//            caller with the group manipulations with the
//            several Link Paths related with the Database
//            Object Path. The procesing functions execute
//            the actions over the single Link Path, if its
//            name is specified. Otherwise, some of the
//            functions execute the actions for all of Link
//            Paths, related with the Database Object for
//            specified path. If the Database Object Path is
//            empty or wasn't specified, some of the
//            functions execute the actions for all of Link
//            Paths in the current drawing.
//
{
  public:
                         CAseLinkPath(CAseApiObj *pObj) ;
                         CAseLinkPath(CAseApiObj &Obj) ;
  virtual                ~CAseLinkPath() ;
  virtual EAsiBoolean    init() ;
  virtual EAseApiClassId isA() const ;
  virtual CAseApiObj    *clone() const ;
  virtual EAsiBoolean    copyFrom(const CAseApiObj *pObj) ;
  virtual CAseApiObj&    operator=(const CAseApiObj &Obj) ;
  virtual int            operator==(const CAseApiObj *pObj) const ;
  virtual int            operator==(const CAseApiObj &Obj) const ;
  virtual EAsiBoolean    isInit() const ;

  EAsiBoolean            initPath(const char *pPath) ;
  EAsiBoolean            initName(const char *pName) ;
  EAsiBoolean            initCurrent() ;
  EAsiBoolean            setName(const char *pName,
                                 EAseDoNameCode NameCode=kAseLpnCode) ;
  EAsiBoolean            getName(char *pBuf, int BufLen,
                                 EAseDoNameCode NameCode=kAseLpnCode) const ;
  int                    getNameSize(EAseDoNameCode NameCode=kAseLpnCode) const;
  int                    cmpName(const char *pName, 
                                 EAseDoNameCode NameCode=kAseLpnCode) const ;
  int                    cmpName(const CAseLinkPath *pLinkPath, 
                                 EAseDoNameCode NameCode=kAseLpnCode) const ;
  EAseDoNameCode         getPathCode() const;
  int                    getStatus() const;
  EAsiBoolean            isUpdatable() const ;
  EAsiBoolean            getKeyDsc(CAsiRow **pKeyDsc) const;
  EAsiBoolean            getKeyDsc(CAsiColumn **pKeyDsc[], int *pColQty) const;
  EAsiBoolean            create(const CAsiRow *pKeyDsc) ;
  EAsiBoolean            create(const CAsiRow &KeyDsc) ;
  EAsiBoolean            create(const CAsiColumn *pKeyDsc[], int ColQty) ;
  EAsiBoolean            erase() ;
  EAsiBoolean            rename(const char *pLinkPathName) ;
  EAsiBoolean            setCurrent() const ;
  struct resbuf         *getLinkNames() const ;
  struct resbuf         *getPaths(int Status) const ;

  protected:
  void                   *mpPath ;    // The internal data pointer
} ;

class CAseLink : public CAseApiObj
//
//  This is the base class, defining the common
//  responsibility of the concrete links.
//
{
  public:
                         CAseLink(CAseApiObj *pObj) ;
                         CAseLink(CAseApiObj &Obj) ;
  virtual                ~CAseLink() ;
  virtual EAsiBoolean    init() ;
  virtual EAsiBoolean    init(EAseLinkType lType) ;
  virtual EAsiBoolean    init(LinkID linkId) ;
  virtual EAsiBoolean    init(const char *pName, 
                              const CAsiRow *pKeyValue, 
                              ads_name EntName) ;
  virtual EAsiBoolean    init(const char *pName, 
                              const CAsiRow &KeyValue, 
                              ads_name EntName) ;
  virtual EAsiBoolean    init(const char *pName, 
                              const CAsiColumn *pKeyValue[],
                              int ColQty,
                              ads_name EntName) ;
  virtual EAsiBoolean    init(const char *pName, 
                              const CAsiRow *pKeyValue,
                              const CAsiRow *pDACols,
                              const struct resbuf *pDAParms) ;
  virtual EAsiBoolean    init(const char *pName, 
                              const CAsiRow &KeyValue,
                              const CAsiRow &DACols,
                              const struct resbuf *pDAParms) ;
  virtual EAsiBoolean    init(const char *pName, 
                              const CAsiColumn *pKeyValue[],
                              int ColQty,
                              const CAsiColumn *pDACols[],
                              int DAColQty,
                              const struct resbuf *pDAParms) ;
  virtual EAseApiClassId isA() const ;
  virtual CAseApiObj    *clone() const ;
  virtual EAsiBoolean    copyFrom(const CAseApiObj *pObj) ;
  virtual CAseApiObj&    operator=(const CAseApiObj &Obj) ;
  virtual int            operator==(const CAseApiObj *pObj) const ;
  virtual int            operator==(const CAseApiObj &Obj) const ;
  virtual EAsiBoolean    isInit() const ;

  LinkID                 getId () const ;
  EAseLinkType           getType() const ;
  EAsiBoolean            getName(char *pBuf, int len) const;
  int                    getNameSize() const ;
  EAsiBoolean            setName(const char *pName) ;
  EAsiBoolean            getEntity(ads_name EntName) const ;
  EAsiBoolean            setEntity(ads_name EntName) ;
  EAsiBoolean            getKey(CAsiRow *pKeyValue) const ;
  EAsiBoolean            getKey(CAsiRow &KeyValue) const ;
  EAsiBoolean            getKey(CAsiColumn *pKeyValue[], int ColQty) const ;
  EAsiBoolean            setKey(const CAsiRow *pKeyValue) ;
  EAsiBoolean            setKey(const CAsiRow &KeyValue) ;
  EAsiBoolean            setKey(const CAsiColumn *pKeyValue[], int ColQty) ;
  EAsiBoolean            setDACols (const CAsiRow *pCols) ;
  EAsiBoolean            setDACols (const CAsiRow &Cols) ;
  EAsiBoolean            setDACols(CAsiColumn *pCols[], int ColQty) ;
  EAsiBoolean            getDACols (CAsiRow **pCols) const ;
  EAsiBoolean            getDACols(CAsiColumn **pCols[], int *pColQty) const ;
  EAsiBoolean            setDAParms(const struct resbuf *pParms) ;
  EAsiBoolean            setDAValues(const CAsiRow *pDAValues) ;
  EAsiBoolean            setDAValues(const CAsiRow &DAValues) ;
  EAsiBoolean            setDAValues(CAsiColumn *pCols[], int ColQty) ;
  EAsiBoolean            create () ;
  EAsiBoolean            remove () ;
  EAsiBoolean            update () ;
  EAsiBoolean            getXName(char *pBuf, int len) const ;
  EAsiBoolean            isUpdatable() const ;

    // Non-documented
    // Gets/sets the link attribute values in the internal representation
  struct resbuf         *getAttr(EAseLinkAttr LinkAttr) const ;
  EAsiBoolean            setAttr(EAseLinkAttr LinkAttr,
                                 const struct resbuf *pAttrValue) ;
  private:
  struct resbuf      *mpBuf ;    // The link result buffer representation
} ;

class CAseLinkSel : public CAseApiObj 
//
//  This class describes the link selection.
//
{
  public:
                         CAseLinkSel(CAseApiObj *pObj) ;
                         CAseLinkSel(CAseApiObj &Obj) ;
  virtual                ~CAseLinkSel() ;
  virtual EAsiBoolean    init() ;
  virtual EAsiBoolean    init(EAseLinkType LinkType) ;
  virtual EAsiBoolean    init(EAseLinkType LinkType,
                              ads_name EntSelName) ;
  virtual EAsiBoolean    init(EAseLinkType LinkType,
                              const char *pXName) ;
  virtual EAsiBoolean    init(ads_name EntName) ;
  virtual EAsiBoolean    init(ads_name EntName,
                              const char *pName) ;
  virtual EAsiBoolean    init(ads_name EntName,
                              const CAseLinkPath *pLinkPath) ;
  virtual EAsiBoolean    init(ads_name EntName,
                              const CAseLinkPath &LinkPath) ;
  virtual EAsiBoolean    init(const char *pName,
                              EAseLinkType LinkType=kAseUnknownLinkType) ;
  virtual EAsiBoolean    init(const CAseLinkPath *pLinkPath,
                              EAseLinkType LinkType=kAseUnknownLinkType) ;
  virtual EAsiBoolean    init(const CAseLinkPath &LinkPath,
                              EAseLinkType LinkType=kAseUnknownLinkType) ;
  virtual EAsiBoolean    init(const char *pName,
                              const CAsiRow *pKeyValue,
                              EAseLinkType LinkType=kAseUnknownLinkType) ;
  virtual EAsiBoolean    init(const char *pName,
                              const CAsiRow &KeyValue,
                              EAseLinkType LinkType=kAseUnknownLinkType) ;
  virtual EAsiBoolean    init(const CAseLinkPath *pLinkPath, 
                              const CAsiRow *pKeyValue,
                              EAseLinkType LinkType=kAseUnknownLinkType) ;
  virtual EAsiBoolean    init(const CAseLinkPath &LinkPath, 
                              const CAsiRow &KeyValue,
                              EAseLinkType LinkType=kAseUnknownLinkType) ;
  virtual EAsiBoolean    init(const char *pName,
                              const CAsiColumn *pKeyValue[],
                              int ColQty,
                              EAseLinkType LinkType=kAseUnknownLinkType) ;
  virtual EAsiBoolean    init(const CAseLinkPath *pLinkPath, 
                              const CAsiColumn *pKeyValue[],
                              int ColQty,
                              EAseLinkType LinkType=kAseUnknownLinkType) ;
  virtual EAsiBoolean    init(const char *pName,
                              const char *pTextCondition,
                              EAseLinkType LinkType=kAseUnknownLinkType) ;
  virtual EAsiBoolean    init(const CAseLinkPath *pLinkPath, 
                              const char *pTextCondition,
                              EAseLinkType LinkType=kAseUnknownLinkType) ;
  virtual EAsiBoolean    init(const CAseLinkPath &LinkPath, 
                              const char *pTextCondition,
                              EAseLinkType LinkType=kAseUnknownLinkType) ;
  virtual EAsiBoolean    init(const char *pName,
                              ads_name EntSelName,
                              EAseLinkType LinkType=kAseUnknownLinkType) ;
  virtual EAsiBoolean    init(const CAseLinkPath *pLinkPath, 
                              ads_name EntSelName,
                              EAseLinkType LinkType=kAseUnknownLinkType) ;
  virtual EAsiBoolean    init(const CAseLinkPath &LinkPath, 
                              ads_name EntSelName,
                              EAseLinkType LinkType=kAseUnknownLinkType) ;

  virtual EAseApiClassId isA() const ;
  virtual CAseApiObj    *clone() const ;
  virtual EAsiBoolean    copyFrom(const CAseApiObj *pObj) ;
  virtual CAseApiObj&    operator=(const CAseApiObj &Obj) ;
  virtual int            operator==(const CAseApiObj *pObj) const ;
  virtual int            operator==(const CAseApiObj &Obj) const ;
  virtual EAsiBoolean    isInit() const ;
  EAsiBoolean            isUpdatable() const ;

  long                   getQty() const ;
  LinkID                 getId(long ItemNum) const ;
  EAsiBoolean            addId(LinkID linkId) ;
  EAsiBoolean            delId(long ItemNum) ;
  long                   membId(LinkID linkId) const ;
  EAsiBoolean            unite(const CAseLinkSel *pSel) ;
  EAsiBoolean            unite(const CAseLinkSel &Sel) ;
  EAsiBoolean            subtract(const CAseLinkSel *pSel) ;
  EAsiBoolean            subtract(const CAseLinkSel &Sel) ;
  EAsiBoolean            intersect(const CAseLinkSel *pSel) ;
  EAsiBoolean            intersect(const CAseLinkSel &Sel) ;

  virtual EAsiBoolean    intersect(EAseLinkType LinkType) ;
  virtual EAsiBoolean    intersect(EAseLinkType LinkType,
                                   ads_name EntSelName) ;
  virtual EAsiBoolean    intersect(EAseLinkType LinkType,
                                   const char *pXName) ;
  virtual EAsiBoolean    intersect(ads_name EntName) ;
  virtual EAsiBoolean    intersect(const char *pName) ;
  virtual EAsiBoolean    intersect(const CAseLinkPath *pLinkPath) ;
  virtual EAsiBoolean    intersect(const CAseLinkPath &LinkPath) ;
  virtual EAsiBoolean    intersect(const char *pName,
                                   const CAsiRow *pKeyValue) ;
  virtual EAsiBoolean    intersect(const char *pName,
                                   const CAsiRow &KeyValue) ;
  virtual EAsiBoolean    intersect(const CAseLinkPath *pLinkPath, 
                                   const CAsiRow *pKeyValue) ;
  virtual EAsiBoolean    intersect(const CAseLinkPath &LinkPath, 
                                   const CAsiRow &KeyValue) ;
  virtual EAsiBoolean    intersect(const char *pName,
                                   const CAsiColumn *pKeyValue[],
                                   int ColQty) ;
  virtual EAsiBoolean    intersect(const CAseLinkPath *pLinkPath, 
                                   const CAsiColumn *pKeyValue[],
                                   int ColQty);
  virtual EAsiBoolean    intersect(const char *pName,
                                   const char *pTextCondition) ;
  virtual EAsiBoolean    intersect(const CAseLinkPath *pLinkPath, 
                                   const char *pTextCondition) ;
  virtual EAsiBoolean    intersect(const CAseLinkPath &LinkPath, 
                                   const char *pTextCondition) ;

  struct resbuf         *getXNames() const ;
  struct resbuf         *getLinkNames() const ;
  EAsiBoolean            getEntSel(ads_name EntSelName) const ;
  EAsiBoolean            erase() ;

  // Non-documented
  // Initializes the link selection by the 
  // internal link filter
  virtual EAsiBoolean    init(struct resbuf *pLinkFilter) ;

  // Filters the link selection by the 
  // internal link filter
  virtual EAsiBoolean    intersect(const struct resbuf *pLinkFilter) ;

  protected:
  void                   freeSel() ;
  struct resbuf         *makeFilter(EAseLinkType LinkType,
                                    const char *pXName,
                                    ads_name EntName,
                                    ads_name EntSelName,
                                    const char *pName,
                                    const CAseLinkPath *pLinkPath,
                                    const CAsiRow *pKeyValue,
                                    const char *pTextCondition) ;

  LinkSelID              mSelId ;    // The link selection identifier
} ;


#endif /* __cplusplus */

#endif /*_ASECLASS_H*/

