/*

        DdeLisp

        Dynamic Data Exchange functions to be called by AutoLISP
        programs, for the transfer of AutoCAD data to and from other 
        programs.  See DDE.DOC for list of functions.

        by Phil Ford
     ________________________________________________________________________

      (C) Copyright 1990-1994 by Autodesk, Inc.

      Permission to use, copy, modify, and distribute this software and its
      documentation for any purpose and without fee is hereby granted.  

      THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY. 
      ALL IMPLIED WARRANTIES OF FITNESS FOR ANY PARTICULAR PURPOSE AND OF 
      MERCHANTABILITY ARE HEREBY DISCLAIMED.                                
     ________________________________________________________________________


*/


#include "options.h"

#include <stdio.h>
#include <math.h>

#include "adslib.h"    
#include <process.h>
#include <string.h>
#include <memory.h>
#include <dos.h>
#include <ctype.h>

#include "winutil.h"
#include "ddewin.h"
#include "windde.h"
#include "ddeconv.h"
#include "spreadsh.h"
#include "acaddefs.h"


extern HANDLE adsw_hInstance;
extern DWORD adsw_idInst;                     /* DDEML handle */        

ddeDATA *pDdeData = NULL;

static char progname[] = "DdeLisp";
static unsigned ModCnt = 0;

/* All Function Prototypes for ddelisp.c */

void main(int argc, char ** argv);
int funcload(void);
static void fddeinit(char *funcname);
static void fappinit(char *funcname);
static void fddedefaults(char *funcname);
static void ffindapp(char *funcname);
static void fstartapp(char *funcname);
static void fapptype(char *funcname);
static void ddeprompt(char *funcname);
static void ddedialog(char *funcname);
static void getddeapp(char *funcname);
static void getddetopic(char *funcname);
static void fsethandles(char *funcname);
static void initiate(char *funcname);
static void terminate(char *funcname);
static void request(char *funcname);
static void poke(char *funcname);
static void execute(char *funcname);
static void advise(char *funcname);
int AdsAdviseFunc(PDDE pdde);
static void unadvise(char *funcname);
static void ddedone(char *funcname);
static void strreal(char *funcname);
static void realstr(char *funcname);
static void cellstr(char *funcname);
static void entcell(char *funcname);
static void cformat(char *funcname);
static void fentpoke(char *funcname);
static void fentpokecell(char *funcname);
static void entlen(char *funcname);
static void freqmod(char *funcname);
static void fpokeset(char *funcname);
static void fpokeall(char *funcname);
static void fprecision(char *funcname);
static void fentformat(char *funcname);
static void listchnls(char *funcname);
static void fddefree(char *funcname);
static void lentbl(char *funcname);
static void poketbl(char *funcname);
static void poketblcell(char *funcname);
static void fpokealltbl(char *funcname);
static void lenalltbl(char *funcname);
static void modcount(char *funcname);
static void reqlarge(char *funcname);
static void modlarge(char *funcname);
static void fsenddrawing(char *funcname);
static void fmoddrawing(char *funcname);
static void getddechnl(char *funcname);
static void setddeformat(char *funcname);
static void getddeformat(char *funcname);
static void listfunc(char *funcname);



/*         AutoCAD/AutoLISP Functions and Commands
   Functions receive multiple parameters and return values, and 
   can be called from AutoLISP programs.  In AutoCAD, enter
   "(xload "ddelisp")" to load this application, then execute the
   initiate function by entering "(initiate "Excel" "sheet1")".
   DDE.LSP has many examples of calling these functions from AutoLISP.
   Names starting with "C:" are commands.  They don't have 
   input parameters, and don't require parenthesis when entering 
   from AutoCAD.  For example, to run the DDE dialog, enter "ddedialog"
   from the AutoCAD command line. */
CMDTAB cmdtab[] = {
        /*   Name    Function  */

        /* Initialize DDE and AutoCAD drawing, remote application
           and work file, set defaults, 
           prompt for remote application name, get DDE info, etc. 
        */

        {"DDEINIT", fddeinit},        /* Set program name, default
                                         DDE app and topic. */
        {"APPINIT", fappinit},        /* Start an application and work
                                         file.  Open DDE channel to it. */
        {"STARTAPP", fstartapp},      /* start an application */
        {"APPTYPE", fapptype},        /* specify Excel (1) or Lotus (2) */
        {"DDEPROMPT", ddeprompt},     /* ask for app and work file,
                                         initiate DDE channel. */
        {"DDEDIALOG", ddedialog},     /* ask for app and work file,
                                         initiate DDE channel. */
        {"SETDDEFORMAT", setddeformat}, /* set dde text format */
        {"GETDDEFORMAT", getddeformat}, /* get dde text format */
        {"DDEDEFAULTS", fddedefaults},/* default app, topic, path */
#ifdef USE_FINDAPP
        {"FINDAPP", ffindapp},        /* find app using classname */
#endif
        {"SETHANDLES", fsethandles},  /* turn handles on */
        {"FPRECISION", fprecision},   /* precision, fuzz factor */
        {"GETDDECHNL", getddechnl},   /* return current dde channel num */
        {"GETDDEAPP", getddeapp},     /* get current app name */
        {"GETDDETOPIC", getddetopic}, /* get current topic name. */
        {"CELLSTR", cellstr},         /* build a spreadsheet cell item 
                                         range string from row and col 
                                         numbers*/
        {"LISTCHNLS", listchnls},     /* print DDE channels in AutoCAD */

        {"DDEDONE", ddedone},         /* All done with DDE--free lists
                                         and remote shared data */



        /* AutoCAD Entity DDE Transfer Functions */
        {"SENDDRAWING", fsenddrawing},/* send tables and drawing */
        {"MODDRAWING", fmoddrawing},  /* request and modify whole drawing */
        {"REQMOD", freqmod},          /* request data from DDE remote and
                                      modify entity with it */
        {"MODCOUNT", modcount},       /* how many entities modified */
        {"ENTFORMAT", fentformat},    /* text format separators */
        {"REQLARGE", reqlarge},       /* request data, return long, which
                                      is actually the address of the 
                                      shared data, for large data
                                      transfers */
        {"MODLARGE", modlarge},       /* modify entities using pointer to
                                      data returned by REQLARGE. */
        {"POKESET", fpokeset},        /* send a selection set */
        {"POKEALL", fpokeall},        /* send entity data from this 
                                      entity to end of drawing (or selection
                                      set), to remote app. */
        {"LENTBL", lentbl},
        {"LENALLTBL", lenalltbl},
        {"POKETBL", poketbl},         /* send a table */
        {"POKETBLCELL", poketblcell}, /* send a table to a spreadsheet */
        {"POKEALLTBL", fpokealltbl},

        /* Single entity transfers */
        {"ENTLEN", entlen},           /* get length of entity */
        {"ENTCELL", entcell},         /* build a cell item string from
                                      an entity, row, and col. */
        {"ENTPOKE", fentpoke},        /* Send entity, ename, to
                                      the remote program */
        {"ENTPOKECELL", fentpokecell},/* Send entity to cell at row
                                      and col */


        /* Low Level DDE functions */
        {"INITIATE", initiate},       /* initialize another program
                                      and work file for data transfer
                                      (initiate a DDE channel) */
        {"TERMINATE", terminate},     /* done with channel */
        {"REQUEST", request},         /* request data from remote prog 
                                      on an item (cell, range of cells,
                                      entities, etc.) */
        {"DDEFREE", fddefree},        /* Free the shared data from a 
                                      DDE request. */
        {"POKE", poke},               /* force data into remote prog */
        {"EXECUTE", execute},         /* execute a command the remote
                                      prog understands */
        {"ADVISE", advise},           /* request automatic data update
                                      on an item (cell, range of cells,
                                      entities, etc.) */
        {"UNADVISE", unadvise},       /* End ADVISE */

        /* Misc. */
        {"CFORMAT", cformat},         /* set the precision desired for
                                      string to real conversions */
        {"STRREAL", strreal},         /* convert a string to a real */
        {"REALSTR", realstr},         /* convert a real to a string, for
                                      DDE transfer. */
        {"C:DDELISP", listfunc}       /* list all the function names 
                                      in this module */

       };


/*

                  DdeLisp Main

*/

void main(int argc, char ** argv)
{
    int scode = RSRSLT;
    int cindex;
    int ads_req; 

    /* Initialize DDE Manager with the program name, 
       the default DDE app and topic, and the instance handle. 
       Pass NULL for those that are not defined. */

    pDdeData = DdeInit(progname, adsw_hInstance, adsw_idInst);

    /* Initialize ADS system: send back the semaphore and shared memory
       info passed in through the "command line", argv. */

    ads_init(argc, argv); 

    for ( ;; ) {
        /* This thread will block here until LISP has a request for us */
        ads_req = ads_link(scode);        

        /* RQQUIT is normal termination.  RTFAIL probably means 
           lisp has died. */
        if (ads_req < 0)
            break;

        scode = RSRSLT;               /* default return code */

        /* Check for AT LEAST the following cases here */
        switch (ads_req) { 

        case RQXLOAD:                 /* Load request. Send function names. */
            scode = -(funcload() ? RSRSLT : RSERR);
            break;

            /* Execute a "loaded" function that was defined via RQXLOAD */
        case RQSUBR:
            cindex = ads_getfuncode();

            if (cindex >= 0 && cindex < ELEMENTS(cmdtab)) {
                (*cmdtab[cindex].cmdfunc)(cmdtab[cindex].cmdname);
            }

            break;

        case RQXUNLD:      
            /* "xunload" or AutoCAD terminating */
            DdeQuit();                /* Terminate DDE channels, free up 
                                         memory, etc */
            break;

        default:
            break;
        }
    }

}




/* Register our functions with the ADS system.  Also,
   any drawing initialization can go here, since this is
   called with each new drawing. 
*/
int funcload(void)
{
    int idx;
    char *cname;

    for (idx = 0; idx < ELEMENTS(cmdtab); ++idx) {
        cname = cmdtab[idx].cmdname;
        ads_defun(cname, (short)idx);
    }

    SetFlatLand();                    /* set local flatland variable, for 
                                      dxftype function. */
    return TRUE;
}


/*
            ADS Functions & Commands
*/

/* Set the current program name, the default DDE application,
   and the default topic name (work file).  No path names here. 
   (ddeinit "DdeLISP" "Excel" "Sheet1") 
*/
static void fddeinit(char *funcname)
{
    ARGLIST arglist[] = {{RTSTR}, {RTSTR}, {RTSTR}, {RTNONE}};
    if (!GetArgList(arglist, funcname)) {
        ads_retint(0);
        return;
    }
    strzcpy(pDdeData->progname, arglist[0].RSTR, MAXNAME);
    strzcpy(pDdeData->app, arglist[1].RSTR, MAXNAME);
    strzcpy(pDdeData->topic, arglist[2].RSTR, MAXNAME);
    ads_retint(1);
}


/* Start up an application and have it load a work file.  Initialize
   a DDE channel to the app and topic.  Returns a pointer to the DDE 
   channel object.  
   (setq chnl (appinit "Excel" "Sheet1" "c:\\excel\\excel"))
*/
static void fappinit(char *funcname)
{
    PDDE pdde;
    ARGLIST arglist[] = {{RTSTR}, {RTSTR}, {RTSTR}, {RTNONE}};

    if (!GetArgList(arglist, funcname)) {
        ads_retint(0);
        return;
    }
    pdde = DdeAppInit(arglist[0].RSTR, arglist[1].RSTR,
                arglist[2].RSTR);
    if (pdde != NULL)
        ads_retint(pdde->channel);
    else
        ads_retint(0);
}


/* Set defaults for DDE conversation init dialog.  Pass 3 string
   parameters: app, topic, appexe.  The 3rd param (appexe) is
   the command line used to start the app and have it load a 
   work file.  For the default work file, no command line should
   follow the exe name.  For work files in the current directory,
   no path is needed for the work file.  Eg., the first 2 of
   the following would attempt to start Excel and use the default
   work file, "Sheet1.xls".

        "Excel"         (Excel directory on PATH)
        "c:\\excel\\excel"     (NOT on PATH)
        "c:\\excel\\excel shaft.xls"     (xls file in current dir)
        "c:\\excel\\excel c:\\excel\\shaft.xls"

   Set topic and appexe to "" to use the default topic name and 
   to search the Windows Registration Database for the application
   exe path (e.g., "f:\excel\excel.exe").

   (ddedefaults "excel" "" "")
   (ddedefaults "excel" "sheet1" "")
   (ddedefaults "excel" "d:\\acadwin\\ads\\win\\shaft.xls" "")
   (ddedefaults "excel" "sheet1" "c:\\excel\\excel.exe")
*/
static void fddedefaults(char *funcname)
{
    ARGLIST arglist[] = {{RTSTR}, {RTSTR}, {RTSTR}, {RTNONE}};

    if (!GetArgList(arglist, funcname)) {
        ads_retint(0);
        return;
    }
    DdeDefaults(arglist[0].RSTR, 
                arglist[1].RSTR,
                arglist[2].RSTR);
    ads_retint(0);
}

#ifdef USE_FINDAPP

/* Find an application path for starting it up using the 
   Windows Registration Database class name.
   (setq appexe (findapp "ExcelWorksheet"))
*/
static void ffindapp(char *funcname)
{
    char path[256];
    ARGLIST arglist[] = {{RTSTR}, {RTNONE}};

    if (!GetArgList(arglist, funcname)) {
        ads_retint(0);
        return;
    }
    if (DdeFindApp(arglist[0].RSTR, path))
        ads_retstr(path);
    else
        ads_retstr(NullStr);
}
#endif


/* Run an application with a command line.  Searches PATH environment.
   Returns 0 if successful.
   E.g., to start Excel and open spreadsheet shaft.xls:
   (setq fail (startapp "excel" "shaft.xls"))   
*/
static void fstartapp(char *funcname)
{
    int status;
    ARGLIST arglist[] = {{RTSTR}, {RTSTR}, {RTNONE}};

    if (!GetArgList(arglist, funcname)) {
        ads_retint(1);
        return;
    }
    status = StartApp(arglist[0].RSTR, arglist[1].RSTR);
    if (status < 32) {
        /* Error */
        status = (status == 0 ? 1 : status);
        ads_retint(status);
    }
    /* Ok */
    ads_retint(0);
}


/* Specify Excel (1) or Lotus (2).  Not necessary if app is named
   "excel" or "123w" */

static void fapptype(char *funcname)
{
    int atype;
    ARGLIST arglist[] = {{RTSHORT}, {RTNONE}};

    if (!GetArgList(arglist, funcname)) {
        ads_retint(0);
        return;
    }
    atype = arglist[0].RINT;
    AppType(atype);
    ads_retint(atype);
}



/* Ask for app and topic name, displaying the defaults.
   Initialize a DDE channel. 
   (setq chnl (ddeprompt))
*/
static void ddeprompt(char *funcname)
{
    PDDE pdde;
    int chnl = 0;

    if (pDdeData == NULL)
        return;
#if 1
    pdde = DdeDlgStart(GetActiveWindow(), FALSE);
#else
    pdde = AdsChnl(pDdeData->app, pDdeData->topic); 
                                      /* Use AutoCAD to prompt user. 
                                      Initiate DDE channel. */
#endif
    if (pdde != NULL)
        chnl = pdde->channel;
    ads_retint(chnl);
}


/* Ask for app and topic name, displaying the defaults.
   Initialize a DDE channel. 
   (setq chnl (ddeprompt))
*/
static void ddedialog(char *funcname)
{
    int ok = 0;
    
    if (pDdeData == NULL)
        ads_retint(0);

    ok = DdeDialog(adsw_hWnd);
                                      /* Could sent unadvise to remote app */
    ads_retint(ok);
}


/* Get the current DDE application name. 
   (setq app (getddeapp))
*/
static void getddeapp(char *funcname)
{
    ads_retstr(pDdeData->app);
}


/* Get the current DDE topic name. 
   (setq app (getddetopic))
*/
static void getddetopic(char *funcname)
{
    ads_retstr(pDdeData->topic);
}


/* Set handles ON.
   (setq oldstate (sethandles)) 
*/
static void fsethandles(char *funcname)
{
    BOOL prevstate;

    prevstate = SetHandles();
    ads_retint(prevstate);
}


/* Check for an open DDE channel on this app and topic.  If none, 
   initialize one. 
   (setq chnl (initiate "AppName" "TopicName")) 
   0 = failure, otherwise, returns channel number 
*/
static void initiate(char *funcname)
{
    PDDE pdde;
    ARGLIST arglist[] = {{RTSTR}, {RTSTR}, {RTNONE}};

    if (!GetArgList(arglist, funcname)) {
        ads_retint(0);
        return;
    }

    pdde = DdeChannel(arglist[0].RSTR, arglist[1].RSTR);
    if (pdde)
        ads_retint(pdde->channel);    /* return the channel number to AutoCAD */
    else
        ads_retint(0);
}



/* done with a channel (remote program and work file):
   (terminate chnl) 
*/
static void terminate(char *funcname)
{
    int ret;
    PDDE pdde;
    ARGLIST arglist[] = {{RTSHORT}, {RTNONE}};

    if (!GetArgList(arglist, funcname)) {
        ads_retint(0);
        return;
    }
    ret = arglist[0].RINT;
    pdde = DdeFindChnlNum(ret);
    if (!pdde)
        ret = 0;
    else
        DdeTerminate(pdde, TRUE);
    ads_retint(ret);
}



/* request data on item (cellstr) from remote program:
   setq ddedata (request chnl cellstr) 
   ddedata is the text data sent by the remote
   DDE program. 
*/
static void request(char *funcname)
{
    char *str2;
    DWORD DdeResult;
    PDDE pdde;
    ARGLIST arglist[] = {{RTSHORT}, {RTSTR}, {RTNONE}};

    if (!GetArgList(arglist, funcname)) {
        ads_retint(0);
        return;
    }
    str2 = arglist[1].RSTR;
    pdde = DdeFindChnlNum(arglist[0].RINT);
    if (!pdde) {
        ads_retstr(NullStr);
        return;
    }
    DdeResult = DdeReqString(pdde, str2);
    if (pdde->DataSize != 0) {  
        ads_retstr(pdde->pData);      /* return the data as a string */
        DdeFreeData(pdde);
    } else
        ads_retstr(NullStr);
}


/* force data into remote program:
   (setq result (poke chnl "item" "Data"))
   1 = success 
*/
static void poke(char *funcname)
{
    char *str2, *str3;
    DWORD DdeResult;
    PDDE pdde;
    ARGLIST arglist[] = {{RTSHORT}, {RTSTR}, {RTSTR}, {RTNONE}};

    if (!GetArgList(arglist, funcname)) {
        ads_retint(0);
        return;
    }
    str2 = arglist[1].RSTR;
    str3 = arglist[2].RSTR;

    pdde = DdeFindChnlNum(arglist[0].RINT);
    if (!pdde) {
        ads_retint(0);
        return;
    }

    DdeResult = DdePokeString(pdde, str2, str3);

    /* return the data as a string */
    if (DdeResult)
        ads_retint(1);
    else
        ads_retint(0);
}


/* execute command in remote program:
   (setq result (execute chnl "ItemName" "Command"))
   1 = success 
*/
static void execute(char *funcname)
{
    char *str2;
    DWORD DdeResult;
    PDDE pdde;
    ARGLIST arglist[] = {{RTSHORT}, {RTSTR}, {RTNONE}};

    if (!GetArgList(arglist, funcname)) {
        ads_retint(0);
        return;
    }
    str2 = arglist[1].RSTR;
    pdde = DdeFindChnlNum(arglist[0].RINT);
    if (!pdde) {
        ads_retint(0);
        return;
    }

    DdeResult = DdeExec(pdde, str2);

    if (DdeResult)
        ads_retint(1);
    else
        ads_retint(0);
}





/* Request dynamic data update.  Pass channel number, item string,
   and AutoCAD command to be executed when remote data changes.
   (setq linkcommand "[(moddrawing 3 20000) ]")
   (setq result (advise chnl "R1C1:R45C4" linkcommand)) 
   1 = success 
*/
static void advise(char *funcname)
{
    char *item, *command;
    DWORD DdeResult;
    PDDE pdde;
    ARGLIST arglist[] = {{RTSHORT}, {RTSTR}, {RTSTR}, {RTNONE}};

    if (!GetArgList(arglist, funcname)) {
        ads_retint(0);
        return;
    }

    pdde = DdeFindChnlNum(arglist[0].RINT);
    if (!pdde) {
        ads_retint(0);
        return;
    }
    item = arglist[1].RSTR;
    if (item && !item[0])
        item = NULL;
    command = arglist[2].RSTR;
    if (command && !command[0])
        command = NULL;

    /*  This can be done from LISP.
    if (pdde->item != NULL)
        DdeResult = DdeUnAdvise(pdde, NULL, CF_TEXT);
    */
    DdeSetAdvise(pdde, item, command, AdsAdviseFunc);
    DdeResult = DdeAdvise(pdde, item, CF_TEXT, WARMLINK);
    if (DdeResult)
        ads_retint(1);
    else
        ads_retint(0);
}           




/* Send pdde->acadadvise, such as "[(moddrawing 3 20000) ]",
   to interrupt AutoCAD to perform command, using special
   DDE message. */
int AdsAdviseFunc(PDDE pdde)
{
    /* Interrupt AutoCAD to execute our function */
    if (pdde && pdde->acadadvise) {
        DdeSendAcadCmd(pdde->acadadvise);
    }
    return 0;
}


/* done with advise link:
   (setq result (unadvise chnl "ItemName")) 
   1 = success 
*/
static void unadvise(char *funcname)
{
    char *str2;
    DWORD DdeResult;
    PDDE pdde;
    ARGLIST arglist[] = {{RTSHORT}, {RTSTR}, {RTNONE}};

    if (!GetArgList(arglist, funcname)) {
        ads_retint(0);
        return;
    }
    str2 = arglist[1].RSTR;
    pdde = DdeFindChnlNum(arglist[0].RINT);
    if (!pdde) {
        ads_retint(0);
        return;
    }
    DdeResult = DdeUnAdvise(pdde, str2, CF_TEXT);
    if (DdeResult)
        ads_retint(1);
    else
        ads_retint(0);
}           


/* all done with DDE channels and data: 
   (ddedone) 
*/
static void ddedone(char *funcname)
{
    int chnls;

    chnls = DdeRemoveAll(TRUE); 
    ads_retint(chnls);
}


/* Convert a string to a real:
   (setq real (strreal "str"))  
*/
static void strreal(char *funcname)
{
    char *str1;
    ads_real rReal;

    ARGLIST arglist[] = {{RTSTR}, {RTNONE}};
    if (!GetArgList(arglist, funcname)) {
        ads_retint(0);
        return;
    }
    str1 = arglist[0].RSTR;
    #if 0
    rRetReal = atof(str1);
    #else
    sscanf(str1, RealScan, &rReal);
    #endif
    ads_retreal(rReal);
}


/* Convert a real to a string: 
   (setq str (realstr real))    
*/
static void realstr(char *funcname)
{
    ads_real rReal;
    char RealStr[40];
    ARGLIST arglist[] = {{RTREAL}, {RTNONE}};

    if (!GetArgList(arglist, funcname)) {
        ads_retint(0);
        return;
    }
    rReal = arglist[0].RREAL;
    RealToStr(rReal, RealStr);

    ads_retstr(RealStr);
}


/* build a spreadsheet cell item name from row & col numbers (1 based).  
   First row, col, number of rows, cols:  
   setq item (cellstr srow scol nrow ncol) 
*/
static void cellstr(char *funcname)
{
    char CellItem[40];
    ARGLIST arglist[] = {{RTSHORT}, {RTSHORT}, {RTSHORT}, {RTSHORT},
                                        {RTNONE}};
    if (!GetArgList(arglist, funcname)) {
        ads_retint(0);
        return;
    }
    CellRange(CellItem, arglist[0].RINT, 
                arglist[1].RINT, 
                arglist[2].RINT, 
                arglist[3].RINT); 
    ads_retstr(CellItem);
}


/* build a cell item string from an entity, row, and col (finds
   length of entity and uses it for number of rows):
   (setq cell (entcell 4 1))  
*/
static void entcell(char *funcname)
{
    USHORT nRow;
    ads_name EntName;
    RESBUF *pEnt = NULL;
    char CellItem[40];
    ARGLIST arglist[] = {{RTENAME}, {RTSHORT}, {RTSHORT}, {RTNONE}};

    if (!GetArgList(arglist, funcname)) {
        ads_retint(0);
        return;
    }
    EntName[0] = arglist[0].RLNAME[0];
    EntName[1] = arglist[0].RLNAME[1];

    /* wait 'til all args are formed before
       calling entxxx functions.  */
    pEnt = ads_entget(EntName);       /* get start node of linked
                                              list of sub-entities */
    if (!pEnt) {
        ads_retstr(NullStr);
        return;
    }
    nRow = GetEntLen(pEnt);
    CellRange(CellItem, arglist[1].RINT, 
                arglist[2].RINT, 
                nRow, 4);
    ads_retstr(CellItem);

    if (pEnt)
        ads_relrb(pEnt);
}


/* C Format string for conversion of reals (doubles) to string.
   Change 6 to whatever precision is needed:
   (precision "%+.6lf") 
*/
static void cformat(char *funcname)
{
    ARGLIST arglist[] = {{RTSTR}, {RTNONE}};

    if (!GetArgList(arglist, funcname)) {
        ads_retint(0);
        return;
    }
    strcpy(RealFmt, arglist[0].RSTR);
    ads_retint(1);
}


/* Send the entity, ename, to the remote program on chnl, item (cell
   range), return number of rows:
   setq numrows 
   (entpoke chnl item ename) 
*/
static void fentpoke(char *funcname)
{
    DWORD DdeResult;
    USHORT NumRows;
    PDDE pdde;
    ads_name EntName;
    RESBUF *pEnt = NULL;
    char *str2;
    ARGLIST arglist[] = {{RTSHORT}, {RTSTR}, {RTENAME}, {RTNONE}};

    if (!GetArgList(arglist, funcname)) {
        ads_retint(0);
        return;
    }
    str2 = arglist[1].RSTR;
    pdde = DdeFindChnlNum(arglist[0].RINT);
    if (!pdde) {
        ads_retint(0);
        return;
    }
    EntName[0] = arglist[2].RLNAME[0];
    EntName[1] = arglist[2].RLNAME[1];
    pEnt = ads_entget(EntName);       /* get start node of linked
                                              list of sub-entities */
    if (!pEnt) {
        ads_retint(0);
        return;
    }

    /* send entity data to remote app */
    DdeResult = EntPoke(pdde, str2, pEnt, ET_NORM);
    if (DdeResult) {
        NumRows = GetEntLen(pEnt);
        ads_retint(NumRows);
    }
    else
        ads_retint(0);
    if (pEnt)
        ads_relrb(pEnt);
}


/* Send the entity, ename, to the remote program on chnl, row, col.
   Return number of rows:
   setq numrows 
   (entpokecell chnl ename row col) 
*/
static void fentpokecell(char *funcname)
{
    int NumRows;
    PDDE pdde;
    ads_name EntName;
    RESBUF *pEnt = NULL;
    ARGLIST arglist[] = {{RTSHORT},{RTENAME},{RTSHORT},{RTSHORT},{RTNONE}};
    if (!GetArgList(arglist, funcname)) {
        ads_retint(0);
        return;
    }
    pdde = DdeFindChnlNum(arglist[0].RINT);
    if (!pdde) {
        ads_retint(0);
        return;
    }

    EntName[0] = arglist[1].RLNAME[0];
    EntName[1] = arglist[1].RLNAME[1];

    if (!EntName[0]) {
        ads_retint(0);
        return;
    }
    pEnt = ads_entget(EntName);       /* get start node of linked
                                      list of sub-entities */
    if (!pEnt) {
        ads_retint(0);
        return;
    }

    NumRows = EntPokeCell(pdde, arglist[2].RINT, 
               arglist[3].RINT, pEnt, ET_NORM);
    ads_retint(NumRows);
    /* NumRows returned. */
    if (pEnt)
        ads_relrb(pEnt);
}


/* get length of an entity (items in sub-list) 
   (setq elen (entlen ename)) 
*/
static void entlen(char *funcname)
{
    int ret;
    ads_name EntName;
    RESBUF *pEnt = NULL;
    ARGLIST arglist[] = {{RTENAME}, {RTNONE}};

    if (!GetArgList(arglist, funcname)) {
        ads_retint(0);
        return;
    }
    EntName[0] = arglist[0].RLNAME[0];
    EntName[1] = arglist[0].RLNAME[1];
    pEnt = ads_entget(EntName);       /* get start node of linked
                                              list of sub-entities */
    ret = (int)GetEntLen(pEnt);
    ads_retint(ret);        
    if (pEnt)
        ads_relrb(pEnt);
}


/* Request data on chnl and item.  Convert to RESBUFs
   and use them to modify a drawing with entmod: 
   setq item (cellstr srow scol nrow ncol) 
   setq rowcnt (reqmod chnl item)
*/
static void freqmod(char *funcname)
{
    int ret;
    PDDE pdde;
    char *str2;
    int more;
    ARGLIST arglist[] = {{RTSHORT}, {RTSTR}, {RTNONE}};

    if (!GetArgList(arglist, funcname)) {
        ads_retint(0);
        return;
    }
    str2 = arglist[1].RSTR;
    pdde = DdeFindChnlNum(arglist[0].RINT);
    if (!pdde) {
        ads_retint(0);
        return;
    }
    ModCnt = ReqMod(pdde, str2, &ret, &more);
    ads_retint(ret);
}



/* poke selection set, starting with
   ename, into remote
   program at row, col.  
   (setq rowcnt (pokeset chnl ename row col)) 
*/
static void fpokeset(char *funcname)
{
    int ret;
    int row, col;
    PDDE pdde;
    ARGLIST arglist[] = {{RTSHORT},{RTENAME},{RTSHORT},{RTSHORT},{RTNONE}};

    if (!GetArgList(arglist, funcname)) {
        ads_retint(0);
        return;
    }
    pdde = DdeFindChnlNum(arglist[0].RINT);
    if (!pdde) {
        ads_retint(0);
        return;
    }
    row = arglist[2].RINT;
    col = arglist[3].RINT;
    ret = PokeSet(pdde, arglist[1].RLNAME, NULL, row, col);
    ads_retint(ret);
}


/* poke all entities, starting with
   ename, into remote
   program at row, col.  
   (setq rowcnt (pokeall chnl ename row col)) 
*/
static void fpokeall(char *funcname)
{
    int ret;
    int row, col;
    PDDE pdde;
    ARGLIST arglist[] = {{RTSHORT},{RTENAME},{RTSHORT},{RTSHORT},{RTNONE}};

    if (!GetArgList(arglist, funcname)) {
        ads_retint(0);
        return;
    }

    pdde = DdeFindChnlNum(arglist[0].RINT);
    if (!pdde) {
        ads_retint(0);
        return;
    }
    row = arglist[2].RINT;
    col = arglist[3].RINT;
    ret = PokeEntList(pdde, row, col, arglist[1].RLNAME, ET_NORM);
    ads_retint(ret);
}



/* Set up conversion parameters.  
   1 - floating point digits of accuracy
   2 - fuzz factor--if difference of reals is less than this,
                                consider them equal
   (fprecision 6 0.0001) 
*/
static void fprecision(char *funcname)
{
    ARGLIST arglist[] = {{RTSHORT}, {RTREAL}, {RTNONE}};

    if (!GetArgList(arglist, funcname)) {
        ads_retint(0);
        return;
    }
    ConvPrecision(arglist[0].RINT, arglist[1].RREAL);
    ads_retint(1);
}


/* Set up text conversion parameters.  
   1 - entity to text format field separator
   2 - entity to text format line  separator
   (entformat "\t" "\r\n") 
*/
static void fentformat(char *funcname)
{
    ARGLIST arglist[] = {{RTSTR}, {RTSTR}, {RTNONE}};

    if (!GetArgList(arglist, funcname)) {
        ads_retint(0);
        return;
    }
    ConvSeparators(arglist[0].RSTR, arglist[1].RSTR);
    ads_retint(1);
}


/* List all the open DDE channels 
*/
static void listchnls(char *funcname)
{
    AcPrintChnlList();
    ads_retint(1);
}



/* Free the data received from the request function
   (ddefree chnl)  
*/
static void fddefree(char *funcname)
{
    PDDE pdde;
    ARGLIST arglist[] = {{RTSHORT}, {RTNONE}};

    if (!GetArgList(arglist, funcname)) {
        ads_retint(0);
        return;
    }
    pdde = DdeFindChnlNum(arglist[0].RINT);
    if (!pdde) {
        ads_retint(0);
        return;
    }
    DdeFreeData(pdde);  
    ads_retint(1);
}





/* Get the length of a table, such as "BLOCK".  
   (tbllen 6)    ;"BLOCK" 

                        VPORT 0
                        LTYPE 1
                        LAYER 2
                        STYLE 3
                        VIEW  4
                        UCS   5
                        BLOCK 6
*/
static void lentbl(char *funcname)
{
    int ret;
    ARGLIST arglist[] = {{RTSHORT}, {RTNONE}};

    if (!GetArgList(arglist, funcname)) {
        ads_retint(0);
        return;
    }
    ret = LenTable(arglist[0].RINT);
    ads_retint(ret);
}


/* Send a table, such as "BLOCK", to a remote program on 
   chnl, item (cell range).
   (setq numrows (poketbl chnl item 6)) ;6 is BLOCK 
*/
static void poketbl(char *funcname)
{
    int ret;
    PDDE pdde;
    char *str2;
    ARGLIST arglist[] = {{RTSHORT}, {RTSTR}, {RTSHORT}, {RTNONE}};

    if (!GetArgList(arglist, funcname)) {
        ads_retint(0);
        return;
    }
    pdde = DdeFindChnlNum(arglist[0].RINT);
    if (!pdde) {
        ads_retint(0);
        return;
    }
    str2 = arglist[1].RSTR;    

    ret = PokeTable(pdde, arglist[2].RINT, str2, 0, 0, FALSE);
    ads_retint(ret);
}


/* Send the table (such as "BLOCK") to a 
   spreadsheet row, col.  Optimized
   for spreadsheet application.
   (setq numrows (poketblcell chnl row col tblnum)) 
*/
static void poketblcell(char *funcname)
{
    int ret;
    PDDE pdde;
    int row, col;
    ARGLIST arglist[] = {{RTSHORT},{RTSHORT},{RTSHORT},{RTSHORT},{RTNONE}};

    if (!GetArgList(arglist, funcname)) {
        ads_retint(0);
        return;
    }
    pdde = DdeFindChnlNum(arglist[0].RINT);
    if (!pdde) {
        ads_retint(0);
        return;
    }
    row = arglist[1].RINT;
    col = arglist[2].RINT;
    ret = PokeTable(pdde, (int)arglist[3].RINT, NULL, row, col, TRUE);
    ads_retint(ret);
}


/* Send all table info to spreadsheet application.
   (setq numrows (pokealltbl chnl row col)) 
*/
static void fpokealltbl(char *funcname)
{
    int ret;
    PDDE pdde;
    ARGLIST arglist[] = {{RTSHORT},{RTSHORT},{RTSHORT},{RTNONE}};

    if (!GetArgList(arglist, funcname)) {
        ads_retint(0);
        return;
    }

    pdde = DdeFindChnlNum(arglist[0].RINT);
    if (!pdde) {
        ads_retint(0);
        return;
    }
    ret = PokeAllTables(pdde, arglist[1].RINT, arglist[2].RINT);
    ads_retint(ret);
}


/* Get row length of all the tables.
   (setq len (lenalltbl)) 
*/
static void lenalltbl(char *funcname)
{
    int ret;

    ret = (USHORT)LenAllTable();
    ads_retint(ret);
}


/* Get number of entities modified on the
   last call to reqmod.
   (setq modnum (modcount)) */
static void modcount(char *funcname)
{
    ads_retint(ModCnt);
}


/* request data on item from remote program:
   setq ddedata (request chnl item) 
   ddedata is the text data sent by the remote
   DDE program. 
*/
static void reqlarge(char *funcname)
{
    DWORD DdeResult;
    PDDE pdde;
    ads_name EntName;
    char *str2;
    ARGLIST arglist[] = {{RTSHORT},{RTSTR},{RTNONE}};

    EntName[0] = EntName[1] = 0L;
    if (!GetArgList(arglist, funcname)) {
        ads_retname(EntName, RTENAME);
        return;
    }
    str2 = arglist[1].RSTR;
    pdde = DdeFindChnlNum(arglist[0].RINT);
    if (!pdde) {
        ads_retname(EntName, RTENAME);
        return;
    }
    DdeResult = DdeReqString(pdde, str2);
    if (pdde->DataSize != 0)   
        EntName[0] = (long)pdde->pData;
    ads_retname(EntName, RTENAME);    /* Just use long of ename
                                      for data pointer return value. */ 
}


/* From a pointer to a stream of text data returned by REQLARGE,
   convert to RESBUFs and use them to modify a drawing
   with entmod: 
        setq item (cellstr srow scol nrow ncol) 
        setq ddedata (reqlarge chnl cellstr) 
        setq rowcnt (modlarge ddedata) 
*/
static void modlarge(char *funcname)
{
    int mod, NumRows;
    ARGLIST arglist[] = {{RTENAME}, {RTNONE}};

    if (!GetArgList(arglist, funcname)) {
        ads_retint(0);
        return;
    }
    mod = ModTextStream((char *)arglist[0].RLNAME[0], &NumRows);
    ads_retint(NumRows);
}


/* Send all tables and drawing entities to a 
   spreadsheet, row, col 1, and return the number of
   rows used.  
   (setq rowcnt (senddrawing row)) 
*/
static void fsenddrawing(char *funcname)
{
    int ret;
    PDDE pdde;
    ARGLIST arglist[] = {{RTSHORT},{RTNONE}};

    if (!GetArgList(arglist, funcname)) {
        ads_retint(0);
        return;
    }
    if (!(pdde = pDdeData->CurChnl)) {
        ads_retint(0);
        return;
    }
    ret = PokeDrawing(pdde, arglist[0].RINT, 1);
    ads_retint(ret);
}


/* Modify the drawing, after requesting data starting with
   row1 and for number of rows, rowcnt.   
   (setq modcnt (moddrawing row1 rowcnt)) 
*/
static void fmoddrawing(char *funcname)
{
    int ret;
    PDDE pdde;
    ARGLIST arglist[] = {{RTSHORT},{RTSHORT},{RTNONE}};

    if (!GetArgList(arglist, funcname)) {
        ads_retint(0);
        return;
    }
    if (!(pdde = pDdeData->CurChnl)) {
        ads_retint(0);
        return;
    }
    ret = ModDrawing(pdde, arglist[0].RINT, 1, arglist[1].RINT);
    ads_printf("\nEntities modified: ");
    ads_retint(ret);
}



/* get current DDE channel number (global)
   (setq chnl (getddechnl)) 
*/
static void getddechnl(char *funcname)
{
    int ret = 0;
    PDDE pdde;

    if (pdde = DdeGetCurChnl())
        ret = pdde->channel;
    ads_retint(ret);
}




/* Set DDE entity-to-text format.  See ddeconv.h for details on format
   numbers.
   (setddeformat 2) 
*/
static void setddeformat(char *funcname)
{
    int Frmt;
    ARGLIST arglist[] = {{RTSHORT},{RTNONE}};

    if (!GetArgList(arglist, funcname)) {
        ads_retint(0);
        return;
    }
    Frmt = arglist[0].RINT;
    SetDdeFormat(Frmt);
    ads_retint(Frmt);
}


/* Get DDE entity-to-text format.  See dde.doc for details on format
   numbers.
   (setq frmt (getddeformat)) 
*/
static void getddeformat(char *funcname)
{
    int Frmt = GetDdeFormat();
    ads_retint(Frmt);
}


/*
                        Commands
*/

/* List function names in this module.  Return number
   of functions. 
*/
static void listfunc(char *funcname)
{
    int idx;
    char *cname;

    for (idx = 0; idx < ELEMENTS(cmdtab); ++idx) {
        cname = cmdtab[idx].cmdname;
        ads_printf(cname);
        ads_printf("\n");
    }
    ads_retint(idx);
}





/* end of file */

