/*

        DdeADS

        Dynamic Data Exchange functions to be called by AutoCAD
        for the transfer of AutoCAD data to and from other 
        programs.

        by Phil Ford
     ________________________________________________________________________

      (C) Copyright 1990-1994 by Autodesk, Inc.

      Permission to use, copy, modify, and distribute this software and its
      documentation for any purpose and without fee is hereby granted.  

      THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY. 
      ALL IMPLIED WARRANTIES OF FITNESS FOR ANY PARTICULAR PURPOSE AND OF 
      MERCHANTABILITY ARE HEREBY DISCLAIMED.                                
     ________________________________________________________________________


/*
See "Dde.DOC".
*/



#include "options.h"

#include <stdio.h>
#include <string.h>
#include <memory.h>
#include <ctype.h>

#include "adslib.h"    
#include "windde.h"
#include "winutil.h"
#include "ddewin.h"
#include "ddeconv.h"
#include "spreadsh.h"
#include "acaddefs.h"

#define STARTROW 3
#define STARTCOL 1


ddeDATA *pDdeData;                    /* Global DDE data */     
PDDE pdde = NULL;                     /* Current channel data */        
char Application[] = "Excel";         /* Default application name */


/* All Function Prototypes for ddeads.c */

void main(int argc, char **argv);
static short funcload(void);
static void fddedlgstart(char *funcname);
static void fddedialog(char *funcname);
static void fddeblocks(char *funcname);
static void fddedrawing(char *funcname);
static void fddesset(char *funcname);
static void fddeupdate(char *funcname);
static void fddeformat(char *funcname);
static void fddedefaults(char *funcname);
static int chk_dde(int pokeflag);
static int AdviseFunc(PDDE pdde);
static void set_advise(PDDE pdde);



/* Table of AutoCAD function (or command) names
   and corresponding C function name 
*/
CMDTAB cmdtab[] = {
    /*   Name    Function  */

    {"DDEDLGSTART", fddedlgstart},
    {"DDEDIALOG", fddedialog},
    {"DDEDRAWING", fddedrawing},
    {"DDEBLOCKS", fddeblocks},
    {"DDESSET", fddesset},
    {"DDEUPDATE", fddeupdate},
    {"DDEFORMAT", fddeformat},
    {"DDEDEFAULTS", fddedefaults},
};


/*

                  DdeADS Main

*/



void main(int argc, char **argv)
{
    short scode = RSRSLT;
    short cindex;
    short ads_req; 

    /* Initialize DDE Manager with our program name, 
       the instance handle, and the DDEML instance handle.  
       Data from DDE dialog will be put in pDdeData data 
       structure */
    pDdeData = DdeInit(adsw_AppName, adsw_hInstance, adsw_idInst);

    ads_init(argc, argv); 

#ifdef SHOW_AS_ICON
    /* Display ADS window as icon */
    ShowWindow(adsw_hWnd, SW_SHOWMINNOACTIVE);
    UpdateWindow(adsw_hWnd);
#endif

    for ( ;; ) {

        /* This process will block (stop) here until AutoCAD has a 
           request for us */

        ads_req = ads_link(scode);        
        if (ads_req < 0)
            break;

        scode = RSRSLT;               /* default return code */

        /* Check for AT LEAST the following cases here */
        switch (ads_req) { 

        case RQXLOAD:          /* Load request. Send function names. */
            scode = -(funcload() ? RSRSLT : RSERR);
            break;

        /* Execute a "loaded" function that was defined via RQXLOAD */
        case RQSUBR:
            cindex = ads_getfuncode();

            if (cindex >= 0 && cindex < ELEMENTS(cmdtab))
                (*cmdtab[cindex].cmdfunc)(cmdtab[cindex].cmdname);
            break;
        case RQSAVE:
        case RQEND:
        case RQQUIT:
            /* Free the list of constant block attributes */    
            FreeConstList(NULL);
            break;

        /* (xunload "adsw_AppName") entered or called in AutoCAD */ 
        case RQXUNLD:      
            /* Free up all memory allocated by DDE system.  Send 
               TERMINATE messages to open channels. */
            DdeQuit();
            break;

        default:
            break;
        }
    }           /* end of ADS loop */

}





/* Register our functions with the ADS system.  Also,
   any drawing initialization can go here, since this is
   called with each new drawing. 
*/
static short funcload(void)
{
    int idx;
    char *cname;

    for (idx = 0; idx < ELEMENTS(cmdtab); ++idx) {
        cname = cmdtab[idx].cmdname;
        ads_defun(cname, (short)idx);
    }

    SetFlatLand();                    /* set local flatland variable, for 
                                      dxftype function. */
    return TRUE;
}



/* Try to link to default application and work file.  If fail,
   put up a dialog asking for app and topic name, try to 
   start app and link up with DDE.  Set forcedlg TRUE to enter
   the dialog immediately without the attempt to find a 
   channel */

static void fddedlgstart(char *funcname)
{
    pdde = DdeDlgStart(adsw_hWnd, FALSE);
}

/* Run DDE dialog */

static void fddedialog(char *funcname)
{
    DdeDialog(adsw_hWnd);
}

/* Export Block data from the tables. */

static void fddeblocks(char *funcname)
{
    int advflag = pDdeData->flags.advise;
    int rows;

    if (!chk_dde(TRUE)) {
        ads_retint(0);
        return;
    }
    /* Send tables, drawing */
    rows = PokeTable(pdde, BLOCK, NULL, STARTROW, STARTCOL, TRUE);
    set_advise(pdde);
    ads_printf("\nTotal rows: ");
    ads_retint(rows);
}

/* Export Drawing, including tables.  Return number of rows used in
   spreadsheet. */

static void fddedrawing(char *funcname)
{
    int advflag = pDdeData->flags.advise;
    int rows;

    if (!chk_dde(TRUE)) {
        ads_retint(0);
        return;
    }
    /* Send tables, drawing */
    rows = PokeDrawing(pdde, STARTROW, 1);
    set_advise(pdde);
    ads_printf("\nTotal rows: ");
    ads_retint(rows);
}

/* Export selection set.  Return number of rows. */

static void fddesset(char *funcname)
{
    int advflag = pDdeData->flags.advise;
    ads_name ss;
    int stat;
    UINT rows;

    if (!chk_dde(TRUE)) {
        ads_retint(0);
        return;
    }
    stat = ads_ssget(NULL, NULL, NULL, NULL, ss);
    if (stat == RTNORM) {
        rows = PokeSet(pdde, ss, NULL, STARTROW, STARTCOL);
        ads_ssfree(ss);
        set_advise(pdde);
    }
    ads_printf("\nTotal rows: ");
    ads_retint(rows);
}


/* Update drawing, return number of entities modified */

static void fddeupdate(char *funcname)
{
    UINT modcnt;

    if (!chk_dde(FALSE)) {
        ads_retint(0);
        return;
    }
    modcnt = ModDrawing(pdde, STARTROW, 1, 20000);
    ads_printf("\nNumber of modifications: ");
    ads_retint(modcnt);
}

/* Set default app, topic, path */

static void fddedefaults(char *funcname)
{
    ARGLIST arglist[] = {{RTSTR}, {RTSTR}, {RTSTR}, {RTNONE}};

    if (!GetArgList(arglist, funcname)) {
        ads_retint(0);
        return;
    }
    DdeDefaults(arglist[0].RSTR, arglist[1].RSTR, arglist[2].RSTR);
    ads_retint(1);
}

static void fddeformat(char *funcname)
{
    ARGLIST arglist[] = {{RTSHORT}, {RTNONE}};
    int frmt;

    if (!GetArgList(arglist, funcname)) {
        ads_retint(0);
        return;
    }
    frmt = arglist[0].RINT;
    SetDdeFormat(frmt);
    ads_retint(frmt);
}


/* Check for channel, attempt to start spreadsheet and initiate 
   DDE channel, put up dialog if fail and start over */

static int chk_dde(int pokeflag)
{

    /* See if we have a channel with default app, topic */
    pdde = DdeFindChnl(pDdeData->app, pDdeData->topic);
    if (pdde == NULL) {
        /* Try to create link, put up dialog if unable */
        pdde = DdeDlgStart(adsw_hWnd, FALSE);
        if (!pdde)
            return FALSE;
    }
    /* Set up hot link command and function.  We don't have an
       item yet, until data is sent and rows are counted. */
    DdeSetAdvise(pdde, NULL, "(ddeupdate) ", AdviseFunc);
    if (pokeflag && pdde->ChannelState.advise_on) {
        /* We're about to change data in the spreadsheet.  Tell other 
           app to forget previous hot link */
        DdeAdviseCur(pdde, FALSE, WARMLINK);
#ifdef CV
        ads_printf("\nAdvise OFF for %s\n", pdde->item);
#endif
    }
    return TRUE;
}


static void set_advise(PDDE pdde)
{
    if (pDdeData->flags.advise) {
        /* Tell other app to start up a warm link */
        DdeAdviseCur(pdde, TRUE, WARMLINK);
#ifdef CV
        ads_printf("\nAdvise ON for %s\n", pdde->item);
#endif
    }
}


/* Send command such as "ddeupdate " to interrupt AutoCAD 
   to perform command (warm link function). */
static int AdviseFunc(PDDE pdde)
{
    /* Interrupt AutoCAD to execute our function */
    if (pdde && pdde->acadadvise) {
        DdeSendAcadCmd(pdde->acadadvise);
    }
    return 0;
}


/* end of file */
