/* 
    ADSSTRNG.H -
    
    This file:

        Declares ADS_STRING.  This may seems excessive given
        most compilers supply their version of CString, 
        But unfortunately some major compiler vendor exclude CString
        from their stand C++ library.

    (C) Copyright 1988-1994 by Autodesk, Inc.

    This program is copyrighted by Autodesk, Inc. and is  licensed
    to you under the following conditions.  You may not distribute
    or  publish the source code of this program in any form.   You
    may  incorporate this code in object form in derivative  works
    provided  such  derivative  works  are  (i.) are  designed and
    intended  to  work  solely  with  Autodesk, Inc. products, and
    (ii.)  contain  Autodesk's  copyright  notice  "(C)  Copyright
    1988-1994 by Autodesk, Inc."

    AUTODESK  PROVIDES THIS PROGRAM "AS IS" AND WITH  ALL  FAULTS.
    AUTODESK  SPECIFICALLY DISCLAIMS ANY IMPLIED WARRANTY OF  MER-
    CHANTABILITY OR FITNESS FOR A PARTICULAR USE.  AUTODESK,  INC.
    DOES  NOT  WARRANT THAT THE OPERATION OF THE PROGRAM  WILL  BE
    UNINTERRUPTED OR ERROR FREE.

*/
#ifndef __ADSSTRNG_H
#define __ADSSTRNG_H

//-----------------------------------------------------------------------------
class   ADS_REFERENCE;
class   STRING_REF;
struct  ADS_STRING;

//-----------------------------------------------------------------------------
const size_t NPOS = size_t(-1);

/******************************************************************************
*                                                                             *
*                            ADS_STRING class                                 *
*    This class is used because the standard CString class is not available   *
*    among all the compilers and platform.  Under MS, using CString requires  *
*    MFC. Note: This is NOT a COMPLETE CString class!!!                       *
*                                                                             *
******************************************************************************/
struct ADS_STRING : ADS_OBJ
{   
    //
    // constructors
    //
                        ADS_STRING();
    					ADS_STRING( const ADS_STRING& );
                        ADS_STRING( const ADS_STRING& s, const char *cp );
                        ADS_STRING( const char *cp );
    // destrcutor
    virtual             ~ADS_STRING();

    //
    // Assignment
    //
    ADS_STRING&         operator = ( const ADS_STRING& );
    ADS_STRING&         Assign( const ADS_STRING  &s );
    ADS_STRING&         Assign( const ADS_STRING  &s, size_t orign, size_t n );

    //
    // Comparision
    //
    int                 Compare(const ADS_STRING  &s) const;

    friend int   operator == ( const char  *cp, const ADS_STRING  &s );
    friend int   operator == ( const ADS_STRING  &s1, const ADS_STRING  &s2 );
    friend int   operator == ( const ADS_STRING  &s, const char  *cp );

    friend int   operator != ( const char  *cp, const ADS_STRING  &s );
    friend int   operator != ( const ADS_STRING  &s1, const ADS_STRING  &s2 );
    friend int   operator != ( const ADS_STRING  &s, const char  *cp );

    friend int   operator <  ( const ADS_STRING  &s1, const ADS_STRING  &s2 );
    friend int   operator <  ( const ADS_STRING  &s, const char  *cp );
    friend int   operator <  ( const char  *cp, const ADS_STRING  &s );

    friend int   operator <= ( const ADS_STRING  &s1, const ADS_STRING  &s2 );
    friend int   operator <= ( const ADS_STRING  &s, const char  *cp );
    friend int   operator <= ( const char  *cp, const ADS_STRING  &s );

    friend int   operator >  ( const ADS_STRING  &s1, const ADS_STRING  &s2 );
    friend int   operator >  ( const ADS_STRING  &s, const char  *cp );
    friend int   operator >  ( const char  *cp, const ADS_STRING  &s );

    friend int   operator >= ( const ADS_STRING  &s1, const ADS_STRING  &s2 );
    friend int   operator >= ( const ADS_STRING  &s, const char  *cp );
    friend int   operator >= ( const char  *cp, const ADS_STRING  &s );

    //
    // Concatenation
    //
    ADS_STRING&         Append( const ADS_STRING   &s );
    ADS_STRING&         Append( const char   *cp );
    ADS_STRING&         Append( const ADS_STRING &s
                                , size_t orig
                                , size_t n = NPOS );

    ADS_STRING&         operator += ( const char *cp );
    ADS_STRING&         operator += ( const ADS_STRING   &s );

    friend ADS_STRING   operator + ( const ADS_STRING &s, const char *cp );
    friend ADS_STRING   operator + ( const ADS_STRING &s, const ADS_STRING &d );

    //
    // Subscripting
    //
    char                GetAt( size_t pos );
    void                PutAt( size_t pos, char c );

    char&               operator[]( size_t pos ) ;
    char&               operator()( size_t pos ) ;

    //
    // Searching
    //
    size_t              Find( const ADS_STRING  &s ) const;
    size_t              Find( const ADS_STRING  &s, size_t pos ) const;
    size_t              Rfind( const ADS_STRING  &s ) const;
    size_t              Rfind( const ADS_STRING  &s, size_t pos ) const;

    int                 Contains( const char  *pat ) const;
    int                 Contains( const ADS_STRING  &s ) const;

    //
    // Character set searching
    //
    size_t              FindFirstOf( const ADS_STRING &s ) const;
    size_t              FindFirstOf( const ADS_STRING &s, size_t pos ) const;
    size_t              FindLastOf( const ADS_STRING &s ) const;
    size_t              FindLastOf( const ADS_STRING &s, size_t pos ) const;

    //
    // Miscellaneous
    //
    size_t              Length() const;
    size_t              Copy( char *cb, size_t n );
    size_t              Copy( char *cb, size_t n, size_t pos );
    const char*         CString() const;
    static size_t       GetInitialCapacity();
    static size_t       GetResizeIncrement();
    static size_t       GetMaxWaste();

    void                CopyOnWrite();
    void                Clone();


    virtual BOOL        Valid();
    virtual void WhoAmI( char *buf )
    {
        if ( buf )
            strcpy( buf, "ADS_STRING" );
    }

    static size_t       initial_capac;
    static size_t       resize_inc;
    static size_t       max_waste;

private:
    STRING_REF         *p;
};

//-----------------------------------------------------------------------------
//      Related global functions
//
istream&    operator >> ( istream &is, ADS_STRING &s );
ostream&    operator << ( ostream &os, const ADS_STRING &s );

/******************************************************************************
*                                                                             *
*                          ADS_REFERENCE                                      *
*                                                                             *
******************************************************************************/
class ADS_REFERENCE : ADS_OBJ
{
public:

                        ADS_REFERENCE( size_t initRef = 0 )
                                    : refs( initRef ) 
                        { }
    void                AddReference() 
                        { 
                            refs++; 
                        }
    size_t              References() 
                        { 
                            return refs; 
                        }
    size_t              RemoveReference() 
                        { 
                            return --refs; 
                        }
    virtual BOOL        Valid()
                        { 
                            return TRUE;
                        }

private:
    size_t refs;
private:
    BASIC_CPP_STUFF(ADS_REFERENCE)
};

/******************************************************************************
*                                                                             *
*                            STRING_REF                                       *
*                                                                             *
******************************************************************************/
class STRING_REF : ADS_REFERENCE
{
    friend struct ADS_STRING;

    //
    // Data
    //
    char            *array;
    size_t          nchars;
    size_t          capacity;

    //
    // Constructors
    //
                    STRING_REF( const char *str1
                            , size_t count1
                            , const char *str2
                            , size_t count2
                            , size_t extra );
    //
    // Destructor
    //
                    ~STRING_REF();

    //
    // Miscellaneous
    //
    void            GrowTo( size_t n );
    size_t          RoundCapacity( size_t nc );
    void            Splice( size_t start
                            , size_t extent
                            , const char *cp
                            , size_t n );

private:
    BASIC_CPP_STUFF(STRING_REF)
};


/******************************************************************************
*                                                                             *
*           INLINES for STRING_REF, ADS_REF, and ADS_STRING                   *
*                                                                             *
******************************************************************************/
//-----------------------------------------------------------------------------
//
//                          ADS_STRING
//
//-----------------------------------------------------------------------------
inline ADS_STRING::~ADS_STRING()
{
    if ( p->RemoveReference() == 0 )
    {
        delete p;
    }
}

//-----------------------------------------------------------------------------
inline ADS_STRING::ADS_STRING( const ADS_STRING& src )
{
    p = src.p;
    p->AddReference();
}

//-----------------------------------------------------------------------------
inline ADS_STRING::ADS_STRING( const char *cp )
{
    p = new STRING_REF( cp, cp?strlen( cp ):0, 0, 0, 0 );
}

//-----------------------------------------------------------------------------
inline int ADS_STRING::Compare(const ADS_STRING  &s) const
{
    return strcmp( CString(), s.CString() );
}

//-----------------------------------------------------------------------------
inline ADS_STRING::ADS_STRING()
{
    p = new STRING_REF( NULL, 0, 0, 0, ADS_STRING::GetInitialCapacity() );
}

//-----------------------------------------------------------------------------
inline ADS_STRING::ADS_STRING( const ADS_STRING& s, const char *cp )
{
    p = new STRING_REF( s.CString(), s.Length(), cp, cp?strlen(cp):0, 0 );
}

//-----------------------------------------------------------------------------
inline size_t ADS_STRING::GetInitialCapacity()
{
    return initial_capac;
}

//-----------------------------------------------------------------------------
inline size_t ADS_STRING::GetResizeIncrement()
{
    return resize_inc;
}

//-----------------------------------------------------------------------------
inline size_t ADS_STRING::GetMaxWaste()
{
    return max_waste;
}

//-----------------------------------------------------------------------------
inline ADS_STRING& ADS_STRING::operator = ( const ADS_STRING& src)
{
    return Assign( src, 0, NPOS );
}

//-----------------------------------------------------------------------------
inline ADS_STRING& ADS_STRING::Assign( const ADS_STRING  &src )
{
    return Assign( src, 0, NPOS );
}

//-----------------------------------------------------------------------------
inline const char * ADS_STRING::CString() const
{
    return p->array;
}

//-----------------------------------------------------------------------------
inline void ADS_STRING::CopyOnWrite()
{
    if ( p->References() > 1 )
    {
        Clone();
    }
}

//-----------------------------------------------------------------------------
inline BOOL ADS_STRING::Valid()
{
    return p != NULL;
}

//-----------------------------------------------------------------------------
inline size_t ADS_STRING::Length() const
{
    return p->nchars;
}

//-----------------------------------------------------------------------------
inline char& ADS_STRING::operator[]( size_t pos )
{
    return (*this)(pos);    // use operator()
}

//-----------------------------------------------------------------------------
inline char& ADS_STRING::operator()( size_t pos )
{
    return p->array[pos];
}

//-----------------------------------------------------------------------------
inline ADS_STRING& ADS_STRING::operator += ( const ADS_STRING &src )
{
    return Append( src, 0, NPOS );
}

//-----------------------------------------------------------------------------
inline ADS_STRING& ADS_STRING::operator += ( const char *cp )
{
    return Append( cp, 0, strlen(cp) );
}

//-----------------------------------------------------------------------------
inline ADS_STRING& ADS_STRING::Append( const ADS_STRING   &s )
{
    return Append(s, 0, NPOS);
}

//-----------------------------------------------------------------------------
inline ADS_STRING& ADS_STRING::Append( const char   *cp )
{
    return Append( cp, 0, strlen(cp) );
}
//-----------------------------------------------------------------------------
inline void ADS_STRING::Clone()
{
    p->RemoveReference();
    p = new STRING_REF( CString(), Length(), 0, 0, 0 );
}

//-----------------------------------------------------------------------------
//  friends...
//
//-----------------------------------------------------------------------------
inline int operator == ( const ADS_STRING &s1, const ADS_STRING &s2 )
{
    return s1.Compare( s2 ) == 0;
}

//-----------------------------------------------------------------------------
inline int operator == ( const ADS_STRING& s1, const char *s2 )
{
    return s1.Compare( s2 ) == 0;
}

//-----------------------------------------------------------------------------
inline int operator == ( const char  *cp, const ADS_STRING  &s )
{
    return ADS_STRING( cp ).Compare( s ) == 0;
}

//-----------------------------------------------------------------------------
inline ADS_STRING operator + ( const ADS_STRING& s, const char *cp )
{
    return ADS_STRING( s, cp );
}

//-----------------------------------------------------------------------------
inline ADS_STRING operator + ( const ADS_STRING &s1, const ADS_STRING &s2 )
{
    return s1 + s2.CString();
}

//-----------------------------------------------------------------------------
inline ostream& operator << ( ostream &os, const ADS_STRING &s )
{
    return os << s.CString();
}

//-----------------------------------------------------------------------------
inline STRING_REF::~STRING_REF()
{
    delete array;
}

#endif
