/*****************************************************************************
    Name: asiclass.h                           ast_asi_inc_appl_asiclass_h

    Description:  Header file for AutoCAD SQL Interface.
                  ASI C++ User Interface.

    Author: Alexander Nikolayev
            Autodesk, Inc.
            Moscow, Russia.

      (C) Copyright 1991-93 by Autodesk, Inc.
   ***************************************************************************
      This program is copyrighted by Autodesk, Inc. and is  licensed
      to you under the following conditions.  You may not distribute
      or  publish the source code of this program in any form.   You
      may  incorporate this code in object form in derivative  works
      provided  such  derivative  works  are  (i.) are  designed and
      intended  to  work  solely  with  Autodesk, Inc. products, and
      (ii.)  contain  Autodesk's  copyright  notice  "(C)  Copyright
      1988-1993 by Autodesk, Inc."
   ***************************************************************************

      THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED
      WARRANTY.  ALL IMPLIED WARRANTIES OF FITNESS FOR ANY PARTICULAR
      PURPOSE AND OF MERCHANTABILITY ARE HEREBY DISCLAIMED.

   ***************************************************************************
                           RESTRICTED RIGHTS LEGEND

      Use, duplication, or disclosure by the U.S. Government is subject
      to restrictions set forth in FAR 52.227-19 (Commerical Computer
      Software - Restricted Rights) and DFAR 252.227-7013 (c) (1) (ii)
      (Rights in Technical Data and Computer Software), as applicable.
   ***************************************************************************


    Entry Points:

    Modification History:
       27 Sep 1993  - ann    -  original written

    Bugs and restriction on use:

    Notes:

*****************************************************************************/
#ifndef ASI_ASICLASS_HH
#define ASI_ASICLASS_HH asiclasshh

///////////////////////////////////////////////////////////////////////////
// INCLUDES 
///////////////////////////////////////////////////////////////////////////
#include <setjmp.h>

///////////////////////////////////////////////////////////////////////////
// DEFINES 
///////////////////////////////////////////////////////////////////////////
#define ASI_TRY           { \
                            CAsiExceptionHandler AsiExceptionHandler; \
                            if ( ::setjmp(AsiExceptionHandler.jbBuffer) == 0 )
#define ASI_CATCH(x)      else { \
                            CAsiException* x = AsiExceptionHandler.pException;
#define ASI_END_CATCH     }}
#define ASI_THROW(x)      AsiExceptionThrow(x);
#define ASI_THROW_LAST    AsiExceptionThrow(0);


///////////////////////////////////////////////////////////////////////////
// FORWARD DEFINITIONS 
///////////////////////////////////////////////////////////////////////////

// ASI Class Hierarchy
//
class CAsiObject;             // Superclass for all classes
  class CAsiIdent;            // Identifier
  class CAsiData;             // ASI Data generic class
    class CAsiNum;            // Numeric data
      class CAsiExactNum;     // Exact numeric data
        class CAsiSmallInt;   // Small Integer
        class CAsiInt;        // Integer
        class CAsiNumeric;    // Numeric
        class CAsiDecimal;    // Decimal
      class CAsiApproxNum;    // Approximate numeric data
        class CAsiFloat;      // Float
        class CAsiReal;       // Real
        class CAsiDouble;     // Double precision
    class CAsiChar;           // Character
    class CAsiBinary;         // Binary 
    class CAsiIQualifier;     // Interval qualifier
      class CAsiInterval;     // Interval
         class CAsiYM;        // Year month intervals
         class CAsiDT;        // Day Time intervals
    class CAsiDatetime;       // Datetime data
      class CAsiDate;         // Date
      class CAsiTime;         // Time
      class CAsiTimeStamp;    // Timestamp
    class CAsiImplDef;        // Implementation defined data

  class CAsiColumn;           // Column definition
  class CAsiRow;              // Array of columns
  class CAsiHostBuffer;       // Host program buffer
    class CAsiParameter;      // Parameter desriptor
  
  class CAsiException;        // ASI Exception

  class CAsiSQLObject;        // ASI User Interface base class
    class CAsiSession;        // SQL Session
    class CAsiStm;            // Generic SQL statement functionality
       class CAsiExecStm;     // Executable SQL statement
          class CAsiNativeStm;// Native DBMS statement 
       class CAsiCsr;         // Cursor

class CAsiUcStr;              // Unicode string
class CAsiUcChar;             // Unicode character
class CAsiExceptionHandler;   // Exception handler
class CAsiCollation;          // Collation


///////////////////////////////////////////////////////////////////////////
//     TYPEDEFS 
///////////////////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////////////////
class CAsiExceptionHandler
// This class is used internaly to support asi exception handling 
// feature
{
public:
    // Constrution/destruction
    // 
    CAsiExceptionHandler(void);
    ~CAsiExceptionHandler(void);

    // Internal data for support TRY/CATCH
    // 
    CAsiException*        pException;
    CAsiExceptionHandler* pPrevHandler;
    jmp_buf               jbBuffer;
};


///////////////////////////////////////////////////////////////////////////
class CAsiObject
//
// This class is a principal base class for ALL ASI classes. It provides
// services for memory management by reloading new and delete operators.
//
{
public:
            CAsiObject                (void);
    virtual ~CAsiObject               (void);

            void*         operator new    (unsigned int);
            void          operator delete (void*);

     // Duplicate 
     //
     virtual CAsiObject  * duplicate  (void) const;

     // Nullify all members
     //
     virtual void          Clear      (void);  

     // Runtime class iteraction functions. Used internaly
     //
     virtual unsigned long packSize   (void) const;
     virtual unsigned long packTo     (void*) const;
     virtual EAsiBoolean   unpackFrom (void*);

};


//////////////////////////////////////////////////////////////////////
class CAsiException : public CAsiObject
//
// This class provides functionality of the SQL exception condition
// It is thrown by ASI member function when an exception condition was 
// raised. (ASI_TRY, ASI_CATCH, ASI_THROW facility).
//
{
public:
    // Constrution/destruction
    // 
            CAsiException (void);
    virtual ~CAsiException (void);

    // Getting Diagnostics Information   
    //
            int           CondQty    (void) const;       // Number of conditions in diag
            AsiError      Err        (int d = 0) const;  // get completion code
            CAsiUcStr    *ErrMsg     (int d = 0) const;  // get error message
            char         *SQLSTATE   (int d = 0) const;  // get SQLSTATE
  
            // Get Diagnostics Parameter
            //
            EAsiBoolean getDiagPar (int, AsiDiag, integer *) const;
            EAsiBoolean getDiagPar (int, AsiDiag, CAsiUcStr *) const;

     // Setting Diagnostics information

            EAsiBoolean setErr      (AsiError);           // set completion code
            EAsiBoolean setErrMsg   (const CAsiUcStr &);  // set error message
            EAsiBoolean setSQLSTATE (const char *);       // set SQLSTATE

            // Set Diagnostics Parameter
            //
            EAsiBoolean setDiagPar (AsiDiag, integer);
            EAsiBoolean setDiagPar (AsiDiag, const CAsiUcStr &);

        
            EAsiBoolean pushCond   (void);       // Push diagnostics information
            EAsiBoolean flushCond  (void);       // Clear diagnostics

    ////////////////////////////////////////////////////
    // Utility
    //
    void*       getDiagData (void) const;

    virtual CAsiObject * duplicate (void) const;         

private:
    void* pExceptionBody;
};


////////////////////////////////////////////////////////////////////////////////
class CAsiIdent : public CAsiObject
//
//  SQL Identifier. Most SQL names are represented as identifiers.
//  Identifier can be regular and delimitered. Regular identifier body 
//  is always converted to upper case. Delimitered identifier body 
//  is not converted to upper case.
//
{
public:
     // Construct identifier
     //
              CAsiIdent (void);
              CAsiIdent (const char *);
              CAsiIdent (const AsiUcStr);
              CAsiIdent (const CAsiUcStr &);
              CAsiIdent (const CAsiUcStr &, const CAsiUcStr &);
              CAsiIdent (const CAsiIdent &);
      virtual ~CAsiIdent (void);

      // Set and get identifier value
      //
              void          set          (EAsiBoolean, const CAsiUcStr &, 
                                                   const CAsiUcStr &);

              void          setDelimited (EAsiBoolean);
              void          setDelimited (void);
              void          setCharSet   (const CAsiUcStr &);
              void          setBody      (const CAsiUcStr &);

              EAsiBoolean   isDelimited  (void) const;   // True if delimitered identifier
        const CAsiUcStr   & CharSet      (void) const;
        const CAsiUcStr   & Body         (void) const;

      // Length of Identifer (inc. introducer, char set name and 
      // identifier itself

              int           Length        (void) const; 

      // Clear components
      //
      virtual void          Clear         (void);


      // Operators
      //
              CAsiIdent    &operator= (const CAsiIdent &);  // Assign another identifier
              CAsiIdent    &operator= (const CAsiUcStr &);  // Assign regular identifier

              int           operator==(const CAsiIdent &) const;

      // Print identifier
      //
              void          ToString  (CAsiUcStr *) const; // Append identifier to string

      // Duplicate 
      // 
      virtual CAsiObject   *duplicate () const;

      // Runtime class iteraction functions
      //
      virtual unsigned long packSize   (void) const;
      virtual unsigned long packTo     (void*) const;
      virtual EAsiBoolean   unpackFrom (void*);

private:
    EAsiBoolean isDelim;      // Is delimited identifier 
    CAsiUcStr   charset;      // Character set name
    CAsiUcStr   ident;        // Identifier body
};




////////////////////////////////////////////////////////////////////////////////
class CAsiSQLObject : public CAsiObject
//
// This is a base class for classes that support SQL Connection-Session-
// Statement-Cursor functionality. It provides storage for the diagnostics 
// information generated by the derived classes.
// Besides diagnostics this class allows to get information about foreground
// DBMS driver, and the current ASI version.
//
// MEMBER FUNCTIONS
// ----------------
//
//  
// Getting Statement Information
//
// EAsiStatement StmType       (void) const    
// EAsiStatement CmdStmType    (void) const    
// long          RowCount      (void) const    
//
// These function retrieve SQL statement type, SQL Command statement type,
// number of rows affected by the <update:searched>, <delete: searched>, or 
// insert statements
//
//
// Syntax error position
//
// int ErrPosition   (void) const     
//
// Getting Diagnostics Information   
//
// int CondQty (void) const
//
// This function returns number of conditions stored in diagnostics area
// associated with the SQL object. Diagnostics area can hold several items
// belonging to the same or different conditions. Eash new condition in the
// diagnostics area introduces a new condition or clarifies a previouss one.
//
//
// EAsiError Err(
//   int icond) const       // Condition number
//
// This function returns error code associated with the condition specified 
// by its number. Condition codes are listed in the asiconst.h header file.
// If a wrong condition number was passed,  kAsiERUndefined is returned.
//
//
// CAsiUcStr *ErrMsg (
//   int icond) const       // Condition number
//
// This function returns pointer to the Unicode string object that contains
// message associated with specified condition. If invalid condition number 
// was specified, 0 is returned instead of it.
//
//
// char * SQLSTATE (
//   int icond) const       // Condition number
//
// This function returns pointer to the character ASCII string containing
// SQLSTATE diagnostics parameter associated with the specified condition.
// (SQLSTATE parameter is a 5 bytes null terminated charater string,
// consisting of class and subclass codes). If an invalid condition number
// was specified, NULL is returned instead of pointer.
//
//
// EAsiBoolean getDiagPar (int icond,              // Condition number
//     EAsiDiag pCode,         // Parameter code
//     integer *ivalue         // Integer parameter value
//    ) const
// EAsiBoolean getDiagPar (int icond,              // Condition number
//     EAsiDiag pCode,         // Parameter code
//     CAsiUcStr *ucvalue      // Character parameter value
//    ) const;
//
// These functions store value of diagnostics parameter specified by its
// code (pCode) for the specified condition into integer or Unicode 
// character string buffers. Return kAsiTrue if succeeded, or kAsiFalse 
// if an invalid condition number was specified.
//
// Parameters
//
// icond      Condition number
// pCode      Diagnostics parameter code
// ivalue     Destination for an integer value
// ucvalue    Destination for an Unicode character string
//
//
// Linked DBMS information
//
// The following functions allow to get information about DBMS driver 
// that maintains link between ASI application and DBMS server.
//
// EAsiBoolean DBMS (
//   CAsiUcStr & dbms         // DBMS Name
//  ) const    
//
// Stores DBMS name into Ucode string. Reference to the Unicode string 
// shall be provided by caller.
//
//
// EAsiBoolean SQLObject (
//   CAsiUcStr &objid         // SQL object identifier
//  ) const    
//
// Stores SQL object identifier into Ucode string. Reference to the destination
// Unicode string shall be provided by caller.
//
//
// EAsiBoolean DrvMessage (
//   CAsiUcStr &mess          // Driver message
//  ) const
//
// Stores driver message into Ucode string. Reference to the destination
// Unicode string shall be provided by caller.
//
//
// EAsiBoolean CheckOp (
//    ulong operation        // Operation codes
//   )                
//
// Return kAsiTrue if operations specified in the parameter are supported 
// by DBMS driver. Operation codes are a bit coded fields, that can be ored.
// kAsiFalse is returned if at least one operation is not supported.
//
//
// ASI version
//
// EAsiBoolean Version (
//    CAsiUcStr*             // Ponter for store ASI version string
//   ) const
//
// const char* Version (void) const
//
// Return pointer to the unocode string containing ASI version.
//
//
{
public:
    // Constrution/destruction
    // 
            CAsiSQLObject();
    virtual ~CAsiSQLObject();

    // Getting Statement Information
    //
            EAsiStatement StmType       (void) const;    // statement type
            EAsiStatement CmdStmType    (void) const;    // command stm type
            long          RowCount      (void) const;    // get row count

    // Syntax error position
            int           ErrPosition   (void) const;     

    // Getting Diagnostics Information   
    //
            int        CondQty     (void) const;       // Number of conditions in diag
            EAsiError  Err         (int d = 0) const;  // get completion code
            CAsiUcStr *ErrMsg      (int d = 0) const;  // get error message
            char      *SQLSTATE    (int d = 0) const;  // get SQLSTATE

            // Get Diagnostics Parameter
            //
            EAsiBoolean getDiagPar (int d,       // Integer condition
                                    EAsiDiag,
                                    integer *) const;
            EAsiBoolean getDiagPar (int d,       // character condition
                                    EAsiDiag,
                                    CAsiUcStr *) const;
            

    // Linked DBMS information
    //
            EAsiBoolean DBMS       (CAsiUcStr *) const;        // DBMS type
            EAsiBoolean SQLObject  (CAsiUcStr *) const;        // SQL object identifier
            EAsiBoolean DrvMessage (CAsiUcStr *) const;        // Driver message
            EAsiBoolean CheckOp    (ulong);                    // Check base operation
            EAsiBoolean GetStatus  (EAsiStatInfo, ulong *);    // Get check status

    //  ASI version
    //
            EAsiBoolean Version    (CAsiUcStr*) const;
            const char* Version    (void) const;

protected:
            EAsiBoolean isInit     (void) const;

protected:
    // Methods for setting diagnosticss information. It may just consist of
    // linking object with DDA.
    //
            ulong        objIdSes;     // Session id
            ulong        objIdStm;     // Statement id
            ulong        objIdCsr;     // Cursor id
            CAsiObject * diag;         // Diagnostics and Descriptors
            CAsiObject * control;      // Control 

private:
    // ASI version string
    //
    static const char* pAsiVersion;

};


////////////////////////////////////////////////////////////////////////////
class CAsiAppl: public CAsiSQLObject
// This class is introduces functionality of ASI application 
// initialization/termination and disconnectinf from all 
// SQL environments. It is derived from the CAsiSQLObject class
// to provide caller with diagnostics info.
//
{
friend class CAsiSession;

public:
      //  Construction/destruction
      //
              CAsiAppl  ();  
      virtual ~CAsiAppl ();

      //  Application initialization/termination, Disconnect all
      //
      EAsiBoolean   Init             (void);  // Initialize application
      EAsiBoolean   Term             (void);  // Terminate application
      EAsiBoolean   Abort            (void);  // Abort application (rollback 
                                              // all active transactions
      EAsiBoolean   DisconnectAll    (void);  // Disconnect from all environmants

protected:
      EAsiBoolean   addConnection    (CAsiSession *);
      EAsiBoolean   removeConnection (CAsiSession *);
      EAsiBoolean   deleteConnection (CAsiSession *);

private: 
      EAsiBoolean   rollbackAll      (void);

private:
      CAsiObject  * cncList;     // Connections
};



////////////////////////////////////////////////////////////////////////////////
class CAsiSession: public CAsiSQLObject
//
// This class provides services of the SQL connection and session management.
// Besides natural connection functionality (connect to and disconnect
// from SQL environment), this class provides straightforward way of executing
// transaction management and session management statements. Of course they
// can be also executed as a ordinary SQL statements by means of prepare and
// execute.
//
// MEMBER FUNCTIONS
// ----------------
//
// Construction/Destruction
//
// CAsiSession (CAsiAppl *)
//
// CAsiSession constructor constructs SQL-session object, but doesn't start
// the SQL session. Session is started when connecting to SQL environemnt
// is performed by means of the Connect () method provided by this class.
//
//
// ~CAsiSession ()
//
// Destroy CAsiSession object. If SQL-session is active when destructor is
// called, then terminate current transaction by ROLLBACK.
//
//
// Starting/Terminating Session
//
// EAsiBoolean Connect (
//   const CAsiUcStr &envname,        // SQL environment name
//   const CAsiUcStr &usrname,        // User name
//   const CAsiUcStr &password)       // Password
//
// Connect ot SQL environment and start SQL-session. Environment name
// is a section in the ASI Configuration File. This function can throw
// ASI exception and sets complition condition. Return kAsiTrue if succeeded
// and kAsiFalse otherwise.
//
//
// EAsiBoolean Disconnect (void)
//
// Terminate SQL session and disconnect from SQL environment if there is
// mo active transaction associated with the session. Otherwise throw
// exception condition (if ASI_TRY was established), and store complition
// condition.
//
//
// Setting Session Context
//
// The function listed below are provided to set context of the current 
// SQL session. Session context items that can be set by means of them 
// comprise the following items:
//
//   catalog name,
//   schema name,
//   character set name,
//   time zone displacement, and
//   authorization
//
// These function can throw ASI exception and set complition condition.
// They return kAsiTrue if succeeded and kAsiFalse otherwise.
//
// EAsiBoolean SetCatalog  (
//   const CAsiUcStr &catalog)     // Catalog name
//
// EAsiBoolean SetSchema   (
//   const CAsiUcStr &schema)      // Schema name
//
// EAsiBoolean SetNames    (
//   const CAsiUcStr &name)        // Character set name
//
// EAsiBoolean SetTimeZone (
//   const CAsiDT &timezone)   // Time Zone 
//
// EAsiBoolean SetAuthor   (
//   const CAsiUcStr & user);      // Authorization
//
//
// Transaction management
//
// EAsiBoolean SetTrans (
//   EAsiTILevel ilevel,         // Isolation level
//   EAsiTMode tmpode,           // Transaction mode
//   uint size                   // Diagnostics area size 
//  )
// Set transaction isolation level and transaction mode for the
// next transaction. If transaction is currently active, then throw ASI
// exception. Store complition condition. 
//
//
// EAsiBoolean DeferrConstr (
//   CAsiIdent *consname)        // Constraint name
//
// Change constraint checking mode to deffered.
//
// EAsiBoolean ImmConstr (
//   const CAsiIdent &consname)  // Constraint name
//
// Change constraint checking mode to immediate
//
// EAsiBoolean Commit       (void);                   // Commit work
//
// Terminate transaction by commit
//
// EAsiBoolean Rollback     (void);                   // Rollback
//
// Terminate transaction by rollback
//            
//
// Misceleneous
//
// EAsiBoolean Cancel (void)
//
// Break execution of the SQL statment (asynchronous break)
//
{
friend class CAsiExecStm;
friend class CAsiNativeStm;
friend class CAsiInterDBStm;
friend class CAsiAppl;

public:
    // Construction/Destruction
    //
            CAsiSession (CAsiAppl *);
    virtual ~CAsiSession ();

    // Starting/Terminating Session
    //
            EAsiBoolean Connect    (const CAsiUcStr &, const CAsiUcStr &,
                                    const CAsiUcStr &);
            EAsiBoolean Disconnect (void);
           
    // Setting Session Context
    //
            EAsiBoolean SetCatalog  (const CAsiUcStr &);
            EAsiBoolean SetSchema   (const CAsiUcStr &);
            EAsiBoolean SetNames    (const CAsiUcStr &);
            EAsiBoolean SetTimeZone (const CAsiDT &);
            EAsiBoolean SetTimeZone (void);          // Set local time zone 
            EAsiBoolean SetAuthor   (const CAsiUcStr &);

    // Transaction management
    //
            EAsiBoolean SetTrans     (EAsiTILevel, EAsiTMode, uint dSize = 64);  // Set transaction
            EAsiBoolean DeferrConstr (CAsiIdent * cnst = 0);   // Deffer constraint
            EAsiBoolean ImmConstr    (CAsiIdent * cnst = 0);   // Immediate constraint

            EAsiBoolean Commit       (void);                   // Commit work
            EAsiBoolean Rollback     (void);                   // Rollback

    // Misceleneous
    //
            EAsiBoolean Cancel       (void);                   // Break execution
     const  CAsiUcStr & Name         (void) const;             // Session/connection name

protected:
            EAsiBoolean Disconnect (CAsiObject *);
            EAsiBoolean iexecute   (const CAsiUcStr &);

private:
    virtual CAsiObject * duplicate (void) const;             // Duplicate

private: 
    EAsiBoolean   isconnected;
    CAsiAppl    * Appl;           // Aplication
    CAsiUcStr     sname;          // Environment name 

};


////////////////////////////////////////////////////////////////////////////////
class CAsiStm: public CAsiSQLObject
//
// This class introduces functionality of the SQL statement descriptors
// Basic categories of services provided by this class are listed below:
//  - Input parameters manipulation (bind buffers)
//  - Output parameter manipulation (resulting columns)
//
//  MEMBER FUNCTIONS
//  ----------------
//
// Construction/Destruction
//
//  CAsiStm (void)
//
//  Allocate SQL statement descriptor.
//
//  virtual ~CAsiStm                   (void);
//
//  Destroy statement descriptor.
//
//
// Input Parameters Manipulation
//
// int ParamQty (void)
//
// Return number of input parameters (bind variables) in the original
// SQL statement. Number of parameter becomes available after SQL statement
// was prepared.
//
//
// CAsiParameter * ParamDsc (
//   int ipar                // Parameter number
//  ) const
//
// Get parameter descriptor. ipar shall be greater than or equal to 0 and 
// less than number of parameters returned by the ParamQty ().
// If wrong parameter number was specified return 0 instead of pointer to it.
//
//
// Bind parameter with host buffer
//
// EAsiBoolean Bind ( ... )
//
// Two basic methods for binding input paramters with the host variable
// buffers are provided by the CAsiStm class. They are
//
// - by parameter number
// - by parameter name
//
// Input parameter is bound with CAsiData object
//
//
// Output Column Descriptors / Column Values
//
// This set of services are provided in order to get description and 
// values of the resulting columns for the single row select statement
// and cursor. For other types of statements these services have no sence
// and always return kAsiBad.
//
// int ColQty (void) const
//
// Return number of columns in resuting rowset. 
//
//
// CAsiColumn *ColDsc (
//   int colnum                    // Column number
// ) const;         
//
// Return pointer to descripttor of the specified column.
//
//
// Two basic methods are provided to retrieve values of the resulting 
// columns:
//
//  - bind output column with the host buffer, and
//  - retrieve column value from descriptor
//
// These two methods are incompatible, i.e. if one method is used, another
// is not allowed.
//
// Both methods are simular and allow to store output value into 
// CAsiData object
//
// EAsiBoolean Sob ( ... )
//
// Bind output column with the host program buffer
//
//
// EAsiBoolean Cvl (...)
//
// Retrieve column data from descriptor
//
//
{

public:
    // Construction/Destruction
    //
            CAsiStm                    (void);
    virtual ~CAsiStm                   (void);


    // Input Parameters Manipulation
    //
    //
            int             ParamQty    (void) const; // Parameter quantity
            CAsiParameter * ParamDsc    (int) const;  // Get parameter descriptor

         // Bind input parameter by number with
         //
    virtual EAsiBoolean Bind (int, CAsiData *);    // CAsiData object
    virtual EAsiBoolean Bind (int, void *, short *,
                              int, EAsiHostType);  // Host buffer

         // Bind input parameter by name with
         //
    virtual EAsiBoolean Bind (const CAsiIdent &, CAsiData *); // CAsiData object
    virtual EAsiBoolean Bind (const CAsiIdent &, void *, short *,
                              int, EAsiHostType);    // Host buffer

    // Output Column Descriptors / Column Values
    //
    //
            int         ColQty (void) const;      // Column quantity
            CAsiColumn *ColDsc (int) const;       // Get column descriptor

            // Bind output parameter (column) with 
            EAsiBoolean Sob    (int, CAsiData *);    // CAsiData object
            EAsiBoolean Sob    (int, void *, short *,// Host buffer
                                int, EAsiHostType); 

            // Retrieve value of output parameter (column) into
            EAsiBoolean Cvl    (int, CAsiData *);    // CAsiData object
            EAsiBoolean Cvl    (int, void *, short *,// Host buffer
                                int, EAsiHostType); 

            // Some methods for recive raw data
    virtual long        Read   (int, char *, long, long) const = 0;
    virtual long        Size   (int) const = 0;

protected:
            EAsiBoolean CvlInt (int, CAsiData *) const;    // CAsiData object
            EAsiBoolean pInput (void);

protected:
    CAsiObject * input;     
    CAsiObject * output;
};


class CAsiExecStm: public CAsiStm
//
// This class introduces functionality of the executable SQL statement. 
// Another type of statement is a cursor that can't be executed, and 
// should be opend.
// Basic categories of services provided by this class are listed below:
//  - Statement preparing
//  - Statement execution
//
//  MEMBER FUNCTIONS
//  ----------------
//
// Construction/Destruction
//
//  CAsiExecStm (void)
//
//  Constructs SQL statement object
//
//  virtual ~CAsiExecStm  (void);
//
//  Destroy statement. If statement was not deallocated explicitly by
//  means of Dealocate method, then statement is deallocated implicitly.
//  
//
// Preparing/Deallocating
//
// EAsiBoolean Prepare (
//   CAsiSession *session,       // Session
//   const CAsiUcStr &stm        // Statement
//  )   

// Prepare standard SQL statement. Associate
// SQL statement with the session and prepares it.
// This function stores complition condtion and 
// throws ASI exception if error condition was encounted. 
//
// EAsiBoolean Deallocate  (void)
//
// Deallocate either standard or interdatabase SQL statement. If statement is 
// a cursor specification associated with cursor in the open state, then
// ASI exception is raised.
//
//
// Execute statement
//
// EAsiBoolean Execute (void)
//
// Execute prepared SQL statement. Complition condition is always set and  
// ASI exception can be thrown.
//
//
// Immediate SQL and Native statement execution
//
// EAsiBoolean ImmediateExecute (
//   CAsiSession *session,     // session
//   const CAsiUcStr &stm)     // SQL statement
//
// Immediate execute standard SQL statement 
// Successful execution results is returning kAsiGood. If execution fails, 
// then ASI exception is thrown, complition
// conditon is stored CAsiSQLObject and kAsiBad is returned. Execution
// of this statement can start transaction.
//
{
friend class CAsiCsr;

public:
    // Construction/Destruction
    //
            CAsiExecStm                 (void);
    virtual ~CAsiExecStm                (void);

    // Preparing/Deallocating
    //
    virtual EAsiBoolean Prepare          (CAsiSession *, const CAsiUcStr &);
    virtual EAsiBoolean Deallocate       (void);  // Deallocate statement

    // Prepared and execute statement (immediate execution)
    //
    virtual EAsiBoolean Execute          (void);
    virtual EAsiBoolean ImmediateExecute (CAsiSession *, const CAsiUcStr &);

    virtual EAsiBoolean Bind (int, CAsiData *);
    virtual EAsiBoolean Bind (int, void *, short *, int, EAsiHostType);
    virtual EAsiBoolean Bind (const CAsiIdent &, CAsiData *);
    virtual EAsiBoolean Bind (const CAsiIdent &, void *, short *, int, EAsiHostType);

            // Some methods for recive raw data
    virtual long        Read   (int, char *, long, long) const;
    virtual long        Size   (int) const;

private:
    virtual CAsiObject * duplicate (void) const;

};  

////////////////////////////////////////////////////////////////////////////
class CAsiNativeStm: public CAsiExecStm
//
//  This class represents functionality of the Native DBMS statement 
//  processing. Native statements are passed to DBMS driver 
//  without preliminary preprocessing. Native statements can be 
//  just immediate executed.
//  CAsiNativeStm is derived from the CAsiExecStm class and inherits 
//  functionality of the statement immediate execution. 
//  
{
public:
    // Construction/Destruction
    //
            CAsiNativeStm                 (void);
    virtual ~CAsiNativeStm                (void);

    // Statement procession
    virtual EAsiBoolean ImmediateExecute (CAsiSession *, const CAsiUcStr &);

private:
    // Overload these methods just to signal that they are not supported
    // by the Native statement
    virtual EAsiBoolean Prepare          (CAsiSession *, const CAsiUcStr &);
    virtual EAsiBoolean Execute          (void);
};


////////////////////////////////////////////////////////////////////////////
class CAsiCsr: public CAsiStm
//
// This class proivides functionality of cursor 
//
// Construction/Destruction
//
// CAsiCsr (void);
//
// Construct cusror object
//
// ~CAsiCsr (void);
//
// Destroy cursor object. If cursor was in open state, close cursor implicitly.
//
//
// Allocate/Deallocate Cursor
//
// EAsiBoolean Allocate  (
//   CAsiExecStm * statement,         // Prepared cursor specification
//   const CAsiIdent & name,          // Cursor name
//   EAsiCurScr Scrollability,        // Cursor Scrollability
//   EAsiCurSns Sensivity);           // Cursor Sensitivity
//
// Allocate cursor. Statement shall be a prepared cursor specification.
// Otherwise ASI exception is thrown.
//
//
//
// Open/Close Cursor
//     
// EAsiBoolean Open (void)
//
// Open cursor. Cursor state becomes open and cursor position becomes before
// the first row of the selection set.
//
//
// EAsiBoolean Close (void)
//
// Close cursor. Cursor state is set to 'close'.
//
//
// Fetching Resulting Rowset        
//
// These services allow to fetch
//   - Next row
//   - Prior row
//   - First row
//   - Last row
//   - by the Relative row number (offset)
//   - by the Absolute row number (direct)
//
//   EAsiBoolean Fetch (void)
//
//   EAsiBoolean FetchPrior (void)
//
//   EAsiBoolean FetchFirst (void)
//   
//   EAsiBoolean FetchLast (void)
//
//   EAsiBoolean FetchRelative (
//     long rrow)                  // Relative row number
//
//   EAsiBoolean FetchAbsolute (
//     long row)                   // Absolute row number
//
//
// Storing Resulting Column Values
//
// CAsiRow   * getCurrentRow (void) const 
//
// Return pointer to the CAsiRow object containing current row 
//
//
// Delete/Update Current Row (Delete: positioned, Update: positioned)
//
// EAsiBoolean Delete  (void)
//
// Delete current row
//
//
// EAsiBoolean Update  (const CAsiRow &) 
//
// Update current row. Only columns that are present in the CAsiRow 
// object are updated. Association is done by column name
//
//
// Misceleneous 
// 
// EAsiBoolean is_updatable  (void);      // Is cursor updatable
// EAsiBoolean is_scrollable (void);      // Is cursor scrollable
// EAsiBoolean is_open       (void);      // Is cursor in open state
//
{
public:
    // Construction/Destruction
    //
             CAsiCsr (void);
    virtual ~CAsiCsr (void);

    // Allocate Cursor
    //
            EAsiBoolean Allocate      (CAsiExecStm *, const CAsiIdent &, 
                                       EAsiCurScr sc = kAsiNonScroll, 
                                       EAsiCurSns sn = kAsiSnsUndef);

    // Open/Close Cursor
    //     
            EAsiBoolean Open          (void);     // Open cursor
            EAsiBoolean Close         (void);     // Close

    // Fetching Resulting Rowset        
    //
            EAsiBoolean Fetch         (void);   // Fetch next row
            EAsiBoolean FetchPrior    (void);   // Fetch Prior row
            EAsiBoolean FetchFirst    (void);   // Fetch First row
            EAsiBoolean FetchLast     (void);   // Fetch Last row
            EAsiBoolean FetchRelative (long);   // Fetch Relative row
            EAsiBoolean FetchAbsolute (long);   // Fetch Absolute row

    // Storing Resulting Column Values
    //
            CAsiRow   * getCurrentRow (void) const;   // Get pointer to current row

    // Delete/Update Current Row (Delete: positioned, Update: positioned)
    //
            EAsiBoolean Delete        (void);            // Delete current row
            EAsiBoolean Update        (const CAsiRow &); // Update current row

    // Misceleneous 
    // 
            EAsiBoolean is_updatable   (void);      // Is cursor updatable
            EAsiBoolean is_scrollable  (void);      // Is cursor scrollable
            EAsiBoolean is_insensitive (void);      // Is cursor insensitive
            EAsiBoolean is_open        (void);      // Is cursor in open state
            EAsiCurPos  position       (void);      // Cursor position

            // Some methods for recive raw data
    virtual long        Read   (int, char *, long, long) const;
    virtual long        Size   (int) const;

private:
    virtual CAsiObject * duplicate (void) const;

private:
            CAsiIdent   csrname;
            EAsiBoolean opnFlag;
            EAsiCurScr  scrFlag;
            EAsiCurSns  snsFlag;
};


////////////////////////////////////////////////////////////////////
//
//  ASI Data classes
//



/////////////////////////////////////////////////////////////////////////
//
//
//
class CAsiData : public CAsiObject
//
//  SQL Data
//
//
// This class is a base class for all ASI data classes.
// The basic functionality of it comprise the foolowing:
//   - Getting data type descriptor;
//   - Getting value in specified format;
//   - Storing value provided in arbitrary format;
//   - NULL value manipulating;
//   - Supporting of the Application/Driver generic data link
//   - Self duplicating
// 
//
//  MEMBER FUNCTIONS:
//  -----------------
//
//  Get data descriptor
//
//  EAsiDataType  Type          (void)  // type of data
//  EAsiDTType    DateTimeType  (void)  // Datetime type
//  EAsiIType     IntervalType  (void)  // Interval type
//  int           ImplDataType  (void)  // Implemetation defined type
//  uint          Length        (void)  // Length of character or bit data
//  uint          Precision     (void)  // Precision
//  uint          Scale         (void)  // Scale
//  uint          TimePrec      (void)  // Time Precision
//  uint          TimeStampPrec (void)  // Timestamp precision
//  uint          LeadFieldPrec (void)  // Leading field precision
//  uint          FracSecPrec   (void)  // Fractional second precision
//  uint          ReturnLength  (void)  // Actual length of data
//
//  The functions listed above allow to retrieve desription of the 
//  the SQL data in terms of SQL. They are pure virtual functions,
//  actual transaltion of internal data descriptioon to the SQL data description
//  is done by the derived classes.
//
//
// ASI application/DBMS driver data exchange
//
// The CAsiData class provides services for reading generic data, 
// without respect to the data internal structure, just as a sequence 
// of bytes. These services provide ASI application and DBMS driver with 
// ability to exchange data in the ASI application/DBMS driver interaction.
// (Note. To use this facility CAsiData shall be bound with input or 
// output parameter of the SQL statement)
// Driver and Application use Client/Server paradigm in their interaction.
// When application needs to read value of the output columns
// from the driver, they are Client and Server respectivly (Application
// asks Driver to provide it with data). On the other hand, when driver 
// needs values for input parameters it asks Application to provide them for
// it. In this case Driver is a Client and Application is a Server. 
// 
// Client shall use member functions size() and read() to get size of
// data and to read block of data. Server shall provide function readFromServer
// that stores specified number of bytes into destination buffer. 
// Functions size and readFromServer shall be reloaded by the derived classes.
// Default versions do nothing.
//
//
//  virtual long  readFromServer (
//    char *dst,         // Destination buffer
//    long start,        // Starting position
//    long len           // Number of bytes to read
//  )
//  
//  Read specified number of bytes into destination buffer starting from 
//  specified position. Return actual number of bytes stored in destination
//  buffer. This function shall be reloaded by the derived classes.
//
//
//  virtual long  size           (void)
//
//  Return size of data in bytes
//
//
//  long read (
//    char *dst,         // Destination buffer
//    long start,        // Starting position
//    long len           // Number of bytes to read
//  )
//
//  Fill in destination buffer by 'len' bytes starting from the 'start'
//  byte of data. This function shall be called when data from Server have 
//  to be retrieved by Client
//
//
//  NULL value manipulation
//
//  void setNull (void)  
//
//    Set internal data to NULL value (absence of value)
//
//
//  EAsiBoolean isNull (void) const
//
//    Return kAsiTrue if data is NULL and kAsiFalse otherwise
//
//
// Store Data value
//
//  virtual EAsiBoolean   storeValue   (...);
//
// These functions allow to store data into class derived from
// the CAsiData. Source data is converted to the actual type of the CAsiData
// If conversion is possible and data was successfully stored and kAsiTrue is
// returned, if conversion is impossible, kAsiFalse is returned. 
//
//
// Get data value
//
//  virtual void getValue (...)
//
// Store value of the CAsiData object into destination buffer and perform all
// needed converisons. If operation succeeded, kAsiTrue is returned, if not
// kSiFalse is returned and contents of destination buffer is undefined.
// If null value is stored, destination buffer contents is undefined.
//
//  EAsiBoolean   SQLType    (CAsiUcStr * dst) const
//  EAsiBoolean   SQLLiteral (CAsiUcStr * dst) const    
//
//  This functions stores SQL data type string and SQL literal into destination
//  buffer. For example, for Year to month interval first function will
//  return INTERVAL YEAR TO MONTH, the second one - INTERVAL 'yy-dd' YEAR TO MONTH
//  
// Duplicate
//
// virtual CAsiObject *duplicate (void) const
//
// Create duplicate of the CAsiData object.
//
//
//  Operations with the Internal Data (Operators)
//
//  Each class derived from CAsiData defines a set of functions that 
//  perform certain operations with it. These functions comprise:
//    - assignment operator;
//    - arithmetics operator;
//    - data specific operations
//
//  For more details see appropriate class definition.
//
//
{
friend class CAsiColumn;

public:

    //  Construction destruction
    //

            CAsiData  (void);
    virtual ~CAsiData (void);

    //  Get data descriptor
    //
    virtual EAsiDataType  Type          (void) const = 0;  // type of data
    virtual EAsiDTType    DateTimeType  (void) const; // Datetime type
    virtual EAsiIType     IntervalType  (void) const; // Interval type
    virtual int           ImplDataType  (void) const; // Implemetation defined type
    virtual uint          Length        (void) const; // Length of character or bit data
    virtual uint          Precision     (void) const; // Precision
    virtual uint          Scale         (void) const; // Scale
    virtual uint          TimePrec      (void) const; // Time Precision
    virtual uint          TimeStampPrec (void) const; // Timestamp precision
    virtual uint          LeadFieldPrec (void) const; // Leading field precision
    virtual uint          FracSecPrec   (void) const; // Fractional second precision
    virtual uint          ReturnLength  (void) const; // Actual length of data

    //  is_ functions
    //
    virtual EAsiBoolean   is_numeric       (void) const;
    virtual EAsiBoolean   is_exactnumeric  (void) const;
    virtual EAsiBoolean   is_approxnumeric (void) const;
    virtual EAsiBoolean   is_character     (void) const;
    virtual EAsiBoolean   is_binary        (void) const;
    virtual EAsiBoolean   is_datetime      (void) const;
    virtual EAsiBoolean   is_interval      (void) const;
    virtual EAsiBoolean   is_yminterval    (void) const;
    virtual EAsiBoolean   is_dtinterval    (void) const;
    virtual EAsiBoolean   is_implement     (void) const;

    virtual EAsiBoolean   is_comparable    (const CAsiData *) const;
    virtual EAsiBoolean   is_storable      (const CAsiData *) const;
    virtual EAsiBoolean   is_castable      (const CAsiData *) const;

    // Check certain operations and return resulting descriptor
    virtual CAsiData *    checkAdd (const CAsiData *) const;
    virtual CAsiData *    checkSub (const CAsiData *) const;
    virtual CAsiData *    checkMul (const CAsiData *) const;
    virtual CAsiData *    checkDiv (const CAsiData *) const;
    virtual CAsiData *    checkCat (const CAsiData *) const;

    // ASI application/DBMS driver data exchange
    //
    virtual long          read           (char *, long, long) const;
    virtual long          size           (void) const;

    //  NULL value manipulation
    //
            void          setNull  (void);            // Assign NULL
            EAsiBoolean   isNull   (void) const;      // Is Null ?

    // Store Data value
    //
    virtual EAsiBoolean   storeValue (real);                 // Real
    virtual EAsiBoolean   storeValue (integer);              // Integer
    virtual EAsiBoolean   storeValue (smallint);             // Smallint
    virtual EAsiBoolean   storeValue (char *, uint);         // ASCII string
    virtual EAsiBoolean   storeValue (const CAsiUcStr &);    // Unicode string
    virtual EAsiBoolean   storeValue (char *, uint, uint *); // Binary Buffer
    virtual EAsiBoolean   storeValue (const CAsiData &);
    virtual EAsiBoolean   storeValue (const AsiUcStr);       // Store AsiUcStr

    //  Retrive data value
    //
    virtual EAsiBoolean   getValue (real *) const;               // Real
    virtual EAsiBoolean   getValue (integer *) const;            // Integer
    virtual EAsiBoolean   getValue (smallint *) const;           // Smallint
    virtual EAsiBoolean   getValue (char *, uint) const;         // ASCII string
    virtual EAsiBoolean   getValue (CAsiUcStr *) const;          // Unicode string
    virtual EAsiBoolean   getValue (char *, uint, uint *) const; // Binary Buffer
    virtual EAsiBoolean   getValue (AsiUcStr, uint) const;       // Unicode string 
    virtual EAsiBoolean   getValue (CAsiData &) const;

    virtual EAsiBoolean   SQLType    (CAsiUcStr *) const = 0;    // SQL data type as string
    virtual EAsiBoolean   SQLLiteral (CAsiUcStr *) const = 0;    // SQL compatible literal

    // Runtime class iteraction functions
    //
    virtual unsigned long packSize   (void) const = 0;
    virtual unsigned long packTo     (void*) const = 0;
    virtual EAsiBoolean   unpackFrom (void*) = 0;

    // Duplicate itself
    //
    virtual CAsiObject   *duplicate (void) const = 0;

    // Nullify all members

    virtual void          Clear     (void);

protected:
    EAsiBoolean isnull;                           // kAsiTrue if data is null
    CAsiObject *pDataLink;                        // Link with server data
};


////////////////////////////////////////////////
//
//
class CAsiNum : public CAsiData
//
// Numeric data
//
{
public:
    // Construction/destruction
    //
            CAsiNum ();           // Constructor
    virtual ~CAsiNum ();          // Destructor

    // Get data descriptor
    //
    virtual uint          Precision     (void) const;
    virtual uint          Scale         (void) const;

    //  is_ functions
    //
    virtual EAsiBoolean   is_numeric       (void) const;
    virtual EAsiBoolean   is_exectnumeric (void) const;
    virtual EAsiBoolean   is_approxnumeric (void) const;
    virtual EAsiBoolean   is_comparable    (const CAsiData *) const;
    virtual EAsiBoolean   is_storable      (const CAsiData *) const;
    virtual EAsiBoolean   is_castable      (const CAsiData *) const;

    virtual CAsiData *    checkAdd (const CAsiData *) const;
    virtual CAsiData *    checkSub (const CAsiData *) const;
    virtual CAsiData *    checkMul (const CAsiData *) const;
    virtual CAsiData *    checkDiv (const CAsiData *) const;

    // Get data value
    //
    virtual integer     intValue  (void) const;
    virtual real        realValue (void) const;


    // virtual Comparision operators
    //
    virtual int         operator>  (const CAsiSmallInt &) const;
    virtual int         operator>  (const CAsiInt &)      const;
    virtual int         operator>  (const CAsiNumeric &)  const;
    virtual int         operator>  (const CAsiDecimal &)  const;
    virtual int         operator>  (const CAsiFloat &)    const;
    virtual int         operator>  (const CAsiReal &)     const;
    virtual int         operator>  (const CAsiDouble &)   const;
    virtual int         operator>  (const smallint)       const;
    virtual int         operator>  (const integer)        const;
    virtual int         operator>  (const real)           const;
    virtual int         operator>  (const int)            const;
                 
    virtual int         operator>= (const CAsiSmallInt &) const;
    virtual int         operator>= (const CAsiInt &)      const;
    virtual int         operator>= (const CAsiNumeric &)  const;
    virtual int         operator>= (const CAsiDecimal &)  const;
    virtual int         operator>= (const CAsiFloat &)    const;
    virtual int         operator>= (const CAsiReal &)     const;
    virtual int         operator>= (const CAsiDouble &)   const;
    virtual int         operator>= (const smallint)       const;
    virtual int         operator>= (const integer)        const;
    virtual int         operator>= (const real)           const;
    virtual int         operator>= (const int)            const;
                
    virtual int         operator== (const CAsiSmallInt &) const;
    virtual int         operator== (const CAsiInt &)      const;
    virtual int         operator== (const CAsiNumeric &)  const;
    virtual int         operator== (const CAsiDecimal &)  const;
    virtual int         operator== (const CAsiFloat &)    const;
    virtual int         operator== (const CAsiReal &)     const;
    virtual int         operator== (const CAsiDouble &)   const;
    virtual int         operator== (const smallint)       const;
    virtual int         operator== (const integer)        const;
    virtual int         operator== (const real)           const;
    virtual int         operator== (const int)            const;
                 
    virtual int         operator<= (const CAsiSmallInt &) const;
    virtual int         operator<= (const CAsiInt &)      const;
    virtual int         operator<= (const CAsiNumeric &)  const;
    virtual int         operator<= (const CAsiDecimal &)  const;
    virtual int         operator<= (const CAsiFloat &)    const;
    virtual int         operator<= (const CAsiReal &)     const;
    virtual int         operator<= (const CAsiDouble &)   const;
    virtual int         operator<= (const smallint)       const;
    virtual int         operator<= (const integer)        const;
    virtual int         operator<= (const real)           const;
    virtual int         operator<= (const int)            const;
                 
    virtual int         operator<  (const CAsiSmallInt &) const;
    virtual int         operator<  (const CAsiInt &)      const;
    virtual int         operator<  (const CAsiNumeric &)  const;
    virtual int         operator<  (const CAsiDecimal &)  const;
    virtual int         operator<  (const CAsiFloat &)    const;
    virtual int         operator<  (const CAsiReal &)     const;
    virtual int         operator<  (const CAsiDouble &)   const;
    virtual int         operator<  (const smallint)       const;
    virtual int         operator<  (const integer)        const;
    virtual int         operator<  (const real)           const;
    virtual int         operator<  (const int)            const;
     
    virtual int         operator!= (const CAsiSmallInt &) const;
    virtual int         operator!= (const CAsiInt &)      const;
    virtual int         operator!= (const CAsiNumeric &)  const;
    virtual int         operator!= (const CAsiDecimal &)  const;
    virtual int         operator!= (const CAsiFloat &)    const;
    virtual int         operator!= (const CAsiReal &)     const;
    virtual int         operator!= (const CAsiDouble &)   const;
    virtual int         operator!= (const smallint)       const;
    virtual int         operator!= (const integer)        const;
    virtual int         operator!= (const real)           const;
    virtual int         operator!= (const int)            const;


protected:
    uint precision;
    uint scale;

};


////////////////////////////////////////////////
//
//
class CAsiExactNum : public CAsiNum
//
// Exact numeric data
//
{
public:
    // Construction/destruction
    //
            CAsiExactNum ();      // Constructor
    virtual ~CAsiExactNum ();     // Destructor

    //  is_ functions
    //
    virtual EAsiBoolean   is_exactnumeric (void) const;

    // Comparision operators
    //
    virtual int         operator>  (const CAsiSmallInt &) const;
    virtual int         operator>  (const CAsiInt &)      const;
    virtual int         operator>  (const CAsiNumeric &)  const;
    virtual int         operator>  (const CAsiDecimal &)  const;
    virtual int         operator>  (const CAsiFloat &)    const;
    virtual int         operator>  (const CAsiReal &)     const;
    virtual int         operator>  (const CAsiDouble &)   const;
    virtual int         operator>  (const smallint)       const;
    virtual int         operator>  (const integer)        const;
    virtual int         operator>  (const real)           const;
    virtual int         operator>  (const int)            const;
                        
    virtual int         operator>= (const CAsiSmallInt &) const;
    virtual int         operator>= (const CAsiInt &)      const;
    virtual int         operator>= (const CAsiNumeric &)  const;
    virtual int         operator>= (const CAsiDecimal &)  const;
    virtual int         operator>= (const CAsiFloat &)    const;
    virtual int         operator>= (const CAsiReal &)     const;
    virtual int         operator>= (const CAsiDouble &)   const;
    virtual int         operator>= (const smallint)       const;
    virtual int         operator>= (const integer)        const;
    virtual int         operator>= (const real)           const;
    virtual int         operator>= (const int)            const;
                        
    virtual int         operator== (const CAsiSmallInt &) const;
    virtual int         operator== (const CAsiInt &)      const;
    virtual int         operator== (const CAsiNumeric &)  const;
    virtual int         operator== (const CAsiDecimal &)  const;
    virtual int         operator== (const CAsiFloat &)    const;
    virtual int         operator== (const CAsiReal &)     const;
    virtual int         operator== (const CAsiDouble &)   const;
    virtual int         operator== (const smallint)       const;
    virtual int         operator== (const integer)        const;
    virtual int         operator== (const real)           const;
    virtual int         operator== (const int)            const;
                        
    virtual int         operator<= (const CAsiSmallInt &) const;
    virtual int         operator<= (const CAsiInt &)      const;
    virtual int         operator<= (const CAsiNumeric &)  const;
    virtual int         operator<= (const CAsiDecimal &)  const;
    virtual int         operator<= (const CAsiFloat &)    const;
    virtual int         operator<= (const CAsiReal &)     const;
    virtual int         operator<= (const CAsiDouble &)   const;
    virtual int         operator<= (const smallint)       const;
    virtual int         operator<= (const integer)        const;
    virtual int         operator<= (const real)           const;
    virtual int         operator<= (const int)            const;
                        
    virtual int         operator<  (const CAsiSmallInt &) const;
    virtual int         operator<  (const CAsiInt &)      const;
    virtual int         operator<  (const CAsiNumeric &)  const;
    virtual int         operator<  (const CAsiDecimal &)  const;
    virtual int         operator<  (const CAsiFloat &)    const;
    virtual int         operator<  (const CAsiReal &)     const;
    virtual int         operator<  (const CAsiDouble &)   const;
    virtual int         operator<  (const smallint)       const;
    virtual int         operator<  (const integer)        const;
    virtual int         operator<  (const real)           const;
    virtual int         operator<  (const int)            const;
     
    virtual int         operator!= (const CAsiSmallInt &) const;
    virtual int         operator!= (const CAsiInt &)      const;
    virtual int         operator!= (const CAsiNumeric &)  const;
    virtual int         operator!= (const CAsiDecimal &)  const;
    virtual int         operator!= (const CAsiFloat &)    const;
    virtual int         operator!= (const CAsiReal &)     const;
    virtual int         operator!= (const CAsiDouble &)   const;
    virtual int         operator!= (const smallint)       const;
    virtual int         operator!= (const integer)        const;
    virtual int         operator!= (const real)           const;
    virtual int         operator!= (const int)            const;



protected:
            int         cmpeqlen(const  CAsiExactNum *)   const;
            int         cmpeqlan(const  CAsiNum *)        const;
            int         cmpeqlenC(const real)             const;
            int         cmpneqen(const  CAsiExactNum *)   const;
            int         cmpneqan(const  CAsiNum *)        const;
            int         cmpneqenC(const real)             const;
            int         cmpltren(const  CAsiExactNum *)   const;
            int         cmpltran(const  CAsiNum *)        const;
            int         cmpltrenC(const real)             const;
            int         cmplteen(const  CAsiExactNum *)   const;
            int         cmpltean(const  CAsiNum *)        const;
            int         cmplteenC(const real)             const;
            int         cmpgtren(const  CAsiExactNum *)   const;
            int         cmpgtran(const  CAsiNum *)        const;
            int         cmpgtrenC(const real)             const;
            int         cmpgteen(const  CAsiExactNum *)   const;
            int         cmpgtean(const  CAsiNum *)        const;
            int         cmpgteenC(const real)             const;

};



////////////////////////////////////////////////
//
//
class CAsiApproxNum : public CAsiNum
//
// Approximate numeric data
//
{
public:
    // Construction/destruction
    //
            CAsiApproxNum ();     // Constructor
    virtual ~CAsiApproxNum ();    // Destructor

    //  is_ functions
    //
    virtual EAsiBoolean   is_approxnumeric    (void) const;

    // Comparision operators
    //
    virtual int         operator>  (const CAsiSmallInt &) const;
    virtual int         operator>  (const CAsiInt &)      const;
    virtual int         operator>  (const CAsiNumeric &)  const;
    virtual int         operator>  (const CAsiDecimal &)  const;
    virtual int         operator>  (const CAsiFloat &)    const;
    virtual int         operator>  (const CAsiReal &)     const;
    virtual int         operator>  (const CAsiDouble &)   const;
    virtual int         operator>  (const smallint)       const;
    virtual int         operator>  (const integer)        const;
    virtual int         operator>  (const real)           const;
    virtual int         operator>  (const int)            const;
                        
    virtual int         operator>= (const CAsiSmallInt &) const;
    virtual int         operator>= (const CAsiInt &)      const;
    virtual int         operator>= (const CAsiNumeric &)  const;
    virtual int         operator>= (const CAsiDecimal &)  const;
    virtual int         operator>= (const CAsiFloat &)    const;
    virtual int         operator>= (const CAsiReal &)     const;
    virtual int         operator>= (const CAsiDouble &)   const;
    virtual int         operator>= (const smallint)       const;
    virtual int         operator>= (const integer)        const;
    virtual int         operator>= (const real)           const;
    virtual int         operator>= (const int)            const;
                        
    virtual int         operator== (const CAsiSmallInt &) const;
    virtual int         operator== (const CAsiInt &)      const;
    virtual int         operator== (const CAsiNumeric &)  const;
    virtual int         operator== (const CAsiDecimal &)  const;
    virtual int         operator== (const CAsiFloat &)    const;
    virtual int         operator== (const CAsiReal &)     const;
    virtual int         operator== (const CAsiDouble &)   const;
    virtual int         operator== (const smallint)       const;
    virtual int         operator== (const integer)        const;
    virtual int         operator== (const real)           const;
    virtual int         operator== (const int)            const;
                        
    virtual int         operator<= (const CAsiSmallInt &) const;
    virtual int         operator<= (const CAsiInt &)      const;
    virtual int         operator<= (const CAsiNumeric &)  const;
    virtual int         operator<= (const CAsiDecimal &)  const;
    virtual int         operator<= (const CAsiFloat &)    const;
    virtual int         operator<= (const CAsiReal &)     const;
    virtual int         operator<= (const CAsiDouble &)   const;
    virtual int         operator<= (const smallint)       const;
    virtual int         operator<= (const integer)        const;
    virtual int         operator<= (const real)           const;
    virtual int         operator<= (const int)            const;
                        
    virtual int         operator<  (const CAsiSmallInt &) const;
    virtual int         operator<  (const CAsiInt &)      const;
    virtual int         operator<  (const CAsiNumeric &)  const;
    virtual int         operator<  (const CAsiDecimal &)  const;
    virtual int         operator<  (const CAsiFloat &)    const;
    virtual int         operator<  (const CAsiReal &)     const;
    virtual int         operator<  (const CAsiDouble &)   const;
    virtual int         operator<  (const smallint)       const;
    virtual int         operator<  (const integer)        const;
    virtual int         operator<  (const real)           const;
    virtual int         operator<  (const int)            const;
     
    virtual int         operator!= (const CAsiSmallInt &) const;
    virtual int         operator!= (const CAsiInt &)      const;
    virtual int         operator!= (const CAsiNumeric &)  const;
    virtual int         operator!= (const CAsiDecimal &)  const;
    virtual int         operator!= (const CAsiFloat &)    const;
    virtual int         operator!= (const CAsiReal &)     const;
    virtual int         operator!= (const CAsiDouble &)   const;
    virtual int         operator!= (const smallint)       const;
    virtual int         operator!= (const integer)        const;
    virtual int         operator!= (const real)           const;
    virtual int         operator!= (const int)            const;



protected:

};




////////////////////////////////////////////////
//
//
class CAsiSmallInt : public CAsiExactNum
//
// SMALL INTEGER
//
{
public:
    // Construction/destruction
    //
            CAsiSmallInt (void);
            CAsiSmallInt (const CAsiSmallInt &);
    virtual ~CAsiSmallInt (void);


    //  Get data descriptor
    //
    virtual EAsiDataType Type         (void) const;
    virtual uint         ReturnLength (void) const;

    // Store Data value
    //
    virtual EAsiBoolean   storeValue (real) ;
    virtual EAsiBoolean   storeValue (integer) ;
    virtual EAsiBoolean   storeValue (smallint) ;
    virtual EAsiBoolean   storeValue (char *, uint) ;
    virtual EAsiBoolean   storeValue (const CAsiUcStr &) ;
    virtual EAsiBoolean   storeValue (const CAsiData &);

    //  Get data value 
    //
    virtual integer       intValue  (void) const;
    virtual real          realValue (void) const;

    virtual EAsiBoolean   getValue (real *) const;
    virtual EAsiBoolean   getValue (integer *) const;
    virtual EAsiBoolean   getValue (smallint *) const;
    virtual EAsiBoolean   getValue (char *, uint) const;
    virtual EAsiBoolean   getValue (CAsiUcStr *) const;

    virtual EAsiBoolean   SQLType    (CAsiUcStr *) const;        // SQL data type as string
    virtual EAsiBoolean   SQLLiteral (CAsiUcStr *) const;        // SQL compatible literal


    // ASI application/DBMS driver data exchange
    //
    virtual long          read           (char *, long, long) const;
    virtual long          size           (void) const;

    // Runtime class iteraction functions
    //
    virtual unsigned long packSize   (void) const;
    virtual unsigned long packTo     (void*) const;
    virtual EAsiBoolean   unpackFrom (void*);

    // Duplicate itself
    //
    virtual CAsiObject   *duplicate (void) const;

    // Operators
    //
            // Assignment
            CAsiSmallInt &operator= (const CAsiSmallInt &);
            CAsiSmallInt &operator= (const CAsiInt &);
            CAsiSmallInt &operator= (const CAsiNumeric &);
            CAsiSmallInt &operator= (const CAsiDecimal &);
            CAsiSmallInt &operator= (const CAsiFloat &);
            CAsiSmallInt &operator= (const CAsiReal &);
            CAsiSmallInt &operator= (const CAsiDouble &);
            CAsiSmallInt &operator= (const smallint);
            CAsiSmallInt &operator= (const integer);
            CAsiSmallInt &operator= (const real);
            CAsiSmallInt &operator= (const int);

            // Arithmetic operators
            CAsiSmallInt &operator+= (const CAsiSmallInt &);
            CAsiSmallInt &operator+= (const CAsiInt &);
            CAsiSmallInt &operator+= (const CAsiNumeric &);
            CAsiSmallInt &operator+= (const CAsiDecimal &);
            CAsiSmallInt &operator+= (const CAsiFloat &);
            CAsiSmallInt &operator+= (const CAsiReal &);
            CAsiSmallInt &operator+= (const CAsiDouble &);
            CAsiSmallInt &operator+= (const smallint);
            CAsiSmallInt &operator+= (const integer);
            CAsiSmallInt &operator+= (const real);
            CAsiSmallInt &operator+= (const int);

            CAsiSmallInt &operator-= (const CAsiSmallInt &);
            CAsiSmallInt &operator-= (const CAsiInt &);
            CAsiSmallInt &operator-= (const CAsiNumeric &);
            CAsiSmallInt &operator-= (const CAsiDecimal &);
            CAsiSmallInt &operator-= (const CAsiFloat &);
            CAsiSmallInt &operator-= (const CAsiReal &);
            CAsiSmallInt &operator-= (const CAsiDouble &);
            CAsiSmallInt &operator-= (const smallint);
            CAsiSmallInt &operator-= (const integer);
            CAsiSmallInt &operator-= (const real);
            CAsiSmallInt &operator-= (const int);

            CAsiSmallInt &operator*= (const CAsiSmallInt &);
            CAsiSmallInt &operator*= (const CAsiInt &);
            CAsiSmallInt &operator*= (const CAsiNumeric &);
            CAsiSmallInt &operator*= (const CAsiDecimal &);
            CAsiSmallInt &operator*= (const CAsiFloat &);
            CAsiSmallInt &operator*= (const CAsiReal &);
            CAsiSmallInt &operator*= (const CAsiDouble &);
            CAsiSmallInt &operator*= (const smallint);
            CAsiSmallInt &operator*= (const integer);
            CAsiSmallInt &operator*= (const real);
            CAsiSmallInt &operator*= (const int);

            CAsiSmallInt &operator/= (const CAsiSmallInt &);
            CAsiSmallInt &operator/= (const CAsiInt &);
            CAsiSmallInt &operator/= (const CAsiNumeric &);
            CAsiSmallInt &operator/= (const CAsiDecimal &);
            CAsiSmallInt &operator/= (const CAsiFloat &);
            CAsiSmallInt &operator/= (const CAsiReal &);
            CAsiSmallInt &operator/= (const CAsiDouble &);
            CAsiSmallInt &operator/= (const smallint);
            CAsiSmallInt &operator/= (const integer);
            CAsiSmallInt &operator/= (const real);
            CAsiSmallInt &operator/= (const int);

            CAsiDouble operator+ (const CAsiSmallInt &) const;
            CAsiDouble operator+ (const smallint)       const;
            CAsiDouble operator+ (const integer)        const;
            CAsiDouble operator+ (const real)           const;
            CAsiDouble operator+ (const int)            const;
     friend CAsiDouble operator+ (const smallint, const CAsiSmallInt &);
     friend CAsiDouble operator+ (const integer,  const CAsiSmallInt &);
     friend CAsiDouble operator+ (const real,     const CAsiSmallInt &);
     friend CAsiDouble operator+ (const int,      const CAsiSmallInt &);

            CAsiDouble operator- (const CAsiSmallInt &) const;
            CAsiDouble operator- (const smallint)       const;
            CAsiDouble operator- (const integer)        const;
            CAsiDouble operator- (const real)           const;
            CAsiDouble operator- (const int)            const;
     friend CAsiDouble operator- (const smallint, const CAsiSmallInt &);
     friend CAsiDouble operator- (const integer,  const CAsiSmallInt &);
     friend CAsiDouble operator- (const real,     const CAsiSmallInt &);
     friend CAsiDouble operator- (const int,      const CAsiSmallInt &);


            CAsiDouble operator* (const CAsiSmallInt &) const;
            CAsiDouble operator* (const smallint)       const;
            CAsiDouble operator* (const integer)        const;
            CAsiDouble operator* (const real)           const;
            CAsiDouble operator* (const int)            const;
     friend CAsiDouble operator* (const smallint, const CAsiSmallInt &);
     friend CAsiDouble operator* (const integer,  const CAsiSmallInt &);
     friend CAsiDouble operator* (const real,     const CAsiSmallInt &);
     friend CAsiDouble operator* (const int,      const CAsiSmallInt &);



            CAsiDouble operator/ (const CAsiSmallInt &) const;
            CAsiDouble operator/ (const smallint)  const;
            CAsiDouble operator/ (const integer)   const;
            CAsiDouble operator/ (const real)      const;
            CAsiDouble operator/ (const int)       const;
     friend CAsiDouble operator/ (const smallint, const CAsiSmallInt &);
     friend CAsiDouble operator/ (const integer,  const CAsiSmallInt &);
     friend CAsiDouble operator/ (const real,     const CAsiSmallInt &);
     friend CAsiDouble operator/ (const int,      const CAsiSmallInt &);


            CAsiSmallInt &operator-- (void);
            CAsiSmallInt &operator++ (void);

            CAsiSmallInt &operator-  (void);

private:
    smallint sivalue;
};


////////////////////////////////////////////////
//
//
class CAsiInt : public CAsiExactNum
//
// INTEGER
//
// This class introduces functionality of the SQL INTEGER data type.
// It redefines all arithmetic and comparision operators in which 
// integer number can be involved in SQL. Memeber functions also allow 
// to set and retrieve value of the internal integer number in double and
// integer formats.
//
{
public:
            CAsiInt ();
            CAsiInt (const CAsiInt &);
    virtual ~CAsiInt ();

    //  Get data descriptor
    //
    virtual EAsiDataType Type         (void) const;
    virtual uint         ReturnLength (void) const;

    // Store Data value
    //
    virtual EAsiBoolean   storeValue (real);
    virtual EAsiBoolean   storeValue (integer);
    virtual EAsiBoolean   storeValue (smallint);
    virtual EAsiBoolean   storeValue (char *, uint);
    virtual EAsiBoolean   storeValue (const CAsiUcStr &);
    virtual EAsiBoolean   storeValue (const CAsiData &);

    //  Get data value
    //
    virtual integer       intValue  (void) const;
    virtual real          realValue (void) const;

    virtual EAsiBoolean   getValue (real *) const;
    virtual EAsiBoolean   getValue (integer *) const;
    virtual EAsiBoolean   getValue (smallint *) const;
    virtual EAsiBoolean   getValue (char *, uint) const;
    virtual EAsiBoolean   getValue (CAsiUcStr *) const;

    virtual EAsiBoolean   SQLType    (CAsiUcStr *) const;        // SQL data type as string
    virtual EAsiBoolean   SQLLiteral (CAsiUcStr *) const;        // SQL compatible literal

    // ASI application/DBMS driver data exchange
    //
    virtual long          read           (char *, long, long) const;
    virtual long          size           (void) const;

    // Runtime class iteraction functions
    //
    virtual unsigned long packSize   (void) const;
    virtual unsigned long packTo     (void*) const;
    virtual EAsiBoolean   unpackFrom (void*);

    // Duplicate itself
    //
    virtual CAsiObject   *duplicate (void) const;

    // Operators
    //
            // Assignment
            CAsiInt &operator= (const CAsiSmallInt &);
            CAsiInt &operator= (const CAsiInt &);
            CAsiInt &operator= (const CAsiNumeric &);
            CAsiInt &operator= (const CAsiDecimal &);
            CAsiInt &operator= (const CAsiFloat &);
            CAsiInt &operator= (const CAsiReal &);
            CAsiInt &operator= (const CAsiDouble &);
            CAsiInt &operator= (const smallint);
            CAsiInt &operator= (const integer);
            CAsiInt &operator= (const real);
            CAsiInt &operator= (const int);

            // Arithmetic operators

            CAsiInt &operator+= (const CAsiSmallInt &);
            CAsiInt &operator+= (const CAsiInt &);
            CAsiInt &operator+= (const CAsiNumeric &);
            CAsiInt &operator+= (const CAsiDecimal &);
            CAsiInt &operator+= (const CAsiFloat &);
            CAsiInt &operator+= (const CAsiReal &);
            CAsiInt &operator+= (const CAsiDouble &);
            CAsiInt &operator+= (const smallint);
            CAsiInt &operator+= (const integer);
            CAsiInt &operator+= (const real);
            CAsiInt &operator+= (const int);

            CAsiInt &operator-= (const CAsiSmallInt &);
            CAsiInt &operator-= (const CAsiInt &);
            CAsiInt &operator-= (const CAsiNumeric &);
            CAsiInt &operator-= (const CAsiDecimal &);
            CAsiInt &operator-= (const CAsiFloat &);
            CAsiInt &operator-= (const CAsiReal &);
            CAsiInt &operator-= (const CAsiDouble &);
            CAsiInt &operator-= (const smallint);
            CAsiInt &operator-= (const integer);
            CAsiInt &operator-= (const real);
            CAsiInt &operator-= (const int);

            CAsiInt &operator*= (const CAsiSmallInt &);
            CAsiInt &operator*= (const CAsiInt &);
            CAsiInt &operator*= (const CAsiNumeric &);
            CAsiInt &operator*= (const CAsiDecimal &);
            CAsiInt &operator*= (const CAsiFloat &);
            CAsiInt &operator*= (const CAsiReal &);
            CAsiInt &operator*= (const CAsiDouble &);
            CAsiInt &operator*= (const smallint);
            CAsiInt &operator*= (const integer);
            CAsiInt &operator*= (const real);
            CAsiInt &operator*= (const int);

            CAsiInt &operator/= (const CAsiSmallInt &);
            CAsiInt &operator/= (const CAsiInt &);
            CAsiInt &operator/= (const CAsiNumeric &);
            CAsiInt &operator/= (const CAsiDecimal &);
            CAsiInt &operator/= (const CAsiFloat &);
            CAsiInt &operator/= (const CAsiReal &);
            CAsiInt &operator/= (const CAsiDouble &);
            CAsiInt &operator/= (const smallint);
            CAsiInt &operator/= (const integer);
            CAsiInt &operator/= (const real);
            CAsiInt &operator/= (const int);

            CAsiDouble operator+ (const CAsiSmallInt &) const;
     friend CAsiDouble operator+ (const CAsiSmallInt &, const CAsiInt &);
            CAsiDouble operator+ (const CAsiInt &) const;
            CAsiDouble operator+ (const smallint)  const;
     friend CAsiDouble operator+ (const smallint, const CAsiInt &);
            CAsiDouble operator+ (const integer) const;
     friend CAsiDouble operator+ (const integer, const CAsiInt &);
            CAsiDouble operator+ (const real) const;
     friend CAsiDouble operator+ (const real, const CAsiInt &);
            CAsiDouble operator+ (const int) const;
     friend CAsiDouble operator+ (const int, const CAsiInt &);


            CAsiDouble operator- (const CAsiSmallInt &) const;
     friend CAsiDouble operator- (const CAsiSmallInt &, const CAsiInt &);
            CAsiDouble operator- (const CAsiInt &) const;
            CAsiDouble operator- (const smallint)  const;
     friend CAsiDouble operator- (const smallint, const CAsiInt &);
            CAsiDouble operator- (const integer) const;
     friend CAsiDouble operator- (const integer, const CAsiInt &);
            CAsiDouble operator- (const real) const;
     friend CAsiDouble operator- (const real, const CAsiInt &);
            CAsiDouble operator- (const int) const;
     friend CAsiDouble operator- (const int, const CAsiInt &);

            CAsiDouble operator* (const CAsiSmallInt &) const;
     friend CAsiDouble operator* (const CAsiSmallInt &, const CAsiInt &);
            CAsiDouble operator* (const CAsiInt &) const;
            CAsiDouble operator* (const smallint)  const;
     friend CAsiDouble operator* (const smallint, const CAsiInt &);
            CAsiDouble operator* (const integer) const;
     friend CAsiDouble operator* (const integer, const CAsiInt &);
            CAsiDouble operator* (const real) const;
     friend CAsiDouble operator* (const real, const CAsiInt &);
            CAsiDouble operator* (const int) const;
     friend CAsiDouble operator* (const int, const CAsiInt &);

            CAsiDouble operator/ (const CAsiSmallInt &) const;
     friend CAsiDouble operator/ (const CAsiSmallInt &, const CAsiInt &);
            CAsiDouble operator/ (const CAsiInt &) const;
            CAsiDouble operator/ (const smallint) const;
     friend CAsiDouble operator/ (const smallint, const CAsiInt &);
            CAsiDouble operator/ (const integer) const;
     friend CAsiDouble operator/ (const integer, const CAsiInt &);
            CAsiDouble operator/ (const real) const;
     friend CAsiDouble operator/ (const real, const CAsiInt &);
            CAsiDouble operator/ (const int) const;
     friend CAsiDouble operator/ (const int, const CAsiInt &);

            CAsiInt &operator-- (void);
            CAsiInt &operator++ (void);

            CAsiInt &operator-  (void);

private:
    integer ivalue;
};


////////////////////////////////////////////////
//
//
class CAsiNumeric: public CAsiExactNum
//
// NUMERIC
//
{
private:
    //  private helper
            EAsiBoolean SetNumeric (real);
//
public:
            CAsiNumeric (uint, uint);
            CAsiNumeric (uint);
            CAsiNumeric (const CAsiNumeric &);
            CAsiNumeric (void);
    virtual ~CAsiNumeric (void);

    //  Get data descriptor
    //
    virtual EAsiDataType Type      (void) const;
    virtual uint         ReturnLength (void) const;

    // Store Data value
    //
    virtual EAsiBoolean   storeValue (real);
    virtual EAsiBoolean   storeValue (integer);
    virtual EAsiBoolean   storeValue (smallint);
    virtual EAsiBoolean   storeValue (char *, uint);
    virtual EAsiBoolean   storeValue (const CAsiUcStr &);
    virtual EAsiBoolean   storeValue (const CAsiData &);

    virtual EAsiBoolean   SQLType    (CAsiUcStr *) const;        // SQL data type as string
    virtual EAsiBoolean   SQLLiteral (CAsiUcStr *) const;        // SQL compatible literal

    //  Get data value
    //
    virtual integer       intValue  (void) const;
    virtual real          realValue (void) const;

    virtual EAsiBoolean   getValue (real *) const;
    virtual EAsiBoolean   getValue (integer *) const;
    virtual EAsiBoolean   getValue (smallint *) const;
    virtual EAsiBoolean   getValue (char *, uint) const;
    virtual EAsiBoolean   getValue (CAsiUcStr *) const;

    // ASI application/DBMS driver data exchange
    //
    virtual long          read           (char *, long, long) const;
    virtual long          size           (void) const;

    // Runtime class iteraction functions
    //
    virtual unsigned long packSize   (void) const;
    virtual unsigned long packTo     (void*) const;
    virtual EAsiBoolean   unpackFrom (void*);

    // Duplicate itself
    //
    virtual CAsiObject     *duplicate (void) const;

    // Operators
    //
            // Assignment
            CAsiNumeric &operator= (const CAsiSmallInt &);
            CAsiNumeric &operator= (const CAsiInt &);
            CAsiNumeric &operator= (const CAsiNumeric &);
            CAsiNumeric &operator= (const CAsiDecimal &);
            CAsiNumeric &operator= (const CAsiFloat &);
            CAsiNumeric &operator= (const CAsiReal &);
            CAsiNumeric &operator= (const CAsiDouble &);
            CAsiNumeric &operator= (const smallint);
            CAsiNumeric &operator= (const integer);
            CAsiNumeric &operator= (const real);
            CAsiNumeric &operator= (const int);

            // Arithmetic operators

            CAsiNumeric &operator+= (const CAsiSmallInt &);
            CAsiNumeric &operator+= (const CAsiInt &);
            CAsiNumeric &operator+= (const CAsiNumeric &);
            CAsiNumeric &operator+= (const CAsiDecimal &);
            CAsiNumeric &operator+= (const CAsiFloat &);
            CAsiNumeric &operator+= (const CAsiReal &);
            CAsiNumeric &operator+= (const CAsiDouble &);
            CAsiNumeric &operator+= (const smallint);
            CAsiNumeric &operator+= (const integer);
            CAsiNumeric &operator+= (const real);
            CAsiNumeric &operator+= (const int);

            CAsiNumeric &operator-= (const CAsiSmallInt &);
            CAsiNumeric &operator-= (const CAsiInt &);
            CAsiNumeric &operator-= (const CAsiNumeric &);
            CAsiNumeric &operator-= (const CAsiDecimal &);
            CAsiNumeric &operator-= (const CAsiFloat &);
            CAsiNumeric &operator-= (const CAsiReal &);
            CAsiNumeric &operator-= (const CAsiDouble &);
            CAsiNumeric &operator-= (const smallint);
            CAsiNumeric &operator-= (const integer);
            CAsiNumeric &operator-= (const real);
            CAsiNumeric &operator-= (const int);

            CAsiNumeric &operator*= (const CAsiSmallInt &);
            CAsiNumeric &operator*= (const CAsiInt &);
            CAsiNumeric &operator*= (const CAsiNumeric &);
            CAsiNumeric &operator*= (const CAsiDecimal &);
            CAsiNumeric &operator*= (const CAsiFloat &);
            CAsiNumeric &operator*= (const CAsiReal &);
            CAsiNumeric &operator*= (const CAsiDouble &);
            CAsiNumeric &operator*= (const smallint);
            CAsiNumeric &operator*= (const integer);
            CAsiNumeric &operator*= (const real);
            CAsiNumeric &operator*= (const int);

            CAsiNumeric &operator/= (const CAsiSmallInt &);
            CAsiNumeric &operator/= (const CAsiInt &);
            CAsiNumeric &operator/= (const CAsiNumeric &);
            CAsiNumeric &operator/= (const CAsiDecimal &);
            CAsiNumeric &operator/= (const CAsiFloat &);
            CAsiNumeric &operator/= (const CAsiReal &);
            CAsiNumeric &operator/= (const CAsiDouble &);
            CAsiNumeric &operator/= (const smallint);
            CAsiNumeric &operator/= (const integer);
            CAsiNumeric &operator/= (const real);
            CAsiNumeric &operator/= (const int);

            CAsiDouble operator+ (const CAsiSmallInt &) const;
            CAsiDouble operator+ (const CAsiInt &)      const;
            CAsiDouble operator+ (const CAsiNumeric &)  const;
     friend CAsiDouble operator+ (const CAsiSmallInt &, const CAsiNumeric &);
     friend CAsiDouble operator+ (const CAsiInt &, const CAsiNumeric &);
            CAsiDouble operator+ (const smallint) const;
            CAsiDouble operator+ (const integer)  const;
            CAsiDouble operator+ (const real)     const;
            CAsiDouble operator+ (const int)      const;
     friend CAsiDouble operator+ (const smallint, const CAsiNumeric &);
     friend CAsiDouble operator+ (const integer, const CAsiNumeric &);
     friend CAsiDouble operator+ (const real, const CAsiNumeric &);
     friend CAsiDouble operator+ (const int, const CAsiNumeric &);

            CAsiDouble operator- (const CAsiSmallInt &) const;
            CAsiDouble operator- (const CAsiInt &)      const;
            CAsiDouble operator- (const CAsiNumeric &)  const;
     friend CAsiDouble operator- (const CAsiSmallInt &, const CAsiNumeric &);
     friend CAsiDouble operator- (const CAsiInt &, const CAsiNumeric &);
            CAsiDouble operator- (const smallint) const;
            CAsiDouble operator- (const integer)  const;
            CAsiDouble operator- (const real)     const;
            CAsiDouble operator- (const int)      const;
     friend CAsiDouble operator- (const smallint, const CAsiNumeric &);
     friend CAsiDouble operator- (const integer, const CAsiNumeric &);
     friend CAsiDouble operator- (const real, const CAsiNumeric &);
     friend CAsiDouble operator- (const int, const CAsiNumeric &) ;

            CAsiDouble operator* (const CAsiSmallInt &) const;
            CAsiDouble operator* (const CAsiInt &)      const;
            CAsiDouble operator* (const CAsiNumeric &)  const;
     friend CAsiDouble operator* (const CAsiSmallInt &, const CAsiNumeric &);
     friend CAsiDouble operator* (const CAsiInt &, const CAsiNumeric &);
            CAsiDouble operator* (const smallint) const;
            CAsiDouble operator* (const integer)  const;
            CAsiDouble operator* (const real)     const;
            CAsiDouble operator* (const int)      const;
     friend CAsiDouble operator* (const smallint, const CAsiNumeric &);
     friend CAsiDouble operator* (const integer, const CAsiNumeric &) ;
     friend CAsiDouble operator* (const real, const CAsiNumeric &)    ;
     friend CAsiDouble operator* (const int, const CAsiNumeric &)     ;

            CAsiDouble operator/ (const CAsiSmallInt &) const;
            CAsiDouble operator/ (const CAsiInt &)      const;
            CAsiDouble operator/ (const CAsiNumeric &)  const;
     friend CAsiDouble operator/ (const CAsiSmallInt &, const CAsiNumeric &);
     friend CAsiDouble operator/ (const CAsiInt &, const CAsiNumeric &)     ;
            CAsiDouble operator/ (const smallint) const;
            CAsiDouble operator/ (const integer)  const;
            CAsiDouble operator/ (const real)     const;
            CAsiDouble operator/ (const int)      const;
     friend CAsiDouble operator/ (const smallint, const CAsiNumeric &);
     friend CAsiDouble operator/ (const integer, const CAsiNumeric &) ;
     friend CAsiDouble operator/ (const real, const CAsiNumeric &)    ;
     friend CAsiDouble operator/ (const int, const CAsiNumeric &)     ;

            CAsiNumeric &operator++ (void);
            CAsiNumeric &operator-- (void);

            CAsiNumeric &operator- (void);

private:
    real nvalue;
};


////////////////////////////////////////////////
//
//
class CAsiDecimal: public CAsiExactNum
//
// DECIMAL
//
{
private:
    //  private helper
            EAsiBoolean SetDecimal (real);

public:
            CAsiDecimal (uint, uint);
            CAsiDecimal (uint);
            CAsiDecimal (const CAsiDecimal &);
            CAsiDecimal ();
    virtual ~CAsiDecimal ();

    //  Get data descriptor
    //
    virtual EAsiDataType Type      (void) const;
    virtual uint         ReturnLength (void) const;

    // Store Data value
    //
    virtual EAsiBoolean   storeValue (real);
    virtual EAsiBoolean   storeValue (integer);
    virtual EAsiBoolean   storeValue (smallint);
    virtual EAsiBoolean   storeValue (char *, uint);
    virtual EAsiBoolean   storeValue (const CAsiUcStr &);
    virtual EAsiBoolean   storeValue (const CAsiData &);

    virtual EAsiBoolean   SQLType    (CAsiUcStr *) const;        // SQL data type as string
    virtual EAsiBoolean   SQLLiteral (CAsiUcStr *) const;        // SQL compatible literal

    //  Get data value 
    //
    virtual integer       intValue  (void) const;
    virtual real          realValue (void) const;

    virtual EAsiBoolean   getValue (real *) const;
    virtual EAsiBoolean   getValue (integer *) const;
    virtual EAsiBoolean   getValue (smallint *) const;
    virtual EAsiBoolean   getValue (char *, uint) const;
    virtual EAsiBoolean   getValue (CAsiUcStr *) const;

    // ASI application/DBMS driver data exchange
    //
    virtual long          read           (char *, long, long) const;
    virtual long          size           (void) const;

    // Runtime class iteraction functions
    //
    virtual unsigned long packSize   (void) const;
    virtual unsigned long packTo     (void*) const;
    virtual EAsiBoolean   unpackFrom (void*);

    // Duplicate itself
    //
    virtual CAsiObject   *duplicate (void) const;

    // Operators
    //
            // Assignment
            CAsiDecimal &operator= (const CAsiSmallInt &);
            CAsiDecimal &operator= (const CAsiInt &);
            CAsiDecimal &operator= (const CAsiNumeric &);
            CAsiDecimal &operator= (const CAsiDecimal &);
            CAsiDecimal &operator= (const CAsiFloat &);
            CAsiDecimal &operator= (const CAsiReal &);
            CAsiDecimal &operator= (const CAsiDouble &);
            CAsiDecimal &operator= (const smallint);
            CAsiDecimal &operator= (const integer);
            CAsiDecimal &operator= (const real);
            CAsiDecimal &operator= (const int);

            // Arithmetic operators

            CAsiDecimal &operator+= (const CAsiSmallInt &);
            CAsiDecimal &operator+= (const CAsiInt &);
            CAsiDecimal &operator+= (const CAsiNumeric &);
            CAsiDecimal &operator+= (const CAsiDecimal &);
            CAsiDecimal &operator+= (const CAsiFloat &);
            CAsiDecimal &operator+= (const CAsiReal &);
            CAsiDecimal &operator+= (const CAsiDouble &);
            CAsiDecimal &operator+= (const smallint);
            CAsiDecimal &operator+= (const integer);
            CAsiDecimal &operator+= (const real);
            CAsiDecimal &operator+= (const int);

            CAsiDecimal &operator-= (const CAsiSmallInt &);
            CAsiDecimal &operator-= (const CAsiInt &);
            CAsiDecimal &operator-= (const CAsiNumeric &);
            CAsiDecimal &operator-= (const CAsiDecimal &);
            CAsiDecimal &operator-= (const CAsiFloat &);
            CAsiDecimal &operator-= (const CAsiReal &);
            CAsiDecimal &operator-= (const CAsiDouble &);
            CAsiDecimal &operator-= (const smallint);
            CAsiDecimal &operator-= (const integer);
            CAsiDecimal &operator-= (const real);
            CAsiDecimal &operator-= (const int);

            CAsiDecimal &operator*= (const CAsiSmallInt &);
            CAsiDecimal &operator*= (const CAsiInt &);
            CAsiDecimal &operator*= (const CAsiNumeric &);
            CAsiDecimal &operator*= (const CAsiDecimal &);
            CAsiDecimal &operator*= (const CAsiFloat &);
            CAsiDecimal &operator*= (const CAsiReal &);
            CAsiDecimal &operator*= (const CAsiDouble &);
            CAsiDecimal &operator*= (const smallint);
            CAsiDecimal &operator*= (const integer);
            CAsiDecimal &operator*= (const real);
            CAsiDecimal &operator*= (const int);

            CAsiDecimal &operator/= (const CAsiSmallInt &);
            CAsiDecimal &operator/= (const CAsiInt &);
            CAsiDecimal &operator/= (const CAsiNumeric &);
            CAsiDecimal &operator/= (const CAsiDecimal &);
            CAsiDecimal &operator/= (const CAsiFloat &);
            CAsiDecimal &operator/= (const CAsiReal &);
            CAsiDecimal &operator/= (const CAsiDouble &);
            CAsiDecimal &operator/= (const smallint);
            CAsiDecimal &operator/= (const integer);
            CAsiDecimal &operator/= (const real);
            CAsiDecimal &operator/= (const int);

            CAsiDouble operator+ (const CAsiSmallInt &) const;
            CAsiDouble operator+ (const CAsiInt &)      const;
            CAsiDouble operator+ (const CAsiNumeric &)  const;
            CAsiDouble operator+ (const CAsiDecimal &)  const;
     friend CAsiDouble operator+ (const CAsiSmallInt &, const CAsiDecimal &);
     friend CAsiDouble operator+ (const CAsiInt &, const CAsiDecimal &)     ;
     friend CAsiDouble operator+ (const CAsiNumeric &, const CAsiDecimal &) ;
            CAsiDouble operator+ (const smallint) const;
            CAsiDouble operator+ (const integer)  const;
            CAsiDouble operator+ (const real)     const;
     friend CAsiDouble operator+ (const smallint, const CAsiDecimal &);
     friend CAsiDouble operator+ (const integer, const CAsiDecimal &) ;
     friend CAsiDouble operator+ (const real, const CAsiDecimal &)    ;
            CAsiDouble operator+ (const int) const;
     friend CAsiDouble operator+ (const int, const CAsiDecimal &);


            CAsiDouble operator- (const CAsiSmallInt &) const;
            CAsiDouble operator- (const CAsiInt &)      const;
            CAsiDouble operator- (const CAsiNumeric &)  const;
            CAsiDouble operator- (const CAsiDecimal &)  const;
     friend CAsiDouble operator- (const CAsiSmallInt &, const CAsiDecimal &);
     friend CAsiDouble operator- (const CAsiInt &, const CAsiDecimal &);
     friend CAsiDouble operator- (const CAsiNumeric &, const CAsiDecimal &);
            CAsiDouble operator- (const smallint) const;
            CAsiDouble operator- (const integer)  const;
            CAsiDouble operator- (const real)     const;
     friend CAsiDouble operator- (const smallint, const CAsiDecimal &);
     friend CAsiDouble operator- (const integer, const CAsiDecimal &) ;
     friend CAsiDouble operator- (const real, const CAsiDecimal &)    ;
            CAsiDouble operator- (const int) const;
     friend CAsiDouble operator- (const int, const CAsiDecimal &);

            CAsiDouble operator* (const CAsiSmallInt &) const;
            CAsiDouble operator* (const CAsiInt &)      const;
            CAsiDouble operator* (const CAsiNumeric &)  const;
            CAsiDouble operator* (const CAsiDecimal &)  const;
     friend CAsiDouble operator* (const CAsiSmallInt &, const CAsiDecimal &);
     friend CAsiDouble operator* (const CAsiInt &, const CAsiDecimal &)     ;
     friend CAsiDouble operator* (const CAsiNumeric &, const CAsiDecimal &) ;
            CAsiDouble operator* (const smallint) const;
            CAsiDouble operator* (const integer)  const;
            CAsiDouble operator* (const real)     const;
     friend CAsiDouble operator* (const smallint, const CAsiDecimal &);
     friend CAsiDouble operator* (const integer, const CAsiDecimal &) ;
     friend CAsiDouble operator* (const real, const CAsiDecimal &)    ;
            CAsiDouble operator* (const int)                          ;
     friend CAsiDouble operator* (const int, const CAsiDecimal &)     ;

            CAsiDouble operator/ (const CAsiSmallInt &) const;
            CAsiDouble operator/ (const CAsiInt &)      const;
            CAsiDouble operator/ (const CAsiNumeric &)  const;
            CAsiDouble operator/ (const CAsiDecimal &)  const;
     friend CAsiDouble operator/ (const CAsiSmallInt &, const CAsiDecimal &);
     friend CAsiDouble operator/ (const CAsiInt &, const CAsiDecimal &);
     friend CAsiDouble operator/ (const CAsiNumeric &, const CAsiDecimal &);
            CAsiDouble operator/ (const smallint) const;
            CAsiDouble operator/ (const integer)  const;
            CAsiDouble operator/ (const real)     const;
     friend CAsiDouble operator/ (const smallint, const CAsiDecimal &);
     friend CAsiDouble operator/ (const integer, const CAsiDecimal &) ;
     friend CAsiDouble operator/ (const real, const CAsiDecimal &)    ;
            CAsiDouble operator/ (const int) const;
     friend CAsiDouble operator/ (const int, const CAsiDecimal &);

            CAsiDecimal &operator++ (void);
            CAsiDecimal &operator-- (void);
            CAsiDecimal &operator- (void);

private:
    real dvalue;
};


////////////////////////////////////////////////
//
//
class CAsiFloat: public CAsiApproxNum
//
// FLOAT
//
{
public:
            CAsiFloat (void);
            CAsiFloat (uint);
            CAsiFloat (const CAsiFloat &);
    virtual ~CAsiFloat (void);

    //  Get data descriptor
    //
    virtual EAsiDataType Type         (void) const;
    virtual uint         ReturnLength (void) const;

    // Store Data value
    //
    virtual EAsiBoolean   storeValue (real);
    virtual EAsiBoolean   storeValue (integer);
    virtual EAsiBoolean   storeValue (smallint);
    virtual EAsiBoolean   storeValue (char *, uint);
    virtual EAsiBoolean   storeValue (const CAsiUcStr &);
    virtual EAsiBoolean   storeValue (const CAsiData &);

    virtual EAsiBoolean   SQLType    (CAsiUcStr *) const;        // SQL data type as string
    virtual EAsiBoolean   SQLLiteral (CAsiUcStr *) const;        // SQL compatible literal

    //  Get data value
    //
    virtual integer       intValue  (void) const;
    virtual real          realValue (void) const;

    virtual EAsiBoolean   getValue (real *) const;
    virtual EAsiBoolean   getValue (integer *) const;
    virtual EAsiBoolean   getValue (smallint *) const;
    virtual EAsiBoolean   getValue (char *, uint) const;
    virtual EAsiBoolean   getValue (CAsiUcStr *) const;

    // ASI application/DBMS driver data exchange
    //
    virtual long          read           (char *, long, long) const;
    virtual long          size           (void) const;

    // Runtime class iteraction functions
    //
    virtual unsigned long packSize   (void) const;
    virtual unsigned long packTo     (void*) const;
    virtual EAsiBoolean   unpackFrom (void*);

    // Duplicate itself
    //
    virtual CAsiObject   *duplicate (void) const;

    // Operators
    //
            // Assignment
            CAsiFloat &operator= (const CAsiSmallInt &);
            CAsiFloat &operator= (const CAsiInt &);
            CAsiFloat &operator= (const CAsiNumeric &);
            CAsiFloat &operator= (const CAsiDecimal &);
            CAsiFloat &operator= (const CAsiFloat &);
            CAsiFloat &operator= (const CAsiReal &);
            CAsiFloat &operator= (const CAsiDouble &);
            CAsiFloat &operator= (const smallint);
            CAsiFloat &operator= (const integer);
            CAsiFloat &operator= (const real);
            CAsiFloat &operator= (const int);

            // Arithmetic operators
            CAsiFloat &operator+= (const CAsiSmallInt &);
            CAsiFloat &operator+= (const CAsiInt &);
            CAsiFloat &operator+= (const CAsiNumeric &);
            CAsiFloat &operator+= (const CAsiDecimal &);
            CAsiFloat &operator+= (const CAsiFloat &);
            CAsiFloat &operator+= (const CAsiReal &);
            CAsiFloat &operator+= (const CAsiDouble &);
            CAsiFloat &operator+= (const smallint);
            CAsiFloat &operator+= (const integer);
            CAsiFloat &operator+= (const real);
            CAsiFloat &operator+= (const int);

            CAsiFloat &operator-= (const CAsiSmallInt &);
            CAsiFloat &operator-= (const CAsiInt &);
            CAsiFloat &operator-= (const CAsiNumeric &);
            CAsiFloat &operator-= (const CAsiDecimal &);
            CAsiFloat &operator-= (const CAsiFloat &);
            CAsiFloat &operator-= (const CAsiReal &);
            CAsiFloat &operator-= (const CAsiDouble &);
            CAsiFloat &operator-= (const smallint);
            CAsiFloat &operator-= (const integer);
            CAsiFloat &operator-= (const real);
            CAsiFloat &operator-= (const int);

            CAsiFloat &operator*= (const CAsiSmallInt &);
            CAsiFloat &operator*= (const CAsiInt &);
            CAsiFloat &operator*= (const CAsiNumeric &);
            CAsiFloat &operator*= (const CAsiDecimal &);
            CAsiFloat &operator*= (const CAsiFloat &);
            CAsiFloat &operator*= (const CAsiReal &);
            CAsiFloat &operator*= (const CAsiDouble &);
            CAsiFloat &operator*= (const smallint);
            CAsiFloat &operator*= (const integer);
            CAsiFloat &operator*= (const real);
            CAsiFloat &operator*= (const int);

            CAsiFloat &operator/= (const CAsiSmallInt &);
            CAsiFloat &operator/= (const CAsiInt &);
            CAsiFloat &operator/= (const CAsiNumeric &);
            CAsiFloat &operator/= (const CAsiDecimal &);
            CAsiFloat &operator/= (const CAsiFloat &);
            CAsiFloat &operator/= (const CAsiReal &);
            CAsiFloat &operator/= (const CAsiDouble &);
            CAsiFloat &operator/= (const smallint);
            CAsiFloat &operator/= (const integer);
            CAsiFloat &operator/= (const real);
            CAsiFloat &operator/= (const int);

            CAsiFloat operator+ (const CAsiSmallInt &) const;
            CAsiFloat operator+ (const CAsiInt &)      const;
            CAsiFloat operator+ (const CAsiNumeric &)  const;
            CAsiFloat operator+ (const CAsiDecimal &)  const;
            CAsiFloat operator+ (const CAsiFloat &)    const;
     friend CAsiFloat operator+ (const CAsiSmallInt &, const CAsiFloat &);
     friend CAsiFloat operator+ (const CAsiInt &, const CAsiFloat &)     ;
     friend CAsiFloat operator+ (const CAsiNumeric &, const CAsiFloat &) ;
     friend CAsiFloat operator+ (const CAsiDecimal &, const CAsiFloat &) ;
            CAsiFloat operator+ (const smallint) const;
            CAsiFloat operator+ (const integer)  const;
            CAsiFloat operator+ (const real)     const;
     friend CAsiFloat operator+ (const smallint, const CAsiFloat &);
     friend CAsiFloat operator+ (const integer, const CAsiFloat &) ;
     friend CAsiFloat operator+ (const real, const CAsiFloat &)    ;
            CAsiFloat operator+ (const int) const;
     friend CAsiFloat operator+ (const int, const CAsiFloat &);

            CAsiFloat operator- (const CAsiSmallInt &) const;
            CAsiFloat operator- (const CAsiInt &)      const;
            CAsiFloat operator- (const CAsiNumeric &)  const;
            CAsiFloat operator- (const CAsiDecimal &)  const;
            CAsiFloat operator- (const CAsiFloat &)    const;
     friend CAsiFloat operator- (const CAsiSmallInt &, const CAsiFloat &);
     friend CAsiFloat operator- (const CAsiInt &, const CAsiFloat &);
     friend CAsiFloat operator- (const CAsiNumeric &, const CAsiFloat &);
     friend CAsiFloat operator- (const CAsiDecimal &, const CAsiFloat &);
            CAsiFloat operator- (const smallint) const;
            CAsiFloat operator- (const integer)  const;
            CAsiFloat operator- (const real)     const;
     friend CAsiFloat operator- (const smallint, const CAsiFloat &);
     friend CAsiFloat operator- (const integer, const CAsiFloat &);
     friend CAsiFloat operator- (const real, const CAsiFloat &);
            CAsiFloat operator- (const int) const;
     friend CAsiFloat operator- (const int, const CAsiFloat &);

            CAsiFloat operator* (const CAsiSmallInt &) const;
            CAsiFloat operator* (const CAsiInt &)      const;
            CAsiFloat operator* (const CAsiNumeric &)  const;
            CAsiFloat operator* (const CAsiDecimal &)  const;
            CAsiFloat operator* (const CAsiFloat &)    const;
     friend CAsiFloat operator* (const CAsiSmallInt &, const CAsiFloat &);
     friend CAsiFloat operator* (const CAsiInt &, const CAsiFloat &);
     friend CAsiFloat operator* (const CAsiNumeric &, const CAsiFloat &);
     friend CAsiFloat operator* (const CAsiDecimal &, const CAsiFloat &);
            CAsiFloat operator* (const smallint) const;
            CAsiFloat operator* (const integer)  const;
            CAsiFloat operator* (const real)     const;
     friend CAsiFloat operator* (const smallint, const CAsiFloat &);
     friend CAsiFloat operator* (const integer, const CAsiFloat &);
     friend CAsiFloat operator* (const real, const CAsiFloat &);
            CAsiFloat operator* (const int) const;
     friend CAsiFloat operator* (const int, const CAsiFloat &);

            CAsiFloat operator/ (const CAsiSmallInt &) const;
            CAsiFloat operator/ (const CAsiInt &)      const;
            CAsiFloat operator/ (const CAsiNumeric &)  const;
            CAsiFloat operator/ (const CAsiDecimal &)  const;
            CAsiFloat operator/ (const CAsiFloat &)    const;
     friend CAsiFloat operator/ (const CAsiSmallInt &, const CAsiFloat &);
     friend CAsiFloat operator/ (const CAsiInt &, const CAsiFloat &);
     friend CAsiFloat operator/ (const CAsiNumeric &, const CAsiFloat &);
     friend CAsiFloat operator/ (const CAsiDecimal &, const CAsiFloat &);
            CAsiFloat operator/ (const smallint) const;
            CAsiFloat operator/ (const integer)  const;
            CAsiFloat operator/ (const real)     const;
     friend CAsiFloat operator/ (const smallint, const CAsiFloat &);
     friend CAsiFloat operator/ (const integer, const CAsiFloat &);
     friend CAsiFloat operator/ (const real, const CAsiFloat &)   ;
            CAsiFloat operator/ (const int) const;
     friend CAsiFloat operator/ (const int, const CAsiFloat &);

            CAsiFloat &operator++ (void);
            CAsiFloat &operator-- (void);
            CAsiFloat &operator- (void);

private:
    real fvalue;
};



////////////////////////////////////////////////
//
//
class CAsiReal: public CAsiApproxNum
//
// REAL
//
// This class introduces functionality of the SQL REAL data.
// It redefines all arithmetic and comparision operators in which 
// real number can be involved in SQL. Memeber functions also allow 
// to set and retrieve value of the internal real number in double and
// integer formats.

{
public:
            CAsiReal (void);
            CAsiReal (const CAsiReal &);
    virtual ~CAsiReal (void);

    //  Get data descriptor
    //
    virtual EAsiDataType Type      (void) const;
    virtual uint         ReturnLength (void) const;

    // Store Data value
    //
    virtual EAsiBoolean   storeValue (real);
    virtual EAsiBoolean   storeValue (integer);
    virtual EAsiBoolean   storeValue (smallint);
    virtual EAsiBoolean   storeValue (char *, uint);
    virtual EAsiBoolean   storeValue (const CAsiUcStr &);
    virtual EAsiBoolean   storeValue (const CAsiData &);

    virtual EAsiBoolean   SQLType    (CAsiUcStr *) const;        // SQL data type as string
    virtual EAsiBoolean   SQLLiteral (CAsiUcStr *) const;        // SQL compatible literal

    //  Get data value 
    //
    virtual integer       intValue  (void) const;
    virtual real          realValue (void) const;

    virtual EAsiBoolean   getValue (real *) const;
    virtual EAsiBoolean   getValue (integer *) const;
    virtual EAsiBoolean   getValue (smallint *) const;
    virtual EAsiBoolean   getValue (char *, uint) const;
    virtual EAsiBoolean   getValue (CAsiUcStr *) const;

    // ASI application/DBMS driver data exchange
    //
    virtual long          read           (char *, long, long) const;
    virtual long          size           (void) const;

    // Runtime class iteraction functions
    //
    virtual unsigned long packSize   (void) const;
    virtual unsigned long packTo     (void*) const;
    virtual EAsiBoolean   unpackFrom (void*);

    // Duplicate itself
    //
    virtual CAsiObject   *duplicate (void) const;

    // Operators
    //
            // Assignment
            CAsiReal &operator= (const CAsiSmallInt &);
            CAsiReal &operator= (const CAsiInt &);
            CAsiReal &operator= (const CAsiNumeric &);
            CAsiReal &operator= (const CAsiDecimal &);
            CAsiReal &operator= (const CAsiFloat &);
            CAsiReal &operator= (const CAsiReal &);
            CAsiReal &operator= (const CAsiDouble &);
            CAsiReal &operator= (const smallint);
            CAsiReal &operator= (const integer);
            CAsiReal &operator= (const real);
            CAsiReal &operator= (const int);

            // Arithmetic operators
            CAsiReal &operator+= (const CAsiSmallInt &);
            CAsiReal &operator+= (const CAsiInt &);
            CAsiReal &operator+= (const CAsiNumeric &);
            CAsiReal &operator+= (const CAsiDecimal &);
            CAsiReal &operator+= (const CAsiFloat &);
            CAsiReal &operator+= (const CAsiReal &);
            CAsiReal &operator+= (const CAsiDouble &);
            CAsiReal &operator+= (const smallint);
            CAsiReal &operator+= (const integer);
            CAsiReal &operator+= (const real);
            CAsiReal &operator+= (const int);

            CAsiReal &operator-= (const CAsiSmallInt &);
            CAsiReal &operator-= (const CAsiInt &);
            CAsiReal &operator-= (const CAsiNumeric &);
            CAsiReal &operator-= (const CAsiDecimal &);
            CAsiReal &operator-= (const CAsiFloat &);
            CAsiReal &operator-= (const CAsiReal &);
            CAsiReal &operator-= (const CAsiDouble &);
            CAsiReal &operator-= (const smallint);
            CAsiReal &operator-= (const integer);
            CAsiReal &operator-= (const real);
            CAsiReal &operator-= (const int);

            CAsiReal &operator*= (const CAsiSmallInt &);
            CAsiReal &operator*= (const CAsiInt &);
            CAsiReal &operator*= (const CAsiNumeric &);
            CAsiReal &operator*= (const CAsiDecimal &);
            CAsiReal &operator*= (const CAsiFloat &);
            CAsiReal &operator*= (const CAsiReal &);
            CAsiReal &operator*= (const CAsiDouble &);
            CAsiReal &operator*= (const smallint);
            CAsiReal &operator*= (const integer);
            CAsiReal &operator*= (const real);
            CAsiReal &operator*= (const int);

            CAsiReal &operator/= (const CAsiSmallInt &);
            CAsiReal &operator/= (const CAsiInt &);
            CAsiReal &operator/= (const CAsiNumeric &);
            CAsiReal &operator/= (const CAsiDecimal &);
            CAsiReal &operator/= (const CAsiFloat &);
            CAsiReal &operator/= (const CAsiReal &);
            CAsiReal &operator/= (const CAsiDouble &);
            CAsiReal &operator/= (const smallint);
            CAsiReal &operator/= (const integer);
            CAsiReal &operator/= (const real);
            CAsiReal &operator/= (const int);

            CAsiReal operator+ (const CAsiSmallInt &) const;
            CAsiReal operator+ (const CAsiInt &)      const;
            CAsiReal operator+ (const CAsiNumeric &)  const;
            CAsiReal operator+ (const CAsiDecimal &)  const;
            CAsiReal operator+ (const CAsiFloat &)    const;
            CAsiReal operator+ (const CAsiReal &)     const;
     friend CAsiReal operator+ (const CAsiSmallInt &, const CAsiReal &);
     friend CAsiReal operator+ (const CAsiInt &, const CAsiReal &);
     friend CAsiReal operator+ (const CAsiNumeric &, const CAsiReal &);
     friend CAsiReal operator+ (const CAsiDecimal &, const CAsiReal &);
     friend CAsiReal operator+ (const CAsiFloat &, const CAsiReal &)  ;
            CAsiReal operator+ (const smallint) const;
            CAsiReal operator+ (const integer)  const;
            CAsiReal operator+ (const real)     const;
     friend CAsiReal operator+ (const smallint, const CAsiReal &);
     friend CAsiReal operator+ (const integer, const CAsiReal &) ;
     friend CAsiReal operator+ (const real, const CAsiReal &)    ;
            CAsiReal operator+ (const int) const;
     friend CAsiReal operator+ (const int, const CAsiReal &);

            CAsiReal operator- (const CAsiSmallInt &) const;
            CAsiReal operator- (const CAsiInt &)      const;
            CAsiReal operator- (const CAsiNumeric &)  const;
            CAsiReal operator- (const CAsiDecimal &)  const;
            CAsiReal operator- (const CAsiFloat &)    const;
            CAsiReal operator- (const CAsiReal &)     const;
     friend CAsiReal operator- (const CAsiSmallInt &, const CAsiReal &);
     friend CAsiReal operator- (const CAsiInt &, const CAsiReal &);
     friend CAsiReal operator- (const CAsiNumeric &, const CAsiReal &);
     friend CAsiReal operator- (const CAsiDecimal &, const CAsiReal &);
     friend CAsiReal operator- (const CAsiFloat &, const CAsiReal &)  ;
            CAsiReal operator- (const smallint) const;
            CAsiReal operator- (const integer)  const;
            CAsiReal operator- (const real)     const;
     friend CAsiReal operator- (const smallint, const CAsiReal &);
     friend CAsiReal operator- (const integer, const CAsiReal &);
     friend CAsiReal operator- (const real, const CAsiReal &);
            CAsiReal operator- (const int) const;
     friend CAsiReal operator- (const int, const CAsiReal &);

            CAsiReal operator* (const CAsiSmallInt &) const;
            CAsiReal operator* (const CAsiInt &)      const;
            CAsiReal operator* (const CAsiNumeric &)  const;
            CAsiReal operator* (const CAsiDecimal &)  const;
            CAsiReal operator* (const CAsiFloat &)    const;
            CAsiReal operator* (const CAsiReal &)     const;
     friend CAsiReal operator* (const CAsiSmallInt &, const CAsiReal &);
     friend CAsiReal operator* (const CAsiInt &, const CAsiReal &);
     friend CAsiReal operator* (const CAsiNumeric &, const CAsiReal &);
     friend CAsiReal operator* (const CAsiDecimal &, const CAsiReal &);
     friend CAsiReal operator* (const CAsiFloat &, const CAsiReal &)  ;
            CAsiReal operator* (const smallint) const;
            CAsiReal operator* (const integer)  const;
            CAsiReal operator* (const real)     const;
     friend CAsiReal operator* (const smallint, const CAsiReal &);
     friend CAsiReal operator* (const integer, const CAsiReal &) ;
     friend CAsiReal operator* (const real, const CAsiReal &)    ;
            CAsiReal operator* (const int) const;
     friend CAsiReal operator* (const int, const CAsiReal &);

            CAsiReal operator/ (const CAsiSmallInt &) const;
            CAsiReal operator/ (const CAsiInt &)      const;
            CAsiReal operator/ (const CAsiNumeric &)  const;
            CAsiReal operator/ (const CAsiDecimal &)  const;
            CAsiReal operator/ (const CAsiFloat &)    const;
            CAsiReal operator/ (const CAsiReal &)     const;
     friend CAsiReal operator/ (const CAsiSmallInt &, const CAsiReal &);
     friend CAsiReal operator/ (const CAsiInt &, const CAsiReal &);
     friend CAsiReal operator/ (const CAsiNumeric &, const CAsiReal &);
     friend CAsiReal operator/ (const CAsiDecimal &, const CAsiReal &);
     friend CAsiReal operator/ (const CAsiFloat &, const CAsiReal &)  ;
            CAsiReal operator/ (const smallint) const;
            CAsiReal operator/ (const integer)  const;
            CAsiReal operator/ (const real)     const;
     friend CAsiReal operator/ (const smallint, const CAsiReal &);
     friend CAsiReal operator/ (const integer, const CAsiReal &) ;
     friend CAsiReal operator/ (const real, const CAsiReal &);
            CAsiReal operator/ (const int) const;
     friend CAsiReal operator/ (const int, const CAsiReal &);

            CAsiReal &operator++ (void);
            CAsiReal &operator-- (void);
            CAsiReal &operator- (void);

private:
    real rvalue;
};


////////////////////////////////////////////////
//
//
class CAsiDouble: public CAsiApproxNum
//
// DOUBLE PRECISION
//
{
public:
            CAsiDouble (void);
            CAsiDouble (const CAsiDouble &);
    virtual ~CAsiDouble (void);

    //  Get data descriptor
    //
    virtual EAsiDataType Type      (void) const;
    virtual uint         ReturnLength (void) const;

    // Store Data value
    //
    virtual EAsiBoolean   storeValue (real);
    virtual EAsiBoolean   storeValue (integer);
    virtual EAsiBoolean   storeValue (smallint);
    virtual EAsiBoolean   storeValue (char *, uint);
    virtual EAsiBoolean   storeValue (const CAsiUcStr &);
    virtual EAsiBoolean   storeValue (const CAsiData &);

    //  Get data value 
    //
    virtual integer       intValue  (void) const;
    virtual real          realValue (void) const;

    virtual EAsiBoolean   getValue (real *) const;
    virtual EAsiBoolean   getValue (integer *) const;
    virtual EAsiBoolean   getValue (smallint *) const;
    virtual EAsiBoolean   getValue (char *, uint) const;
    virtual EAsiBoolean   getValue (CAsiUcStr *) const;

    virtual EAsiBoolean   SQLType    (CAsiUcStr *) const;        // SQL data type as string
    virtual EAsiBoolean   SQLLiteral (CAsiUcStr *) const;        // SQL compatible literal

    // ASI application/DBMS driver data exchange
    //
    virtual long          read           (char *, long, long) const;
    virtual long          size           (void) const;

    // Runtime class iteraction functions
    //
    virtual unsigned long packSize   (void) const;
    virtual unsigned long packTo     (void*) const;
    virtual EAsiBoolean   unpackFrom (void*);

    // Duplicate itself
    //
    virtual CAsiObject   *duplicate (void) const;

    // Operators
    //
            // Assignment
            CAsiDouble &operator= (const CAsiSmallInt &);
            CAsiDouble &operator= (const CAsiInt &);
            CAsiDouble &operator= (const CAsiNumeric &);
            CAsiDouble &operator= (const CAsiDecimal &);
            CAsiDouble &operator= (const CAsiFloat &);
            CAsiDouble &operator= (const CAsiReal &);
            CAsiDouble &operator= (const CAsiDouble &);
            CAsiDouble &operator= (const smallint);
            CAsiDouble &operator= (const integer);
            CAsiDouble &operator= (const real);
            CAsiDouble &operator= (const int);

            // Arithmetic operators
            CAsiDouble &operator+= (const CAsiSmallInt &);
            CAsiDouble &operator+= (const CAsiInt &);
            CAsiDouble &operator+= (const CAsiNumeric &);
            CAsiDouble &operator+= (const CAsiDecimal &);
            CAsiDouble &operator+= (const CAsiFloat &);
            CAsiDouble &operator+= (const CAsiReal &);
            CAsiDouble &operator+= (const CAsiDouble &);
            CAsiDouble &operator+= (const smallint);
            CAsiDouble &operator+= (const integer);
            CAsiDouble &operator+= (const real);
            CAsiDouble &operator+= (const int);

            CAsiDouble &operator-= (const CAsiSmallInt &);
            CAsiDouble &operator-= (const CAsiInt &);
            CAsiDouble &operator-= (const CAsiNumeric &);
            CAsiDouble &operator-= (const CAsiDecimal &);
            CAsiDouble &operator-= (const CAsiFloat &);
            CAsiDouble &operator-= (const CAsiReal &);
            CAsiDouble &operator-= (const CAsiDouble &);
            CAsiDouble &operator-= (const smallint);
            CAsiDouble &operator-= (const integer);
            CAsiDouble &operator-= (const real);
            CAsiDouble &operator-= (const int);

            CAsiDouble &operator*= (const CAsiSmallInt &);
            CAsiDouble &operator*= (const CAsiInt &);
            CAsiDouble &operator*= (const CAsiNumeric &);
            CAsiDouble &operator*= (const CAsiDecimal &);
            CAsiDouble &operator*= (const CAsiFloat &);
            CAsiDouble &operator*= (const CAsiReal &);
            CAsiDouble &operator*= (const CAsiDouble &);
            CAsiDouble &operator*= (const smallint);
            CAsiDouble &operator*= (const integer);
            CAsiDouble &operator*= (const real);
            CAsiDouble &operator*= (const int);

            CAsiDouble &operator/= (const CAsiSmallInt &);
            CAsiDouble &operator/= (const CAsiInt &);
            CAsiDouble &operator/= (const CAsiNumeric &);
            CAsiDouble &operator/= (const CAsiDecimal &);
            CAsiDouble &operator/= (const CAsiFloat &);
            CAsiDouble &operator/= (const CAsiReal &);
            CAsiDouble &operator/= (const CAsiDouble &);
            CAsiDouble &operator/= (const smallint);
            CAsiDouble &operator/= (const integer);
            CAsiDouble &operator/= (const real);
            CAsiDouble &operator/= (const int);

            CAsiDouble operator+ (const CAsiSmallInt &) const;
            CAsiDouble operator+ (const CAsiInt &)      const;
            CAsiDouble operator+ (const CAsiNumeric &)  const;
            CAsiDouble operator+ (const CAsiDecimal &)  const;
            CAsiDouble operator+ (const CAsiFloat &)    const;
            CAsiDouble operator+ (const CAsiReal &)     const;
            CAsiDouble operator+ (const CAsiDouble &)   const;
     friend CAsiDouble operator+ (const CAsiSmallInt &, const CAsiDouble &);
     friend CAsiDouble operator+ (const CAsiInt &, const CAsiDouble &);
     friend CAsiDouble operator+ (const CAsiNumeric &, const CAsiDouble &);
     friend CAsiDouble operator+ (const CAsiDecimal &, const CAsiDouble &);
     friend CAsiDouble operator+ (const CAsiFloat &, const CAsiDouble &);
     friend CAsiDouble operator+ (const CAsiReal &, const CAsiDouble &);
            CAsiDouble operator+ (const smallint) const;
            CAsiDouble operator+ (const integer)  const;
            CAsiDouble operator+ (const real)     const;
     friend CAsiDouble operator+ (const smallint, const CAsiDouble &);
     friend CAsiDouble operator+ (const integer, const CAsiDouble &) ;
     friend CAsiDouble operator+ (const real, const CAsiDouble &)    ;
            CAsiDouble operator+ (const int) const;
     friend CAsiDouble operator+ (const int, const CAsiDouble &);

            CAsiDouble operator- (const CAsiSmallInt &) const;
            CAsiDouble operator- (const CAsiInt &)      const;
            CAsiDouble operator- (const CAsiNumeric &)  const;
            CAsiDouble operator- (const CAsiDecimal &)  const;
            CAsiDouble operator- (const CAsiFloat &)    const;
            CAsiDouble operator- (const CAsiReal &)     const;
            CAsiDouble operator- (const CAsiDouble &)   const;
     friend CAsiDouble operator- (const CAsiSmallInt &, const CAsiDouble &);
     friend CAsiDouble operator- (const CAsiInt &, const CAsiDouble &);
     friend CAsiDouble operator- (const CAsiNumeric &, const CAsiDouble &);
     friend CAsiDouble operator- (const CAsiDecimal &, const CAsiDouble &);
     friend CAsiDouble operator- (const CAsiFloat &, const CAsiDouble &)  ;
     friend CAsiDouble operator- (const CAsiReal &, const CAsiDouble &);
            CAsiDouble operator- (const smallint) const;
            CAsiDouble operator- (const integer)  const;
            CAsiDouble operator- (const real)     const;
     friend CAsiDouble operator- (const smallint, const CAsiDouble &);
     friend CAsiDouble operator- (const integer, const CAsiDouble &);
     friend CAsiDouble operator- (const real, const CAsiDouble &);
            CAsiDouble operator- (const int) const;
     friend CAsiDouble operator- (const int, const CAsiDouble &);

            CAsiDouble operator* (const CAsiSmallInt &) const;
            CAsiDouble operator* (const CAsiInt &)      const;
            CAsiDouble operator* (const CAsiNumeric &)  const;
            CAsiDouble operator* (const CAsiDecimal &)  const;
            CAsiDouble operator* (const CAsiFloat &)    const;
            CAsiDouble operator* (const CAsiReal &)     const;
            CAsiDouble operator* (const CAsiDouble &)   const;
     friend CAsiDouble operator* (const CAsiSmallInt &, const CAsiDouble &);
     friend CAsiDouble operator* (const CAsiInt &, const CAsiDouble &);
     friend CAsiDouble operator* (const CAsiNumeric &, const CAsiDouble &);
     friend CAsiDouble operator* (const CAsiDecimal &, const CAsiDouble &);
     friend CAsiDouble operator* (const CAsiFloat &, const CAsiDouble &);
     friend CAsiDouble operator* (const CAsiReal &, const CAsiDouble &);
            CAsiDouble operator* (const smallint) const;
            CAsiDouble operator* (const integer) const;
            CAsiDouble operator* (const real) const;
     friend CAsiDouble operator* (const smallint, const CAsiDouble &);
     friend CAsiDouble operator* (const integer, const CAsiDouble &);
     friend CAsiDouble operator* (const real, const CAsiDouble &);
            CAsiDouble operator* (const int) const;
     friend CAsiDouble operator* (const int, const CAsiDouble &);

            CAsiDouble operator/ (const CAsiSmallInt &) const;
            CAsiDouble operator/ (const CAsiInt &)      const;
            CAsiDouble operator/ (const CAsiNumeric &)  const;
            CAsiDouble operator/ (const CAsiDecimal &)  const;
            CAsiDouble operator/ (const CAsiFloat &)    const;
            CAsiDouble operator/ (const CAsiReal &)     const;
            CAsiDouble operator/ (const CAsiDouble &)   const;
     friend CAsiDouble operator/ (const CAsiSmallInt &, const CAsiDouble &);
     friend CAsiDouble operator/ (const CAsiInt &, const CAsiDouble &);
     friend CAsiDouble operator/ (const CAsiNumeric &, const CAsiDouble &);
     friend CAsiDouble operator/ (const CAsiDecimal &, const CAsiDouble &);
     friend CAsiDouble operator/ (const CAsiFloat &, const CAsiDouble &);
     friend CAsiDouble operator/ (const CAsiReal &, const CAsiDouble &);
            CAsiDouble operator/ (const smallint) const;
            CAsiDouble operator/ (const integer)  const;
            CAsiDouble operator/ (const real)     const;
     friend CAsiDouble operator/ (const smallint, const CAsiDouble &);
     friend CAsiDouble operator/ (const integer, const CAsiDouble &);
     friend CAsiDouble operator/ (const real, const CAsiDouble &);
            CAsiDouble operator/ (const int) const;
     friend CAsiDouble operator/ (const int, const CAsiDouble &);

            CAsiDouble &operator++ (void);
            CAsiDouble &operator-- (void);
            CAsiDouble &operator- (void);


private:
    real dvalue;
};


////////////////////////////////////////////////
//
//
class CAsiChar: public CAsiData
//
// CHARACTER
//
{
friend class CAsiTempTable;
public:
    // Construction/Destruction
    //
            CAsiChar ();
            CAsiChar (uint);
            CAsiChar (const CAsiChar &);
    virtual ~CAsiChar ();


    // Set Varying, National, Character set name, and Collation
           void           setVarying    (void);
           void           setNational   (void);
           void           setCharSet    (const CAsiUcStr &);
           void           setDefCharSet (void);
           void           setCollName   (const CAsiUcStr &);
           void           setCollIdent  (AsiUcIdCLT);
           void           setCoerc      (EAsiCAttr);

    // Get data descriptor
    //
    virtual EAsiDataType    Type         (void) const;
    virtual uint            Length       (void) const;
    virtual uint            ReturnLength (void) const;
            uint            ByteSize     (void) const;
      const CAsiUcStr   &   CollName     (void) const;      
      const CAsiUcStr   &   CharSetName  (void) const;
            AsiUcIdCLT      CollIdent    (void) const;
            AsiUcIdCCT      CharSetIdent (void) const;
            EAsiCAttr       Coercibility (void) const;

    //  is_ functions
    //
    virtual EAsiBoolean   is_character  (void) const;
    virtual EAsiBoolean   is_comparable (const CAsiData *) const;
    virtual EAsiBoolean   is_storable   (const CAsiData *) const;
    virtual EAsiBoolean   is_castable   (const CAsiData *) const;
            EAsiBoolean   is_national   (void) const;

    virtual CAsiData *    checkCat      (const CAsiData *) const;

    // Store Data value
    //
    virtual EAsiBoolean   storeValue (real);
    virtual EAsiBoolean   storeValue (integer);
    virtual EAsiBoolean   storeValue (smallint);
    virtual EAsiBoolean   storeValue (char *, uint);
            EAsiBoolean   storeValue (char *, uint, AsiUcIdCCT);
    virtual EAsiBoolean   storeValue (const CAsiUcStr &);
    virtual EAsiBoolean   storeValue (const CAsiData &);
    virtual EAsiBoolean   storeValue (const AsiUcStr);

    //  Get data value
    //
        const CAsiUcStr  &charValue (void) const;   // Reference to string

    virtual EAsiBoolean   getValue (real *) const;
    virtual EAsiBoolean   getValue (integer *) const;
    virtual EAsiBoolean   getValue (smallint *) const;
    virtual EAsiBoolean   getValue (char *, uint) const;
    virtual EAsiBoolean   getValue (CAsiUcStr *) const;

    virtual EAsiBoolean   SQLType    (CAsiUcStr *) const;        // SQL data type as string
    virtual EAsiBoolean   SQLLiteral (CAsiUcStr *) const;        // SQL compatible literal

    // ASI application/DBMS driver data exchange
    //
    virtual long          read           (char *, long, long) const;
    virtual long          size           (void) const;

    // Runtime class iteraction functions
    //
    virtual unsigned long packSize   (void) const;
    virtual unsigned long packTo     (void*) const;
    virtual EAsiBoolean   unpackFrom (void*);

    // Duplicate itself
    //
    virtual CAsiObject   *duplicate (void) const;

    // Nullify all members

    virtual void          Clear     (void);

    // Like predicate and string functions
    //
    EAsiBoolean Like       (const CAsiChar &, const CAsiChar &) const;
    CAsiChar    Substring  (uint) const;
    CAsiChar    Substring  (uint, uint) const;
    CAsiChar    Upper      (void) const;
    CAsiChar    Lower      (void) const;
    CAsiChar    Trim       (EAsiTrimSpec, const CAsiChar &) const;
    CAsiNumeric Position   (const CAsiChar &) const;
   

    // Operators
    //
            // Assignment
            CAsiChar  & operator=  (const CAsiChar &);
            CAsiChar  & operator=  (const CAsiUcStr &);

            // Cancatenation
            CAsiChar    operator+  (const CAsiChar &) const;
            CAsiChar    operator+  (const CAsiUcStr &) const;

            // Comparision
            int         operator>  (const CAsiChar &) const;
            int         operator>  (const CAsiUcStr &) const;
     friend int         operator>  (const CAsiUcStr &, const CAsiChar &);

            int         operator>=  (const CAsiChar &) const;
            int         operator>=  (const CAsiUcStr &) const;
     friend int         operator>=  (const CAsiUcStr &, const CAsiChar &);

            int         operator==  (const CAsiChar &) const;
            int         operator==  (const CAsiUcStr &) const;
     friend int         operator==  (const CAsiUcStr &, const CAsiChar &);

            int         operator<=  (const CAsiChar &) const;
            int         operator<=  (const CAsiUcStr &) const;
     friend int         operator<=  (const CAsiUcStr &, const CAsiChar &);

            int         operator<  (const CAsiChar &) const;
            int         operator<  (const CAsiUcStr &) const;
     friend int         operator<  (const CAsiUcStr &, const CAsiChar &);

            int         operator!=  (const CAsiChar &) const;
            int         operator!=  (const CAsiUcStr &) const;
     friend int         operator!=  (const CAsiUcStr &, const CAsiChar &);

private:

    EAsiBoolean     isCompColSet(const AsiUcIdCLT, EAsiCAttr) const;
    AsiUcIdCLT      CompColSet(const AsiUcIdCLT, EAsiCAttr) const;
    EAsiBoolean     isDyadicOpColSet(const AsiUcIdCLT, EAsiCAttr) const;
    AsiUcIdCLT      DyadicOpColSet(const AsiUcIdCLT, EAsiCAttr *, EAsiCAttr) const;

private:

    uint      maxlength; // Length of character data
    uint      retlength; // Length of the stored data
    CAsiUcStr chvalue;   // Value

    short           mod;         // Type modifiers
    AsiUcIdCCT      CharSetId;   // Character set identifier
    CAsiUcStr       CharSet;     // Character set name
    AsiUcIdCLT      ColId;       // Collation identifier
    CAsiUcStr       CName;       // Collation name
    EAsiCAttr       coercibility;// Coercibility attribute
};


////////////////////////////////////////////////
//
//
class CAsiBinary: public CAsiData
//
// BINARY
//
// This class introduces functionality of the internal binary data. 
// It defines assignment and binary concatenation operations which are
// associated with = and + operators.
//
//
{
friend class CAsiTempTable;
public:
    // Construction/Destruction
    //
            CAsiBinary  ();
            CAsiBinary  (uint);
            CAsiBinary  (const CAsiBinary &);
    virtual ~CAsiBinary ();


    // Set Varying
           void          setVarying    (void);

    // Get data descriptor
    //
    virtual EAsiDataType Type         (void) const;
    virtual uint         Length       (void) const; // Binary length
    virtual uint         ReturnLength (void) const;


    //  is_ functions
    //
    virtual EAsiBoolean   is_binary     (void) const;
    virtual EAsiBoolean   is_comparable (const CAsiData *) const;
    virtual EAsiBoolean   is_storable   (const CAsiData *) const;
    virtual EAsiBoolean   is_castable   (const CAsiData *) const;

    virtual CAsiData *    checkCat (const CAsiData *) const;

    // Store Data value
    //
            EAsiBoolean   storeHexValue (char *, uint, uint *);
            EAsiBoolean   storeHexValue (const CAsiUcStr &);
            EAsiBoolean   storeBinValue (char *, uint, uint *);
            EAsiBoolean   storeBinValue (const CAsiUcStr &);
    virtual EAsiBoolean   storeValue    (char *, uint);
    virtual EAsiBoolean   storeValue    (char *, uint, uint *);
    virtual EAsiBoolean   storeValue    (const CAsiData &);

    //  Get data value
    //
            EAsiBoolean   getHexValue (char *, uint, uint *) const;
            EAsiBoolean   getHexValue (CAsiUcStr *) const;
            EAsiBoolean   getBinValue (char *, uint, uint *) const;
            EAsiBoolean   getBinValue (CAsiUcStr *) const;
    virtual EAsiBoolean   getValue    (char *, uint) const;
    virtual EAsiBoolean   getValue    (char *, uint, uint *) const;

    virtual EAsiBoolean   SQLType    (CAsiUcStr *) const;        // SQL data type as string
    virtual EAsiBoolean   SQLLiteral (CAsiUcStr *) const;        // SQL compatible literal

    // ASI application/DBMS driver data exchange
    //
    virtual long          read           (char *, long, long) const;
    virtual long          size           (void) const;

    // Runtime class iteraction functions
    //
    virtual unsigned long packSize   (void) const;
    virtual unsigned long packTo     (void*) const;
    virtual EAsiBoolean   unpackFrom (void*);

    // Duplicate itself
    //
    virtual CAsiObject   *duplicate(void) const;

    // Nullify all members

    virtual void          Clear     (void);

    // Utility 
    //
    CAsiBinary Substring(uint);
    CAsiBinary Substring(uint,uint);

    // Operators
    //
            // Assignment
            CAsiBinary   &operator= (const CAsiBinary &);

            // Cancatenation
            CAsiBinary  operator+  (const CAsiBinary &) const;

            // Comparision
            int         operator>  (const CAsiBinary &) const;
            int         operator>= (const CAsiBinary &) const;
            int         operator== (const CAsiBinary &) const;
            int         operator<= (const CAsiBinary &) const;
            int         operator<  (const CAsiBinary &) const;
            int         operator!= (const CAsiBinary &) const;


private:
    uint       blength;   // Length of binary data
    uint       alength;   // Actual length
    uint       buflen;    // Allocated length
    uchar    * value;     // Value
    short      mod;       // Modifier
};


/////////////////////////////////////////////////////////////////////////////
class CAsiIQualifier : public CAsiData
//
// Interval Qualifier
//
{
friend class CAsiInterval;
friend class CAsiDatetime;

public:
    // Construction/Destruction
    //
               CAsiIQualifier (void);        
               CAsiIQualifier (EAsiIType);        
               CAsiIQualifier (EAsiIType, uint);
               CAsiIQualifier (EAsiIType, uint, uint);
               CAsiIQualifier (const CAsiIQualifier &);
    virtual   ~CAsiIQualifier (void);

    // Set Interval qualifier type and characteristics
    //
    virtual EAsiBoolean  setIType         (EAsiIType);    // Set interval type
            void         setLeadFieldPrec (uint);         // Set Leading field precision
            void         setFracSecPrec   (uint);         // Set Fractional sec prec

    // Get access to interval characteristics
    //
    virtual EAsiDataType Type          (void) const;
    virtual EAsiIType    IntervalType  (void) const;
    virtual uint         LeadFieldPrec (void) const;
    virtual uint         FracSecPrec   (void) const;
    virtual uint         ReturnLength  (void) const;

    // ASI application/DBMS driver data exchange
    //
    virtual long          read           (char *, long, long) const;
    virtual long          size           (void) const;

    // Return interval type
    virtual EAsiBoolean  is_interval     (void) const;
    virtual EAsiBoolean  is_yminterval   (void) const;
    virtual EAsiBoolean  is_dtinterval   (void) const;
    virtual EAsiBoolean  is_storable     (const CAsiData *) const;
    virtual EAsiBoolean  is_comparable   (const CAsiData *) const;
            EAsiBoolean  is_moreaccurate (const CAsiIQualifier *) const;

    virtual EAsiBoolean  SQLType       (CAsiUcStr *) const;        // SQL data type as string

private:
    virtual EAsiBoolean  SQLLiteral    (CAsiUcStr *) const;    // SQL compatible literal

public:
    // Assignment operator
            CAsiIQualifier &operator= (const CAsiIQualifier &);

    // Runtime class iteraction functions
    //
    virtual unsigned long packSize   (void) const;
    virtual unsigned long packTo     (void*) const;
    virtual EAsiBoolean   unpackFrom (void*);

    // Duplicate itself
    //
    virtual CAsiObject   *duplicate(void) const;

protected:
    EAsiIType itype;                 // Interval type
    uint lfp;                        // Leading field precision
    uint fsp;                        // Fractional second precision
};



////////////////////////////////////////////////
//
//
class CAsiInterval : public CAsiIQualifier
//
// INTERVAL
//
//  This is a base class for datetime intervals. The basic functionality 
//  for class includes setting and retrieving interval sign.

{
friend class CAsiDate;
friend class CAsiTime;
friend class CAsiTimeStamp;

public:
    // Constructors/Destructors
    //
            CAsiInterval ();
    virtual ~CAsiInterval ();

    virtual EAsiBoolean   SQLType  (CAsiUcStr *)      const;

    virtual CAsiData *    checkAdd (const CAsiData *) const;
    virtual CAsiData *    checkSub (const CAsiData *) const;
    virtual CAsiData *    checkMul (const CAsiData *) const;
    virtual CAsiData *    checkDiv (const CAsiData *) const;

    //  Set/get interval sign
    //
	         void          setSign  (EAsiBoolean);
            EAsiBoolean   Sign     (void)             const;


protected:
    EAsiBoolean sign;     // Interval sign
};


////////////////////////////////////////////////
//
//
class CAsiYM : public CAsiInterval
//
// Year Month Intervals
//
// Constructors/Destructor
//
// CAsiYM ()
// CAsiYM (
//   EAsiIType itype)                  // Year Month Interval type
// CAsiYM (
//   EAsiIType itype,                  // Year Month Interval type
//   uint lprec                        // Leading fields precision
//   )                  
//
// Different forms of constructor allow to specify interval type, and leading
// field precision
// Interval type shall be a valid Year Month interval type.
// If invalid interval type was specified, interval will be set to kAsiIUnknown.
//
//  EAsiBoolean set (
//    EAsiBoolean sign,                // Interval sign kAsiFalse (+), kAsiTrue (-)
//    uint years,                     // Years
//    uint months)                    // Months
//
//  EAsiBoolean setYears (
//    uint years)                     // Years
//
//  EAsiBoolean setMonths (
//    uint months)                    // Months
//
//  Interval fields set functions allow to set interval. They return
//  boolean value, because they must validate input.
//
//
//  Get interval fields
//
//  void get (
//    EAsiBoolean *sign,               // Interval sign kAsiFalse (+), kAsiTrue (-)
//    uint *years,                    // Years
//    uint *months) const             // Months
//
//  EAsiBoolean getYears (
//    uint *years) const              // Years
//
//  EAsiBoolean getMonths (
//    uint *months) const             // Months
//
//  The functions above retrieve interval fields. Return kAsiTrue if 
//  specified interval field present in interval and kAsiFalse otherwise.
//
//
//  Operators
//
//  Operators overload interval arithmetics operations like +, -, /, *, and 
//  unary minus. Besides them all boolean operators are redefined.
//
//

{
		
friend class CAsiDate;
friend class CAsiTime;
friend class CAsiTimeStamp;
friend class CAsiTempTable;

public:
    // Construction/Destruction
    //
            CAsiYM (void);
            CAsiYM (EAsiIType);
            CAsiYM (EAsiIType, uint);
            CAsiYM (const CAsiYM &);
    virtual ~CAsiYM (void);

    virtual EAsiBoolean   setIType      (EAsiIType);    // Set interval type

    virtual EAsiBoolean   is_castable   (const CAsiData *) const;

    // Store Data value
    //
    virtual EAsiBoolean   storeValue (integer);
    virtual EAsiBoolean   storeValue (smallint);
    virtual EAsiBoolean   storeValue (char *, uint);
    virtual EAsiBoolean   storeValue (const CAsiUcStr &);
    virtual EAsiBoolean   storeValue (const CAsiData &);

    // Set data value
    //
       EAsiBoolean   set       (ASI_IYEAR *);
       EAsiBoolean   set       (ASI_IYEARTOMONTH *);
       EAsiBoolean   set       (ASI_IMONTH *);
	    EAsiBoolean   set       (EAsiBoolean, uint, uint);
	    EAsiBoolean   setYears  (uint);
	    EAsiBoolean   setMonths (uint);

    //  Get data value in the Unicode character string format
    //
    virtual EAsiBoolean   getValue (integer *) const;
    virtual EAsiBoolean   getValue (smallint *) const;
    virtual EAsiBoolean   getValue (char *, uint) const;
    virtual EAsiBoolean   getValue (CAsiUcStr *) const;

    virtual EAsiBoolean   SQLLiteral (CAsiUcStr *) const;        // SQL compatible literal

       EAsiBoolean   get       (ASI_IYEAR *)        const;
       EAsiBoolean   get       (ASI_IYEARTOMONTH *) const;
       EAsiBoolean   get       (ASI_IMONTH *)       const;
	    EAsiBoolean   get       (EAsiBoolean *, uint *, uint *) const;
	    EAsiBoolean   getYears  (uint *) const;
	    EAsiBoolean   getMonths (uint *) const;

    // ASI application/DBMS driver data exchange
    //
    virtual long          read           (char *, long, long) const;
    virtual long          size           (void) const;

    // Runtime class iteraction functions
    //
    virtual unsigned long packSize   (void) const;
    virtual unsigned long packTo     (void*) const;
    virtual EAsiBoolean   unpackFrom (void*);

    // Duplicate itself
    //
    virtual CAsiObject   *duplicate (void) const;

    // Operators
	    CAsiYM       &operator=  (const CAsiYM &);
	    CAsiYM       &operator=  (const CAsiSmallInt &);
	    CAsiYM       &operator=  (const CAsiInt &);
	    CAsiYM       &operator=  (const CAsiNumeric &);
	    CAsiYM       &operator=  (const CAsiDecimal &);
	    CAsiYM       &operator=  (integer);

	    CAsiYM       operator+  (const CAsiYM &) const;
friend CAsiDate     operator+  (const CAsiYM &, const CAsiDate &);
	    CAsiYM       operator-  (const CAsiYM &) const;

       CAsiYM       operator*  (const CAsiSmallInt &) const;
	    CAsiYM       operator*  (const CAsiInt &) const;
	    CAsiYM       operator*  (const CAsiNumeric &) const;
	    CAsiYM       operator*  (const CAsiDecimal &) const;
	    CAsiYM       operator*  (integer) const;
friend CAsiYM       operator*  (const CAsiSmallInt &,const CAsiYM &);
friend CAsiYM       operator*  (const CAsiInt &,     const CAsiYM &);
friend CAsiYM       operator*  (const CAsiNumeric &, const CAsiYM &);
friend CAsiYM       operator*  (const CAsiDecimal &, const CAsiYM &);
friend CAsiYM       operator*  (integer,             const CAsiYM &);

	    CAsiYM       operator/  (const CAsiSmallInt &) const;
	    CAsiYM       operator/  (const CAsiInt &)      const;
	    CAsiYM       operator/  (const CAsiNumeric &)  const;
	    CAsiYM       operator/  (const CAsiDecimal &)  const;
	    CAsiYM       operator/  (integer)              const;

	    CAsiYM       &operator-  (void);

    // Comparision predicates
    //
       int           operator>  (const CAsiYM &) const;
	    int           operator>= (const CAsiYM &) const;
	    int           operator== (const CAsiYM &) const;
	    int           operator<= (const CAsiYM &) const;
	    int           operator<  (const CAsiYM &) const;
	    int           operator!= (const CAsiYM &) const;
private:
       char *        getYMString (char *) const;

protected:
    ulong ivalue;         // Interval value in months

};


////////////////////////////////////////////////
//
//
class CAsiDT : public CAsiInterval
//
// Day Time Intervals
//
// This class introduces functionality of the Day Time intervals.
// 
// MEMBER FUNCTIONS
// ----------------
// Constructors/Destructor
//
// CAsiDT ()
// CAsiDT (
//   EAsiIType itype                   // Interval type
//  )
// CAsiDT (
//   EAsiIType itype,                  // Interval type
//   uint lprec                        // Leading fields precision
//  )
// CAsiDT (
//   EAsiIType itype                   // Interval type
//   uint lprec,                       // Leading fields precision
//   uint fprec                        // Fractional second precision
//  )

// Constructors allow to set interval type, leading field precision and fractional
// seconds precision
// Interval type shall specify valid day time interval type.
// If invalid interval type was specified, interval will be set to kAsiIUnknown.
//
//
//  EAsiBoolean set (
//    EAsiBoolean sign,                 // Interval sign kAsiFalse (+), kAsiTrue (-)
//    uint days,                        // Days
//    uint hours,                       // Hours
//    uint minutes,                     // Minutes
//    double seconds)                   // Seconds
//
//  EAsiBoolean setDays (
//    uint days)                        // Days
//
//  EAsiBoolean setHours (
//    uint hours)                       // Hours
//
//  EAsiBoolean setMinutes (
//    uint minutes)                     // Minutes
//
//  EAsiBoolean getSeconds (
//    double seconds)                   // Seconds
//
//  Interval fields set functions allow to set interval. They return
//  boolean value, because they must validate input.
//
//
//  Get interval fields
//
//  void get (
//    EAsiBoolean *sign,               // Interval sign kAsiFalse (+), kAsiTrue (-)
//    uint *days,                       // Days
//    uint *hours,                      // Hours
//    uint *minutes,                    // Minutes
//    double *seconds) const            // Seconds with fraction
//
//  EAsiBoolean getDays (
//    uint *days                       // Days
//   ) const
//
//  EAsiBoolean getHours (
//    uint *hours                      // Hours
//   ) const;
//
//  EAsiBoolean getMinutes (
//     uint *minutes                    // Minutes
//   ) const;
//
//  EAsiBoolean getSeconds (
//    double *seconds) const;           // Seconds with fraction
//
//  The functions above retrieve interval fields
//
//
//  Operators
//
//  Operators overload interval arithmetics operations like +, -, /, *, and 
//  unary minus. Besides them all boolean operators are redefined.
//
//

{
    // Friend classes
friend class CAsiDate;
friend class CAsiTime;
friend class CAsiTimeStamp;
friend class CAsiTempTable;

private:
    // private helpers

                void        ChkMul     (real);
                void        ChkDiv     (real);
                int         cmpDTI     (const CAsiDT&) const;
                EAsiBoolean ChkSet     (double);
                void        ChkSet2    (double);
                double      LocalToUTC (double) const;
                double      UTCToLocal (double) const;

public:

    // Constructors/Destructors
    //
            CAsiDT (void);
            CAsiDT (EAsiIType);
            CAsiDT (EAsiIType, uint);
            CAsiDT (EAsiIType, uint, uint);
            CAsiDT (const CAsiDT &);
    virtual ~CAsiDT (void);

    virtual EAsiBoolean   setIType      (EAsiIType);    // Set interval type

    virtual EAsiBoolean   is_castable   (const CAsiData *) const;

    // Store Data value
    //
    virtual EAsiBoolean   storeValue (integer);
    virtual EAsiBoolean   storeValue (smallint);
    virtual EAsiBoolean   storeValue (char *, uint);
    virtual EAsiBoolean   storeValue (const CAsiUcStr &);
    virtual EAsiBoolean   storeValue (const CAsiData &);

    // Set data value
    //
            EAsiBoolean   set        (ASI_IDAY *);
            EAsiBoolean   set        (ASI_IHOUR *);
            EAsiBoolean   set        (ASI_IMINUTE *);
            EAsiBoolean   set        (ASI_ISECOND *);
            EAsiBoolean   set        (ASI_IDAYTOHOUR *);
            EAsiBoolean   set        (ASI_IDAYTOMINUTE *);
            EAsiBoolean   set        (ASI_IDAYTOSECOND *);
            EAsiBoolean   set        (ASI_IHOURTOMINUTE *);
            EAsiBoolean   set        (ASI_IHOURTOSECOND *);
            EAsiBoolean   set        (ASI_IMINUTETOSECOND *);
            EAsiBoolean   set        (EAsiBoolean, uint, uint, uint, double);
            EAsiBoolean   setDays    (uint);
            EAsiBoolean   setHours   (uint);
            EAsiBoolean   setMinutes (uint);
            EAsiBoolean   setSeconds (double);

    //  Get data value in the Unicode character string format
    //
    virtual EAsiBoolean   getValue   (integer *)      const;
    virtual EAsiBoolean   getValue   (smallint *)     const;
    virtual EAsiBoolean   getValue   (char *, uint)   const;
    virtual EAsiBoolean   getValue   (CAsiUcStr *)    const;

    virtual EAsiBoolean   SQLLiteral (CAsiUcStr *) const;        // SQL compatible literal

            EAsiBoolean   get        (ASI_IDAY *)            const;
            EAsiBoolean   get        (ASI_IHOUR *)           const;
            EAsiBoolean   get        (ASI_IMINUTE *)         const;
            EAsiBoolean   get        (ASI_ISECOND *)         const;
            EAsiBoolean   get        (ASI_IDAYTOHOUR *)      const;
            EAsiBoolean   get        (ASI_IDAYTOMINUTE *)    const;
            EAsiBoolean   get        (ASI_IDAYTOSECOND *)    const;
            EAsiBoolean   get        (ASI_IHOURTOMINUTE *)   const;
            EAsiBoolean   get        (ASI_IHOURTOSECOND *)   const;
            EAsiBoolean   get        (ASI_IMINUTETOSECOND *) const;
            EAsiBoolean   get        (EAsiBoolean *, uint *, uint *, uint *,
                                      double *) const;
            EAsiBoolean   getDays    (uint *) const;
            EAsiBoolean   getHours   (uint *) const;
            EAsiBoolean   getMinutes (uint *) const;
            EAsiBoolean   getSeconds (double *) const;

    // ASI application/DBMS driver data exchange
    //
    virtual long          read           (char *, long, long) const;
    virtual long          size           (void) const;

    // Runtime class iteraction functions
    //
    virtual unsigned long packSize   (void) const;
    virtual unsigned long packTo     (void*) const;
    virtual EAsiBoolean   unpackFrom (void*);

    // Duplicate itself
    //
    virtual CAsiObject   *duplicate (void) const;

    // Operators
            CAsiDT       &operator=  (const CAsiDT &);

            CAsiDT       &operator=  (const CAsiSmallInt &);
            CAsiDT       &operator=  (const CAsiInt &);
            CAsiDT       &operator=  (const CAsiNumeric &);
            CAsiDT       &operator=  (const CAsiDecimal &);
            CAsiDT       &operator=  (integer);

            CAsiDT       operator+  (const CAsiDT &) const;
            CAsiDT       operator-  (const CAsiDT &) const;
            CAsiDT       operator*  (const CAsiSmallInt &) const;
            CAsiDT       operator*  (const CAsiInt &)      const;
            CAsiDT       operator*  (const CAsiNumeric &)  const;
            CAsiDT       operator*  (const CAsiDecimal &)  const;
            CAsiDT       operator*  (integer)              const;
     friend CAsiDT       operator*  (const CAsiSmallInt &,const CAsiDT &);
     friend CAsiDT       operator*  (const CAsiInt &,     const CAsiDT &);
     friend CAsiDT       operator*  (const CAsiNumeric &, const CAsiDT &);
     friend CAsiDT       operator*  (const CAsiDecimal &, const CAsiDT &);
     friend CAsiDT       operator*  (integer,             const CAsiDT &);

            CAsiDT       operator/  (const CAsiSmallInt &) const;
            CAsiDT       operator/  (const CAsiInt &)      const;
            CAsiDT       operator/  (const CAsiNumeric &)  const;
            CAsiDT       operator/  (const CAsiDecimal &)  const;
            CAsiDT       operator/  (integer)              const;

            CAsiDT       &operator-  (void);

    // Comparision predicates
    //
            int           operator>  (const CAsiDT &) const;
            int           operator>= (const CAsiDT &) const;
            int           operator== (const CAsiDT &) const;
            int           operator<= (const CAsiDT &) const;
            int           operator<  (const CAsiDT &) const;
            int           operator!= (const CAsiDT &) const;

private:
            char        * getDTString (char *) const;


protected:
    double ivalue;        // Interval value in seconds with fraction

};


////////////////////////////////////////////////
//
//
class CAsiDatetime : public CAsiData
//
//  Datetime 
//
{
public:
    // Construction/Destruction
    //
            CAsiDatetime ();
    virtual ~CAsiDatetime ();

    // Get data descriptor
    //
    virtual EAsiDataType Type          (void) const;

    //  is_ functions
    //
    virtual EAsiBoolean   is_datetime   (void) const;
    virtual EAsiBoolean   is_comparable (const CAsiData *) const;

    virtual CAsiData *    checkAdd (const CAsiData *) const;
    virtual CAsiData *    checkSub (const CAsiData *) const;

    //  Get datetime field value
    //
    virtual ushort Year      (void) const = 0;
    virtual ushort Month     (void) const = 0;
    virtual ushort Day       (void) const = 0;
    virtual ushort Hour      (void) const = 0;
    virtual ushort Minute    (void) const = 0;
    virtual double Second    (void) const = 0;

protected:

};


////////////////////////////////////////////////
//
//
class CAsiDate: public CAsiDatetime
//
// DATE
//
{
friend class CAsiTimeStamp;
friend class CAsiTempTable;

public:

    // Construction/Destruction
    //
            CAsiDate ();
            CAsiDate (const CAsiDate &);
    virtual ~CAsiDate (void);

    // Get data descriptor
    //
    virtual EAsiDTType   DateTimeType  (void) const;
    virtual uint         ReturnLength  (void) const;

    // Check storable
    virtual EAsiBoolean  is_storable   (const CAsiData *) const;
    virtual EAsiBoolean  is_castable   (const CAsiData *) const;

    // Store Data value
    //
    virtual EAsiBoolean  storeValue (char *, uint);
    virtual EAsiBoolean  storeValue (const CAsiUcStr &);
    virtual EAsiBoolean  storeValue (const CAsiData &);

    // Set date fields
    //
            EAsiBoolean  set      (ASI_DTDATE *);
            EAsiBoolean  set      (ushort, ushort, ushort);
            EAsiBoolean  setYear  (ushort);
            EAsiBoolean  setMonth (ushort);
            EAsiBoolean  setDay   (ushort);

    //  Get data value
    //
            EAsiBoolean  get      (ASI_DTDATE *)  const;
    virtual EAsiBoolean  getValue (char *, uint)  const;
    virtual EAsiBoolean  getValue (CAsiUcStr *)   const;

    virtual EAsiBoolean  SQLType    (CAsiUcStr *) const;        // SQL data type as string
    virtual EAsiBoolean  SQLLiteral (CAsiUcStr *) const;        // SQL compatible literal

    virtual ushort       Year      (void) const;
    virtual ushort       Month     (void) const;
    virtual ushort       Day       (void) const;
    virtual ushort       Hour      (void) const;
    virtual ushort       Minute    (void) const;
    virtual double       Second    (void) const;

    // ASI application/DBMS driver data exchange
    //
    virtual long          read           (char *, long, long) const;
    virtual long          size           (void) const;

    // Runtime class iteraction functions
    //
    virtual unsigned long packSize   (void) const;
    virtual unsigned long packTo     (void*) const;
    virtual EAsiBoolean   unpackFrom (void*);

    // Duplicate itself
    //
    virtual CAsiObject   *duplicate (void) const;

    // Operators
    //
     // Assignments
            CAsiDate    &operator=     (const CAsiDate &);

     // Arithmatics
            CAsiDate     operator+     (const CAsiYM &) const;
            CAsiDate     operator+     (const CAsiDT &) const;
     friend CAsiDate     operator+     (const CAsiYM &, const CAsiDate &);
     friend CAsiDate     operator+     (const CAsiDT &, const CAsiDate &);

            CAsiDate     operator-     (const CAsiYM &) const;
            CAsiDate     operator-     (const CAsiDT &) const;

            //   Dates difference , result corresponds with EAsiType value

            CAsiYM YMDiff (const CAsiDate &, const CAsiYM &)   const;
            CAsiDT DTDiff (const CAsiDate &, const CAsiDT &)   const;

     // Comparision
            int          operator>     (const CAsiDate &)      const;
            int          operator>=    (const CAsiDate &)      const;
            int          operator==    (const CAsiDate &)      const;
            int          operator<=    (const CAsiDate &)      const;
            int          operator<     (const CAsiDate &)      const;
            int          operator!=    (const CAsiDate &)      const;


protected:
    ulong month;      // Months since 00-00-00
    ushort day;       // Day of the month

};

///////////////////////////////////////////
//
//
class CAsiTime : public CAsiDatetime
//
// TIME
//
{
friend class CAsiTimeStamp;
friend class CAsiTempTable;

public:

    // Construction/Destruction
    //
            CAsiTime  (void);
            CAsiTime  (const CAsiDT&);
            CAsiTime  (uint);
            CAsiTime  (uint, const CAsiDT&);
            CAsiTime  (const CAsiTime &);
    virtual ~CAsiTime (void);


            void         setWithTimeZone (EAsiBoolean f = kAsiTrue );

    // Get data descriptor
    //
    virtual EAsiDTType   DateTimeType  (void) const;
    virtual uint         TimePrec      (void) const;
    virtual uint         ReturnLength  (void) const;

    // Check storable
    virtual EAsiBoolean  is_storable   (const CAsiData *) const;
    virtual EAsiBoolean  is_castable   (const CAsiData *) const;

    // Store Data value
    //
    virtual EAsiBoolean  storeValue   (char *, uint);
    virtual EAsiBoolean  storeValue   (const CAsiUcStr &);
    virtual EAsiBoolean  storeValue   (const CAsiData &);


    // Set date value
    //
            EAsiBoolean  set           (ASI_DTTIME *);
            EAsiBoolean  set           (ushort, ushort, double);
            EAsiBoolean  setHour       (ushort);
            EAsiBoolean  setMinute     (ushort);
            EAsiBoolean  setSecond     (double);
            EAsiBoolean  setTimeZone   (const CAsiDT &);

    //  Get data value
    //
            EAsiBoolean  get          (ASI_DTTIME *) const;
    virtual EAsiBoolean  getValue     (char *, uint) const;
    virtual EAsiBoolean  getValue     (CAsiUcStr *)  const;

    virtual EAsiBoolean  SQLType      (CAsiUcStr *) const;        // SQL data type as string
    virtual EAsiBoolean  SQLLiteral   (CAsiUcStr *) const;        // SQL compatible literal

    virtual ushort       Year         (void) const {return 0;};
    virtual ushort       Month        (void) const {return 0;};
    virtual ushort       Day          (void) const {return 0;};
    virtual ushort       Hour         (void) const;
    virtual ushort       Minute       (void) const;
    virtual double       Second       (void) const;
      const CAsiDT      &TimeZone     (void) const;

    // ASI application/DBMS driver data exchange
    //
    virtual long          read           (char *, long, long) const;
    virtual long          size           (void) const;

    // Runtime class iteraction functions
    //
    virtual unsigned long packSize   (void) const;
    virtual unsigned long packTo     (void*) const;
    virtual EAsiBoolean   unpackFrom (void*);

    // Duplicate itself
    //
    virtual CAsiObject  *duplicate (void) const;

    // Operators
    //
            // Assignments
            CAsiTime    &operator=     (const CAsiTime &);

            // Arithmatics
            CAsiTime     operator+     (const CAsiDT &) const;
     friend CAsiTime     operator+     (const CAsiDT &, const CAsiTime &);
            CAsiTime     operator-     (const CAsiDT &) const;

            // Difference
            CAsiDT       DTDiff        (const CAsiTime &,const CAsiDT &) const;

            // Comparision
            int          operator>     (const CAsiTime &) const;
            int          operator>=    (const CAsiTime &) const;
            int          operator==    (const CAsiTime &) const;
            int          operator<=    (const CAsiTime &) const;
            int          operator<     (const CAsiTime &) const;
            int          operator!=    (const CAsiTime &) const;


private:

    ushort timeprec;   // Time precision
    double seconds;    // Seconds since 0:0 AM with fraction
    short  mod;        // Modifiyer
    CAsiDT timezone;
};

///////////////////////////////////////////
//
//
class CAsiTimeStamp: public CAsiDatetime
//
// TIMESTAMP
//

// This class introduces functionality of the TIMESTAMP SQL data type
//
// MEMBER FUNCTIONS
// ----------------
// Constructors/Destructor
//
//  CAsiTimeStamp ()
//  CAsiTimeStamp (
//    uint tprec)           // Timestamp precision
//
// Two forms of constructor are provided. The second form allows to define 
// timestamp precision. If the first form of constructor
// is used, timestamp precision is set to 0.
//
//
// Set Timestamp fields
//   
// EAsiBoolean set (
//    ushort year,             // Year
//    ushort month,            // Month
//    ushort day,              // Day
//    ushort hour,             // Hour
//    ushort minute,           // Minute
//    double second)           // Second with fraction
//
// EAsiBoolean setYear (
//    ushort year)             // Year
//
// EAsiBoolean setMonth (
//    ushort month)            // Month
//
// EAsiBoolean setDay (
//    ushort day)              // Day
//
// EAsiBoolean setHour (
//    ushort hour)             // Hour
//
// EAsiBoolean setMinute (
//    ushort minute)           // Minute
//
// EAsiBoolean setSecond (
//    double second)           // Second with fraction
//
// These fiunctions are used to set datetime type and datetime fields.
//
// Year shall be greater or equal than 0;
// Month >= 1 and <= 12;
// Day >= 1 and <= actual number of days for each month;
// Hour >= 0 and <= 23
// Minute >= 0 and <= 59
// Second >= 0 and <= 59
// Second fraction is an unsigned short number that holds second fraction.
//
// Most of functions return boolean value bacause they must validate
// input arguments. For example, seconds can't be stored to the DATE or 
// month can be greater than 12. 
// 
//
// void get (
//    ushort *year,             // Year
//    ushort *month,            // Month
//    ushort *day,              // Day
//    ushort *hour,             // Hour
//    ushort *minute,           // Minute
//    double *second)           // Second with fraction
//
// ushort  Year ()
//
// ushort  Month ()
//
// ushort  Day ()
//
// ushort  Hour ()
//
// ushort  Minute ()
//
// double  Second ()
//
// The functions above extracts all or specified datetime fields.
//
//
// Operators
//
// All meaningful arithmetic and comparision operators are redefined by 
// this class. Description of them can be found in SQL/ISO standard (6.14). 
// Besides them special datetime operations are defined for the datetime 
// data.
// 
// CAsiIDateTime & at_time_zone (
//   CAsiDT &tz)         // Time zone value
//
// This function translate TIME or TIMESTAMP datetime according to the 
// time zone value specified as a internal hour or hour to minute day time 
// interval (see 6.14 SQL/ISO). Interval value shall be between -'12:59 and 
// +'13:00'. If wrong interval type was passed or 

{
friend class CAsiDate;
friend class CAsiTime;
friend class CAsiTempTable;
public:

    // Construction/Destruction
    //
            CAsiTimeStamp ();
            CAsiTimeStamp (uint);
            CAsiTimeStamp (const CAsiDT &);
            CAsiTimeStamp (uint, const CAsiDT &);
            CAsiTimeStamp (const CAsiTimeStamp &);
    virtual ~CAsiTimeStamp (void);

            void         setWithTimeZone (EAsiBoolean f = kAsiTrue );

    // Get data descriptor
    //
    virtual EAsiDTType   DateTimeType  (void) const;
    virtual uint         TimeStampPrec (void) const;
    virtual uint         ReturnLength  (void) const;

    // Check storable
    virtual EAsiBoolean  is_storable   (const CAsiData *) const;
    virtual EAsiBoolean  is_castable   (const CAsiData *) const;

    // Store Data value
    //
    virtual EAsiBoolean  storeValue  (char *, uint);
    virtual EAsiBoolean  storeValue  (const CAsiUcStr &);
    virtual EAsiBoolean  storeValue  (const CAsiData &);


    // Set timestamp fields
    //
            EAsiBoolean  set         (ASI_DTTIMESTAMP *);
            EAsiBoolean  set         (const CAsiDate &, const CAsiTime &);
            EAsiBoolean  set         (ushort, ushort, ushort, ushort,
                                      ushort, double);
            EAsiBoolean  setYear     (ushort);
            EAsiBoolean  setMonth    (ushort);
            EAsiBoolean  setDay      (ushort);
            EAsiBoolean  setHour     (ushort);
            EAsiBoolean  setMinute   (ushort);
            EAsiBoolean  setSecond   (double);
            EAsiBoolean  setTimeZone (const CAsiDT &);

    //  Get data value
    //
            EAsiBoolean  get           (ASI_DTTIMESTAMP *) const;
    virtual EAsiBoolean  getValue      (char *, uint)      const;
    virtual EAsiBoolean  getValue      (CAsiUcStr *)       const;

    virtual EAsiBoolean  SQLType       (CAsiUcStr *) const;        // SQL data type as string
    virtual EAsiBoolean  SQLLiteral    (CAsiUcStr *) const;        // SQL compatible literal

    virtual ushort       Year          (void) const;
    virtual ushort       Month         (void) const;
    virtual ushort       Day           (void) const;
    virtual ushort       Hour          (void) const;
    virtual ushort       Minute        (void) const;
    virtual double       Second        (void) const;
      const CAsiDT      &TimeZone      (void) const;

    // ASI application/DBMS driver data exchange
    //
    virtual long          read           (char *, long, long) const;
    virtual long          size           (void) const;

    // Runtime class iteraction functions
    //
    virtual unsigned long packSize   (void) const;
    virtual unsigned long packTo     (void*) const;
    virtual EAsiBoolean   unpackFrom (void*);

    // Duplicate itself
    //
    virtual CAsiObject  *duplicate (void) const;

    // Operators
    //
            // Assignments
            CAsiTimeStamp    &operator=     (const CAsiTimeStamp &);
            CAsiTimeStamp    &operator=     (const CAsiDate &);
            CAsiTimeStamp    &operator=     (const CAsiTime &);

            // Arithmatics
            CAsiTimeStamp    operator+     (const CAsiYM &) const;
            CAsiTimeStamp    operator+     (const CAsiDT &) const;
     friend CAsiTimeStamp    operator+     (const CAsiYM &, CAsiTimeStamp &);
     friend CAsiTimeStamp    operator+     (const CAsiDT &, CAsiTimeStamp &);

            CAsiTimeStamp    operator-     (const CAsiYM &) const;
            CAsiTimeStamp    operator-     (const CAsiDT &) const;

            //  Difference between timestamps
            CAsiYM    YMDiff (const CAsiTimeStamp &, const CAsiYM &) const;
            CAsiDT    DTDiff (const CAsiTimeStamp &, const CAsiDT &) const;

            // Comparision
            int               operator>     (const CAsiTimeStamp &) const;
            int               operator>=    (const CAsiTimeStamp &) const;
            int               operator==    (const CAsiTimeStamp &) const;
            int               operator<=    (const CAsiTimeStamp &) const;
            int               operator<     (const CAsiTimeStamp &) const;
            int               operator!=    (const CAsiTimeStamp &) const;



private:
    CAsiTime time;
    CAsiDate date;
};


class CAsiImplDef: public CAsiData
//
// Implementation defined data
//
// This class is supposed to be a superclass for all user-defined data.
// ASI provides facility for exchanging data in the generic format between
// application and driver.
//
// MEMBER FUNCTIONS
// -----------------
//
//  void  setType (
//   int type         // Implementation type
//  )
//
// Set implementation defined type.
//
{
public:
    //  Construction destruction
    //
            CAsiImplDef  (void);
    virtual ~CAsiImplDef (void);

    //  Get data descriptor
    //
    virtual EAsiDataType  Type          (void) const;
    virtual int           ImplDataType  (void) const;

    // Set implementation data type
    //
            void          setType       (int);
            void          setName       (const CAsiUcStr &);

    //  is_ functions
    //                                
    virtual EAsiBoolean   is_implement  (void) const;


    virtual EAsiBoolean   SQLType    (CAsiUcStr *) const;
    virtual EAsiBoolean   SQLLiteral (CAsiUcStr *) const;

    // ASI application/DBMS driver data exchange
    //
    virtual long          read           (char *, long, long) const;
    virtual long          size           (void) const;

    // Store Data value
    //
    virtual EAsiBoolean   storeValue     (const CAsiData &);

    // Runtime class iteraction functions
    //
    virtual unsigned long packSize   (void) const;
    virtual unsigned long packTo     (void*) const;
    virtual EAsiBoolean   unpackFrom (void*);

    // Duplicate itself
    //
    virtual CAsiObject   *duplicate (void) const;

    // Nullify all members

    virtual void          Clear     (void);

protected:
    int impltype;          // Implementation data type
    CAsiUcStr iname;       // Implemantation defined type name

};




////////////////////////////////////////////////
//
//
class CAsiColumn : public CAsiObject
// This class introduces functionality of a table column. It holds a 
// column descriptor and column data. Column descriptor includes the 
// folowing issues, that can be set and retrieved by the CAsiColumn services:
// 
//  - Column name;
//  - Whether column name is an implementation defined;
//  - Domain, if column is based on it;
//  - Data type descriptor ;
//  - Collation name, if any;
//  - Value of the default option
//  - Nullability characteristics
//  - Ordinal position of column within table
//
//


{
public:
            CAsiColumn (void);
            CAsiColumn (const CAsiIdent &);
            CAsiColumn (const CAsiIdent &, const CAsiSmallInt &);
            CAsiColumn (const CAsiIdent &, const CAsiInt &);
            CAsiColumn (const CAsiIdent &, const CAsiNumeric &);
            CAsiColumn (const CAsiIdent &, const CAsiDecimal &);
            CAsiColumn (const CAsiIdent &, const CAsiFloat &);
            CAsiColumn (const CAsiIdent &, const CAsiReal &);
            CAsiColumn (const CAsiIdent &, const CAsiDouble &);
            CAsiColumn (const CAsiIdent &, const CAsiChar &);
            CAsiColumn (const CAsiIdent &, const CAsiBinary &);
            CAsiColumn (const CAsiIdent &, const CAsiDate &);
            CAsiColumn (const CAsiIdent &, const CAsiTime &);
            CAsiColumn (const CAsiIdent &, const CAsiTimeStamp &);
            CAsiColumn (const CAsiIdent &, const CAsiYM &);
            CAsiColumn (const CAsiIdent &, const CAsiDT &);
            CAsiColumn (const CAsiIdent &, const CAsiIdent &); // By domain name
            CAsiColumn (const CAsiIdent &, CAsiData *);        // By generic data
            CAsiColumn (const CAsiColumn &);                   // copy constructor

    virtual ~CAsiColumn ();
    // Setting Column Descriptor
    //                                             
            void          setName      (const CAsiIdent &);  // Set column name
            void          setNullable  (EAsiBoolean);        // Set nullability
            void          setNative    (EAsiBoolean);        // Set native flag
            void          setCollation (const CAsiUcStr &);  // Set collation name
            void          setDefault   (CAsiData *);         // Set Default data
            void          setDomain    (const CAsiIdent &);  // Set Domain name
            void          setDataDsc   (CAsiData *);         // Set data descriptor
            void          setOffset    (uint);               // Set Ordinal position

    // Read Column Descriptor
    //
      const CAsiIdent   & Name         (void) const;         // Get column name
            EAsiBoolean   Nullable     (void) const;         // Get nullability
            EAsiBoolean   Native       (void) const;         // Get native flag
      const CAsiUcStr   * Collation    (void) const;         // Get collation name
            CAsiData    * Default      (void) const;         // Get Default data
      const CAsiIdent   & Domain       (void) const;         // Get Domain name
            uint          Offset       (void) const;         // Get Ordinal position

    // Get Access to Column Data Descriptor and Data
    //
            CAsiData    * getData (void) const;

    // Overloaded operators
            
            CAsiColumn  & operator= (const CAsiColumn &);
            CAsiColumn  & operator= (CAsiData *);

    // Runtime class iteraction functions
    //
    virtual unsigned long packSize   (void) const;
    virtual unsigned long packTo     (void*) const;
    virtual EAsiBoolean   unpackFrom (void*);

    // Duplicate column
    //
    virtual CAsiObject *duplicate    (void) const;

    // Nullify all members

    virtual void        Clear        (void);  

protected:
    CAsiIdent    name;        // Column name
    CAsiData   * data;        // Column data descriptor and data
    
    EAsiBoolean  nullable;    // Nullability
    EAsiBoolean  native;      // Is native 
    CAsiData   * defval;      // Default value
    CAsiIdent    domain;      // Domain name
    uint         offset;      // Offset within table
};



////////////////////////////////////////////////
//
//
class CAsiRow : public CAsiObject
//
//  This class introduces functionality of a column collection
//  
{
public:
            CAsiRow (int ColNum);
            CAsiRow (const CAsiRow &);
    virtual ~CAsiRow ();

            int          ColNum      (void) const;
            CAsiColumn & operator [] (int) const;
            int          findCol     (const CAsiIdent &) const;
            EAsiBoolean  isDistinct  (void) const;
    virtual CAsiObject * duplicate   (void) const;
    // Nullify all members
    virtual void         Clear        (void);  

private:
    CAsiColumn *columns;    // Columns data
    int         colnum;     // Number of columns in row 
};



////////////////////////////////////////////////
//
//
class CAsiHostBuffer: public CAsiObject
//
// This class holds information about host variable buffer. Host variable
// buffer has a dual usage in ASI: it is either as a source for dynamic 
// parameters input or a destination for column value output.
// CAsiHostBuffer is a base class for CAsiParameter class, designating the
// fact that parameter has to be linked with the host variable buffer.
//
//  virtual EAsiBoolean setBind (
//    CAsiData *pData        // Pointer to host buffer
//   )
//  Bind parameter with data buffer in the host program. Host buffer
//  is represented as a CAsiData object.
//  Function returns kAsiTrue if succeeded and kAsiFalse otherwise.
//
//
//
//  virtual EAsiBoolean getBind (
//    CAsiData  **ppData         // Storage of pointer to host buffer
//   )
//
//  This function stores pointer to the host memory buffer into provided 
//  memory location.
//  kAsiFalse is returned if parameter was not bound with host buffer, or 
//  other abnormal situation occured.
//
{
friend class CAsiCsr;
friend class CAsiStm;

public:
    // Construction/descriptor
    //
             CAsiHostBuffer ();
    virtual ~CAsiHostBuffer ();

    // Getting bound variable address
    //
    virtual EAsiBoolean getBind (CAsiData **) const;

    // Runtime class iteraction functions
    //
    virtual unsigned long packSize   (void) const = 0;
    virtual unsigned long packTo     (void*) const = 0;
    virtual EAsiBoolean   unpackFrom (void*) = 0;

protected:
    // Binding parameter with host variable buffer
    //
    virtual EAsiBoolean setBind (CAsiData *);


protected:
    CAsiData *pData;      // Pointer to data
};


////////////////////////////////////////////////
//
//
class CAsiParameter: public CAsiHostBuffer
//
// This class describes dynamic parameter or variable reference in the SQL 
// statement and its link with the host program buffer. It inherits 
// functionality of the CAsiHostBuffer supercalss and reloads setBind()
// member function.
//
//  EAsiBoolean setBind (
//    CAsiData *pData,            // Pointer to host buffer
//    )
//
//  Bind parameter with the host program buffer. Function also checks
//  whether the host variable type matches the parameter type. Return
//  kAsiTrue if succeded, and kAsiFalse otherwise. Can set and push
//  complition in DDA if types mismatch error was detected.
//
{
// Friend classes
//
friend class CAsiParser;
friend class CAsiStm;
friend class CAsiDDAParams;
friend class CAsiHostLink;
friend class CAsiInterDBCursor;
friend class CAsiEmulCursor;
friend class CAsiEmulTable;

public:
    // Construction/descriptor
    //
    CAsiParameter ();
    virtual ~CAsiParameter ();

protected:
    // Bind parameter with host buffer
    //
    virtual EAsiBoolean setBind     (CAsiData *);

public:
    // Setting parameter characteristics
    //
            void        setName     (const CAsiIdent &);  // name
            void        setNullable (EAsiBoolean);        // nullability characteristics
            void        setDsc      (CAsiData*);          // data descriptor

    // Getting parameter characteristics
    //
      const CAsiIdent  &Name        (void) const;  // name
            EAsiBoolean isNullable  (void) const;  // nullability characteristics
            CAsiData   *Dsc         (void) const;  // data descriptor

    // Runtime class iteraction functions
    //
    virtual unsigned long packSize   (void) const;
    virtual unsigned long packTo     (void*) const;
    virtual EAsiBoolean   unpackFrom (void*);
private:
    virtual CAsiObject * duplicate   (void) const;
    // Nullify all members
    virtual void        Clear        (void);  

private:
    CAsiIdent ucParName;       // Parameter name
    CAsiData *pDataDescriptor; // Parameter data descriptor
    EAsiBoolean eNul;          // Nullability characteristics
};


////////////////////////////////////////////////
//  Extern function prototypes
//
extern "C" {
void                      AsiExceptionThrow (CAsiException*);
}


#endif /* ASI_ASICLASS_HH */
/*EOF*/
