/******************************************************************************/
/*                                                                            */
/* COPYRIGHT:                                                                 */
/* ----------                                                                 */
/* Copyright (C) International Business Machines Corp., 1994.                 */
/* Copyright:(C) Apple Computer, Inc., 1994                                   */
/*                                                                            */  
/* DISCLAIMER OF WARRANTIES:                                                  */
/* -------------------------                                                  */
/* The following [enclosed] code is sample code created by IBM                */
/* Corporation.  This sample code is not part of any standard IBM product     */
/* and is provided to you solely for the purpose of assisting you in the      */
/* development of your applications.  The code is provided "AS IS",           */
/* without warranty of any kind.  IBM shall not be liable for any damages     */
/* arising out of your use of the sample code, even if they have been         */
/* advised of the possibility of such damages.                                */
/*                                                                            */
/******************************************************************************/
/*               
   File:    SimplePart.cpp

   Contains:   implementation of class SimplePart

   Written by: Jason Crawford

   Change History (most recent first):

             4/30/94 JLC   simplify draw part
*/
#ifdef __OS2__
   #define INCL_BASE
   #define INCL_WIN
   #define INCL_GPI
   
   #include <stdlib.h>
   #include <builtin.h>
   #include <string.h>
#endif

#define INCL_ODALTPOINT
#define INCL_ODARBITRAT
#define INCL_ODCANVAS
#define INCL_ODCLIPBD
#define INCL_ODCMDDEFS
#define INCL_ODDGITMITR
#define INCL_ODDISPTCH
#define INCL_ODDOCUMENT
#define INCL_ODDRAFT
#define INCL_ODDRAFT
#define INCL_ODDRAGDROP
#define INCL_ODEVENTS
#define INCL_ODEXCEPT
#define INCL_ODFACET
#define INCL_ODFOCI
#define INCL_ODFOCUSSET
#define INCL_ODFRAME
#define INCL_ODINFO
#define INCL_ODISOSTRING
#define INCL_ODLIBRARYMANAGERCLASSES
#define INCL_ODLIBRARYMANAGERUTILITIES
#define INCL_ODLIMITS
#define INCL_ODNAMRSLVR
#define INCL_ODORDCOLL
#define INCL_ODPASCLSTR
#define INCL_ODPRTPRPAC
#define INCL_ODQUICKDRAW
#define INCL_ODRESOURCES
#define INCL_ODSEMTINFT
#define INCL_ODSEUTILS
#define INCL_ODSHAPE
#define INCL_ODSTDDEFS
#define INCL_ODSTDPROPS
#define INCL_ODSTDPROPS
#define INCL_ODSTDTYPES
#define INCL_ODSTORAGE
#define INCL_ODSTORAGEU
#define INCL_ODSUCURSOR
#define INCL_ODSUVIEW
#define INCL_ODTOOLUTILS
#define INCL_ODTRNSFORM
#define INCL_ODUNDO
#define INCL_ODWINDOW
#define INCL_ODWINSTAT
#define INCL_ODXMPCTR
#define INCL_ODXMPMEM
#define INCL_ODXMPSESSN
#define INCL_ODXMPUTILS

#include "odinclud.h"

#ifndef _SIMPLPRT_
   #include "SimplPrt.h"
#endif

//==============================================================================
// Local functions
//==============================================================================

//XMPBoolean ValueOnClipboard(XMPSession* session);

//==============================================================================
// Constants
//==============================================================================

// #define SSREMOVEWORKAROUND

const XMPSShort kXMPBorderWidth = 5;
const XMPSShort kXMPHandleLenMultiplier = 3;

#ifdef __OS2__
   const RGBColor rgbGray =    { 0x80, 0x80, 0x80 };
   const RGBColor rgbRed =     { 0x00, 0x00, 0x80 };
   const RGBColor rgbGreen =   { 0x00, 0x80, 0x00 };
   const RGBColor rgbYellow =  { 0x00, 0x80, 0x80 };
   const RGBColor rgbBlue =    { 0x80, 0x00, 0x00 };
   const RGBColor rgbMagenta = { 0x80, 0x00, 0x80 };
   const RGBColor rgbCyan =    { 0x80, 0x80, 0x00 };
   const RGBColor rgbWhite =   { 0xc0, 0xc0, 0xc0 };
   const RGBColor rgbBlack =   { 0x00, 0x00, 0x00 };
#endif

enum {
   kGray = 0,
   kRed,
   kGreen,
   kYellow,
   kBlue,
   kMagenta,
   kCyan,
   kWhite,
   kNumColors
};

const    XMPSShort   kMargin = 10;

const short    kSuspendResumeMessage = 0x01; // Resume vs. suspend mask
const short    kMouseMovedMessage    = 0xFA; // High byte mouse-moved event message

//==============================================================================
// Commands
//==============================================================================
   
const XMPCommandID cXMPColor = 2000;
const XMPCommandID cXMPGray = 2001;
const XMPCommandID cXMPRed = 2002;
const XMPCommandID cXMPGreen = 2003;
const XMPCommandID cXMPYellow = 2004;
const XMPCommandID cXMPBlue = 2005;
const XMPCommandID cXMPMagenta = 2006;
const XMPCommandID cXMPCyan = 2007;
const XMPCommandID cXMPWhite = 2008;
const XMPCommandID cXMPOtherColor = 2009;

#ifdef __OS2__

   // constants for drawing background grid
   #define YGRID 32
   #define XGRID 32

   const char *apszColors[] = { "Gray", "Red", "Green", "Yellow", 
                                "Blue", "Magenta", "Cyan", "White" };

   #define IDM_FRAME_BASE 2030
   const XMPCommandID cXMPFrame = IDM_FRAME_BASE+1;
   const XMPCommandID cXMPFreeze = IDM_FRAME_BASE+2;
   const XMPCommandID cXMPDefrost = IDM_FRAME_BASE+3;
#endif

   
//==============================================================================
// 'ternalization 
//==============================================================================
      
const XMPPropertyName  kXMPPropFrameInfo       = "SimplePart:Property:FrameInfo";
const XMPPropertyName  kXMPPropMouseDownOffset = "SimplePart:Property:MouseDownOffset";

const XMPValueType     kQDPoint         = "SimplePart:Type:Point";
const XMPValueType     kXMPMacTPrintRec = "Macintosh:Type:TPrint Record";

const XMPType          kDrawPresNormal  = "SimplePart:Presentation:Normal";

//==============================================================================
// Types
//==============================================================================


XMPBoolean RGBEqual(RGBColor a,RGBColor b)
{
    return (a.bRed == b.bRed &&
            a.bGreen == b.bGreen &&
            a.bBlue == b.bBlue);
}

XMPUShort DetermineNumberOfPagesinDoc(XMPFrame* frame, Rect page);

//==============================================================================
// SimplePart
//==============================================================================
#ifdef __OS2__
   // Helper functions.  These are functions that 
   //  (1) were helped reduce the code changes when
   //      porting the code from Mac to OS/2.
   //  or
   //  (2) routines that we expect Apple to be
   //      add to the API shortly.

   void WinMBox( const char * msg )
   {
        WinMessageBox(HWND_DESKTOP, HWND_DESKTOP,
                   msg, "Caption?", /*id?*/5,
                   MB_OK | MB_ERROR
                   );
   }
#endif

//-------------------------------------------------------------------------
// DLL entry point:
//-------------------------------------------------------------------------
XMPPart* EXPENTRY CreatePart()
{
   return new SimplePart();
}

//-------------------------------------------------------------------------
// constructor/destructor:
//-------------------------------------------------------------------------


//  constructor 

SimplePart::SimplePart()
{
   fDisplayFrames = kXMPNULL;
   fWindowID = 0;

   fFrameGroupIDCounter = 1;

   fContents = kXMPNULL;

   fFocusSet = kXMPNULL;

   fSemtIntf = kXMPNULL;
   fTestDrawSU = kXMPNULL;
   
   fSession = kXMPNULL;
}


//  destructor 

SimplePart::~SimplePart()
{
   if (fContents != kXMPNULL)
      delete fContents;

   if (fDisplayFrames != kXMPNULL)
      delete fDisplayFrames;     // make sure it's empty first

   if (fSemtIntf != kXMPNULL) 
      delete fSemtIntf;
   
   XMPReleaseObject(fTestDrawSU);
}


//  Initialize 

void SimplePart::InitPart(XMPStorageUnit* storageUnit)  // Override
{
   if (fInitialized)
      return;
   
   ImplBasePart::InitPart(storageUnit);
   
   this->CommonInitSimplePart();
   
   XMPStorageUnit* msu = this->GetStorageUnit();
   msu->AddProperty(kXMPPropContents)->AddValue(kXMPKindTestDraw);   
   fTestDrawSU = msu->GetDraft()->CreateStorageUnit();

   fTestDrawSU->AddProperty(kXMPPropDisplayFrames)->AddValue(kXMPIDs);        
   fTestDrawSU->AddProperty(kXMPPropFrameGroup)->AddValue(kXMPULong);         
   fTestDrawSU->AddProperty(kXMPPropContents)->AddValue(kXMPIDs);
}
 

void SimplePart::InitPartFromStorage(XMPStorageUnit* storageUnit)   // Override
{
   if (fInitialized)
      return;
   
   ImplBasePart::InitPartFromStorage(storageUnit);
   
   this->CommonInitSimplePart();
   

   XMPStorageUnit* su;
   XMPStorageUnitRef aSURef;
   XMPFrame* frame;
   XMPShape* newShape;
   XMPULong offset, offsetLimit;
   XMPRgnHandle newRegion;
   MATRIXLF xform;
   XMPTransform* transform;

   XMPUnused(newRegion);
   XMPVolatile(frame);

   su = this->GetStorageUnit();
   su->Focus(kXMPPropContents,kXMPPosSame,kXMPKindTestDraw,1,kXMPPosFirstSib);
   su->GetValue(sizeof(XMPStorageUnitRef),&aSURef);
   fTestDrawSU = su->GetDraft()->GetStorageUnit(su->GetIDFromStorageUnitRef(aSURef));
   su = fTestDrawSU;

   su->Focus(kXMPPropDisplayFrames, kXMPPosSame, 0, 1, kXMPPosFirstSib);
   offsetLimit = su->GetSize();
   for (offset = 0; offset < offsetLimit; offset += sizeof(XMPStorageUnitRef))
   {
      su->SetOffset(offset);
      su->GetValue(sizeof(XMPStorageUnitRef), (XMPValue)&aSURef);
      
      TRY
      
         frame = su->GetDraft()->GetFrame(su->GetIDFromStorageUnitRef(aSURef));
         fDisplayFrames->AddLast((ElementType)frame);
         
      CATCH_ALL
      
         frame = kXMPNULL;
   
      ENDTRY
   }

   su->Focus(kXMPPropFrameGroup,kXMPPosSame,0,1,kXMPPosFirstSib);
   su->GetValue(sizeof(fFrameGroupIDCounter),(XMPValue)&fFrameGroupIDCounter);
}

void SimplePart::CommonInitSimplePart()
{
   fDisplayFrames = new OrderedCollection;

   RGBColor random;
   random.bRed = rand();
   random.bGreen = rand();
   random.bBlue = rand();
   fDefaultColor = random;


   fContents = new OrderedCollection;

   fSession = this->GetStorageUnit()->GetSession();

   fSelectionFocus = fSession->Tokenize(kXMPSelectionFocus);
   fMenuFocus = fSession->Tokenize(kXMPMenuFocus);
   fKeyFocus = fSession->Tokenize(kXMPKeyFocus);
   
   fFocusSet = new XMPFocusSet();
   fFocusSet->InitFocusSet();
   fFocusSet->Add(fSelectionFocus);
   fFocusSet->Add(fMenuFocus);
   fFocusSet->Add(fKeyFocus);
}



//  Release 

void SimplePart::Release()
{
   XMPFrame*   frame;
   
   XMPPersistentObject::Release();
   if (this->GetRefCount() == 0) {
      frame = (XMPFrame*) fDisplayFrames->First();
      while (frame != kXMPNULL) {
         fDisplayFrames->Remove(frame);
         XMPReleaseObject(frame);
         frame = (XMPFrame*) fDisplayFrames->First();
      }
      this->GetStorageUnit()->GetDraft()->ReleasePart(this);
   }
}

//-------------------------------------------------------------------------
// From DragAndDrop protocol
//-------------------------------------------------------------------------


//  FullfillPromise 

void SimplePart::FulfillPromise(XMPStorageUnitView *promiseSUView)
{
   XMPUnused(promiseSUView);
}


//  NotifyDropComplete 

void SimplePart::DropCompleted(XMPPart* destPart, XMPDropResult dropResult)
{
   XMPUnused(destPart);
   XMPUnused(dropResult);
}


//  DragEnter 
//dwf changed return to MRESULT from void
MRESULT SimplePart::DragEnter(XMPDragItemIterator* dragInfo,
                     XMPFacet* facet,
                     XMPPoint where)
{
   XMPUnused(dragInfo);
   XMPUnused(facet);
   XMPUnused(where);

   return MRFROM2SHORT (DOR_NEVERDROP, DO_UNKNOWN);  //dwf

}


//  DragWithin 
//dwf changed void to MRESULT
MRESULT SimplePart::DragWithin(XMPDragItemIterator* dragInfo,
                     XMPFacet* facet,
                     XMPPoint where)
{
   XMPUnused(dragInfo);
   XMPUnused(facet);
   XMPUnused(where);

   return MRFROM2SHORT (DOR_NEVERDROP, DO_UNKNOWN);  //dwf
}


//  DragLeave 

void SimplePart::DragLeave(XMPFacet* facet,
                   XMPPoint where)
{
   XMPUnused(facet);
   XMPUnused(where);
}


//  Drop 

XMPDropResult SimplePart::Drop(XMPDragItemIterator *dropInfo,
                      XMPFacet* facet,
                      XMPPoint where)
{
   return kXMPDropFail;
}


//-------------------------------------------------------------------------
// from Frame protocol:
//-------------------------------------------------------------------------

//  AddDisplayFrame 

void SimplePart::AddDisplayFrame(XMPFrame* frame)
{

   if (frame->GetPart() == this)    // frame belongs to me
   {
      OrderedCollectionIterator  displayFramesIter(fDisplayFrames);
      XMPFrame* displayFrame = (XMPFrame*) displayFramesIter.First();

      // !!! do something with viewType and partInfo...
      PartInfoRec* pInfo = new PartInfoRec;
      pInfo->bgColor = fDefaultColor;
      if (frame->IsRoot())
         pInfo->fNeedsActivating = kXMPTrue;

      frame->SetPartInfo((XMPInfoType) pInfo);
      fDisplayFrames->AddLast(frame);
      frame->IncrementRefCount();
      
      if (frame->GetViewType() == kXMPNullTypeToken)                 // if frame view is set don't change it
         frame->SetViewType(fSession->Tokenize(kXMPViewAsFrame));    // if not, make it viewasframe
      if (frame->GetPresentation() == kXMPNullTypeToken)             
         frame->SetPresentation(fSession->Tokenize(kDrawPresNormal));
      
      if (frame->GetContainingFrame() == kXMPNULL)
      {
         //Wrong place. this->ActivateFrame(frame);
      }
   }
   else
      THROW(kXMPErrInvalidFrame);

   // render in frame?
}



//  RemoveDisplayFrame 

void SimplePart::RemoveDisplayFrame(XMPFrame* oldFrame)
{
   if (oldFrame != kXMPNULL) {
      if (fDisplayFrames->Contains(oldFrame))
      {
         fSession->GetArbitrator()->RelinquishFocusSet(fFocusSet,oldFrame);
   
         PartInfoRec* pInfo = (PartInfoRec*) oldFrame->GetPartInfo();
         oldFrame->SetPartInfo((XMPInfoType) kXMPNULL);
         delete pInfo;
         fDisplayFrames->Remove(oldFrame);
         oldFrame->Release();
         
         OrderedCollectionIterator dIter(fDisplayFrames);
         XMPFrame*   displayFrame = (XMPFrame*) dIter.First();
         if (dIter.IsNotComplete() == kXMPFalse) {
         }
      }
      else
         THROW(kXMPErrInvalidFrame);
   }

   // any display frames left?
}


//  CloseDisplayFrame 

void SimplePart::CloseDisplayFrame(XMPFrame* frame)
{
   if (fDisplayFrames->Contains(frame))
   {
      
      fSession->GetArbitrator()->RelinquishFocusSet(fFocusSet, frame);

      PartInfoRec* pInfo = (PartInfoRec*) frame->GetPartInfo();
      frame->SetPartInfo((XMPInfoType) kXMPNULL);
      delete pInfo;
      fDisplayFrames->Remove(frame);
      XMPReleaseObject(frame);
   }
   else
      THROW(kXMPErrInvalidFrame);
}


//  FrameShapeChanged 

void SimplePart::FrameShapeChanged(XMPFrame* frame)
{
   if (fDisplayFrames->Contains(frame))
   {
      /// !!! should leave UsedShape and ActiveShape null to inherit FrameShape
   }
   else
      THROW(kXMPErrInvalidFrame);
}


//  ViewTypeChanged 

void SimplePart::ViewTypeChanged(XMPFrame* frame)
{
   if (fDisplayFrames->Contains(frame))
      { /* !!! change viewType of frame */ }
   else
      THROW(kXMPErrInvalidFrame);
}


//  PresentationChanged 

void SimplePart::PresentationChanged(XMPFrame* frame)
{
   if (fDisplayFrames->Contains(frame))
      { /* !!! change presentation of frame */ }
   else
      THROW(kXMPErrInvalidFrame);
}


//  WritePartInfo 

void SimplePart::WritePartInfo(XMPPtr partInfo, XMPStorageUnitView* storageUnitView)
{
   if (partInfo)
   {
      storageUnitView->SetValue(sizeof(RGBColor), (XMPValue)&(((PartInfoRec*)partInfo)->bgColor));
      XMPBoolean needsActivating = ((PartInfoRec*)partInfo)->fNeedsActivating 
                           || ((PartInfoRec*)partInfo)->fIsActive;
      storageUnitView->SetValue(sizeof(XMPBoolean),
                           (XMPValue)&needsActivating);
   }
}

//  ReadPartInfo 

XMPPtr SimplePart::ReadPartInfo(XMPFrame* frame, XMPStorageUnitView* storageUnitView)
{
   XMPUnused(frame);
   if (storageUnitView->GetSize())
   {
      PartInfoRec* partInfo = new PartInfoRec;
      
      storageUnitView->GetValue(sizeof(RGBColor),
                           (XMPValue)&(partInfo->bgColor));
      XMPBoolean needsActivating;
      storageUnitView->GetValue(sizeof(XMPBoolean),
                           (XMPValue)&(needsActivating));
      partInfo->fNeedsActivating = needsActivating;
                           
      return partInfo;
   }
   else
      return ((XMPPtr)kXMPNULL);
}


//  Open 

XMPID SimplePart::Open(XMPFrame* frame)
{
   XMPWindow* window = kXMPNULL;

   if (frame) // Doing a View As Window
   {
      window = fSession->GetWindowState()->GetWindow(fWindowID);
      if (window)
         window->Select();
      else
      {
         window = this->CreateWindow(frame);
         fWindowID = window->GetID();
         window->Open();
         window->Show();
         window->Select();       
      }
   }
   else
   {
      window = this->CreateWindow(frame);
      fWindowID = window->GetID();
      window->Open();
      window->Show();
      window->Select();       
   }
   return window->GetID();
}

XMPWindow* SimplePart::CreateWindow(XMPFrame* sourceFrame)
{
   Rect windRect;
   XMPPlatformWindow platformWindow = kXMPNULL;
   XMPWindow* window = kXMPNULL;
   
   _interrupt(3);
   return window;
}


//  ExternalTransformChanged 

void SimplePart::ExternalTransformChanged(XMPFacet* facet)
{
XMPUnused(facet);
   // !!!
}

//-------------------------------------------------------------------------
// From Imaging protocol
//-------------------------------------------------------------------------

//  Draw 

void SimplePart::Draw(XMPFacet* facet, XMPShape* invalShape)
{
   XMPBoolean drawing = facet->GetCanvas()->IsDynamic();
   
   XMPUnused(invalShape);

   XMPFrame* displayFrame = facet->GetFrame();
   if (fDisplayFrames->Contains(displayFrame))  
   {
      // transform grafport

      HPS hpsDraw = facet->GetCanvas()->GetPlatformCanvas();
      GpiSavePS(hpsDraw);


      PartInfoRec* pInfo = (PartInfoRec*) displayFrame->GetPartInfo();

      RgnHandle frameRgn = displayFrame->GetFrameShape()->GetQDRegion();
      Rect frameRect;
      GpiQueryRegionBox(hpsDraw, frameRgn, &frameRect);

      // set up clipping
      RgnHandle saveClip;
      XMPShape* clipShape = new XMPShape;
      clipShape->CopyFrom(facet->GetAggregateClipShape());
      clipShape->Transform(facet->GetContentTransform());
      RgnHandle clip = clipShape->GetQDRegion();
      GpiSetClipRegion(hpsDraw, clip, &saveClip);
      
      if (facet->GetFrame()->GetPresentation() == fSession->Tokenize(kDrawPresNormal))
      {
         XMPTypeToken curView = facet->GetFrame()->GetViewType();
         
         if (curView == fSession->Tokenize(kXMPViewAsFrame)) {
            // paint the background
         
            GpiResetPS(hpsDraw, GRES_ATTRS);
      
            this->SetGrafPortOrigin(facet);

            RGBColor bgRGB = pInfo->bgColor;
            ULONG ulRGB = (bgRGB.bRed << 16) + (bgRGB.bGreen << 8) + bgRGB.bBlue;
            ULONG lColorIndex = GpiQueryColorIndex(hpsDraw, 0, ulRGB);
            GpiSetColor(hpsDraw, lColorIndex);
            POINTL ptl = {frameRect.xRight, frameRect.yTop};
            GpiBox(hpsDraw, DRO_FILL, &ptl, 0, 0);

            GpiSetLineType(hpsDraw, LINETYPE_DASHDOUBLEDOT);
            GpiSetColor(hpsDraw, CLR_BLACK);

            for (int y = 0; y < frameRect.yTop; y += YGRID) {
               ptl.y = y;
               ptl.x = 0;
               GpiMove(hpsDraw, &ptl);
               ptl.x = frameRect.xRight;
               ptl.y += frameRect.xRight;
               GpiLine(hpsDraw, &ptl);
            }
      
            for (int x = XGRID; x < frameRect.xRight; x += XGRID) {
               ptl.x = x;
               ptl.y = 0;
               GpiMove(hpsDraw, &ptl);
               ptl.x += frameRect.yTop;
               ptl.y = frameRect.yTop;
               GpiLine(hpsDraw, &ptl);
            }

         }
      }
      GpiRestorePS(hpsDraw, -1);
   
      GpiSetClipRegion(hpsDraw, 0, &saveClip);
      delete clipShape;
   }
   else
   {
      // !!! signal error: invalid frame
   }
}


//-------------------------------------------------------------------------
// From Memory Management protocol
//-------------------------------------------------------------------------

//  Purge 

XMPSize SimplePart::Purge(XMPSize size)
{
   return this->GetStorageUnit()->Purge(size);
}
     
//-------------------------------------------------------------------------
// From Part Activation protocol
//-------------------------------------------------------------------------

//  RelinquishFocus 

void SimplePart::CommitRelinquishFocus(XMPTypeToken focus, 
                        XMPFrame* currentFrame,
                        XMPFrame* proposedFrame)
{
   XMPUnused(proposedFrame);

   this->FocusLost(focus, currentFrame);
}


//  FocusAcquired 

void SimplePart::FocusAcquired(XMPTypeToken focus,
                        XMPFrame* newOwner)
{
   if (focus == fSelectionFocus) {
      PartInfoRec* pInfo = (PartInfoRec*) newOwner->GetPartInfo();
      pInfo->fIsActive = kXMPTrue;
   }
}

//  FocusLost  

void SimplePart::FocusLost(XMPTypeToken focus,
                     XMPFrame* oldOwner)
{
   if (focus == fSelectionFocus) {
      PartInfoRec* pInfo = (PartInfoRec*) oldOwner->GetPartInfo();
      pInfo->fIsActive = kXMPFalse;
   }
}


//-------------------------------------------------------------------------
// From Storage protocol
//-------------------------------------------------------------------------

//  Externalize 

void SimplePart::Externalize()
{
   ImplBasePart::Externalize();
   
   XMPStorageUnit* su = fTestDrawSU;
   XMPStorageUnitRef aSURef;
   XMPULong offset;
   XMPFrame* frame;
   OrderedCollectionIterator aIter(fDisplayFrames);
   OrderedCollectionIterator cIter(fContents);
   MATRIXLF xform;
 
   su->Focus(kXMPPropDisplayFrames,kXMPPosUndefined,0,1,kXMPPosFirstSib);
   #ifndef SSREMOVEWORKAROUND
      su->Remove();
      su->AddProperty(kXMPPropDisplayFrames);
      su->AddValue(kXMPIDs);
   #else
      XMPULong oldValueSize = su->GetSize();;
   #endif
   offset = 0;
   for (frame = (XMPFrame*)aIter.First(); aIter.IsNotComplete();
         frame = (XMPFrame*)aIter.Next(), offset+=sizeof(XMPStorageUnitRef))
   {
      frame->Externalize();
      aSURef = su->GetWeakStorageUnitRef(frame->GetStorageUnit());
      su->SetOffset(offset);
      su->SetValue(sizeof(XMPStorageUnitRef), (XMPValue)&aSURef);
   }
   #ifdef SSREMOVEWORKAROUND
      if (oldValueSize > offset) su->DeleteValue(oldValueSize-offset);
   #endif
 
   su->Focus(kXMPPropFrameGroup,kXMPPosUndefined,0,1,kXMPPosFirstSib);
   su->SetValue(sizeof(fFrameGroupIDCounter), (XMPValue)&fFrameGroupIDCounter);
 
   su = this->GetStorageUnit();
   su->Focus(kXMPPropContents,kXMPPosSame,kXMPKindTestDraw,1,kXMPPosFirstSib);
   aSURef = su->GetStrongStorageUnitRef(fTestDrawSU);
   su->SetValue(sizeof(XMPStorageUnitRef),&aSURef);
}


//  CloneInto 

void SimplePart::CloneInto(XMPDraftKey key, XMPStorageUnit* storageUnit, XMPStorageUnit* initiatingFrameSU)
{
   XMPStorageUnit*   su = this->GetStorageUnit();
   
   this->Externalize();
   XMPStorageUnit* destTestDrawSU = fTestDrawSU->CloneTo(key, storageUnit->GetDraft(), initiatingFrameSU); 
   
   storageUnit->AddProperty(kXMPPropContents);
   storageUnit->AddValue(kXMPKindTestDraw);
   XMPStorageUnitRef aSURef = storageUnit->GetStrongStorageUnitRef(destTestDrawSU);
   storageUnit->SetValue(sizeof(XMPStorageUnitRef),&aSURef);
      
   destTestDrawSU->Release();
   
   su->CloneInto(key, storageUnit, initiatingFrameSU);
}

//-------------------------------------------------------------------------
// From UI Events protocol
//-------------------------------------------------------------------------


//  HandleEvent  !!!

XMPBoolean SimplePart::HandleEvent(XMPEventData event, XMPFrame* frame,
      XMPFacet* facet)
{
   Point windowPoint;
   XMPBoolean handled = kXMPFalse;
   
   switch (event->what)
   {
      case kXMPEvtMouseDown:
         {
            if (event->msg == WM_BUTTON1DOWN || event->msg == WM_BUTTON2DOWN)
            {
               windowPoint = event->where;
               XMPPoint windowXMPPoint(windowPoint);
               handled = this->HandleMouseDown(facet, windowXMPPoint, event);
            }
         }
         break;
         
      case kXMPEvtMouseUp:
         break;
        
      case kXMPEvtMouseMove:
         break;
         
      case kXMPEvtKeyDown:
      case kXMPEvtAutoKey:
         handled = this->HandleKeyDown(frame, event);
         DosBeep(250,100);
         break;
   
      case kXMPEvtActivate:
         handled = true; // actually ignored by dispatcher
         if (SHORT1FROMMP(event->mp1) != 0)
            this->ActivatingWindow(frame);
         else
            this->DeActivatingWindow(frame);
         break;
      
      case kXMPEvtKeyUp:
         break;
                  
      case kHighLevelEvent:
         break;

      default:
         return kXMPFalse;
   }
   return handled;
}


//  HandleMouseDown 

XMPBoolean SimplePart::HandleMouseDown(  XMPFacet* facet,
                              XMPPoint where,
                              XMPEventData event)
{
   if (!facet->GetWindow()->IsActive())
      facet->GetWindow()->Select();
   else
      this->ActivateFrame(facet->GetFrame());

   Point mouse = facet->GetWindowContentTransform()->InvertPoint(where).AsPOINTL();

   return kXMPTrue;
}


//  HandleKeyDown 

XMPBoolean SimplePart::HandleKeyDown(XMPFrame* focusFrame, XMPEventData event)
{
   XMPUnused(focusFrame);
   XMPUnused(event);
   // cursor keys? !!!

   return kXMPFalse;
}

//-------------------------------------------------------------------------
// private utilities
//-------------------------------------------------------------------------

//  ActivateFrame 

void SimplePart::ActivateFrame(XMPFrame* frame)
{
   if (frame != kXMPNULL) {
      PartInfoRec* pInfo = (PartInfoRec*) frame->GetPartInfo();
      if (!(pInfo->fIsActive))
      {
         XMPBoolean succeeded = false;
                        
         succeeded = fSession->GetArbitrator()->RequestFocusSet(fFocusSet,frame);
               
         if (succeeded)
         {
            this->FocusAcquired(fSelectionFocus, frame);
            this->FocusAcquired(fKeyFocus, frame);
         }
      }
   }
}

//  DeActivateFrame 

void SimplePart::DeActivateFrame(XMPFrame* frame)
{
   if (frame != kXMPNULL) {
      fSession->GetArbitrator()->RelinquishFocusSet(fFocusSet,frame);
      this->FocusLost(fSelectionFocus, frame);
      this->FocusLost(fKeyFocus, frame);
   }
}

//  ActivatingWindow 

void SimplePart::ActivatingWindow(XMPFrame* frame)
{
   PartInfoRec* pInfo = (PartInfoRec*) frame->GetPartInfo();
   if (pInfo->fNeedsActivating)
   {
      this->ActivateFrame(frame);
      pInfo->fNeedsActivating = kXMPFalse;
   }
}

//  DeActivatingWindow 

void SimplePart::DeActivatingWindow(XMPFrame* frame)
{
   PartInfoRec* pInfo = (PartInfoRec*) frame->GetPartInfo();
   if (frame == fSession->GetArbitrator()->GetFocusOwner(fSelectionFocus))
   {
      pInfo->fNeedsActivating = kXMPTrue;
      //this->DeActivateFrame(frame);
   }
   else
      pInfo->fNeedsActivating = kXMPFalse;
}



//  SetBGColor 

void SimplePart::SetBGColor(XMPFrame* frame, XMPCommandID whichColor)
{
   PartInfoRec* pInfo = (PartInfoRec *) frame->GetPartInfo();
   if (!pInfo)
   {  pInfo = new PartInfoRec;
      frame->SetPartInfo((XMPInfoType) pInfo);
   };

   RGBColor newColor = pInfo->bgColor;
   
   switch (whichColor)
   {
      break;
      case  cXMPGray:    newColor = rgbGray;      break;
      case  cXMPRed:     newColor = rgbRed;       break;
      case  cXMPGreen:   newColor = rgbGreen;     break;
      case  cXMPYellow:  newColor = rgbYellow;    break;
      case  cXMPBlue:    newColor = rgbBlue;      break;
      case  cXMPMagenta: newColor = rgbMagenta;   break;
      case  cXMPCyan:    newColor = rgbCyan;      break;
      case  cXMPWhite:   newColor = rgbWhite;     break;
      default: break;
   }
   UserSetBGColor(frame,newColor);
}

struct SetBGColorRec
{
   SetBGColorRec(RGBColor oldColor, RGBColor newColor, XMPFrame* frame)
               {fOldColor = oldColor; fNewColor = newColor;
                  fFrame = frame;}
   RGBColor    fOldColor;
   RGBColor    fNewColor;
   XMPFrame*   fFrame;
};

//  UserSetBGColor 

void SimplePart::UserSetBGColor(XMPFrame* frame, RGBColor whichColor)
{
   PartInfoRec* pInfo = (PartInfoRec *) frame->GetPartInfo();
   if (!pInfo)
   {  pInfo = new PartInfoRec;
      frame->SetPartInfo((XMPInfoType) pInfo);
   }
   if (!RGBEqual(whichColor,pInfo->bgColor))
   {
      this->SetBGColor(frame, whichColor);
   }
}

//  SetBGColor 

void SimplePart::SetBGColor(XMPFrame* frame, RGBColor whichColor)
{
   PartInfoRec* pInfo = (PartInfoRec *) frame->GetPartInfo();
   if (!pInfo)
   {  pInfo = new PartInfoRec;
      frame->SetPartInfo((XMPInfoType) pInfo);
   }
   if (!RGBEqual(whichColor,pInfo->bgColor))
   {
      pInfo->bgColor = whichColor;
      this->GetStorageUnit()->GetDraft()->SetChangedFromPrev();
      frame->Invalidate(NULL);
   }
}

void SimplePart::GetBGColor(XMPUShort whichColor, RGBColor* newColor)
{
   switch (whichColor)
   {
      case  kGray: *newColor = rgbGray;      break;
      case  kRed: *newColor = rgbRed;     break;
      case  kGreen: *newColor = rgbGreen;    break;
      case  kYellow: *newColor = rgbYellow;  break;
      case  kBlue: *newColor = rgbBlue;      break;
      case  kMagenta: *newColor = rgbMagenta;   break;
      case  kCyan: *newColor = rgbCyan;      break;
      case  kWhite: *newColor = rgbWhite;    break;
      
      default:
         WASSERTM(kXMPFalse, "No such color.");
         break;
   }
}

//------------------------------------------------------------------------------
// Frame Shape Transformation Utilities
//------------------------------------------------------------------------------



//  SetGrafPortOrigin 

void SimplePart::SetGrafPortOrigin(XMPFacet* facet)
{
   XMPTransform* localToGlobal = facet->GetContentTransform();
   HPS hps = facet->GetCanvas()->GetPlatformCanvas();
   PMATRIXLF mtx = (PMATRIXLF)facet->GetContentTransform()->GetPlatformTransform();

   // This is a hack until GetPlatformTransform() is fixed to return a 
   // valid MATRIXLF transform (i.e. uses FIXED values where required)
   MATRIXLF mtxTemp = *mtx;
   mtxTemp.fxM11 *= 0x10000;
   mtxTemp.fxM12 *= 0x10000;
   mtxTemp.fxM21 *= 0x10000;
   mtxTemp.fxM22 *= 0x10000;

   GpiSetModelTransformMatrix(hps, 9, &mtxTemp, TRANSFORM_REPLACE);
}

//------------------------------------------------------------------------------
// SimplePart::HasExtension
//------------------------------------------------------------------------------

//  HasExtension 

XMPBoolean  SimplePart::HasExtension(XMPType extensionName)
{
   XMPISOStr   semtIntf = "SemanticInterface";
   if (!XMPISOStrCompare((XMPISOStr)extensionName, semtIntf))
      return kXMPTrue;
   else
      return kXMPFalse;
}

   
//------------------------------------------------------------------------------
// SimplePart::GetExtension
//------------------------------------------------------------------------------


//  GetExtension 

XMPExtension*  SimplePart::GetExtension(XMPType extensionName)
{
   XMPISOStr   semtIntf = "SemanticInterface";
   if (!XMPISOStrCompare((XMPISOStr)extensionName, semtIntf))
      return fSemtIntf;
   else
      return kXMPNULL;
}

