/******************************************************************************/
/*                                                                            */
/* COPYRIGHT:                                                                 */
/* ----------                                                                 */
/* Copyright (C) International Business Machines Corp., 1994.                 */
/* Copyright:(C) Apple Computer, Inc., 1994                                   */
/*                                                                            */
/* DISCLAIMER OF WARRANTIES:                                                  */
/* -------------------------                                                  */
/* The following [enclosed] code is sample code created by IBM                */
/* Corporation.  This sample code is not part of any standard IBM product     */
/* and is provided to you solely for the purpose of assisting you in the      */
/* development of your applications.  The code is provided "AS IS",           */
/* without warranty of any kind.  IBM shall not be liable for any damages     */
/* arising out of your use of the sample code, even if they have been         */
/* advised of the possibility of such damages.                                */
/*                                                                            */
/******************************************************************************/
/*
  File:    ClockPart.cp

  Contains:  Source code for the sample "Clock" part

  Written by:  Steve Smith

  Change History (most recent first):

              5/20/94 CED      Fixes to externalize properly

*/

#define INCL_WIN
#define INCL_GPI
#define INCL_DOSDATETIME
#define INCL_DOSPROCESS
#include <os2.h>

#include <time.h>
#include <string.h>

#include "Clock.h"

POINTL aptlFillet1[]= {{53L ,63L },{ 70L ,50L },{68L,47L}} ;
POINTL aptlFillet2[]= {{68L,47L},{ 70L, 50L },{56L ,67L }} ;
POINTL aptlFillet3[]= {{53L ,64L },{ 56L ,62L },{60L,58L}} ;
POINTL aptlFillet4[]= {{60L,58L},{ 60L, 63L },{56L ,67L }} ;
POINTL ptlLight= { -1L , 1L } ;
POINTL ptlShade= {  2L, -2L } ;
POINTL ptlFace = { 0L, 0L};

static FIXED fxSin [60] =
{
    0x00000000, 0x00001ac2, 0x00003539, 0x00004f1b, 0x0000681f, 0x00007fff,
    0x00009679, 0x0000ab4c, 0x0000be3e, 0x0000cf1b, 0x0000ddb3, 0x0000e9de,
    0x0000f378, 0x0000fa67, 0x0000fe98, 0x0000ffff, 0x0000fe98, 0x0000fa67,
    0x0000f378, 0x0000e9de, 0x0000ddb3, 0x0000cf1b, 0x0000be3e, 0x0000ab4c,
    0x00009679, 0x00008000, 0x00006820, 0x00004f1b, 0x00003539, 0x00001ac2,
    0x00000000, 0xffffe53e, 0xffffcac7, 0xffffb0e5, 0xffff97e1, 0xffff8001,
    0xffff6988, 0xffff54b5, 0xffff41c2, 0xffff30e5, 0xffff224d, 0xffff1622,
    0xffff0c88, 0xffff0599, 0xffff0168, 0xffff0001, 0xffff0167, 0xffff0599,
    0xffff0c88, 0xffff1622, 0xffff224d, 0xffff30e5, 0xffff41c2, 0xffff54b4,
    0xffff6987, 0xffff8000, 0xffff97e0, 0xffffb0e4, 0xffffcac6, 0xffffe53e
} ;


MATRIXLF vmatlfDateTrans = {
     MAKEFIXED ( 1 , 0 ) ,       MAKEFIXED ( 0 , 0 ) ,       0L ,
     MAKEFIXED ( 0 , 0 ) ,       MAKEFIXED ( 1 , 0 ) ,       0L ,
     0L ,                      0L ,                      1L } ;
MATRIXLF vmatlfDateScale  = {
     MAKEFIXED ( 1 , 0 ) ,       MAKEFIXED ( 0 , 0 ) ,       0L ,
     MAKEFIXED ( 0 , 0 ) ,       MAKEFIXED ( 1 , 0 ) ,       0L ,
     0L ,                      0L ,                      1L } ;

/****************************************************************\
 *
 *--------------------------------------------------------------
 *
 *  Name:Clock()
 *
 *  Purpose:Intialize a newly created client window
 *
 *
 *
 *  Usage:
 *
 *  Method:
 *
 *  Returns:
 *          1 - if sucessful execution completed
 *          0 - if error
\****************************************************************/
Clock:: Clock()
{
    SIZEL sizl = { 200 , 200 };

    /* init the Data structure */

    /*
     * Create our off-screen 'buffer'.
     */

    data.hdcBuffer = DevOpenDC ( HAB(0), OD_MEMORY, "*", 0L, NULL, NULL);

    data.hpsBuffer = GpiCreatePS (HAB(0), data.hdcBuffer, &sizl, PU_ARBITRARY |
                               GPIT_MICRO | GPIA_ASSOC);

    // Set the PS into RGB mode!
    GpiCreateLogColorTable (data.hpsBuffer, 0, LCOLF_RGB, 0, 0, (PLONG)NULL);

    // if we don't read this in from storage unit.
    {
        data.cp.usMajorTickPref = CLKTM_ALWAYS;
        data.cp.usMinorTickPref = CLKTM_NOTICONIC;
        data.cp.clrBackground = 0x00008080;
        data.cp.clrFace = 0x00008080;
        data.cp.clrHourHand = RGB_RED;
        data.cp.clrMinuteHand = RGB_RED;
        data.cp.fControlsHidden = FALSE;
        data.cp.usDispMode = DM_TIME | DM_ANALOG | DM_SECONDHAND;
        data.cp.alarm.uchHour = 0;
        data.cp.alarm.uchMinutes = 0;
        data.cp.alarm.usMode = 0;
        SetRGBColors();
    }

    /* get the time in a format for dislaying */
    DosGetDateTime(&data.dt);
    data.dt.hours = (UCHAR )(data.dt.hours * (UCHAR) 5) %
                                           (UCHAR) 60 + data.dt.minutes / (UCHAR)12;


}

/****************************************************************\
 *
 *--------------------------------------------------------------
 *
 *  Name: destructor
 *
 *  Purpose: Destroy clock face.
 *
 *
 *
 *  Usage:
 *
 *  Method:
 *
 *  Returns:VOID
 *
 *
\****************************************************************/
Clock:: ~Clock ()
{
    HBITMAP hbm;

    hbm = GpiSetBitmap (data.hpsBuffer, NULLHANDLE);

    if (hbm != NULLHANDLE)
        GpiDeleteBitmap (hbm);

    GpiDestroyPS (data.hpsBuffer);
    DevCloseDC (data.hdcBuffer);
}

/****************************************************************\
 *
 *--------------------------------------------------------------
 *
 *  Name: CreateRegion
 *
 *  Purpose: Creates a region.
 *
 *
 *
 *  Usage:
 *
 *  Method:
 *
 *  Returns:VOID
 *
 *
\****************************************************************/
HRGN Clock:: CreateRegion( HPS hps )
{
    BOOL f ;
    static POINTL ptlLight= { -2L , 2L } ;
    static POINTL ptlShade= {  1L, -1L } ;
    static POINTL ptlFace = { 0L, 0L};
    static POINTL ptlShadeIn= { -3L , 3L } ;
    static POINTL ptlLightIn= {  1L, -1L } ;
    HRGN hRegion, hRegionTemp;
    RECTL rclBounds;

    /*        1         0      0     *\
    *         0         1      0      *
    \*      100       100      1     */

    static MATRIXLF matlfModel = {
    MAKEFIXED ( 1 , 0 ) ,       MAKEFIXED ( 0 , 0 ) ,       0L ,
    MAKEFIXED ( 0 , 0 ) ,       MAKEFIXED ( 1 , 0 ) ,       0L ,
    100L ,                      100L ,                      1L } ;

    /* center at (100, 100) and draw the clock face */
    f = GpiSetModelTransformMatrix ( hps , ( LONG ) MATLF_SIZE ,
                                     & matlfModel , TRANSFORM_REPLACE ) ;

    DrawFullCircle(hps,
                   &ptlShade,
                   MAKEFIXED(99 ,0));

    hRegion = GpiPathToRegion(hps, 1, FPATH_ALTERNATE);
        GpiQueryRegionBox(hps, hRegion, &rclBounds);

    DrawFullCircle(hps,
                   &ptlLight,
                   MAKEFIXED(98 ,0));

    hRegionTemp = GpiPathToRegion(hps, 1, FPATH_ALTERNATE);
    GpiCombineRegion(hps, hRegion, hRegion, hRegionTemp, CRGN_OR);
    GpiDestroyRegion(hps, hRegionTemp);

    DrawFullCircle(hps,
                   &ptlLightIn,
                   MAKEFIXED(94,0));

    hRegionTemp = GpiPathToRegion(hps, 1, FPATH_ALTERNATE);
    GpiCombineRegion(hps, hRegion, hRegion, hRegionTemp, CRGN_OR);
    GpiDestroyRegion(hps, hRegionTemp);

    DrawFullCircle(hps,
                   &ptlShadeIn,
                   MAKEFIXED(93,0));

    hRegionTemp = GpiPathToRegion(hps, 1, FPATH_ALTERNATE);
    GpiCombineRegion(hps, hRegion, hRegion, hRegionTemp, CRGN_OR);
    GpiDestroyRegion(hps, hRegionTemp);

    DrawFullCircle(hps,
                    &ptlFace ,
                    MAKEFIXED(98 ,0));

    hRegionTemp = GpiPathToRegion(hps, 1, FPATH_ALTERNATE);
    GpiCombineRegion(hps, hRegion, hRegion, hRegionTemp, CRGN_OR);
    GpiDestroyRegion(hps, hRegionTemp);

    return hRegion;
}

/****************************************************************\
 *
 *--------------------------------------------------------------
 *
 *  Name: Size()
 *
 *  Purpose:When the window has been sized, we calculate  a page
 *          rectangle which: (a) fills the window rectangle in either
 *          the x or y dimension, (b) appears square, and (c) is centered
 *          in the window rectangle
 *  Usage:
 *
 *  Method:
 *
 *  Returns:
 *
 *
\****************************************************************/
VOID Clock:: Size( const RECTL& rcl, HRGN *hrgnUsed)
{
    SIZEF            sizef;

    LONG             cxSquare;
    LONG             cySquare;

    LONG             cxEdge;
    LONG             cyEdge;

    LONG             cyHeight;
    LONG             cxWidth;

    HBITMAP          hbm;
    BITMAPINFOHEADER bmp;

    HRGN             hOldRegion;
    RECTL            rclBounds;

    LONG             cxRes;
    LONG             cyRes;

    LONG             cColorPlanes;
    LONG             cColorBitcount;

    BOOL             f;
    SWP              swp;

    this-> rcl = rcl;
    /*
     * First get rid of any buffer bitmap already there.
     */
    hbm = GpiSetBitmap (data.hpsBuffer, NULLHANDLE);

    if (hbm != NULLHANDLE)
        GpiDeleteBitmap (hbm);

    /*
     * Get the width and height of the window rectangle.
     */
    cxWidth = rcl.xRight - rcl.xLeft;
    cyHeight = rcl.yTop - rcl.yBottom;

    /*
     * Now create a bitmap the size of the window.
     */

    //KLS Get device info from desktop window.
    HPS hps;                /* presentation space handle            */
    HDC hdc;                /* device context handle                */
    LONG lWidth, lHeight;

    hps = WinGetScreenPS(HWND_DESKTOP);
    hdc = GpiQueryDevice(hps);
    DevQueryCaps (hdc, CAPS_COLOR_PLANES, 1L, &cColorPlanes);
    DevQueryCaps (hdc, CAPS_COLOR_BITCOUNT, 1L, &cColorBitcount);
    DevQueryCaps (hdc, (LONG)CAPS_VERTICAL_RESOLUTION,(LONG) 1L, &cyRes);
    DevQueryCaps (hdc, CAPS_HORIZONTAL_RESOLUTION, 1L, &cxRes);

    bmp.cbFix = sizeof(BITMAPINFOHEADER);
    bmp.cx = (SHORT)cxWidth;
    bmp.cy = (SHORT)cyHeight;
    bmp.cPlanes = (SHORT)cColorPlanes;
    bmp.cBitCount = (SHORT)cColorBitcount;
    hbm = GpiCreateBitmap(data.hpsBuffer, (PBITMAPINFOHEADER2)&bmp,
                          0x0000, (PBYTE)NULL, (PBITMAPINFO2)NULL);
    GpiSetBitmap (data.hpsBuffer, hbm);

    /*
     * Assume the size of the page rectangle is constrained in the y
     * dimension,compute the x size which would make the rectangle appear
     * square, then check the assumption and do the reverse calculation
     * if necessary.
     */
    cySquare = cyHeight;
    cxSquare = ( cyHeight * cxRes ) / cyRes;

    if (cxWidth < cxSquare)
    {
        cxSquare = cxWidth;
        cySquare = (cxWidth * cyRes) / cxRes;
    }

    /*
     * Fill in the page rectangle and set the page viewport.
     */
    cxEdge = (cxWidth - cxSquare ) / 2;
    cyEdge = (cyHeight - cySquare ) / 2;
    rclPage.xLeft = cxEdge;
    rclPage.xRight = cxWidth - cxEdge;
    rclPage.yBottom = cyEdge;
    rclPage.yTop = cyHeight - cyEdge;

    f = GpiSetPageViewport (data.hpsBuffer, &rclPage);


    *hrgnUsed = CreateRegion(data.hpsBuffer);
    GpiQueryRegionBox(data.hpsBuffer, *hrgnUsed, &rclBounds);

    if (data.hrgnFace != NULL) 
       GpiDestroyRegion(data.hpsBuffer, data.hrgnFace);

    data.hrgnFace = CreateFaceRegion(data.hpsBuffer);
    RECTL rclBox;
    GpiQueryRegionBox(data.hpsBuffer, data.hrgnFace, &rclBox);


    GpiQueryCharBox(data.hpsBuffer, &sizef);
    GpiSetCharBox(data.hpsBuffer, &sizef);

    data.fBufferDirty = TRUE;
}

/****************************************************************\
 *
 *--------------------------------------------------------------
 *
 *  Name: Draw()
 *
 *  Purpose:Draw the clock face ,hand and minute hand,
 *
 *
 *
 *  Usage:
 *
 *  Method:
 *
 *  Returns: void
 *
 *
\****************************************************************/
void Clock:: DrawAnalog()
{
    /* draw the face, the hour hand, and the minute hand */

    DrawRing (data. hpsBuffer, data.vclrRing, data.vclrFace);

    DrawFace (data. hpsBuffer, data.vclrFace);

    DrawHand (data. hpsBuffer, HT_HOUR_SHADE, data.dt.hours, data.vclrFace, data.vclrHands);

    DrawHand (data. hpsBuffer, HT_MINUTE_SHADE, data.dt.minutes, data.vclrFace, data.vclrHands);

    DrawHand (data. hpsBuffer, HT_HOUR, data.dt.hours, data.vclrFace, data.vclrHands);

    DrawHand (data. hpsBuffer, HT_MINUTE, data.dt.minutes, data.vclrFace, data.vclrHands);

    /* draw the tick marks */
    DrawTicks (data. hpsBuffer, CLK_MAJORTICKS, data.vclrMajorTicks, data.vclrMinorTicks);

    DrawTicks (data. hpsBuffer, CLK_MINORTICKS, data.vclrMajorTicks, data.vclrMinorTicks);
}

/****************************************************************\
 *
 *--------------------------------------------------------------
 *
 *  Name: Draw
 *
 *  Purpose: Paint the clock client window.
 *
 *
 *
 *  Usage:
 *
 *  Method:
 *
 *  Returns:  void
 *
 *
\****************************************************************/
void Clock:: Draw( HPS hps, PPOINTL pptlOffset, HRGN hrgnClip)
{
   HRGN hrgnSave;

   // Set the PS into RGB mode!
   GpiCreateLogColorTable (hps, 0, LCOLF_RGB, 0, 0, (PLONG)NULL);

   GpiSetColor(hps, data.vclrFace[SURFACE]);


   if ( data. fBufferDirty )
   {
      GpiPaintRegion(hps, hrgnClip);
      DrawAnalog();
      data. fBufferDirty = FALSE;
    }

   GpiSetClipRegion( hps, hrgnClip, &hrgnSave);
   UpdateScreen (hps, pptlOffset);

   MATRIXLF mtlf;
   POINTL ptlCenter = {0, 0};
   FIXED afxScale[2] = { (rclPage.xRight - rclPage.xLeft) * 0x10000 / 200,
                         (rclPage.yTop - rclPage.yBottom) * 0x10000 / 200 };
   GpiScale(hps, &mtlf, TRANSFORM_REPLACE, afxScale, &ptlCenter);
   GpiTranslate( hps, &mtlf, TRANSFORM_ADD, pptlOffset);
   GpiSetModelTransformMatrix(hps, 9, &mtlf, TRANSFORM_REPLACE);
   DrawHand( hps, HT_SECOND, data.dt.seconds, data.vclrFace, data.vclrHands);

   GpiSetClipRegion(hps, hrgnSave, &hrgnClip);
}

/****************************************************************\
 *  Name:Timer()
 *
 *  Purpose: Handles window timer events
 *
 *  Usage:
 *
 *  Method:
 *
 *  Returns:
\****************************************************************/
HRGN Clock:: Timer ( HPS hps, PPOINTL pptlOffset, HRGN hrgnClip)
{
    HRGN hrgnSave, hrgnInvalid = NULL;

    GpiSetClipRegion( hps, hrgnClip, &hrgnSave);

    // Set the PS into RGB mode!
    GpiCreateLogColorTable (hps, 0, LCOLF_RGB, 0, 0, (PLONG)NULL);

    DATETIME  dtNew;
    static LONG lTimeChangeCheck = 0L;
    LONG  lTime;
    DosGetDateTime ( &dtNew ) ;

    /* get the new time */
    DosGetDateTime (&dtNew);

    /* adjust the hour hand */
    dtNew.hours = (dtNew.hours * (UCHAR) 5 ) % (UCHAR) 60
                  + dtNew.minutes / (UCHAR)12;

    /* if we must move the hour and minute hands, redraw it all */
    if (dtNew.minutes != data. dt.minutes)
    {
        DrawFace (data. hpsBuffer, data.vclrFace);
        DrawHand (data. hpsBuffer, HT_HOUR_SHADE, data.dt.hours, data.vclrFace, data.vclrHands);
        DrawHand (data. hpsBuffer, HT_MINUTE_SHADE, data.dt.minutes, data.vclrFace, data.vclrHands);
        DrawHand (data. hpsBuffer, HT_HOUR, data.dt.hours, data.vclrFace, data.vclrHands);
        DrawHand (data. hpsBuffer, HT_MINUTE, data.dt.minutes, data.vclrFace, data.vclrHands);
        hrgnInvalid = GpiCreateRegion(hps, 0, NULL);
        GpiCombineRegion(hps, hrgnInvalid, data.hrgnFace, 0, CRGN_COPY);
    }
    else /* otherwise just undraw the old second hand and draw the new */
    {
        MATRIXLF mtlf;
        POINTL ptlCenter = {0, 0};
        FIXED afxScale[2] = { (rclPage.xRight - rclPage.xLeft) * 0x10000 / 200,
                              (rclPage.yTop - rclPage.yBottom) * 0x10000 / 200 };
        GpiScale(hps, &mtlf, TRANSFORM_REPLACE, afxScale, &ptlCenter);
        GpiTranslate( hps, &mtlf, TRANSFORM_ADD, pptlOffset);
     
        GpiSetMix(hps, FM_INVERT);
        GpiSetModelTransformMatrix(hps, 9, &mtlf, TRANSFORM_REPLACE);
        DrawHand( hps, HT_SECOND, data. dt.seconds, data.vclrFace, data.vclrHands);
        GpiSetModelTransformMatrix(hps, 9, &mtlf, TRANSFORM_REPLACE);
        DrawHand( hps, HT_SECOND, dtNew.seconds, data.vclrFace, data.vclrHands);
    }
    data. dt = dtNew ;
    GpiSetClipRegion(hps, hrgnSave, &hrgnClip);

    return hrgnInvalid;
}

/****************************************************************\
 *
 *--------------------------------------------------------------
 *
 *  Name:UpdateScreen()
 *
 *  Purpose: Update the screen area.
 *
 *
 *
 *  Usage:
 *
 *  Method:
 *
 *  Returns:
 *
 *
\****************************************************************/
VOID Clock:: UpdateScreen (HPS hps, PPOINTL pptlOffset)
{
    POINTL aptl[3];

    aptl[0].x = this->rclPage.xLeft + pptlOffset->x;
    aptl[0].y = this->rclPage.yBottom + pptlOffset->y;
    aptl[1].x = this->rclPage.xRight + pptlOffset->x;
    aptl[1].y = this->rclPage.yTop + pptlOffset->y;
    aptl[2].x = this->rclPage.xLeft;
    aptl[2].y = this->rclPage.yBottom;

    GpiBitBlt (hps, data.hpsBuffer, 3L, aptl, ROP_SRCCOPY, 0L);
}

/**************************************************************************\
*                                                                          *
*       ROUTINE:    DrawHand                                            *
*                                                                          *
*       COMMENT:    Draws specified hand at specified hour in given PS     *
*                                                                          *
\**************************************************************************/

void Clock:: DrawHand ( HPS hps , SHORT sHandType , SHORT sAngle, LONG vclrFace[], LONG vclrHands[] )
{
    static POINTL aptlHour [ ] = { { 6 , 0 } , { 0 , 62 } , { -6 , 0 } ,
                                   { 0 , -14 } , { 6 , 0 } } ;
    static POINTL aptlHourLine1 [] = {{0L,-10L},{0L,56}};

    static POINTL aptlMinute [ ] = { { 5 , 0 } , { 0 , 77 } , { -5 , 0 } ,
                                     { 0 , -14 } , { 5 , 0 } } ;
    static POINTL aptlMinuteLine1 [] = {{0L,-15L},{0L,72}};

    static POINTL aptlSecond [ ] = { { 0 , -15 } , { 0 , 74 } } ;
    static POINTL ptlOrigin = {0L,0L};

    static LONG cptlHour = sizeof ( aptlHour ) / sizeof ( POINTL ) ;
    static LONG cptlMinute = sizeof ( aptlMinute ) / sizeof ( POINTL ) ;
    static LONG cptlSecond = sizeof ( aptlSecond ) / sizeof ( POINTL ) ;
    BOOL f ;

    static MATRIXLF matlfModel =
    {
        MAKEFIXED ( 1 , 0 ) ,       MAKEFIXED ( 0 , 0 ) ,       0L ,
        MAKEFIXED ( 0 , 0 ) ,       MAKEFIXED ( 1 , 0 ) ,       0L ,
        100L ,                      100L ,                      1L } ;

    static MATRIXLF matlfShade =
    {
        MAKEFIXED ( 1 , 0 ) ,       MAKEFIXED ( 0 , 0 ) ,       0L ,
        MAKEFIXED ( 0 , 0 ) ,       MAKEFIXED ( 1 , 0 ) ,       0L ,
        3L ,                     -3L ,                      1L } ;

    /* prepare a rotation transform and set it into the ps */
    /*      cos x    - sin x    0     *\
    |       sin x      cos x    0      |
    \*      100        100      1     */

    matlfModel.fxM11 =
    matlfModel.fxM22 = fxSin[(sAngle + 15) % 60];
    matlfModel.fxM12 = fxSin[(sAngle + 30) % 60];
    matlfModel.fxM21 = fxSin[sAngle];

    f = GpiSetModelTransformMatrix(hps, (LONG)MATLF_SIZE, &matlfModel,
            (sHandType == HT_SECOND) ? TRANSFORM_PREEMPT : TRANSFORM_REPLACE);

    /* draw the specified hand */

    switch ( sHandType )
    {

        case HT_HOUR:
            GpiSetColor ( hps , vclrHands[SURFACE] ) ;
            GpiBeginPath ( hps , 1L ) ;
            GpiSetCurrentPosition ( hps , aptlHour ) ;
            GpiPolyLine ( hps , cptlHour , aptlHour ) ;
            GpiEndPath ( hps ) ;
            GpiFillPath ( hps , 1L , FPATH_ALTERNATE ) ;
            GpiSetColor ( hps , vclrHands[SHADE]   ) ;
            GpiSetCurrentPosition ( hps , aptlHour ) ;
            GpiPolyLine ( hps , cptlHour , aptlHour ) ;
            GpiSetColor ( hps , vclrHands[SHADE]   ) ;
            GpiSetCurrentPosition ( hps , aptlHourLine1 ) ;
            GpiPolyLine ( hps , 1L , &(aptlHourLine1[1]) ) ;
            break;
        case HT_HOUR_SHADE:
            GpiSetModelTransformMatrix ( hps , ( LONG ) MATLF_SIZE ,
                                              & matlfShade , TRANSFORM_ADD     ) ;
            GpiSetColor ( hps , vclrFace [SHADE]   ) ;
            GpiBeginPath ( hps , 1L ) ;
            GpiSetCurrentPosition ( hps , aptlHour ) ;
            GpiPolyLine ( hps , cptlHour , aptlHour ) ;
            GpiEndPath ( hps ) ;
            GpiFillPath ( hps , 1L , FPATH_ALTERNATE ) ;
            break;

        case HT_MINUTE:
            GpiSetColor ( hps , vclrHands[SURFACE] ) ;
            GpiBeginPath ( hps , 1L ) ;
            GpiSetCurrentPosition ( hps , aptlMinute ) ;
            GpiPolyLine ( hps , cptlMinute , aptlMinute ) ;
            GpiEndPath ( hps ) ;
            GpiFillPath ( hps , 1L , FPATH_ALTERNATE ) ;

            GpiSetColor ( hps , vclrHands[SHADE]   ) ;
            GpiSetCurrentPosition ( hps , aptlMinute ) ;
            GpiPolyLine ( hps , cptlMinute , aptlMinute ) ;
            GpiSetColor ( hps , vclrHands[SHADE]   ) ;
            GpiSetCurrentPosition ( hps , aptlMinuteLine1 ) ;
            GpiPolyLine ( hps , 1L , &(aptlMinuteLine1[1]) ) ;
            GpiSetCurrentPosition ( hps , & ptlOrigin) ;
            GpiFullArc ( hps , DRO_OUTLINEFILL , MAKEFIXED ( 2 , 0 ) ) ;
            break;
        case HT_MINUTE_SHADE:
            GpiSetModelTransformMatrix ( hps , ( LONG ) MATLF_SIZE ,
                                              & matlfShade , TRANSFORM_ADD     ) ;
            GpiSetColor ( hps , vclrFace [SHADE]   ) ;
            GpiBeginPath ( hps , 1L ) ;
            GpiSetCurrentPosition ( hps , aptlMinute ) ;
            GpiPolyLine ( hps , cptlMinute , aptlMinute ) ;
            GpiEndPath ( hps ) ;
            GpiFillPath ( hps , 1L , FPATH_ALTERNATE ) ;
            break;

        case HT_SECOND:
            /* draw in XOR mixmode, so we can undraw later */
            GpiSetMix ( hps , FM_INVERT ) ;
            GpiSetCurrentPosition ( hps , aptlSecond ) ;
            GpiPolyLine ( hps , cptlSecond , aptlSecond ) ;
            GpiSetMix ( hps , FM_OVERPAINT ) ;
            break;
    }
}

/****************************************************************\
 *
 *--------------------------------------------------------------
 *
 *  Name:DrawFace()
 *
 *  Purpose:
 *
 *
 *
 *  Usage:
 *
 *  Method:
 *
 *  Returns: void
 *
 *
\****************************************************************/
void Clock:: DrawFace ( HPS hps, LONG vclrFace[] )
{
    BOOL f ;
    static POINTL ptlLight= { -2L , 2L } ;
    static POINTL ptlShade= {  2L, -2L } ;
    static POINTL ptlFace = { 0L, 0L};
    /*        1         0      0     *\
    *         0         1      0      *
    \*      100       100      1     */

    static MATRIXLF matlfModel = {
        MAKEFIXED ( 1 , 0 ) ,       MAKEFIXED ( 0 , 0 ) ,       0L ,
        MAKEFIXED ( 0 , 0 ) ,       MAKEFIXED ( 1 , 0 ) ,       0L ,
        100L ,                      100L ,                      1L } ;

    /* center at (100, 100) and draw the clock face */
    f = GpiSetModelTransformMatrix ( hps , ( LONG ) MATLF_SIZE ,
                                     & matlfModel , TRANSFORM_REPLACE ) ;


    GpiSetColor ( hps , vclrFace[SURFACE]);
    GpiSetCurrentPosition ( hps , & ptlFace ) ;
    GpiFullArc ( hps , DRO_OUTLINEFILL , MAKEFIXED ( 80, 0 ) ) ;

    FATTRS fat;
    POINTL grad = { 1, 4 };
    POINTL aptlPoints[TXTBOX_COUNT];
    LONG lcid = 101;
    memset(&fat, 0, sizeof(FATTRS));
    fat.usRecordLength = sizeof(FATTRS);
    strcpy(fat.szFacename, "Helv Bold");
    fat.fsFontUse = FATTR_FONTUSE_OUTLINE;
    if (FONT_MATCH == GpiCreateLogFont( hps, NULL, lcid, &fat))
    {
       FONTMETRICS fm;
       GpiSetCharSet(hps, lcid);
       GpiQueryFontMetrics(hps, sizeof(fm), &fm);
       GpiSetCharShear(hps, &grad);
       SIZEF sizfxBox ;
       sizfxBox.cy = MAKEFIXED(20,0);
       sizfxBox.cx = 20 * fm.lEmHeight * 0x10000 / fm.lEmInc;
       GpiSetCharBox(hps, &sizfxBox);
    }
    POINTL ptlString ;
    GpiSetColor ( hps, vclrFace[SHADE]);
    GpiSetTextAlignment(hps, TA_CENTER, TA_HALF);
    ptlString.x = 0;
    ptlString.y = 40;
    GpiCharStringAt ( hps, &ptlString, 7, "OpenDoc");
    ptlString.y = -45;
    GpiCharStringAt ( hps, &ptlString, 4, "OS/2");

    GpiDeleteSetId(hps, lcid);
}

/****************************************************************\
 *
 *--------------------------------------------------------------
 *
 *  Name:CreateFaceRegion()
 *
 *  Purpose:
 *
 *
 *
 *  Usage:
 *
 *  Method:
 *
 *  Returns: void
 *
 *
\****************************************************************/
HRGN Clock:: CreateFaceRegion ( HPS hps )
{
    BOOL f ;
    static POINTL ptlFace = { 0L, 0L};
    /*        1         0      0     *\
    *         0         1      0      *
    \*      100       100      1     */

    static MATRIXLF matlfModel = {
        MAKEFIXED ( 1 , 0 ) ,       MAKEFIXED ( 0 , 0 ) ,       0L ,
        MAKEFIXED ( 0 , 0 ) ,       MAKEFIXED ( 1 , 0 ) ,       0L ,
        100L ,                      100L ,                      1L } ;

    /* center at (100, 100) and draw the clock face */
    f = GpiSetModelTransformMatrix ( hps , ( LONG ) MATLF_SIZE ,
                                     & matlfModel , TRANSFORM_REPLACE ) ;

    GpiBeginPath(hps, 1);
    GpiSetCurrentPosition ( hps , & ptlFace ) ;
    GpiFullArc ( hps , DRO_OUTLINE , MAKEFIXED ( 80, 0 ) ) ;
    GpiEndPath (hps);
    return GpiPathToRegion(hps, 1, FPATH_ALTERNATE);
}

void Clock:: DrawRing ( HPS hps, LONG vclrRing[], LONG vclrFace[] )
{
    BOOL f ;
    static POINTL ptlLight= { -2L , 2L } ;
    static POINTL ptlShade= {  1L, -1L } ;
    static POINTL ptlFace = { 0L, 0L};
    static POINTL ptlShadeIn= { -3L , 3L } ;
    static POINTL ptlLightIn= {  1L, -1L } ;
    /*        1         0      0     *\
    *         0         1      0      *
    \*      100       100      1     */

    static MATRIXLF matlfModel = {
        MAKEFIXED ( 1 , 0 ) ,       MAKEFIXED ( 0 , 0 ) ,       0L ,
        MAKEFIXED ( 0 , 0 ) ,       MAKEFIXED ( 1 , 0 ) ,       0L ,
        100L ,                      100L ,                      1L } ;

    /* center at (100, 100) and draw the clock face */
    f = GpiSetModelTransformMatrix ( hps , ( LONG ) MATLF_SIZE ,
                                     & matlfModel , TRANSFORM_REPLACE ) ;

    DrawFullRing(hps,
                    &ptlShade,
                    MAKEFIXED(95,0),
                    MAKEFIXED(99 ,0),
                    vclrRing[SHADE]);

    DrawFullRing(hps,
                    &ptlLight,
                    MAKEFIXED(95,0),
                    MAKEFIXED(98 ,0),
                    vclrRing[LIGHT]   );

    DrawFullRing(hps,
                    &ptlLightIn,
                    MAKEFIXED(88,0),
                    MAKEFIXED(94,0),
                    vclrRing[LIGHT]   );


    DrawFullRing(hps,
                    &ptlShadeIn,
                    MAKEFIXED(86,0),
                    MAKEFIXED(93,0),
                    vclrRing[SHADE]);

    DrawFullRing(hps,
                    &ptlFace ,
                    MAKEFIXED(94,0),
                    MAKEFIXED(98 ,0),
                    vclrRing[SURFACE]);

    GpiSetColor ( hps , vclrFace[SURFACE]);
    GpiSetCurrentPosition ( hps , & ptlFace ) ;
    GpiFullArc ( hps , DRO_OUTLINEFILL , MAKEFIXED ( 91, 0 ) ) ;
}

void Clock:: DrawFullCircle(HPS hps, PPOINTL pptlCenter, FIXED fxRad)
{
    GpiSetCurrentPosition ( hps , pptlCenter );
    GpiBeginPath(hps,1L);
    GpiFullArc ( hps , DRO_OUTLINE ,  fxRad  ) ;
    GpiEndPath(hps);
}

/****************************************************************\
 *
 *--------------------------------------------------------------
 *
 *  Name:GetArrangedDate()
 *
 *  Purpose:
 *
 *
 *
 *  Usage:
 *
 *  Method:
 *
 *  Returns: VOID
 *
\****************************************************************/
VOID Clock:: GetArrangedDate(CHAR achFinalDate[])
{
    time_t     tTime;
    struct tm  *pLocalTime;

    time(&tTime);
    pLocalTime = localtime(&tTime);
    strftime(achFinalDate, sizeof(achFinalDate), "%m %d %y", pLocalTime);

    /*put date separators*/
    achFinalDate[2] = achFinalDate[5] = '/';
    achFinalDate[8] = '\0';
}

/****************************************************************\
 *
 *--------------------------------------------------------------
 *
 *  Name:DrawTicks()
 *
 *  Purpose:Draws Clock Ticks()
 *
 *
 *
 *  Usage:
 *
 *  Method:
 *
 *  Returns: void
 *
 *
\****************************************************************/
void Clock:: DrawTicks ( HPS hps , USHORT usTicks, LONG vclrMajorTicks[], LONG vclrMinorTicks[] )
{
    BOOL f ;
    USHORT usAngle,usTrapez ;

    /* prepare a transform to use when rotating the ticks */
    /*      cos x    - sin x    0     *\
    |       sin x      cos x    0      |
    \*      100        100      1     */

    static MATRIXLF matlfModel = {
        MAKEFIXED ( 1 , 0 ) ,       MAKEFIXED ( 0 , 0 ) ,       0L ,
        MAKEFIXED ( 0 , 0 ) ,       MAKEFIXED ( 1 , 0 ) ,       0L ,
        100L ,                      100L ,                      1L } ;

    /* define what the tick marks look like */
    static POINTL aptlMajorTick [ ] = { { -3 , 94 } , { 3 , 100 } } ;
    static BYTE   aclr [12][4] = {
            /*12*/        {SHADE,LIGHT,LIGHT,SHADE},
            /*1*/         {LIGHT,LIGHT,LIGHT,SHADE},
            /*2*/         {LIGHT,LIGHT,SHADE,SHADE},
            /*3*/         {LIGHT,LIGHT,SHADE,SHADE},
            /*4*/         {LIGHT,LIGHT,SHADE,LIGHT},
            /*5*/         {LIGHT,SHADE,SHADE,LIGHT},
            /*6*/         {LIGHT,SHADE,SHADE,LIGHT},
            /*7*/         {LIGHT,SHADE,SHADE,LIGHT},
            /*8*/         {LIGHT,SHADE,LIGHT,LIGHT},
            /*9*/         {SHADE,SHADE,LIGHT,LIGHT},
            /*9*/         {SHADE,SHADE,LIGHT,LIGHT},
            /*11*/        {SHADE,LIGHT,LIGHT,SHADE}
                                 };
    static  POINTL aptlMT  [4][4] = {
                                  { {-3,81},{-1,83},{1,83},{3,81  }} ,
                                  { {-3,81},{-1,83},{-1,87},{-3,89 }} ,
                                  { {-3,89},{-1,87},{1,87},{3,89 }} ,
                                  { {3,89},{1,87},{1,83},{3,81  }}
                                 };
    static POINTL aptlMajorTickShadow [ ] = { { -1 , 83 } , { 1 , 87  } } ;
    static POINTL aptlMinorTick [ ] = { { 0 , 83 } , { 0 , 85 } } ;

    /* have we been asked to draw the major ticks? */
    if ( usTicks & CLK_MAJORTICKS )
    {
        for ( usAngle = 0 ; usAngle < 60 ; usAngle += 5 ) 
        {
            /* set the rotation transform */
            matlfModel . fxM11 =
            matlfModel . fxM22 = fxSin [ ( usAngle + 15 ) % 60 ] ;
            matlfModel . fxM12 = fxSin [ ( usAngle + 30 ) % 60 ] ;
            matlfModel . fxM21 = fxSin [ usAngle ] ;
            f = GpiSetModelTransformMatrix ( hps , ( LONG ) MATLF_SIZE ,
                                             & matlfModel ,
                                             TRANSFORM_REPLACE ) ;

            /* draw a major tick mark */
            for (usTrapez = 0; usTrapez < 4; usTrapez++) 
            {
                DrawTrapez(hps,aptlMT[usTrapez],vclrMajorTicks[aclr[usAngle/5][usTrapez]]);
            }
            GpiSetColor ( hps , vclrMajorTicks[SURFACE]) ;
            GpiSetCurrentPosition ( hps , & aptlMajorTickShadow [ 0 ] ) ;
            GpiBox ( hps , DRO_FILL , & aptlMajorTickShadow [ 1 ] , 0L , 0L ) ;
        }
    }

    /* have we been asked to draw the minor ticks? */
    /* draw in the default color */
    GpiSetColor ( hps ,vclrMinorTicks[SHADE]  ) ;
    if ( usTicks & CLK_MINORTICKS )
    {
        for ( usAngle = 0 ; usAngle < 60 ; usAngle ++ ) 
        {
            if ((usAngle % 5) != 0) 
            {
                matlfModel . fxM11 =
                matlfModel . fxM22 = fxSin [ ( usAngle + 15 ) % 60 ] ;
                matlfModel . fxM12 = fxSin [ ( usAngle + 30 ) % 60 ] ;
                matlfModel . fxM21 = fxSin [ usAngle ] ;

                /* set the transform */
                f = GpiSetModelTransformMatrix ( hps , ( LONG ) MATLF_SIZE ,
                                                 & matlfModel ,
                                                 TRANSFORM_REPLACE ) ;

                /* draw a minor tick mark */
                GpiSetCurrentPosition ( hps , & aptlMinorTick [ 0 ] ) ;
                GpiBox ( hps , DRO_OUTLINEFILL , & aptlMinorTick [ 1 ] , 0L , 0L ) ;
            }
       }
    }
}
/****************************************************************\
 *
 *--------------------------------------------------------------
 *
 *  Name:DrawTrapez()
 *
 *  Purpose:
 *
 *
 *
 *  Usage:
 *
 *  Method:
 *
 *  Returns:
 *
 *
\****************************************************************/
void Clock:: DrawTrapez(HPS hps,POINTL *aptl,LONG color)
{
    GpiSetColor ( hps,color) ;
    GpiBeginPath(hps, 1L);                  /* start the path bracket */
    GpiSetCurrentPosition ( hps ,  aptl  ) ;
    GpiPolyLine(hps, 3L,&(aptl[1]) );      /* draw the three sides   */
    GpiCloseFigure(hps);                    /* close the triangle     */
    GpiEndPath(hps);                        /* end the path bracket   */
    GpiFillPath(hps, 1L, FPATH_ALTERNATE);  /* draw and fill the path */
}
/****************************************************************\
 *
 *--------------------------------------------------------------
 *
 *  Name:DrawFullRing()
 *
 *  Purpose: This routine draws the ring for the clock face.
 *
 *
 *
 *  Usage:
 *
 *  Method:
 *
 *  Returns:
 *
 *
\****************************************************************/
void Clock:: DrawFullRing(HPS hps,PPOINTL pptlCenter,FIXED fxRadIn, FIXED fxRadOut,
                     LONG lColor)
{
    GpiSetColor(hps,lColor);
    GpiSetCurrentPosition ( hps , pptlCenter );
    GpiBeginPath(hps,1L);
    GpiFullArc ( hps , DRO_OUTLINE ,  fxRadIn  ) ;
    GpiFullArc ( hps , DRO_OUTLINE ,  fxRadOut  ) ;
    GpiCloseFigure(hps);
    GpiEndPath(hps);
    GpiFillPath ( hps , 1L , FPATH_ALTERNATE ) ;
}

/****************************************************************\
 *
 *--------------------------------------------------------------
 *
 *  Name: SetRGBColors
 *
 *  Purpose: Set the RGB schema so that each time a user changes the
 *           color we update it here.
 *
 *
 *  Usage:
 *
 *  Method:
 *
 *  Returns:VOID
 *
 *
\****************************************************************/
VOID Clock:: SetRGBColors()
{

    data.vclrFace[SURFACE]       = data.cp.clrFace;
    data.vclrBG[SURFACE]         = data.cp.clrBackground;
    data.vclrHands[SURFACE]      = data.cp.clrMinuteHand;
    data.vclrMajorTicks[SURFACE] = data.cp.clrHourHand;

    /*Fill color tables*/
    ShadeLight(data.vclrMajorTicks);

    data.vclrMinorTicks[SURFACE] = data.vclrFace[SURFACE];
    ShadeLight(data.vclrMinorTicks);

    ShadeLight(data.vclrFace);

    data.vclrRing[SURFACE] = data.vclrFace[SURFACE];
    ShadeLight(data.vclrRing);

    ShadeLight(data.vclrHands);
    data.vclrHands[SHADE] = RGB_BLACK;

    ShadeLight(data.vclrBG);
}
/****************************************************************\
 *
 *--------------------------------------------------------------
 *
 *  Name:ShadeLight()
 *
 *  Purpose: Find the shade and light color index values and
 *           install them in the colors   array of the element
 *
 *
 *
 *  Usage:
 *
 *  Method:
 *
 *  Returns:
 *
 *
\****************************************************************/
VOID Clock:: ShadeLight(PLONG nplColors)
{
   typedef  union  _RGBLONG
   {
        RGB rgb;
        LONG lng;
   } RGBLONG ;
   RGBLONG  rgbSurface,rgbShade,rgbLight;

   rgbSurface.lng = rgbShade.lng = rgbLight.lng = 0L;

   rgbSurface.lng = nplColors[SURFACE];
   rgbShade.rgb.bBlue = ShadeRGBByte(rgbSurface.rgb.bBlue);
   rgbShade.rgb.bRed = ShadeRGBByte(rgbSurface.rgb.bRed);
   rgbShade.rgb.bGreen = ShadeRGBByte(rgbSurface.rgb.bGreen);
   rgbLight.rgb.bBlue = LightRGBByte(rgbSurface.rgb.bBlue);
   rgbLight.rgb.bRed = LightRGBByte(rgbSurface.rgb.bRed);
   rgbLight.rgb.bGreen = LightRGBByte(rgbSurface.rgb.bGreen);
   nplColors[SHADE] = rgbShade.lng;
   nplColors[LIGHT] = rgbLight.lng;
}


/****************************************************************\
 *
 *--------------------------------------------------------------
 *
 *  Name:ShadeRGBByte
 *
 *  Purpose:
 *
 *
 *
 *  Usage:
 *
 *  Method:
 *
 *  Returns:
 *
 *
\****************************************************************/

BYTE Clock:: ShadeRGBByte(BYTE brgb)
{
  #define SHADER   ( (BYTE) 0x50)

  if (brgb > SHADER)
  {
     return (brgb - SHADER);
  }
  else
  {
     return (0);
  }

}
/****************************************************************\
 *
 *--------------------------------------------------------------
 *
 *  Name:LightRGBByte
 *
 *  Purpose:
 *
 *
 *
 *  Usage:
 *
 *  Method:
 *
 *  Returns:
 *
 *
\****************************************************************/
BYTE Clock:: LightRGBByte(BYTE brgb)
{

  #define LIGHTER  ( (BYTE) 0x40)

  if (brgb < (0xff - LIGHTER) )
  {
     return (brgb + LIGHTER);
  }

  else return (0xff);

}

