/********************************************************************/ 
/*  Licensed Materials - Property of IBM                            */ 
/*                                                                  */ 
/*                                                                  */ 
/* Copyright (C) International Business Machines Corp., 1994.       */ 
/* Copyright (C) Apple Computer, Inc., 1994                         */ 
/*                                                                  */ 
/*  US Government Users Restricted Rights -                         */ 
/*  Use, duplication, or disclosure restricted                      */ 
/*  by GSA ADP Schedule Contract with IBM Corp.                     */ 
/*                                                                  */ 
/********************************************************************/ 
/*
  File:    XMPMem.h

  Contains:  Memory allocation routines

  Written by:  Nick Pilch

*/

#ifndef _XMPMEM_
#define _XMPMEM_

#ifndef _XMPTYPES_
#include "XMPTypes.h"
#endif

//==============================================================================
// Theory of Operation
//==============================================================================

/*
  Abstract memory routines.
*/

//==============================================================================
// Constants
//==============================================================================

//==============================================================================
// Scalar Types
//==============================================================================

typedef const void *ConstXMPPtr;

//==============================================================================
// Classes defined in this interface
//==============================================================================

//==============================================================================
// Classes used by this interface
//==============================================================================

class XMPObject;

//==============================================================================
// Routines
//==============================================================================

XMPPtr XMPNewPtr(XMPULong howBig);
XMPPtr XMPNewPtr(XMPULong howBig, XMPHeap heap);
    // Create a pointer in the specified heap
XMPPtr XMPNewPtrClear(XMPULong howBig);
XMPPtr XMPNewPtrClear(XMPULong howBig, XMPHeap heap);
XMPPtr XMPNewPtrSys(XMPULong howBig);
    // Create the pointer in the system heap
void XMPDisposePtr(XMPPtr pointer);
XMPHeap  XMPGetHeap(XMPPtr pointer);
    // Get the heap from which the pointer was allocated
XMPHeap XMPGetSystemHeap();
    // Get a reference to the system heap.
void XMPBlockMove(ConstXMPPtr from, XMPPtr to, XMPULong size);
    // Move data. !!Parameters are in opposite order from memcpy/memmove!!
XMPHandle XMPNewHandle(XMPULong howBig);
void XMPDisposeHandle(XMPHandle handle);
XMPHandle XMPCopyHandle( XMPHandle handle );
XMPULong XMPGetHandleSize(XMPHandle handle);
void XMPSetHandleSize(XMPHandle handle, XMPULong size);
void* XMPLockHandle(XMPHandle handle);
void XMPUnlockHandle(void* ptr);
void XMPUnlockHandle(XMPHandle handle);

#if XMPDebug
  // Debugging Utilities:
  
  XMPULong  XMPGetPtrSize( XMPPtr );
  
  XMPBoolean  XMPValidateHeap( XMPHeap heap = kXMPNULL );
  XMPBoolean  XMPValidateAllHeaps( );
  XMPBoolean  XMPValidateAllObjects( );
  XMPBoolean  XMPValidatePtr( XMPPtr block );
  XMPBoolean  XMPValidateObject( XMPObject *obj );
  XMPBoolean  XMPValidateHandle( XMPHandle h );
  
  void    XMPZapMem( XMPPtr anywhere, XMPULong size );
  XMPBoolean  XMPZapBlock( XMPPtr block );
  
  // Wrap these two calls around code you want to torture-test for memory problems.
  //  * All input pointers and handles will be checked for validity.
  //  * ASLM heaps will be checked for validity on every memory call.
  //  * New & disposed blocks will be filled with a repeating pattern of 0x50FF8001.
  //  * Warning if XMPUnlockHandle(void*) called on already-unlocked handle.
  void    XMPBeginMemValidation( );
  void    XMPEndMemValidation( );
#endif

#endif // _XMPMEM_