/********************************************************************/ 
/*  Licensed Materials - Property of IBM                            */ 
/*                                                                  */ 
/*                                                                  */ 
/* Copyright (C) International Business Machines Corp., 1994.       */ 
/* Copyright (C) Apple Computer, Inc., 1994                         */ 
/*                                                                  */ 
/*  US Government Users Restricted Rights -                         */ 
/*  Use, duplication, or disclosure restricted                      */ 
/*  by GSA ADP Schedule Contract with IBM Corp.                     */ 
/*                                                                  */ 
/********************************************************************/ 
/*
  File:    StrHshTb.h

  Contains:  Interface to XMPHashTable class.

  Written by:  Nick Pilch

  Copyright:   1993 by Apple Computer, Inc., all rights reserved.

  Change History (most recent first):

    <15>    2/7/94  JA    Tiger Team Makeover!
    <14>    2/3/94  CG    Added fHeap to XMPStringHashTable.
    <13>  11/15/93  NP    Added GetNumEntries method.
    <12>   11/5/93  NP    Added comments and GetNumEntries method.
    <11>   11/2/93  NP    Updated comments.
    <10>  10/22/93  NP    Changed for new protocol-only throw for out
                  of memory.
     <9>   9/21/93  CG    Added valuelength param to methods.
     <8>    8/9/93  NP    Added class id string.
     <7>   7/21/93  NP    Added a virtual keyword (temporary for
                  ASLM).
     <6>    7/2/93  NP    Mods to prevent shared memory problems.
     <5>   6/22/93  NP    Changed iterator semantics. Added comments.
     <4>    6/4/93  NP    Added iterator class. Fixed comments.
     <3>   4/28/93  NP    File name changes.
     <2>   4/26/93  NP    It's working now.
     <1>   4/16/93  NP    first checked in

  To Do:
*/

#ifndef _STRHSHTB_
#define _STRHSHTB_

#ifndef _XMPTYPES_
#include "XMPTypes.h"
#endif

#ifndef __STDIO__
#include <stdio.h>
#endif

//==============================================================================
// Theory of Operation
//==============================================================================

/*
  A simple hash table for keys of variable length. Value is 8 bytes. First
  four bytes are a pointer to the storage for the value, the next four bytes
  are the length of that storage. The caller must allocate the storage.
  Implemented with chaining. Keys are bytes streams. Embedded NULLs are
  allowed. However, keys of zero length are not allowed.
*/

//==============================================================================
// Scalar Types
//==============================================================================

typedef XMPULong (*XMPStringHashTableFunc)(XMPUByte* string,
                      XMPULong stringLength,
                      XMPULong& hashValueLowerBounds,
                      XMPULong& hashValueUpperBounds);

  // this is the hash function prototype.

//==============================================================================
// Classes defined in this interface
//==============================================================================

class  XMPStringHashTable;
class  XMPStringHashTableIterator;

//==============================================================================
// Classes used by this interface
//==============================================================================

class  LinkedNodesIterator;
struct  HashEntryRec;
struct  EntryNode;

//==============================================================================
// XMPStringHashTable
//==============================================================================

#define kXMPStringHashTableID "appl:xmpstringhashtable$class,1.0.0"

class XMPStringHashTable
{
  friend class LinkedNodesIterator;
  friend class XMPStringHashTableIterator;

  public:

  XMPStringHashTable(XMPULong numEntries);
    // numEntries should be the expected number of table entries. The table
    //  will not grow or shrink. If zero is passed, the table is initialized
    //  with 1 entry. (not very interesting).

  XMPVMethod void Initialize(XMPHeap heap);
    // kXMPErrOutOfMemory will be thrown if the table could not be created.

  virtual ~XMPStringHashTable();


  XMPVMethod void      Insert(XMPUByte* string, XMPULong stringLength,
                    XMPPtr value, XMPULong valueLength);
    // If this key already exists, that value is replaced by the new value.
    //   kXMPErrOutOfMemory will be thrown if the entry could not be inserted.
    //  kXMPErrInvalidKey will be thrown if stringLength is zero.
    //  The key is copied into the table, the value is not.
    //  If an exception is 

  XMPVMethod void      Remove(XMPUByte* string, XMPULong stringLength);
    // No exception is thrown if the key does not exist.
    //  kXMPErrInvalidKey will be thrown if stringLength is zero.

  XMPVMethod XMPBoolean     Find(XMPUByte* string, XMPULong stringLength,
                    XMPPtr* value, XMPULong* valueLength);  
    // kXMPTrue is returned if the value is found, kXMPFalse otherwise.
    //  If the entry is not found. *value and *valueLength are not
    //  guaranteed to contain any useful information.
    //  kXMPErrInvalidKey will be thrown if stringLength is zero.

  XMPVMethod XMPULong     GetNumEntries();  
    // Return the number of entries.

  XMPVMethod void    PrintTable(FILE* outfile);
  XMPVMethod void    PrintDistribution(FILE* outfile);
  
    // for debugging / testing

  XMPStringHashTableFunc  GetHashFunc();
  void          SetHashFunc(XMPStringHashTableFunc func);

    // get and set the hash function.

  protected:
    XMPHeap     fHeap;

  HashEntryRec*  fTable; // pointer to start of table,
              //  fNumSlots * sizeof(HashEntry) long. Initialized to
              //  all zeroes.

  XMPULong    fNumSlots; // Number of slots in the table, not number of
                //  entries.

  XMPULong    fNumEntries;  // Actual number of entries.

  XMPStringHashTableFunc  fHashFunc;
  
  XMPVMethod HashEntryRec*  GetTable();
  XMPVMethod XMPULong      GetNumSlots();

  static XMPULong    StdHash(XMPUByte* string, XMPULong stringLength,
                      XMPULong& hashValueLowerBounds,
                      XMPULong& hashValueUpperBounds);

  XMPVMethod XMPULong  MapToTableIndex(XMPULong hash,
                      XMPULong hashValueLowerBounds,
                      XMPULong hashValueUpperBounds);

  XMPVMethod XMPULong  HashAndMap(XMPUByte* string, XMPULong stringLength);
  XMPVMethod void    InsertAtIndex(XMPULong index, XMPUByte* string,
                        XMPULong stringLength,
                        XMPPtr value,
                        XMPULong valueLength);
  XMPVMethod void    RemoveAtIndex(XMPULong index, XMPUByte* string,
                        XMPULong stringLength);
  EntryNode*      MakeNewNode(XMPUByte* string, XMPULong stringLength,
                    XMPPtr value, XMPULong valueLength);
  XMPUByte*      MakeStringFromBytes(XMPUByte* string,
                          XMPULong stringLength);
};

//==============================================================================
// XMPStringHashTableIterator
//
//  This iterator is only meant to be used in the the context of a for loop,
//  e.g.:
//
//  XMPStringHashTableIterator iter;
//  for (iter.First(string, len, value, valueLen);
//      iter.IsNotComplete();
//      iter.Next(string, len, value, valueLen))
//  {
//    ...
//  }
//
//==============================================================================

#define kXMPStringHashTableIteratorID "appl:xmpstringhashtableiterator$class,1.0.0"

class XMPStringHashTableIterator
{
  public:
  XMPStringHashTableIterator(XMPStringHashTable* tb);
  ~XMPStringHashTableIterator();
  void    First(XMPUByte** string, XMPULong* len, XMPPtr* value,
            XMPULong* valueLen);
    // A pointer to the private copy of the string is returned. The user
    //  must copy the storage if he or she desires to change it.
    //  If there are no entries in the table, *string will be set to
    //  kXMPNULL and *len set to 0.
  void    Next(XMPUByte** string, XMPULong* len, XMPPtr* value,
            XMPULong* valueLen);
    // A pointer to the private copy of the string is returned. The user
    //  must copy the storage if he or she desires to change it.
    //  If there are no more entries in the table, *string will be set to
    //  kXMPNULL and *len set to 0.
  XMPBoolean  IsNotComplete();
  private:
  XMPBoolean  GetNext(XMPUByte** string, XMPULong* len,
            XMPPtr* value, XMPULong* valueLen);
  XMPStringHashTable*  fTable;
  XMPULong      fTableIndex;
  EntryNode*      fCurNode;
  XMPBoolean      fDone;
};

#endif // _STRHSHTB_