/********************************************************************/ 
/*  Licensed Materials - Property of IBM                            */ 
/*                                                                  */ 
/*                                                                  */ 
/* Copyright (C) International Business Machines Corp., 1994.       */ 
/* Copyright (C) Apple Computer, Inc., 1994                         */ 
/*                                                                  */ 
/*  US Government Users Restricted Rights -                         */ 
/*  Use, duplication, or disclosure restricted                      */ 
/*  by GSA ADP Schedule Contract with IBM Corp.                     */ 
/*                                                                  */ 
/********************************************************************/ 
/*
  File:    StorageU.h

  Contains:  definition of XMPStorageUnit

  Written by: Vincent Lo, Tantek I. elik

*/

#ifndef _STORAGEU_
#define _STORAGEU_

#ifndef _XMPTYPES_
#include "XMPTypes.h"
#endif

#ifndef _REFCTOBJ_
#include "RefCtObj.h"
#endif

#ifndef __CM_API__
#include "CMAPI.h"
#endif

//==============================================================================
// Constants
//==============================================================================

typedef enum {
  kXMPPosUndefined    = -1,
  kXMPPosSame        = 0x00,
  kXMPPosAll        = 1,
  kXMPPosFirstSib      = 2,
  kXMPPosLastSib      = 3,
  kXMPPosNextSib      = 4,
  kXMPPosPrevSib      = 5,
  kXMPPosFirstBelow    = 6,
  kXMPPosLastBelow    = 7,
  kXMPPosFirstAbove     = 8,
  kXMPPosLastAbove    = 9,
  kXMPPosReserved11     = 10,
  kXMPPosReserved12    = 11,
  kXMPPosReserved13    = 12,
  kXMPPosReserved14    = 13,
  kXMPPosReserved15    = 14,
  
  kXMPPosMWrap      = 0x10,
  kXMPPosMOtherTypes    = 0x20
} XMPPositionCode;

const  XMPID      kXMPIDAll  = 0;
const  XMPID      kXMPIndexAll= 0;
extern const XMPValueType  kXMPTypeAll; // = 0;
const  XMPID      kXMPIDWild  = 0;

//==============================================================================
// Scalar Types
//==============================================================================

typedef  void*    XMPValue;
typedef  XMPID    XMPValueIndex;
typedef  XMPISOStr  XMPPropertyName;
typedef XMPID    XMPStorageUnitID;
typedef  XMPISOStr  XMPStorageUnitName;
typedef XMPID    XMPStorageUnitRef;
typedef XMPULong  XMPStorageUnitKey;
typedef XMPULong  XMPDraftKey;

//==============================================================================
// Classes defined in this interface
//==============================================================================

class  XMPStorageUnit;

//==============================================================================
// Classes used by this interface
//==============================================================================

class  XMPDraft;
class  XMPSession;
class  XMPPart;
class  XMPStorageUnitCursor;
class  XMPStorageUnitView;
class  XMPAEHashTable;

class  XMPRefTable;
class  XMPRefCollection;

//==============================================================================
// XMPStorageUnit
//==============================================================================

#define kXMPStorageUnitID "appl:xmpstorageunit$class,1.0.0"

class XMPStorageUnit : public XMPRefCntObject {

  friend class XMPStorageUnitRefIterator;

public:

  XMPMethod  XMPDraft*    GetDraft();
  
  XMPMethod  XMPSession*    GetSession();
      
  XMPMethod  void IncrementRefCount();
      
  XMPMethod  void Release();

  XMPMethod XMPSize Purge(XMPSize size);
  
  XMPMethod  XMPBoolean  Exists(XMPPropertyName propertyName,
                  XMPValueType valueType,
                  XMPValueIndex valueIndex);

  XMPMethod  XMPBoolean  Exists(XMPStorageUnitCursor* cursor);

  XMPMethod  XMPULong  CountProperties();
  XMPMethod  XMPULong  CountValues();
  
  XMPMethod  XMPStorageUnit*  Focus(XMPPropertyName propertyName,
                    XMPPositionCode propertyPosCode,
                    XMPValueType valueType,
                    XMPValueIndex valueIndex,
                    XMPPositionCode  valuePosCode);

  XMPMethod  XMPStorageUnit*  Focus(XMPStorageUnitCursor* cursor);
  
  XMPMethod  XMPStorageUnitCursor*  CreateCursor();
  
  XMPMethod  XMPStorageUnit*  Externalize();  
      
  XMPMethod   XMPStorageUnit*  Internalize();  
    
  XMPMethod  XMPID      GetID();    

  XMPMethod  XMPStorageUnitName GetName();    
    
  XMPMethod  void SetName(XMPStorageUnitName name);

  XMPMethod  XMPStorageUnit*  AddProperty(XMPPropertyName propertyName);  
  
  XMPMethod  XMPStorageUnit* AddValue(XMPValueType type);

  XMPMethod  XMPStorageUnit*  Remove();

  XMPMethod void      CopyTo(XMPStorageUnit* toSU);
    
  XMPMethod XMPStorageUnit*  CloneTo(XMPDraftKey key, XMPDraft* destDraft, XMPStorageUnit* initiatingFrame);

  XMPMethod void      CloneInto(XMPDraftKey key, XMPStorageUnit* destStorageUnit, XMPStorageUnit* initiatingFrame);
  
  XMPMethod  XMPStorageUnitView*  CreateView(XMPStorageUnitCursor* cursor);

  XMPMethod XMPPropertyName  GetProperty();

  XMPMethod XMPValueType  GetType();
  XMPMethod void      SetType(XMPValueType valueType);

  XMPMethod void      SetOffset(XMPULong offset);
  XMPMethod XMPULong    GetOffset();
  
  XMPMethod XMPULong    GetValue(XMPULong length, XMPValue value);
  XMPMethod void      SetValue(XMPULong length, XMPValue value);
  XMPMethod void      InsertValue(XMPULong length, XMPValue value);
  XMPMethod void      DeleteValue(XMPULong length);
  
  XMPMethod XMPULong    GetSize();
  
  XMPMethod XMPStorageUnitRef  GetStrongStorageUnitRef(XMPStorageUnit* embeddedSU);
  XMPMethod XMPStorageUnitRef  GetWeakStorageUnitRef(XMPStorageUnit* embeddedSU);
  XMPMethod XMPBoolean    IsStrongStorageUnitRef(XMPStorageUnitRef ref);
  XMPMethod XMPBoolean    IsWeakStorageUnitRef(XMPStorageUnitRef ref);
  XMPMethod XMPStorageUnit*  RemoveStorageUnitRef(XMPStorageUnitRef aRef);
  XMPMethod XMPStorageUnitID  GetIDFromStorageUnitRef(XMPStorageUnitRef aRef);
  
  XMPMethod XMPStorageUnitRefIterator* GetStorageUnitRefIterator();
  
  XMPMethod  XMPULong      GetGenerationNumber();
  XMPMethod  XMPULong      IncrementGenerationNumber();

  XMPMethod  XMPBoolean  IsPromiseValue();
  XMPMethod  void    SetPromiseValue(XMPValueType valueType,
                      XMPULong offset,
                      XMPULong length,
                      XMPValue value,
                      XMPPart *sourcePart);
  XMPMethod  XMPULong  GetPromiseValue(XMPValueType valueType,
                      XMPULong offset,
                      XMPULong length,
                      XMPValue value,
                      XMPPart **sourcePart);
  XMPMethod  void    ClearAllPromises();

  XMPStorageUnitKey  Lock(XMPStorageUnitKey key);
  void        Unlock(XMPStorageUnitKey key);

protected:

  CMContainer      fCMContainer;
  CMObject      fObject;
  
  XMPVMethod  void    SetChangedFromPrevFlag(XMPBoolean changed);
  XMPVMethod  CMContainer  GetCMContainer();

private:

  void        CleanupAndFail(XMPError  err);
  XMPStorageUnitRef   GetStorageUnitRef(XMPStorageUnit* embeddedSU, XMPBoolean strong);
  XMPStorageUnitRef  GetNextStorageUnitRef();
  void         GetClonedSUs(XMPAEHashTable* clonedSUIDs, XMPStorageUnit* initiatingFrameSU);
  XMPStorageUnitRef   SetStorageUnitRef(XMPStorageUnit* embeddedSU, XMPStorageUnitRef ref);
  
  XMPDraft*      fDraft;
  XMPStorageUnitID  fID; 
  
  CMObjectID      fObjectID;
  CMProperty      fCurProperty;
  XMPBoolean      fHasPropertyLooped;
  XMPValueIndex    fCurValueIndex;
  CMValue        fCurValue;
  XMPValueType    fCurValueType;
  XMPULong      fOffset;
  CMType        fCurType;
  XMPBoolean      fHasValueLooped;
  XMPULong      fPromiseCount;
  
  XMPULong      fCurrentKey;
  XMPULong      fLockCount;
  
  XMPULong      fCurrentSURef;
  
  XMPHeap        fHeap;
  

public: // Private by convention calls:

  XMPStorageUnit();
  
  XMPVMethod  ~XMPStorageUnit();

  XMPVMethod  void InitStorageUnit(XMPDraft* draft, XMPStorageUnitID suid);

  XMPMethod  void    Clone(XMPDraftKey key, XMPStorageUnit* fromSU, XMPAEHashTable* clonedSUIDs);
  XMPMethod  void    ResolvePromise();
  XMPMethod  void    ResolveAllPromises();
  XMPMethod  XMPPtr    GetPromiseInfo();
  XMPMethod  void    SetPromiseInfo(XMPPtr theInfo);
  XMPMethod  void    IncrementPromiseCount();
  XMPMethod  void    DecrementPromiseCount();
  
  XMPMethod  XMPHeap    GetHeap();
  
  XMPMethod  XMPID    GetObjectID();
  
  XMPMethod  CMValue    GetCurValue();
};


#endif  // _STORAGEU_
