/********************************************************************/ 
/*  Licensed Materials - Property of IBM                            */ 
/*                                                                  */ 
/*                                                                  */ 
/* Copyright (C) International Business Machines Corp., 1994.       */ 
/* Copyright (C) Apple Computer, Inc., 1994                         */ 
/*                                                                  */ 
/*  US Government Users Restricted Rights -                         */ 
/*  Use, duplication, or disclosure restricted                      */ 
/*  by GSA ADP Schedule Contract with IBM Corp.                     */ 
/*                                                                  */ 
/********************************************************************/ 
/*
  File:    OrdColl.h

  Contains:  Definition of class OrderedCollection

  Written by:  Richard Rodseth

*/

#ifndef _ORDCOLL_
#define _ORDCOLL_

#ifndef _XMPTYPES_
#include "XMPTypes.h"
#endif

#ifndef _LIST_
#include "List.h"
#endif

//==============================================================================
// Theory of Operation
//==============================================================================

// OrdereCollection is an ordered collection of elements of type void* (since
// we can't use templates)
// Duplicates are allowed.

//==============================================================================
// Constants
//==============================================================================

//==============================================================================
// Scalar Types
//==============================================================================

typedef void* ElementType;

//=====================================================================================
// Classes defined in this interface
//=====================================================================================

class OrderedCollection;  // An ordered (not sorted) collection of ElementTypes
class OrderedCollectionIterator;

//=====================================================================================
// Classes used by this interface
//=====================================================================================

class ValueLink;       // A link plus a value of type ElementType.

//=====================================================================================
// Global Variables
//=====================================================================================

//=====================================================================================
// Class ValueLink - Definition
//=====================================================================================

#define kValueLinkID "appl:valuelink$class,1.0.0"

class ValueLink : public Link {
public:
              ValueLink(ElementType value);    
  XMPVMethod          ~ValueLink()            {}
  XMPNVMethod  ElementType  GetValue()            { return fValue;}
  XMPNVMethod void    SetValue(ElementType v)      { fValue = v;}

private:
  ElementType     fValue;
};

//=====================================================================================
// Class OrderedCollection
//=====================================================================================

#define kOrderedCollectionID "appl:orderedcollection$class,1.0.0"

class OrderedCollection
{
public:

  OrderedCollection();
  OrderedCollection(XMPHeap where);
  virtual ~OrderedCollection();

  XMPNVMethod XMPULong Count() const       { return fImplementation.Count(); };
  
  XMPVMethod void  AddFirst(ElementType element);
  XMPVMethod void  AddLast(ElementType element);
  XMPVMethod void  AddBefore(ElementType existing, ElementType tobeadded);
  XMPVMethod void  AddAfter(ElementType existing, ElementType tobeadded);

  XMPVMethod ElementType  After(ElementType existing);
  XMPVMethod ElementType  Before(ElementType existing);

  XMPVMethod ElementType  First();
    // Returns kXMPNULL if there is no first element.
  XMPVMethod ElementType  Last();

  XMPVMethod ElementType  RemoveFirst();
    // Don't call if there are no elements. Crash will result.
  XMPVMethod ElementType  RemoveLast();
  XMPVMethod void  RemoveAll();
  
    // Called from the destructor. Removes all elements, deleting the links
    // Does not delete the elements themselves
    
  XMPVMethod void  DeleteAll();
  
    // Removes and deletes all elements
    
  XMPVMethod void  Remove(ElementType existing);
  XMPVMethod XMPBoolean  Contains(ElementType existing);
  
  XMPVMethod OrderedCollectionIterator* CreateIterator();

protected:
   XMPVMethod ValueLink*   CreateNewLink(ElementType value) const;
   XMPVMethod XMPBoolean  ElementsMatch(ElementType v1,ElementType v2) const;
     // Does a pointer comparison by default 

private:
  LinkedList    fImplementation;
  XMPHeap      fHeap; // if kXMPNULL, use default heap.

  friend class OrderedCollectionIterator;
  friend class ListIterator;
};

//=====================================================================================
// Class OrderedCollectionIterator
//=====================================================================================

#define kOrderedCollectionIteratorID "appl:orderedcollectioniterator$class,1.0.0"

class OrderedCollectionIterator {
public:
  OrderedCollectionIterator(OrderedCollection* collection);
  virtual ~OrderedCollectionIterator();
  ElementType  First();
  ElementType  Next();
  ElementType  Last();
  ElementType  Previous();
  XMPBoolean  IsNotComplete();
  
private:
    OrderedCollection*  fCollection;
  LinkedListIterator  fImplementation;
};

#endif // _ORDCOLL_