/********************************************************************/ 
/*  Licensed Materials - Property of IBM                            */ 
/*                                                                  */ 
/*                                                                  */ 
/* Copyright (C) International Business Machines Corp., 1994.       */ 
/* Copyright (C) Apple Computer, Inc., 1994                         */ 
/*                                                                  */ 
/*  US Government Users Restricted Rights -                         */ 
/*  Use, duplication, or disclosure restricted                      */ 
/*  by GSA ADP Schedule Contract with IBM Corp.                     */ 
/*                                                                  */ 
/********************************************************************/ 
/*
  File:    XMPLinkedList.h

  Contains:  Primitive linked list class

  Written by:  Richard Rodseth

*/

#ifndef _LIST_
#define _LIST_

#ifndef _XMPTYPES_
#include "XMPTypes.h"
#endif


//=====================================================================================
// Theory of Operation
//=====================================================================================

/*

 Note: These classes are private to the implementation. They are not available
 to part handlers.

 Note: These are primitive classes for implementing higher-level collections
 For example, to create a FrameList class, subclass Link to add a field to
 store the frame. The FrameList class would use a LinkedList in its implementation
 and would manufacture the Link objects internally.
 
 Note: SimpleLinkedList is a list without a "seed" for consistency checking.
 It cannot be iterated over.

*/

//==============================================================================
// Constants
//==============================================================================

//==============================================================================
// Scalar Types
//==============================================================================

//==============================================================================
// Classes defined in this interface
//==============================================================================

class Link;
class LinkedList;
class LinkedListIterator;

//==============================================================================
// Classes used by this interface
//==============================================================================


//==============================================================================
// Link
//==============================================================================

#define kLinkID "appl:link$class,1.0.0"

class  Link {

  public:
  
    Link();
    
    Link(Link* next, Link* previous);
    
    virtual ~Link();
    
    Link* GetNext() const          {return fNext;}
    
    Link* GetPrevious() const        {return fPrevious;}
  
  // The following operations are provided for efficiency, but DO NOT USE THEM
  // if there are any iterators active on a list. These operations don't bump
  // the list's fSeed and the iterators will not be able to detect that they
  // are out of sync!
    
    void  Remove( );
    
    void  AddBefore( Link *aLink );
    
    void  AddAfter( Link *aLink );
  
  //private-by-convention:
      
    void  SetNext(Link* aLink)        {fNext = aLink;}
    
    void  SetPrevious(Link* aLink)      {fPrevious = aLink;}

  private:
  
    Link*    fNext;
    Link*    fPrevious;
};


//==============================================================================
// LinkedList
//==============================================================================

#define kLinkedListID "appl:linkedlist$class,1.0.0"

class LinkedList {

  public:
  
      LinkedList();
      
      ~LinkedList();
      
      XMPBoolean    IsEmpty( )                    const;
      
      XMPULong    Count()                      const;
      
      XMPBoolean    Includes( const Link* )              const;
      
      void      Remove(Link&);
      
      void      RemoveAll();

      void      DeleteAllLinks();
      
      Link*      RemoveFirst();
      
      Link*      RemoveLast();
      
      void      AddBefore(Link& existing, Link* link);
      
      void      AddAfter(Link& existing, Link* link);
      
      void      AddFirst(Link* link);
      
      void      AddLast(Link* link);
      
      void      AddLast( LinkedList &list );
      
      void      AddLastUnique( LinkedList &list );
      
      Link*      After(const Link& link)              const;
      
      Link*       Before(const Link& link)            const;
      
      Link*      First()                      const;
      
      Link*       Last()                      const;

protected:

    Link      fSentinel;  // Marks the head & tail
    XMPULong    fSeed;    // Used to detect out-of-sync iterators
    
    Link*      GetSentinel( )
                        {return &fSentinel;}
    const Link*    GetSentinel( )                  const
                        {return &fSentinel;}
    XMPBoolean    IsSentinel( const Link* link )          const
                        {return link==this->GetSentinel();}
    XMPBoolean    NotSentinel( const Link* link )          const
                        {return link!=this->GetSentinel();}

private:          
    friend class LinkedListIterator;
};


//=====================================================================================
// LinkedListIterator
//=====================================================================================

#define kLinkedListIteratorID "appl:linkedlistiterator$class,1.0.0"

class LinkedListIterator {

  public:
  
    LinkedListIterator(LinkedList*  list);
    
    ~LinkedListIterator();
    
    Link*      First();
    
    Link*      Next();
    
    Link*      Last();
    
    Link*      Previous();

    Link*      Current();

    XMPBoolean     IsNotComplete();
    
  private:
  
    LinkedList*  fList;
    Link*          fCurrent;
    Link*          fSentinel;
    XMPULong    fSeed;    // Used to detect out-of-sync iterators
    
};


#endif // _LIST_