/********************************************************************/ 
/*  Licensed Materials - Property of IBM                            */ 
/*                                                                  */ 
/*                                                                  */ 
/* Copyright (C) International Business Machines Corp., 1994.       */ 
/* Copyright (C) Apple Computer, Inc., 1994                         */ 
/*                                                                  */ 
/*  US Government Users Restricted Rights -                         */ 
/*  Use, duplication, or disclosure restricted                      */ 
/*  by GSA ADP Schedule Contract with IBM Corp.                     */ 
/*                                                                  */ 
/********************************************************************/ 
/*
   File:    LinkSrc.h

   Contains:   The abstract XMPLinkSource class

   Written by: Nick Pilch & Craig Carper

*/

#ifndef _LINKSRC_
#define _LINKSRC_

#ifndef _LINK_
#include "Link.h"
#endif

#ifndef _PLFMDEF_
#include "PlfmDef.h"
#endif

#ifndef _PSTOBJ_
#include "PstObj.h"
#endif

//==============================================================================
// Theory of Operation
//==============================================================================

/*
   This class is used to represent the source side of OpenDoc links. XMPDrafts
   create and own these links. The source XMPPart will ask the draft to create
   an XMPLinkSource and an XMPLink pair, and return the XMPLink object to the
   the destination part.  The destination part uses the XMPLink object to
   extract the contents of the link.
*/

//==============================================================================
// Classes defined in this interface
//==============================================================================

class XMPAbsLinkSource;

//==============================================================================
// Classes used by this interface
//==============================================================================

class XMPPart;
class XMPStorageUnit;

//==============================================================================
// XMPAbsLinkSource
//==============================================================================

class XMPAbsLinkSource : public XMPPersistentObject
{
  public:

   XMPVMethod XMPBoolean Lock(XMPULong wait, XMPLinkKey* key)
      = 0;

      // Acquire exclusive access to the content storage unit of the link
      // for the current thread. The value kXMPTrue is returned if the lock is
      // granted; the key argument is set to a valid link key.  Access is
      // granted if the current thread already holds the lock.  Nested calls
      // to Lock must be balanced by an equal number of calls to Unlock to 
      // relinquish the lock.

   XMPVMethod void Unlock(XMPLinkKey key)
      =0;
      
      // Relinquish exclusive access to the content storage unit of the link.
      // The argument should be a non-zero key returned by Lock.

   XMPVMethod XMPStorageUnit* GetContentStorageUnit(XMPLinkKey key)
      = 0;

      // used by the source part to get the storage unit containing the 
      //  contents of the link.  The key parameter should be obtained by 
      //  aquiring the lock for this thread. 
      //  formerly named: GetStorageUnit

   XMPVMethod void ContentChanged(XMPChangeID id)
      = 0;

      // used by the source to make sure all destinations are notified
      //  of the updated link contents
      //  formerly named: Updated

   XMPVMethod void Clear(XMPChangeID id, XMPLinkKey key)
      = 0;

      // Empties the contents of the link. The key parameter
      // should be obtained by aquiring the lock for this thread.
      // The next call to GetContentStorageUnit returns an empty storage unit.
   
   XMPVMethod XMPChangeID GetChangeID()
      =0;

      // Returns the change id of the link content.  Source parts will most likely
      //  cache this value internally rather than use this method.  The returned 
      //  value should be tested for equality only, as there is no implicit 
      //  ordering of change ids.

   XMPVMethod void SetAutoExport(XMPBoolean automatic)
      = 0;

      // If automatic is kXMPTrue, export updates to other documents when the 
      // link's draft is saved.  If automatic is kXMPFalse, updates are exported
      // only when the Export method is called. This setting defaults to kXMPTrue.

   XMPVMethod XMPBoolean IsAutoExport()
      = 0;

      // Returns kXMPTrue if updates are automatically exported to other documents
      // when the link's draft is saved.

   XMPVMethod XMPLink* GetLink()
      = 0;
      
      // Return the link object associated with this link source.

   //---------------------------------
   // For XMPLink objects only

   XMPVMethod void ShowSourceContent()
      = 0;

      // Causes the source part of the link to become active, makes the source content
      // visible, and displays the link border.  Throws an exception if the source
      // part of the link cannot be located, or if the link has been broken at the
      // source.

   //-------------------------------------------------------------------------
   // From Memory protocol
   //-------------------------------------------------------------------------
   
   XMPVMethod XMPSize Purge(XMPSize size)
      = 0;

      // Gets rid of any unnecessary memory by purging memory blocks
      // or flushing structures out to disk
               
   //-------------------------------------------------------------------------
   // From Persistent Object protocol
   //-------------------------------------------------------------------------

   XMPVMethod void Externalize()
      = 0;

      // used by the Draft to get the link to externalize the necessary
      // parts of its persistent store.
   
   //-------------------------------------------------------------------------
   // constructors/destructors
   //-------------------------------------------------------------------------

   XMPAbsLinkSource(){}

   XMPVMethod  ~XMPAbsLinkSource() {}
};

#ifndef _LINKSRCM_
#include "LinkSrcM.h"
#endif

#endif // _LINKSRC_
