/********************************************************************/ 
/*  Licensed Materials - Property of IBM                            */ 
/*                                                                  */ 
/*                                                                  */ 
/* Copyright (C) International Business Machines Corp., 1994.       */ 
/* Copyright (C) Apple Computer, Inc., 1994                         */ 
/*                                                                  */ 
/*  US Government Users Restricted Rights -                         */ 
/*  Use, duplication, or disclosure restricted                      */ 
/*  by GSA ADP Schedule Contract with IBM Corp.                     */ 
/*                                                                  */ 
/********************************************************************/ 
/*
   File:    LinkMgr.h

   Contains:   Class definition for XMPLinkManager.

   Written by: Craig Carper 

*/

#ifndef _LINKMGR_
#define _LINKMGR_

#ifndef _PLFMDEF_
#include "PlfmDef.h"
#endif

#ifndef _XMPTYPES_
#include "XMPTypes.h"
#endif

#ifndef _STORAGEU_
#include "StorageU.h"
#endif

#ifndef _ORDCOLL_
#include "OrdColl.h"
#endif

#define APPLEINCL_AppleEvents
#ifndef APPLEINCLUDED
   #include <Apple.h>
#endif

#ifndef __EDITIONS__
#include "Editions.h"
#endif

//=====================================================================================
// Theory of Operation
//=====================================================================================

/*
   This file describes the Link Manager, an object specific to the Macintosh
   implementation of OpenDoc.
   
   The link manager assists in the maintenance of cross-document (external) links.
   There is one link manager object created for each OpenDoc session.
   
   The link manager is used by link objects, draft objects, and the document shell.
   It is not available to parts.
   
   Link objects use the link manager to register unique ID for sections.
   
   The document shell also uses the link manager to register unique IDs for sections.

*/

//=====================================================================================
// Classes defined in this interface
//=====================================================================================

class XMPLinkManager;

//=====================================================================================
// Classes used by this interface
//=====================================================================================

class XMPSession;
class XMPDocument;
class XMPDraft;
class XMPLink;
class XMPLinkSpec;
class OrderedCollection;

//=====================================================================================
// Private struct DraftInfo
//=====================================================================================
typedef struct
{
   XMPDraft*         draft;
   OrderedCollection*   linksToExport;
   OrderedCollection*   newExportedLinks;
   XMPBoolean        linksImported;    // True if any link automatically imported
}
DraftInfo;

//=====================================================================================
// XMPLinkManager
//=====================================================================================

#define kXMPLinkManagerID "appl:xmplinkmanager$class,1.0.0"

class XMPLinkManager
{
public:
   // ----------------------------------------
   // 'structors
   
   XMPLinkManager();
   
   virtual ~XMPLinkManager();
   
   XMPNVMethod void InitLinkManager(XMPSession* session, XMPHeap heap);

   XMPVMethod XMPULong NewSectionID(XMPDocument* document);
   
      // Return a new ID guaranteed to be unique for the lifetime of the document.

   // ----------------------------------------
   // For use by document shells only

   XMPVMethod XMPBoolean ReserveSectionID(XMPULong sectionID, XMPDocument* document);

      // Reserves a section ID guaranteed to be unique for the lifetime
      // of the document. ReserveSectionID returns kXMPTrue if the
      // argument sectionID is not already reserved and kXMPFalse otherwise.
      // If kXMPFalse is returned, the caller should request a different ID
      // or call NewSectionID().

   XMPVMethod XMPBoolean UnsavedExportedLinks(XMPDraft* draft);

      // Returns kXMPTrue if cross-document links have been created since the
      // draft was last saved.  This routine should be called by the document
      // shell prior to reverting or closing the draft without saving.
      // If this method returns kXMPTrue, cross-document links will be broken
      // if the document is not saved.  The document shell should alert the
      // user before proceeding with the rever or close.

   XMPVMethod XMPBoolean AnyLinkImported(XMPDraft* draft);

      // Returns kXMPTrue if one or more links to content in other documents
      // have automatically updated since the draft was last saved.
      // The document shell should call this method before closing a draft
      // that has not been changed by the user.  If this method returns
      // kXMPTrue, the shell should save the draft without prompting the
      // user.

   // ----------------------------------------
   // For use by drafts only

   XMPVMethod void DraftOpened(XMPDraft* draft);

      // Notifies the link manager that a draft has been opened.  The link manager
      // should internalize all link objects; each link publishing or subscribing
      // to an edition file will register with the edition manager.
      
   XMPVMethod void DraftSaved(XMPDraft* draft);

      // Links that automatically update will be called to write their edition file.
      // Cross-document links cannot be established until the first draft in the
      // document has been saved.

   XMPVMethod void DraftClosing(XMPDraft* draft);

      // Should be called before a draft is reverted or closed.
      // The link manager will delete any editon files created since the last save.
      // The link manager can discard any transient information for the argument draft.

   // ----------------------------------------
   // For use by links only

   XMPVMethod void ExportOnSave(XMPLink* link, XMPBoolean doExport);

      // If doExport is kXMPTrue, call the Export method of the argument link object
      // the next time the draft is saved.  If doExport is kXMPFalse, do not call
      // Export on save.  This method should be called by publishing link objects 
      // that export linked data automatically when the draft is saved.  If the draft 
      // is closed without saving, or is reverted, the Export method will not be called.
      // Once the draft has been saved, the link needs to call this method again when
      // the link is updated.

   // ----------------------------------------
   // private by convention

   XMPVMethod XMPLink* CreateLink(XMPDraft* draft, XMPLinkSpec* linkSpec);

      // Create a link in the argument draft from a linkSpec originating in
      // another document.

private:

   XMPStatic XMPPascal OSErr GetLinkAEHandler(
         const AppleEvent* theAppleEvent,
         const AppleEvent* reply,
         long refCon);

      // Handler for GetLink events, sent from one document to another to
      // resolve a link spec.

   // ----------------------------------------
   // Handlers for Edition Manager section events.

   XMPStatic XMPPascal OSErr SectionReadAEHandler(
         const AppleEvent *theAppleEvent,
         const AppleEvent *reply,
         long handlerRefcon);

   XMPStatic XMPPascal OSErr SectionWriteAEHandler(
         const AppleEvent *theAppleEvent,
         const AppleEvent *reply,
         long handlerRefcon);

   XMPStatic XMPPascal OSErr SectionScrollAEHandler(
         const AppleEvent *theAppleEvent,
         const AppleEvent *reply,
         long handlerRefcon);

   #if 1
      // I don't know where it's defined, so I'll just declare it here.
      typedef struct { } EditionContainerSpec;
      typedef struct { } SectionHandle;
   #endif

   // ----------------------------------------
   // Methods for supporting Edition files

   XMPVMethod void NewEditionFile(
         XMPFileSpec documentSpec,
         EditionContainerSpec* editionContainer,
         XMPDocument* document);
   
   XMPVMethod SectionHandle GetSectionFromEvent(const AppleEvent *theAppleEvent);

   XMPVMethod void EditionFileName(char* fileName, XMPUShort numb);

   XMPVMethod XMPULong NewEditionID(XMPDocument* document);

   XMPVMethod XMPULong NextGlobalID(XMPDocument* document, XMPPropertyName globalProp);

   // ----------------------------------------
   // Misc

   XMPMethod void NewDraftInfo(XMPDraft* draft);
   XMPMethod DraftInfo* GetDraftInfo(XMPDraft* draft);
   XMPMethod void DeleteDraftInfos();

   XMPHeap           fHeap;
   OrderedCollection*   fDraftInfos;   // Info for each open draft
};

#endif
