/********************************************************************/ 
/*  Licensed Materials - Property of IBM                            */ 
/*                                                                  */ 
/*                                                                  */ 
/* Copyright (C) International Business Machines Corp., 1994.       */ 
/* Copyright (C) Apple Computer, Inc., 1994                         */ 
/*                                                                  */ 
/*  US Government Users Restricted Rights -                         */ 
/*  Use, duplication, or disclosure restricted                      */ 
/*  by GSA ADP Schedule Contract with IBM Corp.                     */ 
/*                                                                  */ 
/********************************************************************/ 
/*
   File:    Link.h

   Contains:   The abstract XMPLink class

   Written by: Nick Pilch & Craig Carper

*/

#ifndef _LINK_
#define _LINK_

#ifndef _PLFMDEF_
#include "PlfmDef.h"
#endif

#ifndef _PSTOBJ_
#include "PstObj.h"
#endif

//==============================================================================
// Theory of Operation
//==============================================================================

/*
   This class is used to represent the destination side of OpenDoc links.
   XMPDrafts create and own these links. The source XMPPart will ask the draft
   to create an XMPLinkSource and an XMPLink pair, and return the XMPLink
   object to the the destination part.  The destination part uses the XMPLink
   object to extract the contents of the link.
*/

//==============================================================================
// Scalar types defined in this interface
//==============================================================================
typedef XMPULong XMPLinkKey;

//==============================================================================
// Classes defined in this interface
//==============================================================================

class XMPAbsLink;

//==============================================================================
// Classes used by this interface
//==============================================================================

class XMPPart;
class XMPStorageUnit;

//==============================================================================
// XMPLink
//==============================================================================

class XMPAbsLink : public XMPPersistentObject
{
  public:

   XMPVMethod XMPBoolean Lock(XMPULong wait, XMPLinkKey* key)
      = 0;

      // Acquire exclusive access to the content storage unit of the link
      // for the current thread. The value kXMPTrue is returned if the lock is
      // granted; the key argument is set to a valid link key.  Access is
      // granted if the current thread already holds the lock.  Nested calls
      // to Lock must be balanced by an equal number of calls to Unlock to 
      // relinquish the lock.

   XMPVMethod void Unlock(XMPLinkKey key)
      =0;
      
      // Relinquish exclusive access to the content storage unit of the link.
      // The argument should be a non-zero key returned by Lock.

   XMPVMethod XMPStorageUnit* GetContentStorageUnit(XMPLinkKey key)
      = 0;

      // used by destination parts to get the storage unit
      //  containing the contents of the link.  The key parameter
      //  should be obtained by aquiring the lock for this thread. 
      //  formerly named: GetStorageUnit

   XMPVMethod void RegisterDependent(XMPPart* clientPart, XMPChangeID id)
      = 0;

      // used by the destination to inform the link of its interest in the
      //  contents of the link. The parts LinkUpdated() method is called
      //  immediately if the links change id differs from the argument.
      //  The constant kChangeUnknown may be supplied if the part does not
      //  keep track of link change ids.
      //  formerly named: Notify
      
   XMPVMethod void UnregisterDependent(XMPPart* clientPart)
      = 0;

      // unregister dependent part previously registered with RegisterDependent.
      
   XMPVMethod XMPChangeID GetChangeID()
      =0;

      // Returns the change id of the link content.  Can be used by destinations
      //  to determine when new data is available.  The returned value should be
      //  tested for equality only, as there is no implicit ordering of change ids.

   XMPVMethod XMPError GetStatus()
      = 0;

      // Returns the status of the link.  A non-zero value indicates a non-fatal
      // problem which the part may ignore or bring to the users attention.
      // The most common problem is a broken cross-document destination.
      // This situation may be temporary if the source document re-establishes
      // the cross-document link.

   XMPVMethod void ShowSourceContent()
      = 0;

      // Causes the source part of the link to become active, makes the source content
      // visible, and displays the link border.  Throws an exception if the source
      // part of the link cannot be located, or if the link has been broken at the
      // source.

   //---------------------------------
   // For XMPLinkSource

   XMPVMethod void UpdateDependents(XMPChangeID id)
      = 0;
      
      // Notify dependent parts of a change to link contents.

   //-------------------------------------------------------------------------
   // From Memory protocol
   //-------------------------------------------------------------------------
   
   XMPVMethod XMPSize Purge(XMPSize size)
      = 0;

      // Gets rid of any unnecessary memory by purging memory blocks
      // or flushing structures out to disk
               
   //-------------------------------------------------------------------------
   // From Persistent Object protocol
   //-------------------------------------------------------------------------

   XMPVMethod void Externalize()
      = 0;

      // used by the Draft to get the link to externalize the necessary
      // parts of its persistent store.
   
   //-------------------------------------------------------------------------
   // constructors/destructors
   //-------------------------------------------------------------------------

   XMPAbsLink(){}

   XMPVMethod  ~XMPAbsLink() {}
};

#ifndef _LINKM_
#include "LinkM.h"
#endif

#endif // _LINK_
