/********************************************************************/ 
/*  Licensed Materials - Property of IBM                            */ 
/*                                                                  */ 
/*                                                                  */ 
/* Copyright (C) International Business Machines Corp., 1994.       */ 
/* Copyright (C) Apple Computer, Inc., 1994                         */ 
/*                                                                  */ 
/*  US Government Users Restricted Rights -                         */ 
/*  Use, duplication, or disclosure restricted                      */ 
/*  by GSA ADP Schedule Contract with IBM Corp.                     */ 
/*                                                                  */ 
/********************************************************************/ 
/*
  File:    XMPFrame.h

  Contains:  Definition of class XMPFrame

  Written by:  Joshua Susser & Sepp Friedrich

*/

#ifndef _FRAME_
#define _FRAME_

#ifndef _PSTOBJ_
#include "PstObj.h"  // base class
#endif

#ifndef _XMPTYPES_
#include "XMPTypes.h"
#endif

#ifndef _DRAFT_
#include "Draft.h"  // For XMPDraftKey
#endif

//==============================================================================
// Theory of Operation
//==============================================================================
/*
If an XMPParts contents needs to become visible, it provides to the embedding
XMPPart an XMPFrame with a desired frameShape and an XMPCanvas,  in which
coordinate space the frameShape is defined, and an XMPWindow, in which
eventually the XMPParts contents will be rendered.

The embedding XMPParts Part Handler then reshapes the XMPFrames bounding
FrameShape and also sets the ClipShape to be used for clipping drawings or
cached images rendered within the XMPFrame.  It also decides about proper
alignment within its own space and, therefore, sets the XMPFrames External
Transform (c.f. below) accordingly.

The Part Handler having created an XMPFrame object adds it to the XMPWindows
list of visible XMPFrames.

Invalidation concerns two cases:  an XMPFrames XMPTransfom or a (partial) area
of the XMPFrame can be affected as described below.

First, an XMPFrames position and shape within its containing Root XMPFrame
(which is equal to the XMPWindows frame) is determined by its Aggregate
Transfoms, which consists of the Frames External and Internal Transform
pre-composed with the containing Frames Aggregate Transforms.  For
mapping a point given in Window coordinates to a frames local space, XMPFrame
transforms it by using its Aggregate Internal Transform. Similarly, for
representing an XMPShape object, given in a containing Frames coordinate
space, into this Frames coordinate space, the given shape becomes transformed
by applying this Frames External Transform. For this release, it is required
that the local coordinate spaces have the same metric as the Root Frames
canvas. In other words, the Frames associated XMPCanvas objects must have the
same type as the Root Frames XMPCanvas object.  Independently, a part handler
can come up with its own private canvases and mappings to the XMPCanvas object
given to its XMPFrame object.

Whenever a Part Handler for a containing XMPPart changes a contained XMPFrames
External Transform (e.g. moving the XMPFrame to a new position) or the
XMPFrames own Part Handler changes the Internal Transform (e.g. scrolling the
visible XMPPart  contents within the XMPFrame), the XMPFrames Aggregate
Transforms are invalidated or get updated.

Furthermore, to assist a containing Part Handler in finding a fine-tuned
layout between adjacent contained Frames, the UsedShape, set by its Part, may
be used by the embedding Part.  It describes the XMPShape of an XMPFrame which
is used by its actual contents, versus the FrameShape, that reflects rather the
required XMPShape as defined by the contained Part Handler.

ActiveShape, also set by this XMPFrames XMPParts Part Hanlder, defines the
area within the UsedShape which reacts to mouse clicks which it receives when
the Frame is not set to Frozen.

All Shapes are given in their Frames local coordinate space.
*/

                
//==============================================================================
// Types
//==============================================================================

enum XMPLinkStatus
{
  kXMPInLinkDestination,
  kXMPInLinkSource,
  kXMPNotInLink
};

//==============================================================================
// Classes defined in this interface
//==============================================================================

class XMPFrame;
class XMPFrameFacetIterator;

//==============================================================================
// Classes used by this interface
//==============================================================================

class OrderedCollection;
class OrderedCollectionIterator;
class XMPCanvas;
class XMPDraft;
class XMPFacet;
class XMPPart;
class XMPPersistentObject;
class XMPShape;
class XMPStorageUnit;
class XMPTransform;
class XMPDragItemIterator;
class XMPWindow;

struct XMPPoint;

//==============================================================================
// XMPFrame
//==============================================================================

#define kXMPFrameID "appl:xmpframe$class,1.0.0"

class XMPFrame : public XMPPersistentObject
{
  friend class XMPDraft;
  friend class XMPFrameFacetIterator;

public:

// getters/setters:

  XMPVMethod XMPFrame*  GetContainingFrame();
  XMPVMethod void     SetContainingFrame(XMPFrame* frame);

  XMPVMethod XMPULong    GetFrameGroup();
  XMPVMethod void      SetFrameGroup(XMPULong groupID);

  XMPVMethod XMPBoolean  IsRoot();

  XMPVMethod XMPBoolean  IsSubframe();
  XMPVMethod void      SetSubframe(XMPBoolean isSubframe);

  XMPVMethod XMPBoolean  IsOverlaid();

  XMPVMethod XMPBoolean  IsFrozen();
  XMPVMethod void      SetFrozen(XMPBoolean isFrozen);

  XMPVMethod XMPBoolean  DoesPropagateEvents();
  XMPVMethod void      SetPropagateEvents(XMPBoolean doesPropagateEvents);


// part

  XMPVMethod XMPPart* GetPart();

  XMPVMethod void ChangePart(XMPPart* part);
    // change the part this frame views

  XMPVMethod XMPInfoType  GetPartInfo();
  XMPVMethod void      SetPartInfo(XMPInfoType partInfo);

  XMPVMethod XMPTypeToken GetViewType();
  XMPVMethod void      SetViewType(XMPTypeToken viewType);
    // should be called only by my part
  XMPVMethod void      ChangeViewType(XMPTypeToken viewType);
    // called by container, etc. to request part to change type

  XMPVMethod XMPTypeToken GetPresentation();
  XMPVMethod void      SetPresentation(XMPTypeToken presentation);
    // should be called only by my part
  XMPVMethod void      ChangePresentation(XMPTypeToken presentation);
    // called by container, etc. to request part to change presentation kind

// facets

  XMPVMethod void FacetAdded(XMPFacet* facet);

  XMPVMethod void FacetRemoved(XMPFacet* facet);

  XMPVMethod XMPFrameFacetIterator* CreateFacetIterator();

// shapes:

  // frame shape
  
  XMPVMethod XMPShape* GetFrameShape();

  XMPVMethod void ChangeFrameShape(XMPShape* shape);
    // Containing Part picks a new Frame Shape for this embedded frame.

  XMPVMethod XMPShape* RequestFrameShape(XMPShape* shape);
    // Embedded Part asks to change the Frame Shape for this frame.

  // used shape
  
  XMPVMethod XMPShape* GetUsedShape();

  XMPVMethod void ChangeUsedShape(XMPShape* shape);
    // Embedded Part changes the Used Shape of this frame.

// transformations:

  XMPVMethod XMPTransform* GetInternalTransform();

  XMPVMethod void ChangeInternalTransform(XMPTransform* transform);
    // change internalTransform to new value, notify facets of change

// drag&drop:

  XMPVMethod XMPBoolean  IsDroppable();
  XMPVMethod void      SetDroppable(XMPBoolean isDroppable);

  XMPVMethod XMPBoolean  IsDragging();
  XMPVMethod void      SetDragging(XMPBoolean isDragging);

// linking:

  XMPVMethod void        ContentChanged(XMPChangeID change);
  XMPVMethod void        ChangeLinkStatus(XMPLinkStatus status);
  XMPVMethod XMPLinkStatus  GetLinkStatus();

// invalidation/draw:

  XMPVMethod void Invalidate(XMPShape* invalidShape);
    // invalidate the area in all my facets

  XMPVMethod void Validate(XMPShape* validShape);
    // validate the area in all my facets

  XMPVMethod void InvalidateActiveBorder();
    // invalidate the active border on all my facets

  XMPVMethod void DrawActiveBorder();
    // draw the active border on all my facets

// memory management:

  XMPVMethod XMPSize Purge(XMPSize numBytes);    // override
    //  gets rid of any unnecessary memory by purging memory blocks
    //  or flushing structures out to disk.

  XMPVMethod XMPStorageUnit*  CloneTo(XMPDraftKey key, XMPDraft* destDraft);
  
  XMPVMethod void Externalize();          // override

// ref counting
  
  XMPVMethod void Release();            // override

  XMPVMethod void Close();
    
    // tell the frame to release references

  XMPVMethod void Remove();
    
    // tell the frame to release references and remove from Draft

protected:  // only to be called by XMPDraft

  XMPFrame();
  XMPVMethod ~XMPFrame();

  XMPNVMethod void  InitFrame(
        XMPStorageUnit*  storageUnit,
        XMPFrame*     containingFrame,
        XMPShape*     frameShape,
        XMPPart*     part,
        XMPTypeToken  viewType,
        XMPTypeToken  presentation,
        XMPULong     frameGroup,
        XMPBoolean    isRoot,
        XMPBoolean     isOverlaid);
        
  XMPNVMethod void  InitFrameFromStorage(XMPStorageUnit*  storageUnit);

protected:  // only to be called by XMPFrameFacetIterator

  XMPVMethod OrderedCollection* GetFacets();

protected:

  void  CommonInitFrame();

  XMPFrame*      fContainingFrame;  // parent in frame hierarchy
  OrderedCollection*  fFacets;      // list of current facets

  XMPShape*    fFrameShape;      // basic shape of frame
  XMPShape*    fUsedShape;        // region to wrap around

  XMPTransform*  fInternalTransform;    // viewing transform

  XMPPart*    fPart;          // part the frame is displaying
  XMPInfoType    fPartInfo;        // data type of part
  XMPTypeToken  fViewType;        // type of view of part (icon, etc.)
  XMPTypeToken  fPresentation;      // kind of frame (content, palette, etc.)

  XMPULong    fFrameGroup;      // ID of group this frame is in

  XMPBoolean    fIsRoot;        // is this a window's root frame?
  XMPBoolean    fIsSubframe;      // is this subframe of conglomerate frame?
  XMPBoolean    fIsOverlaid;      // is this an overlaid frame?
  XMPBoolean    fIsFrozen;        // mouse events go to my containing frame
  XMPBoolean    fIsDroppable;      // accepts drag/drop events
  XMPBoolean    fIsDragging;      // is being dragged
  XMPBoolean    fDoesPropagateEvents;  // frame can't handle delegated events
  
  XMPBoolean    fGettingPart;      // Semaphore to ensure that GetPart is not called recursively.
};

//==============================================================================
// XMPFrameFacetIterator
//==============================================================================

#define kXMPFrameFacetIteratorID "appl:xmpframefacetiterator$class,1.0.0"

class XMPFrameFacetIterator
{

public:

  XMPFrameFacetIterator(XMPFrame* frame);

  virtual ~XMPFrameFacetIterator();

  XMPVMethod XMPFacet* First();
    // Returns the first visible facet
  
  XMPVMethod XMPFacet* Next();
    // Returns the next visible facet
  
  XMPVMethod XMPBoolean IsNotComplete();
    // Returns TRUE as long as there are more facets

  XMPFrame*          fFrame;
  OrderedCollectionIterator*  fIter;
};

#endif // _XMPFRAME_