/********************************************************************/ 
/*  Licensed Materials - Property of IBM                            */ 
/*                                                                  */ 
/*                                                                  */ 
/* Copyright (C) International Business Machines Corp., 1994.       */ 
/* Copyright (C) Apple Computer, Inc., 1994                         */ 
/*                                                                  */ 
/*  US Government Users Restricted Rights -                         */ 
/*  Use, duplication, or disclosure restricted                      */ 
/*  by GSA ADP Schedule Contract with IBM Corp.                     */ 
/*                                                                  */ 
/********************************************************************/ 
/*
  File:    Facet.h

  Contains:  Definition of class XMPAbsFacet

  Written by:  Joshua Susser

*/

#ifndef _FACET_
#define _FACET_

#ifndef _XMPTYPES_
#include "XMPTypes.h"
#endif

#ifndef _XMPOBJ_
#include "XMPObj.h"
#endif

//=====================================================================================
// Theory of Operation
//=====================================================================================

/*
  Facets hold the ephemeral information associated with the display of
  a particular Frame.
*/

//==============================================================================
// Types
//==============================================================================

typedef enum {
  kXMPNoHighlight = 0,
  kXMPFullHighlight,
  kXMPDimHighlight
} XMPHighlight;

//=====================================================================================
// Classes defined in this interface
//=====================================================================================

class XMPAbsFacet;
class XMPFacetIterator;

//=====================================================================================
// Classes used by this interface
//=====================================================================================

class XMPFacet;
class FacetNode;
class List;
class NodeTraverser;
class XMPCanvas;
class XMPDragItemIterator;
class XMPFrame;
class XMPObject;
class XMPShape;
class XMPTransform;
class XMPWindow;

//=====================================================================================
// XMPAbsFacet
//=====================================================================================

class XMPAbsFacet : public XMPObject
{
  friend class XMPFacetIterator;

public:
  
  //---------------------------------
  // constructor

  XMPAbsFacet() {}
      
  XMPVMethod ~XMPAbsFacet() {}
  
  XMPVMethod XMPSize Purge(XMPSize size)
    = 0;
  
  //---------------------------------
  // frames

  XMPVMethod XMPFrame* GetFrame()
    = 0;

  XMPVMethod XMPFacet* CreateEmbeddedFacet(XMPFrame* frame,
              XMPShape* clipShape, XMPTransform* externalTransform,
              XMPFacet* siblingFacet, XMPFramePosition position)
    = 0;
    // Add a frame to my list of visible facets, creating a new Facet for it
        
  XMPVMethod void RemoveFacet(XMPFacet* facet)
    = 0;
    // Remove a facet which is no longer visible. 

  XMPVMethod void MoveBefore(XMPFacet* child, XMPFacet* sibling)
    = 0;
    // Move child facet to be in front of sibling.
    // If sibling is NULL, move child to front of all children.

  XMPVMethod void MoveBehind(XMPFacet* child, XMPFacet* sibling)
    = 0;
    // Move child facet to be in back of sibling.
    // If sibling is NULL, move child to back of all children.

  XMPVMethod XMPFacet* GetContainingFacet()
    = 0;
  
  XMPVMethod XMPFacetIterator* CreateFacetIterator(XMPTraversalType traversalType, 
                           XMPSiblingOrder siblingOrder)
    = 0;

  //---------------------------------
  // clip shape
  
  XMPVMethod XMPShape* GetClipShape()
    = 0;

  XMPVMethod void ChangeClipShape(XMPShape* clipShape)
    = 0;
    // set the value of my clip shape and notify my part

  XMPVMethod XMPShape* GetAggregateClipShape()
    = 0;
    // returns clipShape clipped to all containing
    // frames' clipShapes on facet's canvas

  XMPVMethod XMPShape* GetWindowAggregateClipShape()
    = 0;
    // returns clipShape clipped to all containing
    // frames' clipShapes in facet's window

  XMPVMethod void InvalidateAggregateClipShape()
    = 0;

  //---------------------------------
  // active shape
  
  XMPVMethod XMPShape* GetActiveShape()
    = 0;

  XMPVMethod void ChangeActiveShape(XMPShape* activeShape)
    = 0;
    // set the value of my active shape

  //---------------------------------
  // transformations:

  XMPVMethod XMPTransform* GetExternalTransform()
    = 0;

  XMPVMethod void ChangeExternalTransform(XMPTransform* transform)
    = 0;
    // if current fExternalTransform differs from new transform, then
    // invalidate the cached external and internal aggregate transforms
    // and also for all contained frames.

  XMPVMethod void InternalTransformChanged()
    = 0;
    // notify me that my frame's internalTransform changed
    // I need to invalidate my content transforms, and
    // my children's transforms as well

  XMPVMethod XMPTransform* GetFrameTransform()
    = 0;

  XMPVMethod XMPTransform* GetContentTransform()
    = 0;

  XMPVMethod XMPTransform* GetWindowFrameTransform()
    = 0;

  XMPVMethod XMPTransform* GetWindowContentTransform()
    = 0;

  XMPVMethod void InvalidateAggregateTransforms()
    = 0;

  //---------------------------------
  // canvas

  XMPVMethod XMPBoolean HasCanvas()
    = 0;
    // return whether this facet has its own canvas
    // (as opposed to using its parent's)

  XMPVMethod XMPCanvas* GetCanvas()
    = 0;
    // return my canvas.  if fCanvas is NULL, return my parent's canvas.

  XMPVMethod void SetCanvas(XMPCanvas* canvas)
    = 0;

  //---------------------------------
  // window

  XMPVMethod XMPWindow* GetWindow()
    = 0;
    // return my window - if NULL, inherit from containing facet

  XMPVMethod void SetWindow(XMPWindow* window)
    = 0;
    // only root facets should have a window

  //---------------------------------
  // PartInfo

  XMPVMethod XMPInfoType GetPartInfo()
    = 0;

  XMPVMethod void SetPartInfo(XMPInfoType partInfo)
    = 0;

  //---------------------------------
  // drag&drop

  XMPVMethod MRESULT DragEnter(XMPPoint point, XMPDragItemIterator* dragInfo)  //dwf
    = 0;
    // point in window coordinates

  XMPVMethod MRESULT DragWithin(XMPPoint point, XMPDragItemIterator* dragInfo)   //dwf
    = 0;
    // point in window coordinates

  XMPVMethod void DragLeave(XMPPoint point)
    = 0;
    // point in window coordinates

  XMPVMethod XMPDropResult Drop(XMPPoint point, XMPDragItemIterator* dropInfo)
    = 0;
    // point in window coordinates

  //---------------------------------
  // drawing

  XMPVMethod void Update(XMPShape* invalidShape)
    = 0;
    // draw any of (me + children) intersecting invalidShape

  XMPVMethod void Invalidate(XMPShape* invalidShape)
    = 0;
    // tell underlying graphics system to invalidate the area in invalidShape.
    // invalidShape is in my frame's coordinate system

  XMPVMethod void Validate(XMPShape* validShape)
    = 0;
    // tell underlying graphics system to validate the area in validShape.
    // validShape is in my frame's coordinate system

  XMPVMethod void Draw(XMPShape* invalidShape)
    = 0;
    // Asks my frame's part to draw its contents within my clipShape
    // clipped to the transformed invalidShape on my frame's canvas

  XMPVMethod void DrawChildren(XMPShape* invalidShape)
    = 0;
    // Draw all of my child facets which need updating, and all of their
    // children too.

  XMPVMethod void DrawChildrenAlways(XMPShape* invalidShape)
    = 0;
    // Force drawing of all of my child facets, and all of their children too.

  XMPVMethod void DrawnIn(XMPShape* shape)
    = 0;
    // mark shape as having been drawn in, but not in response to
    // an update event.  notify owner of canvas to update parent
    // canvas with altered contents

  XMPVMethod void InvalidateActiveBorder()
    = 0;

  XMPVMethod void DrawActiveBorder()
    = 0;

  //---------------------------------
  // mouse tracking and hit detection:

  XMPVMethod void MouseEnter(XMPPoint where)
    = 0;
    // Called when the mouse is moved over a facet (mouse button up)
    // Transform point to facet coords and pass to part

  XMPVMethod void MouseWithin(XMPPoint where)
    = 0;
    // Called when the mouse is moved within a facet (mouse button up)
    // Transform point to facet coords and pass to part
      
  XMPVMethod void MouseLeave()
    = 0;
    // Called when the mouse is moved out of a facet (mouse button up)

  XMPVMethod XMPBoolean ContainsPoint(XMPPoint point)
    = 0;
    // within activeShape ^ clipShape
    // point in window coordinate space

  XMPVMethod XMPBoolean ActiveBorderContainsPoint(XMPPoint point)
    = 0;
    // within active border shape
    // point in window coordinate space

  //---------------------------------
  // selection and highlighting:

  XMPVMethod XMPBoolean IsSelected()
    = 0;
    // return selection state

  XMPVMethod void SetSelected(XMPBoolean isSelected)
    = 0;
    // set selection state

  XMPVMethod XMPHighlight  GetHighlight()
    = 0;
    // return highlight state
  
  XMPVMethod void ChangeHighlight(XMPHighlight highlight)
    = 0;
    // Change the highlight state of this facet.


protected:

  XMPVMethod FacetNode* GetNode()
    = 0;

  XMPVMethod XMPBoolean NeedsUpdate()
    = 0;

  XMPVMethod void SetNeedsUpdate(XMPBoolean needsUpdate)
    = 0;
};


//=====================================================================================
// XMPFacetIterator
//=====================================================================================

#define kXMPFacetIteratorID "appl:xmpfacetiterator$class,1.0.0"

class XMPFacetIterator
{

public:

  XMPFacetIterator(XMPFacet* facet,
           XMPTraversalType traversalType, 
           XMPSiblingOrder siblingOrder);

  virtual ~XMPFacetIterator();
  
  XMPMethod XMPFacet* First();
  
    // Returns the first visible facet in traversal order.
  
  XMPMethod XMPFacet* Next();
  
    // Returns the next visible facet in traversal order.
    // Step transparently into nested Canvases.
  
  XMPMethod void SkipChildren();
  
    // advance past all the children of the current facet.
    // only has an effect for kXMPTopDown traversalType.

  XMPMethod XMPBoolean IsNotComplete();
  
    // Returns TRUE as long as there are more facets

private:

  XMPFacet*      fFacet;
   XMPTraversalType  fTraversalType;
   XMPSiblingOrder    fSiblingOrder;
  NodeTraverser*    fNodeTraverser;
};

#if PLATFORM_MACINTOSH
#ifndef _FACETM_
#include "FacetM.h"
#endif
#endif

#endif // _FACET_