/********************************************************************/ 
/*  Licensed Materials - Property of IBM                            */ 
/*                                                                  */ 
/*                                                                  */ 
/* Copyright (C) International Business Machines Corp., 1994.       */ 
/* Copyright (C) Apple Computer, Inc., 1994                         */ 
/*                                                                  */ 
/*  US Government Users Restricted Rights -                         */ 
/*  Use, duplication, or disclosure restricted                      */ 
/*  by GSA ADP Schedule Contract with IBM Corp.                     */ 
/*                                                                  */ 
/********************************************************************/ 
/*
  File:    DArray.h

  Contains:  Implementation of a dynamic array

  Written by:  Vincent Lo

*/

#ifndef _DARRAY_
#define _DARRAY_

#ifndef _XMPTYPES_
#include "XMPTypes.h"
#endif

//==============================================================================
// Constants
//==============================================================================
const XMPULong  kXMPUnlimited = 0xFFFFFFF;

//==============================================================================
// Scalar Types
//==============================================================================
typedef XMPSByte*  DArrayItemPtr;

//==============================================================================
// Classes defined in this interface
//==============================================================================

class DynamicArray;
class DynamicArrayEntry;
class DynamicArrayIterator;

//==============================================================================
// DynamicArray
//==============================================================================

class DynamicArray
{
  friend class DynamicArrayIterator;
  
public:

  DynamicArray(XMPSLong maxEntryCount, XMPULong itemSize);
  ~DynamicArray();
  
  XMPMethod void      Initialize();
  
  XMPMethod void      Add(XMPULong key, XMPSByte* item);
  XMPMethod void      Remove(XMPULong key);
  XMPMethod XMPBoolean  Get(XMPULong key, XMPSByte* item);  // "GetItem" conflicts w/Mac Menu Mgr
  XMPMethod XMPSLong    GetEntryCount();
  XMPMethod XMPSLong    GetMaxEntryCount();
  
  XMPMethod void      MoveEntriesTo(DynamicArray* array, XMPULong numEntries);
  
private:

  XMPBoolean  Find(XMPULong key, XMPSLong* index, XMPULong* numMatched);
  void     Allocate(XMPSLong fromIndexInclusive, XMPSLong deltaIndex);
  XMPULong  EntryKey(XMPSLong index);
  XMPSByte*  EntryItem(XMPSLong index);
  XMPSByte*  EntryKeyAddress(XMPSLong index);
  XMPSByte*  EntryItemAddress(XMPSLong index);
  void    SetEntry(XMPSLong index, XMPULong key, XMPSByte* item);
  
  XMPULong  fKeySize;
  XMPULong  fItemSize;
  XMPULong  fNumEntriesUsed;
  XMPULong  fNumEntriesAllocated;
  XMPULong  fMaxEntries;
  XMPSByte*  fArray;
};

//==============================================================================
// DynamicArrayEntry
//==============================================================================

class DynamicArrayEntry
{
  friend DynamicArrayIterator;

public:

  DynamicArrayEntry(XMPULong itemSize);
  ~DynamicArrayEntry();
  
  XMPULong  GetKey();
  void    Get(XMPPtr itemPtr);        // "GetItem" conflicts w/Mac Menu Mgr
  void    Set(XMPULong key, XMPPtr item);

private:

  XMPULong  fKey;
  XMPSByte*  fItemAddress;
  XMPULong  fItemSize;
};


//==============================================================================
// DynamicArrayIterator
//==============================================================================
// User of DynamicArrayIterator has to ensure that the DynamicArray is not
// changed when there is still outstanding Iterators. Otherwise, something
// bad may happen.

class DynamicArrayIterator
{

public:

  DynamicArrayIterator(DynamicArray* array);
  
  ~DynamicArrayIterator();

  XMPMethod void Initialize();

  XMPMethod DynamicArrayEntry*  First();
    
  XMPMethod DynamicArrayEntry*  Next();
      
  XMPBoolean IsNotComplete();

private:
  
  DynamicArray*    fDynamicArray;
  XMPULong      fIndex;
  DynamicArrayEntry*  fEntry;
  
};
  

#endif  // _DARRAY_