/********************************************************************/ 
/*  Licensed Materials - Property of IBM                            */ 
/*                                                                  */ 
/*                                                                  */ 
/* Copyright (C) International Business Machines Corp., 1994.       */ 
/* Copyright (C) Apple Computer, Inc., 1994                         */ 
/*                                                                  */ 
/*  US Government Users Restricted Rights -                         */ 
/*  Use, duplication, or disclosure restricted                      */ 
/*  by GSA ADP Schedule Contract with IBM Corp.                     */ 
/*                                                                  */ 
/********************************************************************/ 
/*
   File:    Binding.h

   Contains:   Class definition for XMPBinding

   Written by: Tantek Celik and Craig Carper 

*/

#ifndef _BINDING_
#define _BINDING_

#ifndef _PLFMDEF_
#include "PlfmDef.h"
#endif

#ifndef _XMPOBJ_
#include "XMPObj.h"
#endif

#ifndef _XMPTYPES_
#include "XMPTypes.h"
#endif

//=====================================================================================
// Theory of Operation
//=====================================================================================

/*
   This file describes the XMPBinding class. 
   XMPBinding is a system service that is used to bind part data to part editors.
   XMPBinding also includes a number of utility functions, including mapping content
   kinds to categories, providing a list of part editors supporting a given kind, and 
   routines to access and set system-wide editor preferences.
*/

//=====================================================================================
// Scalar Types
//=====================================================================================
//typedef XMPType XMPCategory;      // ISO string type for categories
//typedef void* XMPEditor;       // An opaque type identifying a specific part editor
//typedef XMPClassID* XMPEditor;    // An opaque type identifying a specific part editor

//=====================================================================================
// Classes defined in this interface
//=====================================================================================
class XMPEditorSet;
class XMPEditorSetIterator;
class XMPBinding;

//=====================================================================================
// Classes used by this interface
//=====================================================================================
class XMPStorageUnit;
class XMPDraft;
class XMPTypeSet;
class XMPBaseSession;
class XMPNameSpace;
class OrderedCollection;
class OrderedCollectionIterator;
class XMPFile;
class TMacFileSpec;

//=====================================================================================
// Class XMPEditorSet
//=====================================================================================
// Should these routines have more generic names like "Add" and "Remove", so there's more
// consistency with other set classes, like XMPTypeSet?

#define kXMPEditorSetID "appl:xmpeditorset$class,1.0.0"

class XMPEditorSet
{
public:

   XMPEditorSet();
   XMPEditorSet(XMPHeap where);
   virtual ~XMPEditorSet();

   XMPMethod void AddEditor(XMPEditor editor);
   
      // Adds an type to the end of the set, creating a copy of the argument in the set.

   XMPMethod void AddEditorSet(XMPEditorSet* editors);
   
      // Unions the two into this (leaving the argument unchanged).

   XMPMethod void RemoveEditor(XMPEditor editor);
   
      // Removes an editor from a set.  Does nothing if the argument editor is not present.
   
   XMPMethod void RemoveEditor(const XMPEditorSet& set);
   
      // Removes all editors present in the argument set.

   XMPMethod XMPBoolean ContainsEditor(XMPEditor editor);
   
      // Returns true if the set contains the specified editor.
      // Otherwise, returns false.

   XMPMethod XMPULong GetEditorCount();
      
   XMPMethod XMPEditorSetIterator* CreateIterator();
   
      // Returns an iterator for the set, which can be used to retrieve editors
      // from the set in order.

private:

   XMPHeap           fHeap;

   OrderedCollection*   fSet;

   friend class XMPEditorSetIterator;
};
   
//=====================================================================================
// Class XMPEditorSetIterator
//=====================================================================================

#define kXMPEditorSetIteratorID "appl:xmpeditorsetiterator$class,1.0.0"

class XMPEditorSetIterator
{
   public:
      XMPEditorSetIterator(XMPEditorSet* set);
      virtual ~XMPEditorSetIterator();

      XMPVMethod XMPEditor First();
      XMPVMethod XMPEditor Next();
      XMPVMethod XMPBoolean   IsNotComplete();
      
         // These methods return a pointer to a string within the set.
         // The client must not dispose this memory, and must be aware that
         // These methods return nil if the requested item does not exist.
      
   private:
      OrderedCollectionIterator* fIterator;
      XMPHeap           fHeap;
};


//=====================================================================================
// Class XMPBinding
//=====================================================================================

#define kXMPBindingID "appl:xmpbinding$class,1.0.0"

class XMPBinding : public XMPObject
{
public:
   
   XMPBinding();
   
      // XMPSession creates one instance of XMPBinding
      // which is then shared among clients within the process.
      
   XMPVMethod ~XMPBinding();
   
   XMPNVMethod void InitBinding(XMPBaseSession* session, XMPHeap heap);
             
   XMPMethod XMPSize Purge(XMPSize size);
   
   //---------------------------------
   // Direct editor bindings

   XMPMethod XMPEditor ChooseEditorForPart(XMPStorageUnit* thePartSU, XMPType newKind );
      // Returns the "best" available editor for the part
      // may invoke automatic translation

   XMPMethod XMPEditor TranslateAndChooseEditorForPart(XMPStorageUnit* thePartSU);
      // attempt to intelligently translate one of the kinds in the part into 
      // a kind which an available editor supports, and return that editor

   XMPMethod void          UpdatePreferences();

   XMPMethod XMPBoolean GetAllEditorsForKind(XMPType kind, XMPEditorSet* theEditorSet);
      // Returns the set of Editors supporting the argument data kind without translation.

   XMPMethod XMPBoolean GetAllEditorsForCategory( XMPType category,
         XMPEditorSet* theEditorSet ) ;
      // Returns the set of Editors supporting the argument data category without translation.

   //---------------------------------
   // access to system preferred editors

   XMPMethod XMPEditor SetSysPrefEditorForKind(XMPType kind, XMPEditor editor);
      // Sets the system-wide preferred editor for the argument data kind.
      // This routine will overwrite any preference set by the user.

   XMPMethod XMPEditor SetSysPrefEditorForCategory(XMPType category,XMPEditor editor);
      // Sets the system-wide preferred editor for the argument data category.
      // This routine will overwrite any preference set by the user.

   XMPMethod XMPEditor GetSysPrefEditorForKind(XMPType kind);
      // Returns the system-wide preferred editor for the argument data kind.
      // A null value is returned if no preference has been specified.
      // This routine checks to ensure that the preferred editor actually exists;
      // if it does not, the preference is canceled and null is returned.

   XMPMethod XMPEditor GetSysPrefEditorForCategory(XMPType category);
      // Returns the system-wide preferred editor for the argument data category.
      // A null value is returned if no preference has been specified.

   XMPMethod XMPBoolean GetUserKindFromKind( XMPType kind, XMPName** name );
   XMPMethod XMPBoolean GetUserEditorFromEditor( XMPType editor, XMPName** name );
   XMPMethod XMPBoolean GetUserCatFromCat( XMPType category, XMPName** name );

private:

   XMPMethod XMPEditor ChooseDirectEditorForPart(XMPStorageUnit* thePartSU, XMPType newKind );
      // Returns the "best" available editor for the part
      // without translation.
   
   XMPMethod XMPEditor ChooseDirectEditorForKind(XMPType aKind, XMPStorageUnit* thePartSU);
      // Returns the "best" available editor which supports kind aKind of the part
      // without translation.

   //---------------------------------
   // Editor related utility methods

   XMPMethod XMPBoolean EditorExists(XMPEditor editor);
      // Returns true if the argument editor is present on this machine.
      
   XMPMethod XMPEditor GetPreferredEditorForPart(XMPStorageUnit* thePartSU);
      // Returns the editor that last externalized the argument storage unit.
      // Null is returned if that editor is not present on the current machine.

   XMPMethod XMPTypeSet* GetKindsSupported(XMPEditor editor);
      // Returns the set of data types supported (without translation) by the
      // argument editor.

   XMPMethod XMPBoolean EditorSupportsKind(XMPEditor editor, XMPType kind);
      // Returns true if the argument editor supports the argument data type
      // without translation.

   XMPMethod XMPEditor AlphaNumChooseEditor(XMPEditorSet* editorSet);
      // Returns the (alphanum preferred editor over viewer) from 
      // the set of Editors
      
   XMPMethod XMPEditor AlphaNumChooseAnEditorFromTwo(XMPEditor e1, XMPEditor e2);
      // Returns the (alphanum preferred editor over viewer) of the two editors
      
   XMPMethod XMPBoolean DraftRestrictsEditors(XMPDraft* draft);
      // Returns true if the argument draft restricts the editors that may be
      // used in this document.

   XMPMethod XMPEditorSet* GetAllRestrictedEditorsForKind(XMPDraft* draft, XMPType kind);
      // returns subset of restricted editors of draft which support kind.
      
   //---------------------------------
   // Category and kind mappings

   XMPMethod XMPTypeSet* GetCategoriesOfKind(XMPType kind);
      // Returns the categories of the argument data kind, in the content of the
      // argument storage unit.  The client must dispose of the result.


   XMPMethod XMPNameSpace* ViewerNameSpace();

   XMPHeap           GetHeap();

   XMPNVMethod void        LocateASLMLibFiles(); // TEMPORARY!!!!
   XMPNVMethod void        CreateNameSpaces();
   XMPNVMethod void        LoadNameSpaceTable();
   XMPNVMethod void        ReadNMAPResources(XMPFileSpec *theFileSpec);
   XMPNVMethod void        LoadNMAPResource(XMPPtr mappingPair, XMPSize bufferSize);
   XMPNVMethod void        GetOpenDocPrefs();
   XMPNVMethod void        LocatePrefsFile();
   XMPNVMethod void        LoadPreferences();
   
// Fields:

   XMPBaseSession*   fSysSession;
   XMPHeap           fHeap;

   XMPFile*       fPrefsFile;
   XMPULong       fPrefsModDate;
      // for ASLM builds
   TMacFileSpec*     fRegLibFileFolder;

   
};

#endif // _BINDING_
