/********************************************************************/ 
/*  Licensed Materials - Property of IBM                            */ 
/*                                                                  */ 
/*                                                                  */ 
/* Copyright (C) International Business Machines Corp., 1994.       */ 
/* Copyright (C) Apple Computer, Inc., 1994                         */ 
/*                                                                  */ 
/*  US Government Users Restricted Rights -                         */ 
/*  Use, duplication, or disclosure restricted                      */ 
/*  by GSA ADP Schedule Contract with IBM Corp.                     */ 
/*                                                                  */ 
/********************************************************************/ 
/*
   File:    Arbitrat.h

   Contains:   XMPArbitrator interface

   Written by: Richard Rodseth

*/

#ifndef _ARBITRAT_
#define _ARBITRAT_

#ifndef _XMPOBJ_
#include "XMPObj.h"
#endif

#ifndef _XMPTYPES_
#include "XMPTypes.h"
#endif

//==============================================================================
// Theory of Operation
//==============================================================================

/*
   The arbitrator is used to manage competition for facilities which can only
   be controlled by a single part at a time. These facilities are referred to
   as "focus types". An example is the keystroke focus. Keyboard events are
   sent to the part which has the keystroke focus. Another example is the menu
   focus. One part owns the menu bar at any time. Many focus types are related
   to event distribution, but focus types could also be associated with ports,
   for example.
   
   Rather than having an all-encompassing notion of "active" part, parts simply
   request ownership of a set of focus types. It is possible, in principle, to
   have one part receive keystroke events while another receives sound input,
   though the human interface guidelines may encourage parts to request and 
   relinquish most focus types along with the menu focus.
      
   New focus types can be added. This might be done by a part handler which is
   associated with some exotic input device.
   
   Parts request ownership of a set of focus types. This is necessary to 
   avoid deadlock.
      
*/

//==============================================================================
// Constants
//==============================================================================


//==============================================================================
// Scalar Types
//==============================================================================

//=====================================================================================
// Class defined in this interface
//=====================================================================================

class XMPArbitrator;

//=====================================================================================
// Classes used by this interface
//=====================================================================================

class XMPFocusModule;
class XMPFocusSet;
class XMPFocusOwnerIterator;
class XMPSession;
class XMPFrame;
class DictionaryList;

//=====================================================================================
// Global Variables
//=====================================================================================

//=====================================================================================
// Class XMPArbitrator
//=====================================================================================

#define kXMPArbitratorID "appl:xmparbitrator$class,1.0.0"

 class XMPArbitrator : public XMPObject
{
public:

   XMPArbitrator();
         
   virtual ~XMPArbitrator();
   
   XMPNVMethod void InitArbitrator(XMPSession* session);

   XMPMethod   XMPSize  Purge(XMPSize size); 
      
   XMPMethod XMPBoolean RequestFocusSet(XMPFocusSet* focusSet, XMPFrame* requestingFrame);
      
   XMPMethod XMPBoolean RequestFocus(XMPTypeToken focus, XMPFrame* requestingFrame);
            
   XMPMethod void RelinquishFocusSet(XMPFocusSet* focusSet, XMPFrame* relinquishingFrame);
            
   XMPMethod void RelinquishFocus(XMPTypeToken focus, XMPFrame* relinquishingFrame);
   
   XMPMethod void TransferFocus(XMPTypeToken focus, 
                         XMPFrame* transferringFrame,
                         XMPFrame* newOwner);
                         
   XMPMethod void TransferFocusSet(XMPFocusSet* focusSet, 
                         XMPFrame* transferringFrame,
                         XMPFrame* newOwner);
                         
   XMPMethod XMPFrame* GetFocusOwner(XMPTypeToken focus);
      
   XMPMethod XMPFocusOwnerIterator* CreateOwnerIterator(XMPTypeToken focus);
               
   XMPMethod void RegisterFocus(XMPTypeToken focus, XMPFocusModule* focusModule);
         
   XMPMethod void UnregisterFocus(XMPTypeToken focus);

   XMPMethod XMPBoolean IsFocusRegistered(XMPTypeToken focus);
   
   XMPMethod XMPBoolean IsFocusExclusive(XMPTypeToken focus);
   
protected:

   XMPMethod void SetFocusOwnership(XMPTypeToken focus, XMPFrame* frame);
   
   XMPMethod XMPFocusModule* GetFocusModule(XMPTypeToken focus);
         
   XMPMethod XMPBoolean BeginRelinquishFocus(
                        XMPTypeToken focus, 
                        XMPFrame* requestingFrame);
                                                
   XMPMethod void CommitRelinquishFocus(XMPTypeToken focus, 
                      XMPFrame* requestingFrame);
                                           
   XMPMethod void AbortRelinquishFocus(XMPTypeToken focus, 
                      XMPFrame* requestingFrame);
                      
protected:

   XMPSession* fSession;   // Access to globals
         
private:

   DictionaryList* fFocusModules;   

};



#endif // _ARBITRAT_
