/*************************************************************************
 * Source Id :
 *
 * $Id: dberr.cc,v 1.18 1994/02/03 05:54:19 kevinl Exp $
 *-------------------------------------------------------------------------
 * Project Notes :
 *
 *  Diamond Base
 *  ============
 *      A solid database implementation, spurred on by the continuing
 *  Metal (Lead) Base saga.
 *
 *  Project Team :
 *        A. Davison
 *        K. Lentin
 *        D. Platt
 *
 *	Project Commenced : 05-02-1993
 *
 *-------------------------------------------------------------------------
 *  Module Notes :
 *
 *	dbErr : A global error handler.
 *
 *  Original Author : Daz / Andy
 *
 **************************************************************************/

#pragma implementation

#include <iostream.h>
#include <stdio.h>
#include <string.h>

#include <diamond/dbconf.h>

#if HAVE_MALLOC_H && !MALLOC_H_INCLUDED
extern "C" {
#include <malloc.h>
}
#define MALLOC_H_INCLUDED 1
#endif

#include <diamond/dbconf.h>
#include <diamond/dberr.h>

// This is the status returned by the last dbErr call
// extraErr can be optionally added and is cleared by subsequent calls

dbError dbErrorStatus = db_ok;
dbString extraErr;

//----------------------------------------
// Register an error by remembering the
// error code. Then return the error
// code.

dbError
dbErr(const dbError err, const dbString& extra)
{
	if (extra.empty())
		extraErr.dispose();
	else
		extraErr = extra;
	dbErrorStatus = err;
	return err;
}

// Display a particular error message. temp is used to reinstate the
// the last error code.

void
dbErr(const dbString& msg, const dbError err)
{
	dbError temp = dbErrorStatus;
	dbErrorStatus = err;
	dbErr(msg);
	dbErrorStatus = temp;
}

//--------------------------------------------
// Display the current error status, preceeded
// by an optional error message.

void
dbErr(const dbString& msg)
{
	if (msg.empty())
		cerr << "Error : " << endl;
	else
		cerr << msg << " : " ;

	if (extraErr.len())
		cerr << extraErr << " : ";

	cerr << dbErrStr(dbErrorStatus) << endl;
}

// Return a string representation of the last error

dbString
dbErrStr(void)
{
	dbString s;
	if (extraErr.len())
		s += extraErr + " : ";

	return s + dbErrStr(dbErrorStatus);
}
	
// Return the string representation of err

dbString
const
dbErrStr(const dbError err)
{
	dbString errStr;

	switch (err)
	{
		case db_ok :
			errStr = "No error.";
			break;
		case db_alreadyregistered:
			errStr = "Relation already registered.";
			break;
		case db_unimp:
			errStr = "Not yet implemented.";
			break;
		case db_nfound:
			errStr = "Not Found.";
			break;
		case db_range:
			errStr = "Range Error.";
			break;
		case db_toomany:
			errStr = "Too many.";
			break;
		case db_noquery:
			errStr = "No (such) query currently in progress.";
			break;
		case db_querylocked:
			errStr = "Query has a lock placed on it.";
			break;
		case db_reclocked:
			errStr = "Record is locked.";
			break;
		case db_dup:
			errStr = "Duplicate data found in database.";
			break;
		case db_badName:
			errStr = "Bad Name. - who knows?";
			break;
		case db_notopen:
			errStr = "Data file not open.";
			break;
		case db_err:
			errStr = "An unnamed error occured.";
			break;
		case db_alreadyopen:
			errStr = "Data file is already open.";
			break;
		case db_locked:
			errStr = "Data base is locked.";
			break;
		case db_unlocked:
			errStr = "Database unlocked.";
			break;
		case db_nobuckets:
			errStr = "Cannot create a bucket.";
			break;
		case db_eof:
			errStr = "End of file reached.";
			break;
		case db_nomem:
			errStr = "Out of memory. MV";
			break;
		case db_comm:
			errStr = "Comm. error. MV";
			break;
		case db_refid:
			errStr = "Bad reference id. MV";
			break;
		case db_protocol:
			errStr = "Old protocol. MV";
			break;
		default:
			errStr.setSize(80);
			sprintf((char*)errStr,"Danger, Will Robinson ! Unspecified error %d approaching !", err);
			break;
	}
	return errStr;
}

//////////////////////////////////////////////////////////////////////////
// Revision History:
//
// $Log: dberr.cc,v $
// Revision 1.18  1994/02/03  05:54:19  kevinl
// Moved include files to diamond/ and fixed HAVE_MALLOC_H
//
// Revision 1.17  1993/12/23  22:58:48  kevinl
// dbErr uses dbString, autoconf stuff, moved logs, added const and size_t
//
// Revision 1.16  1993/11/18  02:54:47  kevinl
// Fixed passing/returning of most complex types
//
// Revision 1.15  1993/11/09  04:05:44  kevinl
// Added dbErrStr(void)
// Replaced all char* with dbString
// Moved logs
//
// Revision 1.14  1993/11/05  13:46:57  kevinl
// Protocol and fixes
//
// Revision 1.13  1993/11/03  13:48:27  kevinl
// Fixed something little
//
// Revision 1.12  1993/11/03  10:45:26  kevinl
// Needs string.h
//
// Revision 1.11  1993/08/05  11:52:08  darrenp
// added db_refid error for MV.
//
// Revision 1.10  1993/07/05  05:36:19  darrenp
// Added db_nomem and db_comms for the multi version.
//
// Revision 1.9  1993/06/23  05:21:22  kevinl
// Mallocs are now in angular brackets
//
// Revision 1.8  1993/05/26  01:01:39  kevinl
// MALLOC_H_MISSING
//
// Revision 1.7  1993/05/06  04:00:19  kevinl
// SASC define for malloc.h
//
// Revision 1.6  1993/05/03  01:34:07  kevinl
// 2 cases left out of switch
//
// Revision 1.5  1993/04/25  11:21:28  kevinl
// Comments
//
// Revision 1.4  1993/03/30  14:37:48  kevinl
// Modified error handling again. now we can get a string back (if we want)
//
// Revision 1.3  1993/03/29  23:26:47  kevinl
// Support for diaRel's error handling
//
// Revision 1.2  1993/03/29  08:21:19  darrenp
// Added malloc library support
//
// Revision 1.1  1993/03/28  10:39:20  kevinl
// Initial revision
//
//////////////////////////////////////////////////////////////////////////
