/************************************************************************/
/*				SAMPLE3.C							05-01-94			*/
/*													ver 1.00			*/
/* The following sample cource code for C compilers for DOS shows the	*/
/* use of the trigger buffer to store and autonomously start motion 	*/
/* trajectories to absolute target postions on a single axis. This 		*/
/* program will load four trajectories into the trigger buffer.			*/
/* These trajectories will execute in series at each trigger_io			*/
/* command or valid, enabled trigger io input signal. This program 		*/
/* simulates the input signal by using the trigger io command. Once a 	*/
/* move is complete, the program waits for a carriage		 			*/
/* return from the user to fire the next trigger buffer sequence, which */
/* will automatically send the commands to move the motor to the next 	*/
/* position stored in the trigger buffer.		 						*/
/************************************************************************/

#include <stdio.h>
#include <conio.h>
#include <math.h> 		/*additional header file needed for math functions */
#include <nulib02.h>

#define PROF_COMPL		0x400			/*profile complete bit in 'per Axis' hardware status */
#define MOTOR_OFF		0x04			/*motor off bit */
#define MOTOR_RUNNING	0x80           	/*motor running bit */

int WaitForMoveComplete(WORD BoardAddr, WORD BoardType, BYTE Axis, WORD TimeOutSecs);

int LoadTrajects(WORD BoardAddr, WORD BoardType, BYTE Axis, WORD StepsPerRev,
								double *VelArray, long int *PosArray, WORD NumPoints);

void main(void) {
  int i, ch, retstat;
  WORD BoardAddr, BoardType;
  WORD StepsPerRev;
  WORD TimeOutSecs;
  double Vel, Accel;
  long int Position;
  BYTE Axis;
  long int PosArray[4];
  double VelArray[4];
  WORD NumPoints;

/* * * * * * * * * * * * * *  SETUP SPECIFICS * * * * * * * * * * * * * * * * * * */
	/*Set up parameters for this board and motors*/
  	BoardAddr = 0x398;		/*Board address is set with DIP switches on the board */
  	BoardType = 3;			/*Board Type 3 means an 4 Axis Servo Motor Board */
  	StepsPerRev = 1000;     /*Steps Per Revolution(step)/ Encoder Counts(servo) on your motor */

  	/*consider this to be your target position, vector velocity, and vector acceleration */
  	Axis = 1;				/*Controlling Axis 1	*/
  	Vel = 500; /*500 RPM for vector, the X and Y components will be less */
  	Accel = 200; /*200 RPS/s for vector, the X and Y components will be less */

    /*assign some points that you would like to reach by using the trigger buffer */
    NumPoints = 4;
    PosArray[0] = 10000; VelArray[0] = 120; /*go to 10000 counts at a velocity of 120 rpm */
    PosArray[1] = 13000; VelArray[1] = 240; /*go to 13000 counts at a velocity of 240 rpm */
    PosArray[2] = 17777; VelArray[2] = 88;  /*go to 17777 counts at a velocity of 88  rpm */
    PosArray[3] = -9999; VelArray[3] = 280; /*go to -9999 counts at a velocity of 280 rpm */

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * **/

	/*First, you want to read your present position*/
  	retstat = read_pos(BoardAddr, BoardType, Axis, &Position);
  	if (retstat) printf("Read Pos Error!");
	printf("Position is now = %d\n",Position);

    retstat = LoadTrajects(BoardAddr, BoardType, Axis, StepsPerRev, VelArray, PosArray, NumPoints);
	if (retstat) printf("LoadTrajects Error!");

	for(i=0;i<NumPoints;i++) {

        printf("Press <Enter> to trigger the next sequence...");
        ch = getchar();

		/*This will start the next move by playing everything in the trigger buffer,*/
		/* up until the trigger buffer delimiter.									*/
		retstat = trigger_io(BoardAddr, BoardType, 1);
		if (retstat) printf("Trigger_IO Error!");

 		/*							Wait for move complete.  								*/
    	TimeOutSecs = 30; /*wait 30 seconds before giving up and returning a timeout error	*/
    	retstat = WaitForMoveComplete(BoardAddr, BoardType, Axis, TimeOutSecs);
 		if (retstat) printf("Timeout Error!");

		/*Last, you want to read your new position*/
  		retstat = read_pos(BoardAddr, BoardType, Axis, &Position);
  		if (retstat) printf("Read Pos Error!");
		printf("Position is now = %d\n",Position);

	}

	printf("Moves are done!\n"); /*all finished*/

}

/************************************************************************************/
/* Load Trajectories - This function takes <NumPoints> points for  one axis, and	*/
/* sets up the trigger buffer to issue trajectory commands to get to those points. 	*/
/* Here we load trigger buffer one with the load_rpm, load_target_pos, and 			*/
/* start_motion commands for each target, and then we just need to send one command */
/* to begin the new trajectory, which is done from the main routine.				*/
/************************************************************************************/
/* BoardAddr - The ISA bus board address											*/
/* BoardType - The nuLogic board type (1-stepper ol, 2-stepper cl, 3-servo4a, etc.)	*/
/* Axis - The axis on which all the motion will occur.								*/
/* StepsPerRev - The Steps Per Revolution for that motor.							*/
/* VelArray - Pointer to an array of velocities at which to approach their respec-	*/
/* 		tive targets.																*/
/* PosArray - Pointer to an array of target positions to reach upon each firing of	*/
/*  	the trigger buffer.															*/
/* NumPoints - The number of target points that are set up in the arrays.			*/
/* Returns 0 status if everything loaded okay, and an error status if one of the	*/
/*  	commands failed.															*/
/************************************************************************************/

int LoadTrajects(WORD BoardAddr, WORD BoardType, BYTE Axis, WORD StepsPerRev,
								double *VelArray, long int *PosArray, WORD NumPoints)
{
  WORD i;
  int retstat;
  int k;

  for (i=0; i<NumPoints; i++) {

  	if (i==0) { 		/*begin recording the trigger buffer contents */
  		retstat = begin_prestore(BoardAddr, BoardType, 1); /*begin prestore on buffer 1*/
		if (retstat) return(retstat);
	}

  						/* load a single trajectory sequence here */
  	retstat = load_rpm(BoardAddr, BoardType, Axis, StepsPerRev, *(VelArray+i));
	if (retstat) break;
  	retstat = load_target_pos(BoardAddr, BoardType, Axis, *(PosArray+i));
	if (retstat) break;
  	retstat = start_motion(BoardAddr, BoardType, Axis);
	if (retstat) break;

	if (i!=NumPoints-1) {
    	retstat = trig_buff_delim(BoardAddr, BoardType);
		if (retstat) break;
	}
  }
  					/* Always end the buffer recording session with end_prestore.*/
  k = end_prestore(BoardAddr, BoardType);
  if (k) return(k);
  if (retstat) return(retstat); /*any previous errors will also return error status */

  return(0);
}



/************************************************************************************/
/* Wait For Move Complete -- This function waits for an Axis to complete its move	*/
/*  and returns a 0 status. If the number of timeout seconds specified is greater 	*/
/*  than 10000, this routine will return an error (-1) . 							*/
/* If you give a timeout of 0, the routine will only return when the move completes.*/
/* In other words a timeout value of 0 is effectively an infinite timeout.			*/
/* If the timeout actually happens, we return a (-2);								*/
/* Other error numbers can be returned from the library call, see your library doc- */
/* ument for more detail.															*/
/************************************************************************************/
/* BoardAddr - The ISA bus board address											*/
/* BoardType - The nuLogic board type (1-stepper ol, 2-stepper cl, 3-servo4a, etc.	*/
/* TimeOutSecs - Number of seconds before routine quits looking for move complete	*/
/* Returns a 0 if successful, -1 if timeout parm too large, -2 if timeout occurred	*/
/************************************************************************************/
int WaitForMoveComplete(WORD BoardAddr, WORD BoardType, BYTE Axis, WORD TimeOutSecs)
{
  int retstat;
  WORD AxisStat;
  DWORD TimeOutCycles, i;

  	if (TimeOutSecs>10000) return(-1); /*too big a number */

  	TimeOutCycles = 240 * TimeOutSecs; /*this is a rough estimate of cycles per second*/
  	/*approx 240 read_axis_stat's per sec, can vary by how many axes are moving, etc. */
    /*since this is just an error condition limit, it doesn't need to be very accurate*/

    AxisStat = 0; /*if timeoutsecs = 0, then timeout is infinite */
    for (i=0;((i<TimeOutCycles)||(TimeOutSecs==0));i++)  {

    	retstat = read_axis_stat(BoardAddr, BoardType, Axis, &AxisStat);
    	if (retstat) return(retstat); /*error checking */

		/* check if motor is not running ,and either profile is complete or motor is off*/
    	if ( (AxisStat&(PROF_COMPL|MOTOR_OFF)) && (!(AxisStat&MOTOR_RUNNING)) )
    	  return(0);
    }

    return(-2); /*timeout*/
}
