#include <windows.h>
#include <conio.h>
#include "comm.h"

/******************************************************************************/
/*							SEND COMMAND		   			   			      */
/******************************************************************************/
int send_command(WORD boardaddr, BYTE axis, BYTE cmd, BYTE wcount,
						    DWORD cmddata)
{
int retstat;
WORD packet[4];

/*build the command packet and send it to the board*/
if (wcount == 2) {
	packet[0] = (WORD) (((axis | 0x20) << 8) | cmd);  /*command word*/
	packet[1] = TERMINATOR;
	retstat = sendpacket(boardaddr,packet,2);
	}
else if (wcount == 3) {
	packet[0] = (WORD) (((axis | 0x30) << 8) | cmd);  /*command word*/
	packet[1] = (WORD) cmddata;
	packet[2] = TERMINATOR;
	retstat = sendpacket(boardaddr,packet,3);
	}
else {
	packet[0] = (WORD) (((axis | 0x40) << 8) | cmd);  /*command word*/
	packet[1] = (WORD) ((cmddata >> 16) & 0x0000FFFF); /* data high word*/
	packet[2] = (WORD) cmddata;		/*data low word*/
	packet[3] = TERMINATOR;
	retstat = sendpacket(boardaddr,packet,4);
    }

return(retstat);
}

/******************************************************************************/
/*						READ THE RETURN DATA BUFFER 	   			   		  */
/******************************************************************************/
/*Generic command to read the return data buffer.						      */
/*Intended to be called by the application at some asynchronous time after    */
/*a 'READ' command has been issued which in turn has left data in the         */
/*return data buffer.                                                         */
/*Using pointers to axis, command, and rdb, three arrays will be filled with  */
/*sequentially corresponding data for the number of packets specified.		  */
/*If fewer than the number of packets requested can be read, then the number  */
/*of packets read will be indicated in the location pointed to by numptr. */
/******************************************************************************/
int read_rdb(WORD boardaddr, BYTE FAR *numptr, BYTE FAR *axisptr,
						  BYTE FAR *cmdptr, DWORD FAR *rdbptr)
{
WORD packetID,dataloword,datahiword,trash;
BYTE wordcount,axisin,cmdin;
int retstat,i;
BYTE j;
DWORD datahidword;

/*read status port and check for an existing board level error*/
retstat = checkpreviouserr(boardaddr);
if (retstat) return(retstat);

/*check for data pending*/
if (!datapending(boardaddr)) return(3);  /*status 3: no data pending*/

for (j = 0; j < *numptr; j++) {
	/*read the first word: the packet identifier*/
	packetID = byteswap(inpw(boardaddr));

	/*check the word count for validity*/
	wordcount = (BYTE) (packetID >> 12);  /*isolate the data word count*/
	if ((wordcount != 2) && (wordcount != 3)) {
		/*try to flush data buffer if packetID is bad*/
		for (i = 0; i < 10000; i++) {
			if (!datapending(boardaddr)) break;
			trash = inpw(boardaddr);
    		}
		if (i == 10000) return(15); /*status 15: unable to flush RDB*/
		else return(11);     /*status 11: corrupted return packet ID*/
		}
	/*set the return axis and command*/
	*(axisptr + j) = (BYTE) ((packetID & 0x0F00) >> 8);
	*(cmdptr + j) = (BYTE) (packetID & 0x00ff);

	/*read the second word*/
	if (!datapending(boardaddr)) return(4); /*status 4: incomplete packet*/
	dataloword = byteswap(inpw(boardaddr));

	/*read the third word, if needed, and build the return data double word*/
	if (wordcount == 3) {
		if (!datapending(boardaddr)) return(4); /*status 4: incomplete packet*/
    	datahiword = byteswap(inpw(boardaddr));
    	datahidword = (DWORD) datahiword;
    	*(rdbptr + j) = (DWORD) ((datahidword << 16) | dataloword);
    	}
	else *(rdbptr + j) = (DWORD) dataloword;   /*no high word if only one data*/
	/*check to see if more data is available before continuing*/
	if (!datapending(boardaddr)) {
		*numptr = (BYTE) (j + 1);
		return(0);
		}
} /*end for (numpackets)*/

return(0);      /*status 0: successful reading of data buffer*/
}

/******************************************************************************/
/*						READ THE COMMUNICATIONS STATUS REGISTER	   			  */
/******************************************************************************/
/* generic command to read the communications status register			      */
/******************************************************************************/
int read_csr(WORD boardaddr,BYTE FAR *csrptr)
{
int retstat;
DWORD boardconf;

/*get the CSR from I/O port*/
*csrptr = (BYTE) inpw(boardaddr + 6);

return(0);
}

/******************************************************************************/
/*							SENDPACKET					   			 	      */
/******************************************************************************/
/*For all routines that do not require an immediate read of the data 	      */
/*buffer, this routine is used to send a command packet to the board.	      */
/*A packet is sent followed by a short wait for command processing.		      */
/*Command error checking is also provided. 								      */
/*A zero status is returned if no errors have occurred.                       */
/******************************************************************************/
int sendpacket(WORD boardaddr, WORD FAR *packetptr, BYTE packetsize)
{
BYTE i;
WORD swappedword;
int retstat;

/*read status port and check for an existing board level error*/
retstat = checkpreviouserr(boardaddr);
if (retstat) return(retstat);

/*send the packet: check RTR and CE before each word is sent*/
for (i = 0; i < packetsize; i++) {
	retstat = pollrtrandce(boardaddr);    /*poll RTR and check cmd err*/
	if (retstat) return(retstat);
    swappedword = byteswap(*(packetptr + i));    /*all words are byte swapped*/
    outpw(boardaddr,swappedword);
	}
retstat = chkcmdprocess(boardaddr);
if (retstat) return(retstat);
else return(0);    /*status 0: successful transmition of packet*/
}

/******************************************************************************/
/*							POLLRTRANDCE					   			   	  */
/******************************************************************************/
/*this routine is called from sendpacket()									  */
/******************************************************************************/
int pollrtrandce(WORD boardaddr)
{
long i;
WORD csr;
int retstat;

for (i = 0; i < 40000; i++) {
	csr = inpw(boardaddr + 6);

	/*check command error bit*/
	if (csr & 0x0004) {
		retstat = clearcmderr(boardaddr);
		if (retstat) return (retstat);
		else return(2); }	/*status 2: command error on current packet*/

	/*check ready receive*/
	if (csr & 0x0001) {
		waitforpulse(boardaddr);  /*small delay to allow bit setting pulse to complete*/
        return(0);      /*0: no command error, ready to receive*/
        }
	} /*end for*/

return(1);  /*status 1: RTR timeout*/
}

/******************************************************************************/
/*							READYTORECEIVE					   			      */
/******************************************************************************/
/*return 1 if board is ready to receive, else return 0					      */
/******************************************************************************/
int readytoreceive(WORD boardaddr)
{
long i;

for (i = 0; i < 40000; i++) {
	if (inpw(boardaddr + 6) & 0x0001) {  /*bit 0 set means ready to receive*/
		waitforpulse(boardaddr);  /*small delay to allow bit setting pulse to complete*/
        return(1);
        }
	}

return(0);
}

/******************************************************************************/
/*							DATAPENDING						   			      */
/******************************************************************************/
/*return 1 if data is pending, 0 otherwise					 			      */
/*called prior to any read of the data buffer							      */
/*1/18: waits up to 200msec													  */
/******************************************************************************/
int datapending(WORD boardaddr)
{
long i;

for (i = 0; i < 40000; i++) {
	if (inpw(boardaddr + 6) & 0x0002) {   /*bit 1 on means data pending*/
		waitforpulse(boardaddr);  /*small delay to allow bit setting pulse to complete*/
        return(1);
        }
	}

return(0);
}

/******************************************************************************/
/*							CHKCMDPROCESS						   			  */
/******************************************************************************/
/*checks ready to receive to indicate that last word has been taken by board  */
/*then polls command error for up to 200 usec (486/66)						  */
/******************************************************************************/
int chkcmdprocess(WORD boardaddr)
{
long i;
int retstat;

retstat = pollrtrandce(boardaddr);
if (retstat) return(retstat);

/*check for command error*/
/*1/18 loop takes about 400 usec on 486/33*/
/*normally command error is set in about 150 usec*/
for (i = 0; i < 80; i++) {
	if (inpw(boardaddr + 6) & 0x0004) {
		retstat = clearcmderr(boardaddr);
		if (retstat) return (retstat);
		else return(2); }	/*status 2: command error on current packet*/
     }

return(0);   /*0: no errors*/
}

/******************************************************************************/
/*							CHECKPREVIOUSERR					   		      */
/******************************************************************************/
/*This routine is called before sending a command packet, or before 	      */
/*processing any application issued call. This error checking is provided     */
/*just in case a board level error has occurred since the last completion     */
/*of a command.                                                               */
/*The command error and system failure bits are checked, and the 		      */
/*appropriate status is returned.										      */
/******************************************************************************/
int checkpreviouserr(WORD boardaddr)
{
int retstat;

/*read status port and check for an existing board level error*/
retstat = checkboarderr(boardaddr);
switch (retstat) {
	case 0:
		return(0);
	case 2:
		retstat = clearcmderr(boardaddr); /*try to clear error if needed*/
		if (retstat) return(retstat); /*if unable to clear, return*/
		else return(8); /*status 8: command error occurred in between packets*/
	case 5:
		return(5);  /*system failure bit was set*/
	}
}

/******************************************************************************/
/*							CHECKBOARDERR					   			      */
/******************************************************************************/
/*This routine is called from checkcurrenterr() and checkpreviouserr()	      */
/*The command error and system failure bits are checked, and the 		      */
/*appropriate status is returned.										      */
/******************************************************************************/
int checkboarderr(WORD boardaddr)
{
WORD csr;

/*get contents of communications status register*/
csr = inpw(boardaddr + 6);
/*check system failure bit, return status 5 if set*/
if (csr & 0x0010) return(5);
/*check command error bit, return status 2 if set*/
if (csr & 0x0004) return(2);
/*return 0 if all is OK*/
return(0);
}

/******************************************************************************/
/*							CLEARCMDERR					   				      */
/******************************************************************************/
int clearcmderr(WORD boardaddr)
{
WORD swappedterm,csr;
int gotrtr;
long i,j;

swappedterm = 0x0A00;  /*byte swapped terminator word*/

/*since the maximum packet size is 4 words*/
/*sending 5 terminators should be more than sufficient*/
for(i = 0; i < 5; i++) {
	if (!readytoreceive(boardaddr)) return (9); /*status 9: unable to clear command error*/
	outpw(boardaddr,swappedterm);     /*send terminator word*/

	/*check RTR -- to make sure terminator was taken*/
	gotrtr = 0;
	for (j = 0; j < 40000; j++) {
		if (inpw(boardaddr + 6) & 0x0001) {
			gotrtr = 1;
			break;
        	}
		} /*end for*/
	if (!gotrtr) return(9);   /*status 9: unable to clear command error*/

    /*check command error*/
    csr = inpw(boardaddr + 6);		  /*get contents of csr*/
	if (!(csr & 0x0004)) return(0);   /*status 0: successfull*/
    }

return(9);   /*unable to clear command error bit*/
}

/******************************************************************************/
/*							WAITFORPULSE					   			      */
/******************************************************************************/
/*this is a very small wait used to allow the hardware to settle after        */
/*bits have been set in the CSR, this avoids bits getting stuck			      */
/******************************************************************************/
void waitforpulse(WORD boardaddr)
{
int i;
WORD csr;
/*1/18 loop takes about 25 usec on 486/33*/
/*which is about twice what we may actually need*/


for (i = 0;i < 3; i++) {
	csr = inpw(boardaddr + 6); }
}

/******************************************************************************/
/*							BYTESWAP						   			      */
/******************************************************************************/
WORD byteswap(WORD wordtoswap)
{
WORD swappedword;

/*all words sent to the board need to be byte swapped*/
swappedword = (wordtoswap << 8 | wordtoswap >> 8);

return(swappedword);
}

